// -*- mode: C++ -*-
//
// PDF-Cube source file - pdfcube.cc
// 
// Copyright (C) 2006 Mirko Maischberger <mirko.maischberger@gmail.com>
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WAPRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

#include <iostream>

#include <cstdio>
#include <cstdlib>
#include <cmath>
#include <cassert>

// Gtk+ (pkg-config gtk+-2.0)
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

// GtkGLExt (pkg-config gtkglext-1.0)
#include <gtk/gtkgl.h>

// OpenGL (-lglut)
#include <GL/gl.h>
#include <GL/glu.h>

// PDF to GdkPixbuf (pkg-config poppler-glib)
#include <poppler.h>

using namespace std;

//////////////////////////////////////////////////////////////////////////
// Macros

#define DEFAULT_WIDTH 640
#define DEFAULT_HEIGHT 480
#define DEFAULT_TITLE  "PDF-Cube"

#define TIMEOUT_INTERVAL 38

//////////////////////////////////////////////////////////////////////////
// Globals (will be moved inside classes some day)

static gboolean fullscreen = TRUE;
static gboolean animating = FALSE;

enum animation { ANIM_NONE, 
                 CUBE, 
                 ZOOM0, ZOOM1, ZOOM2, ZOOM3, ZOOM4, ZOOMC, 
                 SWITCH_FW, SWITCH_BW 
};

animation active_animation = ANIM_NONE;
animation previous_animation = ANIM_NONE;
animation last_animation = ANIM_NONE;

// Cube Transitions on the command line 
// (space advances simply or with the rotating cube)
// depending on the values in this array
static bool *page_transition;

//////////////////////////////////////////////////////////////////////////
// Forward declarations

static void         timeout_add       (GtkWidget   *widget);
static void         timeout_remove    (GtkWidget   *widget);

static void         start_animation  (GtkWidget   *widget, animation);
static void         stop_animation  (GtkWidget   *widget);

static GdkGLConfig *configure_gl      (void);

static GtkWidget   *create_window     (GdkGLConfig *glconfig);

static bool sleeping() { return !animating && active_animation == ANIM_NONE; }

//////////////////////////////////////////////////////////////////////////
//
// pdfcube
//
// This is an attemp to move some dirtyness inside a class
// some work still needs to be done to transform this quick 
// hack in a serious pdf-viewer
//
class pdfcube 
{
public:
  pdfcube(PopplerDocument *d)
    : doc(d),
      current_page(0),
      current_face(0),
      total_pages(poppler_document_get_n_pages(d)),
      frame(0), 
      lookposx(0.0), lookposy(0.0), lookposz(3.48), 
      atx(0.0), aty(0.0), atz(0.0), 
      persp(44.0), angle(0.0),
      pixmap(0)
  {
    texmap[0] = 0; texmap[1] = 1; texmap[2] = 2;
    pixmap = 
      gdk_pixbuf_new(GDK_COLORSPACE_RGB, true, 8, tex_width, tex_height);
  }

  int page() { return current_page; }

  int pages() { return total_pages; }

  void restart(GtkWidget* widget) 
  { current_page = 0; update_textures(widget); }

  void go_to(GtkWidget* widget, int page) 
  { 
    if(page>=0 && page < total_pages)
      {
	current_page = page; 
	update_textures(widget); 
      }
  }

  void section(GtkWidget* widget, int section)
  {
    cerr << "Section: " << section << " total pages: " << total_pages << endl;
    int ii;
    for(ii = 0; ii < total_pages; ++ii)
      {
       if(page_transition[ii]) section--;
       if(section == 0) break;
      }
    cerr << "Page: " << ii << endl;
    if(ii < total_pages)
      {
       current_page = ii;
       update_textures(widget);
      }
  }

  void 
  initialize(GtkWidget *widget)
  {
    glClearColor (0.4, 0.0, 0.0, 0.0);
    
    // glEnable(GL_NORMALIZE);
    glShadeModel(GL_FLAT);
    glEnable(GL_TEXTURE_RECTANGLE_ARB);

    // glEnable(GL_DEPTH_TEST);
    // or
    glEnable(GL_CULL_FACE);
    glCullFace(GL_FRONT);

    glPixelStorei(GL_UNPACK_ALIGNMENT, 1);
    
    glGenTextures (3, textures);

    glTexEnvi (GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_DECAL);
    glTexParameteri( GL_TEXTURE_RECTANGLE_ARB, GL_TEXTURE_WRAP_S,
                    GL_CLAMP_TO_EDGE );
    glTexParameteri( GL_TEXTURE_RECTANGLE_ARB, GL_TEXTURE_WRAP_T,
                    GL_CLAMP_TO_EDGE );

    glTexParameteri (GL_TEXTURE_RECTANGLE_ARB, GL_TEXTURE_MIN_FILTER, 
		     GL_LINEAR_MIPMAP_LINEAR);
    glTexParameteri (GL_TEXTURE_RECTANGLE_ARB, GL_TEXTURE_MAG_FILTER, 
		     GL_LINEAR);
    
    update_textures(widget);
    
    GLint size;
    glGetIntegerv(GL_MAX_TEXTURE_SIZE, &size);
    printf("%u\n", size);
    assert(size >= 512);
    assert(glIsTexture(textures[0]));
    
    // Cube vertex
    v[0][0] = v[1][0] = v[2][0] = v[3][0] = -1;
    v[4][0] = v[5][0] = v[6][0] = v[7][0] = 1;
    v[0][1] = v[1][1] = v[4][1] = v[5][1] = -1;
    v[2][1] = v[3][1] = v[6][1] = v[7][1] = 1;
    v[0][2] = v[3][2] = v[4][2] = v[7][2] = 1;
    v[1][2] = v[2][2] = v[5][2] = v[6][2] = -1;
    
    
    glMatrixMode(GL_PROJECTION);
    gluPerspective( persp,
		    1.0,
		    0.5, 
		    10.0);
    
    glMatrixMode(GL_MODELVIEW);

    gluLookAt(lookposx, lookposy, lookposz,
	      atx, aty, atz,
	      0.0, 1.0, 0.0);      // up is in positive Y direction 
    
  }

  
  
  void 
  redraw(GtkWidget *widget)
  {
    int frames = 17;
    double yoffset = 0.1;
    if(animating)
      {
	switch(active_animation)
	  {
	  case ANIM_NONE:
	    cerr << "No animation... stopping right now." << endl;
	    frame = 0;
	    stop_animation(widget);
	    break;
	  case CUBE:
	    cerr << "cube " << frame << endl;
	    if(frame == 17)
	      {
		frame = 0;
		stop_animation(widget);
	      } 
	    else 
	      {
		glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
		glMatrixMode(GL_MODELVIEW);
		glLoadIdentity();
		lookposz -= zsteps[frame]*4;
		lookposy = 6*xsteps[frame];
		gluLookAt(lookposx, lookposy, lookposz, atx,aty,atz, 0,1,0);
		angle -= steps[frame];
		glRotatef(angle, 0.0, 1.0, 0.0);
		drawCube();
		frame++;
	      }
	    break;
	  case ZOOM0:
	    cerr << "zoom0 " << frame << endl;
	    if(frame == 17)
	      {
		frame = 0;
		stop_animation(widget);
	      } 
	    else 
	    {
	      glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	      switch(previous_animation)
		{
		case ZOOM1:
		  persp = perspsteps[16-frame];
		  atx = lookposx = -(1.3*zoomsteps[16-frame]);
		  aty = lookposy = zoomsteps[16-frame] - 
                    yoffset/frames * (16-frame);
		  break;
		case ZOOM2:
		  persp = perspsteps[16-frame];
		  atx = lookposx = 1.3*zoomsteps[16-frame];
		  aty = lookposy = zoomsteps[16-frame] - 
                    yoffset/frames * (16-frame);
		  break;
		case ZOOM3:
		  persp = perspsteps[16-frame];
		  atx = lookposx = -1.3*zoomsteps[16-frame];
		  aty = lookposy = -zoomsteps[16-frame] - 
                    yoffset/frames * (16-frame);
		  break;
		case ZOOM4:
		  persp = perspsteps[16-frame];
		  atx = lookposx = 1.3*zoomsteps[16-frame];
		  aty = lookposy = -zoomsteps[16-frame] - 
                    yoffset/frames * (16-frame);
		  break;
		case ZOOMC:
		  persp = perspstepsc[16-frame];
		  aty = lookposy = - zoomsteps[16-frame]*0.38;
		  break;
		default:
		  cerr << "Should not reach" << endl;
		  break;
		}
	      glMatrixMode(GL_PROJECTION);
	      glLoadIdentity();
	      gluPerspective( persp,
			      1.0,
			      0.5, 
			      10.0);
	      glMatrixMode(GL_MODELVIEW);
	      glLoadIdentity();
	      gluLookAt(lookposx, lookposy, lookposz, atx, aty, atz, 0,1,0);
	      glRotatef(angle, 0.0, 1.0, 0.0);
	      drawCube();
	      frame++;
	    }
	    break;
	  case ZOOM1:
	    cerr << "zoom1 " << frame << endl;
	    if(frame == 17)
	      {
		frame = 0;
		stop_animation(widget);
	      } 
	    else 
	      {	
		glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
		glMatrixMode(GL_PROJECTION);
		glLoadIdentity();
		persp = perspsteps[frame];
		gluPerspective(persp,
			       1.0,
			       0.5, 
			       10.0);
		glMatrixMode(GL_MODELVIEW);
		glLoadIdentity();
		atx = lookposx = -1.3*zoomsteps[frame];
		aty = lookposy = zoomsteps[frame] - yoffset/frames * (frame);
		gluLookAt(lookposx, lookposy, lookposz, atx, aty, atz, 0,1,0);
		glRotatef(angle, 0.0, 1.0, 0.0);
		drawCube();
		frame++;
	      }
	    break;
	  case ZOOM2:
	    cerr << "zoom1 " << frame << endl;
	    if(frame == 17)
	      {
		frame = 0;
		stop_animation(widget);
	      } 
	    else 
	      {	
		glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
		glMatrixMode(GL_PROJECTION);
		glLoadIdentity();
		persp = perspsteps[frame];
		gluPerspective(persp,
			       1.0,
			       0.5, 
			       10.0);
		glMatrixMode(GL_MODELVIEW);
		glLoadIdentity();
		atx = lookposx = 1.3*zoomsteps[frame];
		aty = lookposy = zoomsteps[frame] - yoffset/frames * (frame);
		gluLookAt(lookposx, lookposy, lookposz, atx, aty, atz, 0,1,0);
		glRotatef(angle, 0.0, 1.0, 0.0);
		drawCube();
		frame++;
	      }
	    break;
	  case ZOOM3:
	    cerr << "zoom1 " << frame << endl;
	    if(frame == 17)
	      {
		frame = 0;
		stop_animation(widget);
	      } 
	    else 
	      {	
		glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
		glMatrixMode(GL_PROJECTION);
		glLoadIdentity();
		persp = perspsteps[frame];
		gluPerspective(persp,
			       1.0,
			       0.5, 
			       10.0);
		glMatrixMode(GL_MODELVIEW);
		glLoadIdentity();
		atx = lookposx = -1.3*zoomsteps[frame];
		aty = lookposy = -zoomsteps[frame] - yoffset/frames * (frame);
		gluLookAt(lookposx, lookposy, lookposz, atx, aty, atz, 0,1,0);
		glRotatef(angle, 0.0, 1.0, 0.0);
		drawCube();
		frame++;
	      }
	    break;
	  case ZOOM4:
	    cerr << "zoom1 " << frame << endl;
	    if(frame == 17)
	      {
		frame = 0;
		stop_animation(widget);
	      } 
	    else 
	      {	
		glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
		glMatrixMode(GL_PROJECTION);
		glLoadIdentity();
		persp = perspsteps[frame];
		gluPerspective(persp,
			       1.0,
			       0.5, 
			       10.0);
		glMatrixMode(GL_MODELVIEW);
		glLoadIdentity();
		atx = lookposx = 1.3*zoomsteps[frame];
		aty = lookposy = -zoomsteps[frame] -yoffset/frames * (frame);
		gluLookAt(lookposx, lookposy, lookposz, atx, aty, atz, 0,1,0);
		glRotatef(angle, 0.0, 1.0, 0.0);
		drawCube();
		frame++;
	      }
	    break;
	  case ZOOMC:
	    cerr << "zoomc " << frame << endl;
	    if(frame == 17)
	      {
		frame = 0;
		stop_animation(widget);
	      } 
	    else 
	      {	
		glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
		persp = perspstepsc[frame];
		aty = lookposy = - zoomsteps[frame]*0.38;
		glMatrixMode(GL_PROJECTION);
		glLoadIdentity();
		gluPerspective(persp,
			       1.0,
			       0.5, 
			       10.0);
		glMatrixMode(GL_MODELVIEW);
		glLoadIdentity();
		gluLookAt(lookposx, lookposy, lookposz, atx, aty, atz, 0,1,0);
		glRotatef(angle, 0.0, 1.0, 0.0);
		drawCube();
		frame++;
	      }
	    break;
	  case SWITCH_FW:
	    cerr << "fw " << frame << endl;
	    if(frame == 1)
	      {
		frame = 0;
		stop_animation(widget);
	      }
	    else 
	      {
		glClear (GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
		glMatrixMode(GL_MODELVIEW);
		glLoadIdentity();
		gluLookAt(lookposx, lookposy, lookposz, atx, aty, atz, 0,1,0);
		angle -= 90;
		glRotatef(angle, 0.0, 1.0, 0.0);
		drawCube();
		frame++;
	      }
	    break;
	  case SWITCH_BW:
	    cerr << "bw " << frame << endl;
	    if(frame == 1)
	      {
		frame = 0;
		stop_animation(widget);
	      }
	    else
	      {
		glClear (GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
		glMatrixMode(GL_MODELVIEW);
		glLoadIdentity();
		gluLookAt(lookposx, lookposy, lookposz, atx, aty, atz, 0,1,0);
		angle += 90;
		glRotatef(angle, 0.0, 1.0, 0.0);
		drawCube();
		frame++;
	      }
	    break;
	  }
      }
    else
      {
	switch(active_animation)
	  {
	  case ANIM_NONE:
	    cerr << "Redrawing" << endl;
	    break;
	  case CUBE:
	    cerr << "cube stop" << endl;
	    forward(widget);
	    current_face = next_face();
	    break;
	  case SWITCH_FW:
	    cerr << "fw stop" << endl;
	    forward(widget);
	    current_face = next_face();
	    break;
	  case SWITCH_BW:
	    cerr << "bw stop" << endl;
	    backward(widget);
	    current_face = prev_face();
	    break;
	  case ZOOM0:
	  case ZOOM1:
	  case ZOOM2:
	  case ZOOM3:
	  case ZOOM4:
	  case ZOOMC:
	  default:
	    cerr << "default stop" << endl;
	    break;
	    
	  }

	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	gluPerspective( persp,
			1.0,
			0.5, 
			10.0);
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();
	gluLookAt(lookposx, lookposy, lookposz, atx, aty, atz, 0,1,0);	
	glRotatef(angle, 0.0, 1.0, 0.0);
	drawCube();

	glRasterPos3f(0, -1.4, 0);
	GLuint rcube[] = {
	  0,0,0,127
	};
	glDrawPixels(1, 1, GL_RGBA, GL_UNSIGNED_BYTE, rcube);

	active_animation = ANIM_NONE;
	
	cerr << "Ok!" << endl;
      }    
  }

  int prev_face()
  {
    if(current_face-1 < 0) return 3;
    else return current_face -1;
  }

  int next_face()
  {
    if(current_face+1 > 3) return 0;
    else return current_face+1;  
  }

  int prev_page()
  {
    assert(current_page >= 0);
    if(current_page == 0) return total_pages-1;
    else return current_page-1;
  }
  
  int next_page()
  {
    assert(current_page < total_pages);
    if(current_page == total_pages-1) return 0;
    else return current_page+1;
  }
  
  void
  forward(GtkWidget *widget)
  {
    update_textures_dir(widget, true);
    cerr << "Current page: " << current_page << endl;
  }
  
  void
  backward(GtkWidget *widget)
  {
    update_textures_dir(widget, false);
    cerr << "Current page: " << current_page << endl;
  }

  void
  reset(GtkWidget *widget)
  {
    animating = FALSE;
    frame = 0; 
    lookposx = 0.0;
    lookposy = 0.0;
    lookposz = 3.48; 
    atx = 0.0;
    aty = 0.0;
    atz = 0.0;
    persp = 44.0; 
    angle = 0.0;
    current_face = 0;
    active_animation = ANIM_NONE;
    previous_animation = ANIM_NONE;
    last_animation = ANIM_NONE;
    update_textures(widget);
  }

  // shift old textures and render the new page
  // texmap[0] -> current page
  // texmap[1] -> prev page
  // texmap[2] -> next page
  void update_textures_dir(GtkWidget *widget, bool forward)
  {
    assert(current_page >= 0);
    assert(current_page < total_pages);

    if(forward)
      {
	current_page=next_page();
	int tmp = texmap[2];
	texmap[2] = texmap[1];
	texmap[1] = texmap[0];
	texmap[0] = tmp;
	render_page(pixmap, next_page(), tex_width, tex_height);
      }
    else
      {
	current_page=prev_page();
	int tmp = texmap[0];
	texmap[0] = texmap[1];
	texmap[1] = texmap[2];
	texmap[2] = tmp;
	render_page(pixmap, prev_page(), tex_width, tex_height);
      }

    glBindTexture (GL_TEXTURE_RECTANGLE_ARB, 
                   textures[texmap[forward ? 2 : 1]]);
    glTexImage2D (GL_TEXTURE_RECTANGLE_ARB,
		  0,
		  GL_RGBA,
		  tex_width,
		  tex_height,
		  0,
		  GL_RGBA,
		  GL_UNSIGNED_BYTE,
		  gdk_pixbuf_get_pixels(pixmap));
    
    gdk_window_invalidate_rect (widget->window, &widget->allocation, FALSE);
    
  }

  // render all (3) textures
  void update_textures(GtkWidget *widget)
  {
    assert(current_page >= 0);
    assert(current_page < total_pages);

    render_page(pixmap, current_page, tex_width, tex_height);
    glBindTexture (GL_TEXTURE_RECTANGLE_ARB, textures[texmap[0]]);
    glTexImage2D (GL_TEXTURE_RECTANGLE_ARB,
		  0,
		  GL_RGBA,
		  tex_width,
		  tex_height,
		  0,
		  GL_RGBA,
		  GL_UNSIGNED_BYTE,
		  gdk_pixbuf_get_pixels(pixmap));
    
    render_page(pixmap, prev_page(), tex_width, tex_height);
    glBindTexture (GL_TEXTURE_RECTANGLE_ARB, textures[texmap[1]]);
    glTexImage2D (GL_TEXTURE_RECTANGLE_ARB,
		  0,
		  GL_RGBA,
		  tex_width,
		  tex_height,
		  0,
		  GL_RGBA,
		  GL_UNSIGNED_BYTE,
		  gdk_pixbuf_get_pixels(pixmap));
    
    render_page(pixmap, next_page(), tex_width, tex_height);
    glBindTexture (GL_TEXTURE_RECTANGLE_ARB, textures[texmap[2]]);
    glTexImage2D (GL_TEXTURE_RECTANGLE_ARB,
		  0,
		  GL_RGBA,
		  tex_width,
		  tex_height,
		  0,
		  GL_RGBA,
		  GL_UNSIGNED_BYTE,
		  gdk_pixbuf_get_pixels(pixmap));

    
      gdk_window_invalidate_rect (widget->window, &widget->allocation, FALSE);
      
  }

protected:
  PopplerDocument* doc;
  int current_page;
  int current_face;
  const int total_pages;
  int frame;
  double lookposx, lookposy, lookposz;
  double atx, aty, atz;
  double persp, angle;
  GdkPixbuf* pixmap;
  int texmap[3];

  // OpenGL Textures
  GLuint textures[3];
  
  // Width and Height of the rendered pixmap (aspect
  // ratio is fixed, should instead depend on the 
  // aspect ratio of the pdf page)
  static const gint tex_width = (gint)(3*1024/2);
  static const gint tex_height = (gint)(3*768/2);

  // Cube Normals
  static const GLfloat n[6][3];
  // Cube Faces
  static const GLint faces[6][4];
  // Cube vertex (filled in pdfcube->initialize())
  GLfloat v[8][3];  
  // Cube texture mapping
  static const GLfloat mapping[6][8];
  // Cube Rotation Animation steps (17 frames)
  // Cube rotation at each frame
  static const GLfloat steps[17];
  // x camera movement 
  static const double xsteps[17];
  // z camera movement
  static const double zsteps[17];
  static const double zoomsteps[17];
  static const double perspsteps[17];
  static const double perspstepsc[17];

  void 
  render_page(GdkPixbuf* pm, int i, gint iWidth, gint iHeight)
  {
    PopplerPage *page;
    page = poppler_document_get_page(doc, i);
    double w, h;
    poppler_page_get_size(page, &w, &h);
    poppler_page_render_to_pixbuf(page, 0,0,iWidth,iHeight,1.0*iWidth/w,0,pm);
  }

  void
  drawCube(void)
  {
    int i;

    for (i = 0; i < 6; i++) 
      {
	if(i == current_face)
	  {
	    glEnable (GL_TEXTURE_RECTANGLE_ARB);
	    glBindTexture (GL_TEXTURE_RECTANGLE_ARB, textures[texmap[0]]);
	  }
	else if(i == prev_face())
	  {
	    glEnable (GL_TEXTURE_RECTANGLE_ARB);
	    glBindTexture (GL_TEXTURE_RECTANGLE_ARB, textures[texmap[1]]);
	  }
	else if(i == next_face()) 
	  {
	    glEnable (GL_TEXTURE_RECTANGLE_ARB);
	    glBindTexture (GL_TEXTURE_RECTANGLE_ARB, textures[texmap[2]]);
	  }
	else if(i <= 3)
	  {
	    glDisable(GL_TEXTURE_RECTANGLE_ARB);
	    glColor4f(0.4, 0.0, 0.0, 1.0);
	  } 
	else
	  {
	    glDisable(GL_TEXTURE_RECTANGLE_ARB);
	    glColor4f(1.0, 1.0, 1.0, 1.0);
	  }
	glBegin(GL_QUADS);
	glNormal3fv(&n[i][0]);
	glTexCoord2f((1.0-mapping[i][4]) * tex_width, 
		     mapping[i][5] * tex_height); 
	glVertex3fv(&v[faces[i][0]][0]);
	
	glTexCoord2f((1.0-mapping[i][6]) * tex_width, 
		     mapping[i][7] * tex_height); 
	glVertex3fv(&v[faces[i][1]][0]);
	
	glTexCoord2f((1.0-mapping[i][0]) * tex_width, 
		     mapping[i][1] * tex_height); 
	glVertex3fv(&v[faces[i][2]][0]);
	
	glTexCoord2f((1.0-mapping[i][2]) * tex_width, 
		     mapping[i][3] * tex_height); 
	glVertex3fv(&v[faces[i][3]][0]);
	
	glEnd();
      }
  }
};

const GLfloat pdfcube::n[6][3] = { 
  {0.0, 0.0, -1.0}, {1.0, 0.0, 0.0}, {0.0, 0.0, 1.0},  {-1.0, 0.0, 0.0},  
  {0.0, 1.0, 0.0}, {0.0, -1.0, 0.0}
};
const GLint pdfcube::faces[6][4] = {
  {7, 4, 0, 3}, {7, 6, 5, 4}, {5, 6, 2, 1}, {0, 1, 2, 3}, 
  {3, 2, 6, 7}, {4, 5, 1, 0}
};
const GLfloat pdfcube::mapping[6][8] = {
    {1.0, 1.0, 1.0, 0.0, 0.0, 0.0, 0.0, 1.0}, 
    {0.0, 1.0, 1.0, 1.0, 1.0, 0.0, 0.0, 0.0}, 
    {0.0, 0.0, 0.0, 1.0, 1.0, 1.0, 1.0, 0.0}, 
    {1.0, 0.0, 0.0, 0.0, 0.0, 1.0, 1.0, 1.0}, 
    {1.0, 0.0, 0.0, 0.0, 0.0, 1.0, 1.0, 1.0}, // top
    {1.0, 1.0, 0.0, 1.0, 0.0, 0.0, 1.0, 0.0}, // bottom
};
const GLfloat pdfcube::steps[17] = 
  {  2.0,   2.5,   3,     4,     5,     6,     8,     10,   
     14,   10,   8,    6,    4,    3,    2.5,   2.0, 0.0  };
const double pdfcube::xsteps[17] = 
  {  0.01,  0.03,  0.07,  0.12,  0.16,  0.18,  0.20,  0.21, 
     0.21, 0.20, 0.18, 0.16, 0.12, 0.07, 0.03, 0.01, 0.00 };
const double pdfcube::zsteps[17] = 
  { -0.01, -0.02, -0.04, -0.05, -0.04, -0.02, -0.02, -0.01, 
    0.01, 0.02, 0.02, 0.04, 0.05, 0.04, 0.02, 0.01, 0.00 };
const double pdfcube::zoomsteps[17] =
  { 0.00, 0.01, 0.02, 0.03, 0.05, 0.07, 0.10, 0.13, 0.17, 
    0.21, 0.25, 0.29, 0.32, 0.35, 0.37, 0.38, 0.38 };
const double pdfcube::perspsteps[17] =
  { 44.0, 44.0, 44.0, 44.00, 44.00, 43.00, 42.00, 40.00, 
    38.00, 36.00, 33.00, 30.00, 27.00, 24.00, 22.00, 21.00, 21.00 };
const double pdfcube::perspstepsc[17] =
  { 44.0, 44.0, 44.0, 44.00, 44.00, 43.00, 42.00, 40.00, 
    38.00, 36.00, 34.00, 32.00, 31.00, 30.00, 30.00, 30.00, 30.00 };

pdfcube* pc;

//////////////////////////////////////////////////////////////////////////
// Callbacks

/***
 *** The "realize" signal handler. All the OpenGL initialization
 *** should be performed here, such as default background colour,
 *** certain states etc.
 ***/
static void
realize (GtkWidget *widget,
	 gpointer   data)
{
  GdkGLContext *glcontext = gtk_widget_get_gl_context (widget);
  GdkGLDrawable *gldrawable = gtk_widget_get_gl_drawable (widget);

  g_print ("%s: \"realize\"\n", gtk_widget_get_name (widget));

  //g_mutex_lock (gl_mutex);

  /*** OpenGL BEGIN ***/
  if (!gdk_gl_drawable_gl_begin (gldrawable, glcontext))
    return;

  pc->initialize(widget);

  gdk_gl_drawable_gl_end (gldrawable);
  /*** OpenGL END ***/
  //g_mutex_unlock (gl_mutex);
}

/***
 *** The "configure_event" signal handler. Any processing required when
 *** the OpenGL-capable drawing area is re-configured should be done here.
 *** Almost always it will be used to resize the OpenGL viewport when
 *** the window is resized.
 ***/
static gboolean
configure_event (GtkWidget         *widget,
		 GdkEventConfigure *event,
		 gpointer           data)
{
  GdkGLContext *glcontext = gtk_widget_get_gl_context (widget);
  GdkGLDrawable *gldrawable = gtk_widget_get_gl_drawable (widget);

  GLsizei w = widget->allocation.width;
  GLsizei h = widget->allocation.height;

  g_print ("%s: \"configure_event\"\n", gtk_widget_get_name (widget));

  //g_mutex_lock (gl_mutex);
  /*** OpenGL BEGIN ***/
  if (!gdk_gl_drawable_gl_begin (gldrawable, glcontext))
    return FALSE;

  glViewport (0, 0, w, h);

  gdk_gl_drawable_gl_end (gldrawable);

 /*** OpenGL END ***/
  //g_mutex_unlock (gl_mutex);

  return TRUE;
}

/***
 *** The "expose_event" signal handler. All the OpenGL re-drawing should
 *** be done here. This is repeatedly called as the painting routine
 *** every time the 'expose'/'draw' event is signalled.
 ***/
static gboolean
expose_event (GtkWidget      *widget,
	      GdkEventExpose *event,
	      gpointer        data)
{
  GdkGLContext *glcontext = gtk_widget_get_gl_context (widget);
  GdkGLDrawable *gldrawable = gtk_widget_get_gl_drawable (widget);

  //g_mutex_lock (gl_mutex);
  /*** OpenGL BEGIN ***/
  if (!gdk_gl_drawable_gl_begin (gldrawable, glcontext))
    return FALSE;

  glDrawBuffer(GL_BACK);  

  pc->redraw(widget);
  
  /* Swap buffers */
  if (gdk_gl_drawable_is_double_buffered (gldrawable))
    gdk_gl_drawable_swap_buffers (gldrawable);
  else
    glFlush ();
  

  gdk_gl_drawable_gl_end (gldrawable);
  /*** OpenGL END ***/

  //g_mutex_unlock (gl_mutex);

  return TRUE;
}

/***
 *** The timeout function. Often in animations,
 *** timeout functions are suitable for continous
 *** frame updates.
 ***/
static gboolean
timeout (GtkWidget *widget)
{
  /* Invalidate the whole window. */
  gdk_window_invalidate_rect (widget->window, &widget->allocation, FALSE);

  /* Update synchronously. */
  gdk_window_process_updates (widget->window, FALSE);

  return TRUE;
}

/***
 *** The "unrealize" signal handler. Any processing required when
 *** the OpenGL-capable window is unrealized should be done here.
 ***/
static void
unrealize (GtkWidget *widget,
	   gpointer   data)
{
  GdkGLContext *glcontext = gtk_widget_get_gl_context (widget);
  GdkGLDrawable *gldrawable = gtk_widget_get_gl_drawable (widget);

  g_print ("%s: \"unrealize\"\n", gtk_widget_get_name (widget));

  //g_mutex_lock (gl_mutex);

  /*** OpenGL BEGIN ***/
  if (!gdk_gl_drawable_gl_begin (gldrawable, glcontext))
    return;

  gdk_gl_drawable_gl_end (gldrawable);
  /*** OpenGL END ***/
  //g_mutex_unlock (gl_mutex);
}

/***
 *** The "motion_notify_event" signal handler. Any processing required when
 *** the OpenGL-capable drawing area is under drag motion should be done here.
 ***/
static gboolean
motion_notify_event (GtkWidget      *widget,
		     GdkEventMotion *event,
		     gpointer        data)
{
  g_print ("%s: \"motion_notify_event\": button", gtk_widget_get_name (widget));

  if (event->state & GDK_BUTTON1_MASK)
    {
      g_print (" 1");
    }

  if (event->state & GDK_BUTTON2_MASK)
    {
      g_print (" 2");
    }

  if (event->state & GDK_BUTTON3_MASK)
    {
      g_print (" 3");
    }

  g_print ("\n");

  return FALSE;
}

/***
 *** The "button_press_event" signal handler. Any processing required when
 *** mouse buttons (only left and middle buttons) are pressed on the OpenGL-
 *** capable drawing area should be done here.
 ***/
static gboolean
button_press_event (GtkWidget      *widget,
		    GdkEventButton *event,
		    gpointer        data)
{
  g_print ("%s: \"button_press_event\": ", gtk_widget_get_name (widget));

  if (event->button == 1)
    {
      g_print ("button 1\n");

      return TRUE;
    }

  if (event->button == 2)
    {
      g_print ("button 2\n");

      return TRUE;
    }

  g_print ("\n");

  return FALSE;
}

/***
 *** The "key_press_event" signal handler. Any processing required when key
 *** presses occur should be done here.
 ***/
static gboolean
key_press_event (GtkWidget   *widget,
		 GdkEventKey *event,
		 gpointer     data)
{
  g_print ("%s: \"key_press_event\": ", gtk_widget_get_name (widget));

  if(event->state == GDK_CONTROL_MASK)

    switch (event->keyval)
      {
      case GDK_1:
      case GDK_2:
      case GDK_3:
      case GDK_4:
      case GDK_5:
      case GDK_6:
      case GDK_7:
      case GDK_8:
      case GDK_9:
	g_print ("n key\n");
	if(sleeping())
	  pc->section(widget, event->keyval - GDK_1 + 1);
	break;

	// Let's quit
      case GDK_q:
	g_print ("Escape key\n");
	gtk_main_quit ();
	break;

	// Update all textures
      case GDK_l:
	g_print ("u key\n");
	cerr << "Pagina: " << pc->page() << endl;
	pc->reset(widget);
	break;
      }
  else
    switch (event->keyval)
      {
	
	// return to page 1
      case GDK_1:
      case GDK_2:
      case GDK_3:
      case GDK_4:
      case GDK_5:
      case GDK_6:
      case GDK_7:
      case GDK_8:
      case GDK_9:
	g_print ("n key\n");
	if(sleeping())
	  pc->go_to(widget, (event->keyval - GDK_1) * 5);
	break;
	
	// Animated Cube Advancement
      case GDK_a:
      case GDK_c:
	g_print ("c key\n");
	if(sleeping())
	  start_animation(widget, CUBE);
	break;
	
      // Quick switch to next page
      case GDK_Page_Down:
      case GDK_Right:
	g_print ("s key\n");
	if(sleeping())
	  start_animation(widget, SWITCH_FW);
	break;
	
      // Quick switch to previous page
      case GDK_Page_Up:
      case GDK_Left:
	g_print ("q key\n");
	if(sleeping())
	  start_animation(widget, SWITCH_BW);
	break;



      case GDK_g:
	if(sleeping())
	  if(last_animation >= ZOOM1 and last_animation <= ZOOMC)
	    start_animation(widget, ZOOM0);
	break;

      case GDK_h:
	if(sleeping())
	  if(last_animation >= ZOOM1 and last_animation <= ZOOMC)
	    start_animation(widget, ZOOM0);
	  else
	    start_animation(widget, ZOOM1);
	break;

      case GDK_j:
	if(sleeping())
	  if(last_animation >= ZOOM1 and last_animation <= ZOOMC)
	    start_animation(widget, ZOOM0);
	  else
	    start_animation(widget, ZOOM2);
	break;

      case GDK_k:
	if(sleeping())
	  if(last_animation >= ZOOM1 and last_animation <= ZOOMC)
	    start_animation(widget, ZOOM0);
	  else
	    start_animation(widget, ZOOM3);
	break;

      case GDK_l:
	if(sleeping())
	  if(last_animation >= ZOOM1 and last_animation <= ZOOMC)
	    start_animation(widget, ZOOM0);
	  else
	    start_animation(widget, ZOOM4);
	break;
	
      case GDK_z:
	if(sleeping())
	  if(last_animation >= ZOOM1 and last_animation <= ZOOMC)
	    start_animation(widget, ZOOM0);
	  else
	    start_animation(widget, ZOOMC);
	break;
	
	
	// Automatic advance (you should se the Animated slides on the command line)
      case GDK_space:
	if(page_transition[pc->page()] and sleeping())
	  start_animation(widget, CUBE);
	else if(sleeping())
	  start_animation(widget, SWITCH_FW);
	
	break;
	
	// switch fullscreen
      case GDK_f:
	if( (fullscreen = !fullscreen) == true)
	  gtk_window_fullscreen((GtkWindow*)(data));
	else
	  gtk_window_unfullscreen((GtkWindow*)(data));
	break;
	
	// Let's quit
      case GDK_Escape:
	g_print ("Escape key\n");
	gtk_main_quit ();
	break;

      default:
	g_print("\n");
	return FALSE;
    }

  return TRUE;
}


//////////////////////////////////////////////////////////////////////////
// Timeout functions

/***
 *** Helper functions to add or remove the timeout function.
 ***/

static guint timeout_id = 0;

static void
timeout_add (GtkWidget *widget)
{
  if (timeout_id == 0)
    {
      timeout_id = g_timeout_add (TIMEOUT_INTERVAL,
                                  (GSourceFunc) timeout,
                                  widget);
    }
}

static void
timeout_remove (GtkWidget *widget)
{
  if (timeout_id != 0)
    {
      g_source_remove (timeout_id);
      timeout_id = 0;
    }
}

/***
 *** The "map_event" signal handler. Any processing required when the
 *** OpenGL-capable drawing area is mapped should be done here.
 ***/
static gboolean
map_event (GtkWidget *widget,
	   GdkEvent  *event,
	   gpointer   data)
{
  g_print ("%s: \"map_event\":\n", gtk_widget_get_name (widget));
  if (animating)
    timeout_add (widget);

  return TRUE;
}

/***
 *** The "unmap_event" signal handler. Any processing required when the
 *** OpenGL-capable drawing area is unmapped should be done here.
 ***/
static gboolean
unmap_event (GtkWidget *widget,
	     GdkEvent  *event,
	     gpointer   data)
{
  g_print ("%s: \"unmap_event\":\n", gtk_widget_get_name (widget));
  timeout_remove (widget);

  return TRUE;
}

/***
 *** The "visibility_notify_event" signal handler. Any processing required
 *** when the OpenGL-capable drawing area is visually obscured should be
 *** done here.
 ***/
static gboolean
visibility_notify_event (GtkWidget          *widget,
			 GdkEventVisibility *event,
			 gpointer            data)
{
  if (animating)
    {
      if (event->state == GDK_VISIBILITY_FULLY_OBSCURED)
	timeout_remove (widget);
      else
	timeout_add (widget);
    }

  return TRUE;
}


/**************************************************************************
 * The following section contains some miscellaneous utility functions.
 **************************************************************************/

/***
 *** Toggle animation.
 ***/
static void
start_animation(GtkWidget *widget, enum animation a)
{
  if(sleeping())
    {
      animating = true;
      previous_animation = last_animation;
      last_animation = active_animation = a;
      timeout_add(widget);
    }
}

static void
stop_animation(GtkWidget *widget)
{
  animating = false;
  timeout_remove(widget);
  gdk_window_invalidate_rect(widget->window, &widget->allocation, FALSE);
  gdk_window_process_updates (widget->window, FALSE);
}

//////////////////////////////////////////////////////////////////////////
// GTK+ GUI Functions

/***
 *** Creates the simple application window with one
 *** drawing area that has an OpenGL-capable visual.
 ***/
static GtkWidget *
create_window (GdkGLConfig *glconfig)
{
  GtkWidget *window;
  GtkWidget *vbox;
  GtkWidget *drawing_area;

  /*
   * Top-level window.
   */

  window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title (GTK_WINDOW (window), DEFAULT_TITLE);

  /* Get automatically redrawn if any of their children changed allocation. */
  gtk_container_set_reallocate_redraws (GTK_CONTAINER (window), TRUE);

  /* Connect signal handlers to the window */
  g_signal_connect (G_OBJECT (window), "delete_event",
		    G_CALLBACK (gtk_main_quit), NULL);

  /*
   * VBox.
   */

  vbox = gtk_vbox_new (FALSE, 0);
  gtk_container_add (GTK_CONTAINER (window), vbox);
  gtk_widget_show (vbox);

  /*
   * Drawing area to draw OpenGL scene.
   */

  drawing_area = gtk_drawing_area_new ();
  gtk_widget_set_size_request (drawing_area, DEFAULT_WIDTH, DEFAULT_HEIGHT);

  /* Set OpenGL-capability to the widget */
  gtk_widget_set_gl_capability (drawing_area,
				glconfig,
				NULL,
				TRUE,
				GDK_GL_RGBA_TYPE);

  gtk_widget_add_events (drawing_area,
			 GDK_BUTTON1_MOTION_MASK    |
			 GDK_BUTTON2_MOTION_MASK    |
			 GDK_BUTTON_PRESS_MASK      |
			 GDK_VISIBILITY_NOTIFY_MASK);

  /* Connect signal handlers to the drawing area */
  g_signal_connect_after (G_OBJECT (drawing_area), "realize",
                          G_CALLBACK (realize), NULL);
  g_signal_connect (G_OBJECT (drawing_area), "configure_event",
		    G_CALLBACK (configure_event), NULL);
  g_signal_connect (G_OBJECT (drawing_area), "expose_event",
		    G_CALLBACK (expose_event), NULL);
  g_signal_connect (G_OBJECT (drawing_area), "unrealize",
		    G_CALLBACK (unrealize), NULL);

  g_signal_connect (G_OBJECT (drawing_area), "motion_notify_event",
		    G_CALLBACK (motion_notify_event), NULL);
  g_signal_connect (G_OBJECT (drawing_area), "button_press_event",
		    G_CALLBACK (button_press_event), NULL);

  /* key_press_event handler for top-level window */
  g_signal_connect_swapped (G_OBJECT (window), "key_press_event",
			    G_CALLBACK (key_press_event), drawing_area);

  /* For timeout function. */
  g_signal_connect (G_OBJECT (drawing_area), "map_event",
		    G_CALLBACK (map_event), NULL);
  g_signal_connect (G_OBJECT (drawing_area), "unmap_event",
		    G_CALLBACK (unmap_event), NULL);
  g_signal_connect (G_OBJECT (drawing_area), "visibility_notify_event",
		    G_CALLBACK (visibility_notify_event), NULL);

  gtk_box_pack_start (GTK_BOX (vbox), drawing_area, TRUE, TRUE, 0);

  gtk_widget_show (drawing_area);

  return window;
}

/***
 *** Configure the OpenGL framebuffer.
 ***/
static GdkGLConfig *
configure_gl (void)
{
  GdkGLConfig *glconfig;

  /* Try double-buffered visual */
  glconfig = gdk_gl_config_new_by_mode ((GdkGLConfigMode)
					(
					 GDK_GL_MODE_RGBA    |
					 GDK_GL_MODE_ALPHA  |
					 GDK_GL_MODE_DEPTH  |
					 GDK_GL_MODE_DOUBLE));
  if (glconfig == NULL)
    {
      g_print ("\n*** Cannot find the double-buffered visual.\n");
      g_print ("\n*** Trying single-buffered visual.\n");

      /* Try single-buffered visual */
      glconfig = gdk_gl_config_new_by_mode ((GdkGLConfigMode)
					    (
					     GDK_GL_MODE_RGB   |
					     GDK_GL_MODE_DEPTH));
      if (glconfig == NULL)
	{
	  g_print ("*** No appropriate OpenGL-capable visual found.\n");
	  exit (1);
	}
    }


  return glconfig;
}

///
/// @brief Gets the absolute path of a filename.
///
/// This function checks if the given @a fileName is an absolute path. If
/// it is then it returns a copy of it, otherwise it prepends the current
/// working directory to it.
///
/// @param fileName The filename to get the absolute path from.
///
/// @return A copy of the absolute path to the file name. This copy must be
///         freed when no longer needed.
///
gchar *
get_absolute_file_name (const gchar *fileName)
{
    gchar *absoluteFileName = NULL;
    if ( g_path_is_absolute (fileName) )
    {
        absoluteFileName = g_strdup (fileName);
    }
    else
    {
        gchar *currentDir = g_get_current_dir ();
        absoluteFileName = g_build_filename (currentDir, fileName, NULL);
        g_free (currentDir);
    }

    return absoluteFileName;
}

//////////////////////////////////////////////////////////////////////////
// Main function: should we use getopts? (who doesn't?)

int
main (int   argc,
      char *argv[])
{
  GtkWidget *window;
  GdkGLConfig *glconfig;

  /* Initialize GTK. */
  gtk_init (&argc, &argv);

  /* Initialize GtkGLExt. */
  gtk_gl_init (&argc, &argv);

  /* Configure OpenGL framebuffer. */
  glconfig = configure_gl();

  if(argc < 2)
    {
      perror("usage: pdfcube file_uri [cube_page ...]");
      exit(1);
    }
  gchar *absoluteFileName = get_absolute_file_name (argv[1]);
  gchar *filename_uri = g_filename_to_uri (absoluteFileName, NULL, NULL);
  g_free (absoluteFileName);
  if ( NULL == filename_uri )
    {
      cerr << "Errore nel nome del file" << endl;
    }
  PopplerDocument* document = 
    poppler_document_new_from_file(filename_uri, NULL, NULL);

  if(document == NULL)
    {
      perror("invaild pdf file");
      exit(2);
    }
  
  pc = new pdfcube(document);
  
  page_transition = new bool[pc->pages()];
  
  for(int ii = 0; ii < pc->pages(); ii++)    
    {
      page_transition[ii] = false;
    }
  
  for(int ii = 2; ii < argc; ii++)    
    {
      page_transition[atoi(argv[ii])] = true;
    }

  /* Create and show the application window. */
  window = create_window(glconfig);

  if(fullscreen)
    gtk_window_fullscreen((GtkWindow*)(window));

  gtk_widget_show (window);

  gtk_main ();

  return 0;
}

// EOF
//////////////////////////////////////////////////////////////////////////
