//
// C++ Interface: vocabularymodel
//
// Description: 
//
//
// Author: Benjamin Mesing <bensmail@gmx.net>, (C) 2006
//
// Copyright: See COPYING file that comes with this distribution
//
//
#ifndef __NTAGMODEL_VOCABULARYMODEL_H_2006_10_01
#define __NTAGMODEL_VOCABULARYMODEL_H_2006_10_01

#include <set>
#include <vector>

#include <QAbstractListModel>

// #include <ept/apt/apt.h>
#include <ept/debtags/tag.h>
// #include <ept/cache/debtags/vocabulary.h>


#include <helpers.h>

namespace NTagModel
{

using namespace std;

struct FacetData;
struct TagData;

struct ItemData
{
	virtual ~ItemData() {};
	virtual bool isFacet() const = 0;
	virtual QString name() const = 0;
	virtual QString fullname() const = 0;
	virtual QString description() const = 0;
	virtual const FacetData* toFacetData() const = 0;
	virtual const TagData* toTagData() const = 0;
	virtual FacetData* toFacetData() = 0;
	virtual TagData* toTagData() = 0;
};

struct FacetData : public ItemData
{
	typedef ept::debtags::Facet Facet;
	
	Facet facet;
	bool hidden;
	int row;
	FacetData(Facet facet_, int row_) :
		hidden(false),
		facet(facet_),
		row(row_)
	{

	}
	virtual bool isFacet() const { return true; }
	virtual QString name() const { return toQString(facet.shortDescription()); }
	virtual QString fullname() const { return toQString(facet.name()); }
	virtual QString description() const { return toQString(facet.shortDescription()); }
	virtual const FacetData* toFacetData() const { return this; };
	virtual const TagData* toTagData() const { return 0; };
	virtual FacetData* toFacetData() { return this; };
	virtual TagData* toTagData() { return 0; };
};

struct TagData : public ItemData
{
	typedef ept::debtags::Tag Tag;
	
	int facetIndex;
	/** @brief Stores if the tag was selected to be searched for.
	  *
	  * This has nothing to do with user selection like it is happening
	  * through the views.
	  */
	bool selected;
	Tag tag;
	TagData(const Tag tag_, int facetIndex_) :
		facetIndex(facetIndex_),
		selected(false),
		tag(tag_)
	{
	}
	virtual bool isFacet() const { return false; }
	virtual QString name() const { return toQString(tag.shortDescription()); }
	virtual QString fullname() const { return toQString(tag.fullname()); }
	virtual QString description() const { return toQString(tag.longDescription()); }
	virtual const FacetData* toFacetData() const { return 0; };
	virtual const TagData* toTagData() const { return this; };
	virtual FacetData* toFacetData() { return 0; };
	virtual TagData* toTagData() { return this; };
};


/**
  *	@author Benjamin Mesing <bensmail@gmx.net>
  */
class VocabularyModel : public QAbstractItemModel
{
// Q_OBJECT
	typedef ept::debtags::Tag Tag;
	typedef ept::debtags::Facet Facet;

	/** @brief The object used to access the tag collection.
	  *
	  * If this is 0, the tag selection will not be accessible, and no companion tags
	  * will be determined (i.e. if you select a tag, the selection will not be limited 
	  * according to this tag.)
	  */

	vector<FacetData> _facets; 
 	vector< vector<TagData> > _tags; 
 	/** Maps the tag to the pair (facetIndex, tagIndex) identifying the tag. */
	map<Tag, pair<int, int> > _tagToTagData;

public:
	VocabularyModel();
	~VocabularyModel();

	/** @name AbstractItemModel interface
	  * 
	  * Implementation of the AbstractItemModel interface
	  */
	//@{
	virtual int rowCount(const QModelIndex & parent = QModelIndex() ) const;
	virtual int columnCount(const QModelIndex & = QModelIndex() ) const { return 2; };
	virtual QVariant data(const QModelIndex & index, int role = Qt::DisplayRole) const;
	virtual QVariant headerData ( int section, Qt::Orientation orientation, int role = Qt::DisplayRole ) const;
	virtual QModelIndex parent(const QModelIndex & index ) const;
	virtual QModelIndex index(int row, int column, const QModelIndex & parent = QModelIndex() ) const;
	//@}
	/** @brief Returns an index for the handed tag. 
	  *
	  * @param tag the tag to get the index for
	  * @param column the column for the row with the given tag
	  * @returns an invalid index if no such tag/facet exists
	  */
	virtual QModelIndex indexForTag(const Tag& tag, int column) const;

	
	/** Allows to set the data for the VocabularyModeRole::Selected and the 
	  * VocabularyModeRole::Hidden role.
	  *
	  * @returns true if the data was set successfully, else false
	  * @pre index.isValid()
	  */
	virtual bool setData(const QModelIndex& index, const QVariant& value, int role);

	/** Gets the index'th facet in the current model. 
	  *
	  * Precondition: 0 &lt;= i &lt; facetNum
	  */
	Facet getFacet(int index) const;

	// TODO implement
	/** @brief Returns the facets that are hidden. */
	set<Facet> hiddenFacets() const { qDebug("hiddenFacets() - to be implemented"); };
	/** @brief Returns the facets that are hidden. */
	set<Facet> shownFacets() const { qDebug("shownFacets() - to be implemented"); };

	/** Returns the tags selected for this model. */
	const set<Tag>& selectedTags() const;

	/** Gets the index'th object in the given set. 
	  *
	  * Precondition: 0 &lt;= i &lt; collection.size()
	  */
	template<class T> const T& getElement(std::set<T> collection, int index) const
	{
		typedef set<T> set_type;
		int i = 0;
		typename set_type::const_iterator it = collection.begin();
		while (i != index)
		{
			++i;
			++it;
		}
		return *it;
	}
protected:
	/** Collects the tags from the model that are selected. 
	  * 
	  * @pre parent != QModelIndex() ( which means that the children of parent 
	  * must point be tags (i.e. no facets)
	  */
	std::set<Tag> collectSelectedChildItems(const QModelIndex& parent) const;
private:
	/** Keeps track of the tags which are selected. */
	std::set<Tag> _selectedTags;
	// TODO: access for companion tags to be implemented (including caching)
	volatile bool _companionTagsValid;
	// TODO: access for companion tags to be implemented (including caching)
	volatile std::set<Tag> _companionTagsChache;
	void emitAllDataChanged();
};

}

struct TagWrapper
{
	typedef ept::debtags::Tag Tag;
public:
	Tag tag;
	TagWrapper();
	TagWrapper(Tag tag_);
};

Q_DECLARE_METATYPE(TagWrapper)


#endif	// __NTAGMODEL_VOCABULARYMODEL_H_2006_10_01
