/*************************************************
**      To: qt_nvclock.cpp
** Project: nvclock
**  Author: Jan Prokop
**  e-mail: jprokop@ibl.sk
*************************************************/

#include "qt_nvclock.h"
#include "../../config.h"

#include <qlabel.h>
#include <qlayout.h>
#include <qspinbox.h>
#include <qcombobox.h>
#include <qcheckbox.h>
#include <qpushbutton.h>
#include <qmessagebox.h>

#define QMB       QMessageBox
#define QPIX(pix) QPixmap((const char**)pix)

#include "nv.xpm"

struct card card[MAX_CARDS];

//********************************************************************************

CTabNVidia::CTabNVidia(QWidget *parent, char *name)
    : QWidget(parent, name)
{
    l = new QGridLayout(this, 2, 3, 4, 4);

    int yy = 0;
    addLabel(tr("Card number"), yy);
    comboCardNo = new QComboBox(this, "comboCardNo");
    l->addWidget(comboCardNo, yy++, 1);

    addLabel(tr("Memory speed"), yy);
    spinMemory = new QSpinBox(100, 1000, 1, this, "spinMemory");
    spinMemory->setSuffix("MHz");
    l->addWidget(spinMemory, yy++, 1);

    addLabel(tr("Core speed"), yy);
    spinCore = new QSpinBox(100, 1000, 1, this, "spinCore");
    spinCore->setSuffix("MHz");
    l->addWidget(spinCore, yy++, 1);

    checkDebug = new QCheckBox(tr("Debug info"), this, "checkDebug");
    l->addWidget(checkDebug, yy++, 1);
    
    QPushButton *button = new QPushButton(tr("GO"), this);
    l->addMultiCellWidget(button, 0, 2, 2, 2);
    connect(button, SIGNAL(clicked()), SLOT(slotGo()));

    connect(comboCardNo, SIGNAL(activated(int)),
            SLOT(slotLoad(int)));
    
    if(getCards()) slotLoad(0);
}

//********************************************************************************

void
CTabNVidia::addLabel(const QString &text, int yy)
{
    QLabel *label = new QLabel(text, this);
    l->addWidget(label, yy, 0, AlignRight | AlignVCenter);
}

//********************************************************************************

int
CTabNVidia::getCards()
{
    //*** get list of cards ***

    if ( getuid() != 0 && geteuid() != 0) {
	total_cards = FindAllCards_ioctl(); 
    }

    else {
	total_cards = FindAllCards_proc();
    }


    if(total_cards < 0) {
        QMB::warning(this, tr("Error"),
                     tr("No nVidia card found in your system !"));
	setEnabled(FALSE);
        return(FALSE);
    }

    for(int i=0; i<=total_cards; i++)
        comboCardNo->insertItem(card[i].device_string);
    
    return(TRUE);
}

//********************************************************************************

int
CTabNVidia::slotLoad(int card_number)
{
    float memclk, nvclk;
    if(GetSpeed(&memclk, &nvclk, card[card_number].number,
                checkDebug->isChecked()) != 0) {
        QMB::warning(this, tr("Warning"), tr("Can't load data !"));
        return(FALSE);
    }

    const pci_ids *p = GetCardInfo(card_number);
    if(!p) {
        QMB::warning(this, tr("Error"), tr("Can't get card info !"));
        spinMemory->setRange(100, 1000);
        spinCore->setRange(100, 1000);
    } else {
        spinMemory->setRange(p->mem_min, p->mem_max);
        spinCore->setRange(p->nv_min, p->nv_max);
    }

    spinMemory->setValue((int)memclk);
    spinCore->setValue((int)nvclk);
    return(TRUE);
}

//********************************************************************************

int
CTabNVidia::slotGo()
{
    int id = comboCardNo->currentItem();
    qDebug("overclocking card: %d", id);
    
    if(!ChangeSpeeds(spinMemory->value(),
                     spinCore->value(),
                     card[id].number,
                     checkDebug->isChecked())) return(FALSE);
    
    return(slotLoad(id));
}

//********************************************************************************

CNVclock::CNVclock(QWidget *parent, char *name)
: QTabDialog(parent, name)
{
    setCaption(tr("Linux overclocker - NVclock QT " VERSION));
    
    setOkButton(0);
    setCancelButton(tr("&Close"));

    CTabNVidia *tabNVidia  = new CTabNVidia(this, "tabNVidia");
    
    addTab(tabNVidia, QPIX(nv_xpm), tr("NVidia"));
    addTab(initLinks(), tr("Links"));
    addTab(initAbout(), tr("About"));
}

//********************************************************************************

QTextBrowser *
CNVclock::initLinks()
{
    QTextBrowser *tab = new QTextBrowser(this, "tabAbout");
    tab->setText(tr("<qt><b>NVidia</b><br>"
                    "<ul>"
                    "<li>home page http://www.nvidia.com</li>"
		    "<li>news http://www.evil3d.net</li>"
                    "<li>news http://news.nvidia.com</li>"
                    "</ul>"
                    "<b>Accelerated games</b><ul>"
                    "<li>Quake3 arena</li>"
                    "</ul>"
                    "</qt>"));
    return(tab);
}

//********************************************************************************

QTextBrowser *
CNVclock::initAbout()
{
    QTextBrowser *tab = new QTextBrowser(this, "tabAbout");
    tab->setText(tr("<qt>Authors<br><br>"
                    "<b>Roderick Colenbrander thunderbird@evil3d.nospam.net</b>"
                    "<ul>"
                    "<li>author of nvclock</li>"
		    "<li>overclocking/pci code</li>"
		    "<li>support for multiple cards</li>"
		    "<li>GTK gui</li>"
                    "<li>documentation</li>"
                    "</ul>"
                    "<b>Jan Prokop jprokop@ibl.nospam.sk</b>"
                    "<ul>"
                    "<li>Qt dialog</li>"
                    "</ul>"
                    "<b>Christian Zander phoenix@minion.nospam.de</b>"
		    "<br>(Christian only worked on nvclock 0.1 and 0.2)"
                    "<ul>"
                    "<li>cleanups</li>"
                    "<li>autoconf/automake</li>"
                    "<li>restructuring</li>"
                    "</ul>"
                    "</qt>"));
    return(tab);
}
