// file kernel/n/ppc32/mmod.S: operations on residues modulo BASE^n + 1
/*-----------------------------------------------------------------------+
 |  Copyright 2005-2006, Michel Quercia (michel.quercia@prepas.org)      |
 |                                                                       |
 |  This file is part of Numerix. Numerix is free software; you can      |
 |  redistribute it and/or modify it under the terms of the GNU Lesser   |
 |  General Public License as published by the Free Software Foundation; |
 |  either version 2.1 of the License, or (at your option) any later     |
 |  version.                                                             |
 |                                                                       |
 |  The Numerix Library is distributed in the hope that it will be       |
 |  useful, but WITHOUT ANY WARRANTY; without even the implied warranty  |
 |  of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU  |
 |  Lesser General Public License for more details.                      |
 |                                                                       |
 |  You should have received a copy of the GNU Lesser General Public     |
 |  License along with the GNU MP Library; see the file COPYING. If not, |
 |  write to the Free Software Foundation, Inc., 59 Temple Place -       |
 |  Suite 330, Boston, MA 02111-1307, USA.                               |
 +-----------------------------------------------------------------------+
 |                                                                       |
 |                  Arithmtique modulo BASE^n + 1                       |
 |                                                                       |
 +-----------------------------------------------------------------------*/

                   ; +------------------------------------+
                   ; |  Multiplication modulo BASE^n + 1  |
                   ; +------------------------------------+

;  void xn(mmul)(chiffre *a, chiffre *b, long n)
;
;  entre :
;  a = naturel de longueur n+1
;  b = naturel de longueur n+1 non confondu avec a
;
;  contrainte : n > 0
;
;  sortie :
;  a <- (a*b) mod (BASE^n + 1), le chiffre de poids fort vaut 0 ou 1
;  b <- b mod (BASE^n + 1)

#ifdef assembly_sn_mmul
#ifdef debug_mmul
.globl _sn_mmul_buggy
_sn_mmul_buggy:
#else
.globl _sn_mmul
_sn_mmul:
Lsn_mmul:
#endif

#define L(x) Lsn_mmul_##x

        ; normalise a
	slwi   r8,   r5,   2	; r8 <- 4n
	lwzx   r6,   r3,  r8	; r6 <- a[n]
	lwzx   r9,   r4,  r8	; r9 <- b[n]
	lwz    r7, 0(r3)	; a[0..n-1] -= a[n]
	subfc  r7,   r6,  r7
	stw    r7, 0(r3)
	subfe. r6,   r6,  r6
	beq    L(a_ok)
	mr     r6,   r3		; propage la retenue
	addic  r7,   r5,  -1	; r7 <- n-1, CA <- 1
	mtctr  r7
1:
	lwz    r7, 4(r6)
	and.   r7,   r7,  r7
	subi   r7,   r7,   1
	stwu   r7, 4(r6)
	bdnzt  eq,   1b
	bne    L(a_ok)
	subi   r6,   r3,   4	; si elle ressort, ajoute BASE^n + 1
	addic  r7,   r5,  -1	; r7 <- n-1, CA <- 1
	mtctr  r7
2:
	lwz    r7, 4(r6)
	addze. r7,   r7
	stwu   r7, 4(r6)
	bdnzt  eq,   2b
	beq    L(neg_b)		; si a = BASE^n, res = -b
	
        ; normalise b
L(a_ok):
	lwz    r7, 0(r4)	; b[0..n-1] -= b[n]
	subfc  r7,   r9,  r7
	stw    r7, 0(r4)
	li     r9,   0		; r9 <- a[n] (= 0)
	stwx   r9,   r4,  r8	; b[n] <- 0
	subfe. r6,   r6,  r6
	beq    L(b_ok)
	mr     r6,   r4		; propage la retenue
	addic  r7,   r5,  -1	; r7 <- n-1, CA <- 1
	mtctr  r7
1:
	lwz    r7, 4(r6)
	and.   r7,   r7,  r7
	subi   r7,   r7,   1
	stwu   r7, 4(r6)
	bdnzt  eq,   1b
	bne    L(b_ok)
	subi   r6,   r4,   4	; si elle ressort, ajoute BASE^n + 1
	addic  r7,   r5,  -1	; r7 <- n-1, CA <- 1
	mtctr  r7
2:
	lwz    r7, 4(r6)
	addze. r7,   r7
	stwu   r7, 4(r6)
	bdnzt  eq,   2b
	bne    L(b_ok)

        ; ici b = BASE^n, donc le produit vaut -a mod BASE^n + 1
	li     r7,   1		; b[n] <- 1
	stwx   r7,   r4,   r8
	mr     r4,   r3		; r4 <- &a
L(neg_b):
	li     r6,   0		; r6:r9 <- b[n] + 1
	addic  r9,   r9,   1
	addze  r6,   r6
	lwz    r7,  0(r4)	; a <- BASE^n + b[n] + 1 - b[0..n-1]
	subfc  r7,   r7,   r9
	stw    r7,  0(r3)
	lwzu   r7,  4(r4)
	subfe  r7,   r7,   r6
	stwu   r7,  4(r3)
	subi   r7,   r5,    2
	mtctr  r7
1:
	lwzu   r7,  4(r4)
	subfze r7,   r7
	stwu   r7,  4(r3)
	bdnz   1b
	li     r7,    1
	addme  r7,   r7
	stw    r7,  4(r3)
	blr

        ; ici a et b sont normaliss et tiennent sur n chiffres
        ; n est-il divisible par 3 et suffisament grand ?
L(b_ok):
	cmpwi  cr0,  r5,   mmul_lim
        ble    L(simple_mul)
	li     r6,   3
	divwu  r6,   r5,   r6	; r6 <- floor(n/3)
	mulli  r7,   r6,    3
	cmpw   cr0,  r5,   r7
        beq    L(trois)

        ; cas n petit ou non divisible par 3 : multiplication dans N
L(simple_mul):

        ; variables locales
	#define _c_  32(r1)
	#define _a_  r31
	#define _n_  r30
	#define _ra_ r29

	; rserve 2n chiffres + cadre de pile
	stmw   _ra_, 4(r1)
	slwi   r6,   r5,   3
	addi   r6,   r6,  40
	clrrwi r6,   r6,   4	; arrondi  un multiple de 16 octets
	neg    r6,   r6
	stwux  r1,   r1,  r6

	mflr   _ra_		; sauve les paramtres
	mr     _a_,  r3
	mr     _n_,  r8		; n <- 4n

	; c <- a*b
	mr     r6,   r5		; r6 <- n
	mr     r5,   r4		; r5 <- &b
	mr     r4,   r6		; r4 <- n
	la     r7,   _c_
	bl     Lsn_toommul
        
        ; point de chute pour msqr
Lsn_mmul_aux_simple:

	la     r3,   _c_
	srwi   r4,   _n_,   2	; r4 <- n
	add    r5,   r3,   _n_	; r5 <- &c[n]
	srwi   r6,   _n_,   2	; r6 <- n
	mr     r7,   _a_
	bl     Lsn_sub		; a[0..n-1] <- c[0..n-1] - c[n..2n-1]
	li     r4,   0
	stwx   r4,   _a_,  _n_	; a[n] <- 0
	addic. r3,   r3,   -1	; s il n y a pas de retenue, c est fini
	bne    2f
	subi   _a_,  _a_,   4	; sinon, ajoute BASE^n + 1
1:
	lwz    r3, 4(_a_)
	addze. r3,   r3
	stwu   r3, 4(_a_)
	beq    1b
2:
	
	mtlr   _ra_
	lwz    r1,   0(r1)	; nettoie la pile
	lmw    _ra_, 4(r1)
	blr

	#undef _c_
	#undef _a_
	#undef _n_
	#undef _ra_
        
        ; cas n divisible par 3 : multiplie modulo BASE^p + 1
        ; et modulo BASE^(2p) - BASE^p + 1
L(trois):

        ; variables locales
	#define _a_   r31
	#define _b_   r30
	#define _c_   r29
	#define _p_   r28
	#define _add_ r27
	#define _sub_ r26
	#define _ctr_ r25
	#define _ra_  r24

	; rserve 6p chiffres + 8 registres + cadre de pile
	slwi   r7,   r5,   3
	addi   r7,   r7,  72
	clrrwi r7,   r7,   4	; arrondi  un multiple de 16 octets
	neg    r7,   r7
	stwux  r1,   r1,  r7
	stmw   _ra_, 32(r1)
	mflr   _ra_

	; prpare le droulement des boucles internes
	slwi   _p_,  r6,   2	; p <- 4p
	addi   _ctr_, r6,  31	; ctr <- ceil(p/32)
	srwi   _ctr_, _ctr_, 5
	neg    r6,   r6		; r6 <- 4*((-p) % 32)
        clrlslwi r6, r6,  27,2
	subf   _a_,  r6,  r3	; cadre a,b,c sur le multiple de 32 prcdent
	subf   _b_,  r6,  r4
	subf   _c_,  r6,  r1
	addi   _c_,  _c_, 64
	bcl    20,31, L(here)
L(here):
	mflr   _add_
	slwi   r6,   r6,  2	; r6 <- dplacement % entres de boucles
	addi   _sub_, _add_, lo16(Lsn_subloop - L(here))
	addi   _add_, _add_, lo16(Lsn_addloop - L(here))
/*	addis  _sub_, _sub_, ha16(Lsn_subloop - L(here)) */
/*	addis  _add_, _add_, ha16(Lsn_addloop - L(here)) */
	add    _sub_, _sub_, r6
	add    _add_, _add_, r6

        ; dcompose a modulo BASE^(2p) - BASE^p + 1 et BASE^p + 1
        ; c0 <- a0 - a1
	mtlr   _sub_
	mtctr  _ctr_
	mr     r10,  _a_	; r10 <- &a0
	add    r11,  _a_,  _p_	; r11 <- &a1
	mr     r12,  _c_
	subfc  r3,   r3,   r3	; CA <- 1
	blrl
	li     r2,    0		; r2 <- CA, CA <- 0
	addze  r2,   r2

        ; a1 <- a1 + a2
	mtlr   _add_
	mtctr  _ctr_
	add    r10,  _a_,  _p_	; r10 <- &a1
	add    r11,  r10,  _p_	; r11 <- &a2
	add    r12,  _a_,  _p_	; r12 <- &a1
	blrl
	li     r5,   -1		; r5 <- CA - 1, CA <- 1 - CA
	addze  r5,   r5
	srawi  r5,   r5,   1

        ; a0 <- a0 - a2
	mtlr   _sub_
	mtctr  _ctr_
	mr     r10,  _a_	; r10 <- &a0
	add    r11,  _a_,  _p_	; r11 <- &a2
	add    r11,  r11,  _p_
	mr     r12,  _a_
	blrl

	; propage la somme des retenues sur a1 (-1,0,1, pas de dbordement)
	addze  r5,   r5		; r5 <- somme des retenues
	subi   r12,  r12, 4
1:
	lwz    r3, 4(r12)
	addc   r3,   r3,  r5
	stwu   r3, 4(r12)
	addze  r5,   r5
	srawi. r5,   r5,  1
	bne    1b

        ; a2 <- a2 + (a0 - a1)
	mtlr   _add_
	mtctr  _ctr_
	mr     r10,  _c_
	add    r11,  _a_,  _p_	; r11 <- &a2
	add    r11,  r11,  _p_
	mr     r12,  r11	; r12 <- &a2
	subfc  r3,   r3,   r3	; CA <- 1
	blrl
	addze  r2,   r2		; dernier chiffre = somme des retenues
	stw    r2,   0(r12)

        ; dcompose b modulo BASE^(2p) - BASE^p + 1 et BASE^p + 1
        ; c2 <- b0 - b1
	mtlr   _sub_
	mtctr  _ctr_
	mr     r10,  _b_	; r10 <- &b0
	add    r11,  _b_,  _p_	; r11 <- &b1
	add    r12,  _c_,  _p_	; r12 <- &c2
	add    r12,  r12,  _p_
	subfc  r3,   r3,   r3	; CA <- 1
	blrl
	li     r2,    0		; r2 <- CA, CA <- 0
	addze  r2,   r2

        ; c1 <- b1 + b2
	mtlr   _add_
	mtctr  _ctr_
	add    r10,  _b_,  _p_	; r10 <- &b1
	add    r11,  r10,  _p_	; r11 <- &b2
	add    r12,  _c_,  _p_	; r12 <- &c1
	blrl
	li     r5,   -1		; r5 <- CA - 1, CA <- 1 - CA
	addze  r5,   r5
	srawi  r5,   r5,   1

        ; c0 <- b0 - b2
	mtlr   _sub_
	mtctr  _ctr_
	mr     r10,  _b_	; r10 <- &b0
	add    r11,  _b_,  _p_	; r11 <- &b2
	add    r11,  r11,  _p_
	mr     r12,  _c_
	blrl

	; propage la somme des retenues sur c1 (-1,0,1, pas de dbordement)
	addze  r5,   r5		; r5 <- somme des retenues
	subi   r12,  r12, 4
1:
	lwz    r3, 4(r12)
	addc   r3,   r3,  r5
	stwu   r3, 4(r12)
	addze  r5,   r5
	srawi. r5,   r5,  1
	bne    1b

        ; c2 <- b2 + (b0 - b1)
	mtlr   _add_
	mtctr  _ctr_
	add    r10,  _c_,  _p_	; r10 <- &c2
	add    r10,  r10,  _p_
	add    r11,  _b_,  _p_	; r11 <- &b2
	add    r11,  r11,  _p_
	mr     r12,  r10	; r12 <- &c2
	subfc  r3,   r3,   r3	; CA <- 1
	blrl
	addze  r2,   r2		; dernier chiffre = somme des retenues
	stw    r2,   0(r12)

        ; a[2p..3p] <- (a*b) mod BASE^p + 1
	subf   r4,   _p_,  r12	; r4 <- &c2
	subf   r3,   _c_,  r4	; r3 <- &a2
	add    r3,   _a_,  r3
	srwi   r5,   _p_,  2	; r5 <- p
	bl     Lsn_mmul

        ; c[2p..6p-1] <- (a*b) mod (BASE^(2p) - BASE^p + 1), non rduit
	la     r5, 64(r1)	; r5 <- &c0
	subf   r3,   _c_,   r5	; r3 <- &a
	add    r3,   _a_,   r3
	srwi   r4,   _p_,   1	; r4 <- 2p
	srwi   r6,   _p_,   1	; r6 <- 2p
	add    r7,   r5,   _p_	; r7 <- &c2
	add    r7,   r7,   _p_
        bl     Lsn_toommul
        
        ; point de chute pour msqr
Lsn_mmul_aux_trois:

        ; a[0..2p-1] <- (a*b) mod (BASE^(2p) - BASE^p + 1)
	; a1 <- c1 + c2
	mtlr   _add_
	mtctr  _ctr_
	add    _c_,  _c_,  _p_	; c += 2p
	add    _c_,  _c_,  _p_
	add    r10,  _c_,  _p_	; r10 <- &c1
	add    r11,  r10,  _p_	; r11 <- &c2
	add    r12,  _a_,  _p_	; r12 <- &a1
	li     r3,   0		; CA <- 0
	addc   r3,   r3,   r3
	blrl
	subfe  r2,   r2,   r2	; r2 <- ret(c1+c2) - 1

	; c1 <- c2 + c3
	mtlr   _add_
	mtctr  _ctr_
	add    r12,  _c_,  _p_	; r12 <- &c1
	add    r10,  r12,  _p_	; r10 <- &c2
	add    r11,  r10,  _p_	; r11 <- &c3
	blrl
	subfze r2,   r2		; r2 <- ret(c2+c3) - ret(c1+c2)

	; a0 <- c0 - (c2 + c3)
	mtlr   _sub_
	mtctr  _ctr_
	mr     r10,  _c_	; r10 <- &c0
	add    r11,  _c_,  _p_	; r11 <- &c1
	mr     r12,  _a_
	subfc  r3,   r3,   r3	; CA <- 1
	blrl
	subfze  r2,   r2	; r2 <- ret(c1+c2) - ret(c2+c3) - ret(c0-(c2+c3))

	; propage la retenue (-2,-1,0,1) sur a1
	srwi   r3,   _p_,   2	; ctr <- p
	mtctr  r3
	subi   r12,  r12,   4	; r12 <- &a1[-1]
	srwi   r3,   r2,    1	; r3 <- sgn(r2)
	or     r3,   r3,   r2
1:
	lwz    r4, 4(r12)
	addc   r4,   r4,  r2
	stwu   r4, 4(r12)
	addze  r3,   r3
	srawi. r2,   r3,   1
	bdnzf  eq,   1b

	; si la retenue ressort a0++, a1-- (ne peut dborder)
	beq    3f
	subf   r12,  r12, _p_	; r12 <- &a1[-1]
	subf   r11,  r12, _p_	; r11 <- &a0(-1]
1:
	lwz    r3, 4(r12)
	and.   r3,   r3,  r3
	subi   r3,   r3,   1
	stwu   r3, 4(r12)
	beq    1b
2:
	lwz    r3, 4(r11)
	addze. r3,   r3
	stwu   r3, 4(r11)
	bne    1b
3:

        ; a[2p..3p] <- (a[0..2p-1] - a[2p..3p]) mod (BASE^p + 1), normalis
	; a2 <- a0 - a2
	mtlr   _sub_
	mtctr  _ctr_
	mr     r10,  _a_
	add    r11,  _a_,  _p_	; r11 <- &a2
	add    r11,  r11,  _p_
	mr     r12,  r11	; r12 <- &a2
	subfc  r3,   r3,   r3	; CA <- 1
	blrl
	subfe  r2,   r2,   r2	; r2 <- retenue
	
	; a2 <- (a0 - a2) - a1
	mtlr   _sub_
	mtctr  _ctr_
	add    r11,  _a_,  _p_	; r11 <- &a1
	add    r10,  r11,  _p_	; r10 <- &a2
	mr     r12,  r10	; r12 <- &a2
	subfc  r3,   r3,   r3	; CA <- 1
	blrl
	addme  r2,   r2		; r2 += retenue

	; recycle la retenue (0,1,2)
	lwz    r4, 0(r12)	; r4 <- somme des retenues
	subf   r4,   r2,   r4
	li     r3,   0		; a2[p] <- 0
	stw    r3, 0(r12)
	subi   r12,  r11,  4	; r12 <- &a2[-1]
1:
	lwz    r3, 4(r12)
	addc   r3,   r3,  r4
	stwu   r3, 4(r12)
	li     r4,   0
	addze. r4,   r4
	bne    1b
	

; algorithme de division par -3 modulo BASE^p + 1
; -----------------------------------------------
;
; On note a pour a2 et B pour BASE. On a 0 <= a <= B^p et on cherche
; q tel que a = -3q mod (B^p+1) avec 0 <= q <= 2B^p - 1 (ie. le chiffre
; de poids fort de q vaut 0 ou 1). q existe car -3 est premier  B.
;
; Soit m = (B-1)/3 (entier). l algorithme consiste  calculer m*a et  le
; diviser par 1-B (division suivant les puissances de B croissantes).
; On obtient donc b compris entre 0 et B^p-1 et x relatif tels que :
;
;     m*a = (1-B)*b + B^p*x
;
; Les deux premiers termes sont divisibles par m, donc le troisime aussi.
; Soit x = m*y, d o a = -3b + B^p*y. Comme 0 <= a <= B^p et 0 <= b < B^p,
; on a 0 <= y <= 3.
;
; Si y = 0: (a=0 dans ce cas) alors q = b (= 0).
; Si y = 1: a = -3b +  B^p = -3(b + 1 +  m*(1+B+..+B^(p-1))) mod (B^p+1)
; Si y = 2: a = -3b + 2B^p = -3(b + 2 + 2m*(1+B+..+B^(p-1))) mod (B^p+1)
; Si y = 3: a = -3n + 3B^p = -3(b+1) mod (B^p+1).
;
; Calcul de b et y: on note a = sum(a.i*B^i), b = sum(b.i*B^i) et on
; calcule les nombres b.i,c.i,d.i de proche en proche par :
;
;     d.0 = 0
;     m*(a.i + d.i) = b.i + B*c.i    (division euclidienne)
;     d.(i+1) = (b.i + c.i)/m        (entier)
;
; Donc m*(a+d) = b + B*c + m*B^p*(a.p+d.p) et m*d = B*(b+c)
; d o m*a = (1-B)*b + B^p*m*(a.p+d.p) comme annonc, et y = a.p+d.p.
;
; Le calcul de b.i et c.i  partir de a.i et d.i est immdiat. En ce qui
; concerne d.i, on montre par rcurrence que d.i <= 3 et comme m = 1 mod 4,
; on en dduit d.i = (b.i + c.i) mod 4.

        ; a[2p..3p] <- -a[2p..3p]/3 mod BASE^p + 1

	subi   r12,  r11,    4	; r12 <- &a2[-1]
	li     r2,  0x5555	; r2 <- m = (BASE-1)/3
	addis  r2,   r2,   0x5555
	li     r3,   0		; r3 <- reste initial
	srwi   r4,   _p_,    2	; ctr <- p
	mtctr  r4
1:
	lwz    r4, 4(r12)	; r4 <- a.i
	addc   r4,   r4,   r3	; r4 <- (a.i + d.i) mod B
	mulhwu r5,   r4,   r2	; r5 <- high(m*r4)
	mullw  r4,   r4,   r2	; r4 <- b.i
	stwu   r4, 4(r12)
	adde   r3,   r4,   r5	; r3 <- (b.i + c.i) mod 4
	andi.  r3,   r3,    3
	bdnz   1b
	lwz    r4, 4(r12)	; dernier chiffre
	add.   r4,   r4,   r3
	li     r5,    0		; r5 <- 0 (= a2[p])
	beq    6f		; pas de retenue -> fini

	srwi   r3,   _p_,   2	; r3 <- p
	cmpwi  cr0,  r4,   2
	bgt    3f
	blt    1f		; y = 1 => r4 = 1, r2 = m
	add    r2,   r2,   r2	; y = 2 => r4 = 2, r2 = 2m
1:
	add    r4,   r2,   r4	; r4 <- 1er chiffre de retenue
	subi   r3,   r3,    1	; ctr <- p-1
	mtctr  r3
	lwz    r3, 0(r11)
	addc   r3,   r3,   r4	; a2[0] += r4
	stw    r3, 0(r11)
2:
	lwz    r3, 4(r11)	; a2[i] += r2
	adde   r3,   r3,   r2
	stwu   r3, 4(r11)
	bdnz   2b
	b      5f

3:
	subi   r11,  r11,   4	; r11 <- &a2[-1]
	mtctr  r3		; ctr <- p
	subfc  r3,   r3,   r3	; CA <- 1
4:
	lwz    r3, 4(r11)	; a2++
	addze. r3,   r3
	stwu   r3, 4(r11)
	bdnzt  eq,   4b
5:	
	addze  r5,   r5		; r5 <- retenue finale
6:
	stw    r5, 4(r12)	; a2[p] <- retenue
     
        ; a <- a - (BASE^p - 1)*a[2p..3p]
	; a0 <- a0 + a2
	mtlr   _add_
	mtctr  _ctr_
	mr     r10,  _a_
	add    r11,  _a_,  _p_	; r11 <- &a2
	add    r11,  r11,  _p_
	mr     r12,  _a_
	li     r3,   0		; CA <- 0
	addc   r3,   r3,   r3
	blrl
	addze  r2,   r5		; r2 <- CA + a2(p]

	; a1 <- a1 - a2
	mtlr   _sub_
	mtctr  _ctr_
	add    r10,  _a_,  _p_	; r10 <- &a1
	add    r11,  r10,  _p_	; r11 <- &a2
	add    r12,  _a_,  _p_	; r12 <- &a1
	subfc  r3,   r3,   r3	; CA <- 1
	blrl
	lwz    r3, 0(r10)	; propage la retenue
	subfe  r3,   r5,   r3
	stw    r3, 0(r10)
	subfe. r3,   r3,   r3
	beq    2f
1:
	lwz    r3, 4(r10)
	and.   r3,   r3,   r3
	subi   r3,   r3,    1
	stwu   r3, 4(r10)
	beq    1b
2:

	; propage la retenue sur a1
	subf   r12,  _p_,  r12	; r12 <- &a1
	lwz    r3, 0(r12)
	addc   r3,   r2,   r3
	stw    r3, 0(r12)
	subfe. r3,   r3,   r3
	bne    2f
1:
	lwz    r3, 4(r12)
	addze. r3,   r3
	stwu   r3, 4(r12)
	beq    1b
2:
	
        ; termin
	mtlr   _ra_
	lmw    _ra_, 32(r1)
	lwz    r1,    0(r1)
	blr

	#undef _a_
	#undef _b_
	#undef _c_
	#undef _p_
	#undef _add_
	#undef _sub_
	#undef _ctr_
	#undef _ra_

#undef L
#endif /* assembly_sn_mmul */
#if !defined(assembly_sn_mmul) || defined(debug_mmul)
REPLACE(sn_mmul)
#endif



                        ; +---------------------------+
                        ; |  Carr modulo BASE^n + 1  |
                        ; +---------------------------+

;  void xn(msqr)(chiffre *a, long n)
;
;  entre :
;  a = naturel de longueur n+1
;
;  contrainte : n > 0
;
;  sortie :
;  a <- a^2 mod (BASE^n + 1), le chiffre de poids fort vaut 0 ou 1

#ifdef assembly_sn_msqr
#ifdef debug_mmul
.globl _sn_msqr_buggy
_sn_msqr_buggy:
#else
.globl _sn_msqr
_sn_msqr:
Lsn_msqr:
#endif

#define L(x) Lsn_msqr_##x

        ; normalise a
	slwi   r8,   r4,   2	; r8 <- 4n
	lwzx   r6,   r3,  r8	; r6 <- a[n]
	lwz    r7, 0(r3)	; a[0..n-1] -= a[n]
	subfc  r7,   r6,  r7
	stw    r7, 0(r3)
	subfe. r6,   r6,  r6
	beq    L(a_ok)
	mr     r6,   r3		; propage la retenue
	addic  r7,   r4,  -1	; r7 <- n-1, CA <- 1
	mtctr  r7
1:
	lwz    r7, 4(r6)
	and.   r7,   r7,  r7
	subi   r7,   r7,   1
	stwu   r7, 4(r6)
	bdnzt  eq,   1b
	bne    L(a_ok)
	subi   r6,   r3,   4	; si elle ressort, ajoute BASE^n + 1
2:
	lwz    r7, 4(r6)
	addze. r7,   r7
	stwu   r7, 4(r6)
	bdnzt  eq,   2b
	bne    L(a_ok)

	; si a = BASE^n, res = 1
	stw    r7, 4(r6)	; a[n] <- 0
	li     r4,   1
	stw    r4, 0(r3)
	blr
	
        ; ici a est normalis et tient sur n chiffres
        ; n est-il divisible par 3 et suffisament grand ?
L(a_ok):
	cmpwi  cr0,  r4,   msqr_lim
        ble    L(simple_sqr)
	li     r6,   3
	divwu  r6,   r4,   r6	; r6 <- floor(n/3)
	mulli  r7,   r6,    3
	cmpw   cr0,  r4,   r7
        beq    L(trois)

        ; cas n petit ou non divisible par 3 : carr dans N
L(simple_sqr):

        ; variables locales
	#define _c_  32(r1)
	#define _a_  r31
	#define _n_  r30
	#define _ra_ r29

	; rserve 2n chiffres + cadre de pile
	stmw   _ra_, 4(r1)
	slwi   r6,   r4,   3
	addi   r6,   r6,  40
	clrrwi r6,   r6,   4	; arrondi  un multiple de 16 octets
	neg    r6,   r6
	stwux  r1,   r1,  r6

	mflr   _ra_		; sauve les paramtres
	mr     _a_,  r3
	mr     _n_,  r8		; n <- 4n

	; c <- a^2
	la     r5,   _c_
	bl     Lsn_toomsqr
	b      Lsn_mmul_aux_simple ; continue vec mmul

	#undef _c_
	#undef _a_
	#undef _n_
	#undef _ra_
        
        ; cas n divisible par 3 : lve au carr modulo BASE^p + 1
        ; et modulo BASE^(2p) - BASE^p + 1
L(trois):

        ; variables locales
	#define _a_   r31
	#define _b_   r30
	#define _c_   r29
	#define _p_   r28
	#define _add_ r27
	#define _sub_ r26
	#define _ctr_ r25
	#define _ra_  r24

	; rserve 6p chiffres + 8 registres + cadre de pile
	slwi   r7,   r4,   3
	addi   r7,   r7,  72
	clrrwi r7,   r7,   4	; arrondi  un multiple de 16 octets
	neg    r7,   r7
	stwux  r1,   r1,  r7
	stmw   _ra_, 32(r1)
	mflr   _ra_

	; prpare le droulement des boucles internes
	slwi   _p_,  r6,   2	; p <- 4p
	addi   _ctr_, r6,  31	; ctr <- ceil(p/32)
	srwi   _ctr_, _ctr_, 5
	neg    r6,   r6		; r6 <- 4*((-p) % 32)
        clrlslwi r6, r6,  27,2
	subf   _a_,  r6,  r3	; cadre a,c sur le multiple de 32 prcdent
	subf   _c_,  r6,  r1
	addi   _c_,  _c_, 64
	bcl    20,31, L(here)
L(here):
	mflr   _add_
	slwi   r6,   r6,  2	; r6 <- dplacement % entres de boucles
	addi   _sub_, _add_, lo16(Lsn_subloop - L(here))
	addi   _add_, _add_, lo16(Lsn_addloop - L(here))
/*	addis  _sub_, _sub_, ha16(Lsn_subloop - L(here)) */
/*	addis  _add_, _add_, ha16(Lsn_addloop - L(here)) */
	add    _sub_, _sub_, r6
	add    _add_, _add_, r6

        ; dcompose a modulo BASE^(2p) - BASE^p + 1 et BASE^p + 1
        ; c0 <- a0 - a1
	mtlr   _sub_
	mtctr  _ctr_
	mr     r10,  _a_	; r10 <- &a0
	add    r11,  _a_,  _p_	; r11 <- &a1
	mr     r12,  _c_
	subfc  r3,   r3,   r3	; CA <- 1
	blrl
	li     r2,    0		; r2 <- CA, CA <- 0
	addze  r2,   r2

        ; a1 <- a1 + a2
	mtlr   _add_
	mtctr  _ctr_
	add    r10,  _a_,  _p_	; r10 <- &a1
	add    r11,  r10,  _p_	; r11 <- &a2
	add    r12,  _a_,  _p_	; r12 <- &a1
	blrl
	li     r5,   -1		; r5 <- CA - 1, CA <- 1 - CA
	addze  r5,   r5
	srawi  r5,   r5,   1

        ; a0 <- a0 - a2
	mtlr   _sub_
	mtctr  _ctr_
	mr     r10,  _a_	; r10 <- &a0
	add    r11,  _a_,  _p_	; r11 <- &a2
	add    r11,  r11,  _p_
	mr     r12,  _a_
	blrl

	; propage la somme des retenues sur a1 (-1,0,1, pas de dbordement)
	addze  r5,   r5		; r5 <- somme des retenues
	subi   r12,  r12, 4
1:
	lwz    r3, 4(r12)
	addc   r3,   r3,  r5
	stwu   r3, 4(r12)
	addze  r5,   r5
	srawi. r5,   r5,  1
	bne    1b

        ; a2 <- a2 + (a0 - a1)
	mtlr   _add_
	mtctr  _ctr_
	mr     r10,  _c_
	add    r11,  _a_,  _p_	; r11 <- &a2
	add    r11,  r11,  _p_
	mr     r12,  r11	; r12 <- &a2
	subfc  r3,   r3,   r3	; CA <- 1
	blrl
	addze  r2,   r2		; dernier chiffre = somme des retenues
	stw    r2,   0(r12)

        ; a[2p..3p] <- a^2 mod BASE^p + 1
	subf   r3,   _p_,  r12	; r4 <- &a2
	srwi   r4,   _p_,  2	; r4 <- p
	bl     Lsn_msqr

        ; c[2p..6p-1] <- a^2 mod (BASE^(2p) - BASE^p + 1), non rduit
	la     r5, 64(r1)	; r5 <- &c0
	subf   r3,   _c_,   r5	; r3 <- &a
	add    r3,   _a_,   r3
	srwi   r4,   _p_,   1	; r4 <- 2p
	add    r5,   r5,   _p_	; r5 <- &c2
	add    r5,   r5,   _p_
        bl     Lsn_toomsqr
	b      Lsn_mmul_aux_trois ; continue avec mmul

	#undef _a_
	#undef _b_
	#undef _c_
	#undef _p_
	#undef _add_
	#undef _sub_
	#undef _ctr_
	#undef _ra_

#undef L
#endif /* assembly_sn_msqr */
#if !defined(assembly_sn_msqr) || defined(debug_mmul)
REPLACE(sn_msqr)
#endif


                      ; +------------------------------+
                      ; |  Papillon modulo BASE^n + 1  |
                      ; +------------------------------+

; void xn(butterfly1)(chiffre *a, chiffre *b, long n, long q, int s)
;
;  entre :
;  a = naturel de longueur n+1
;  b = naturel de longueur n+1 non confondu avec a
;  q = entier positif ou nul
;  s = 0 ou 1
;
;  contraintes : n >= 3 et si q est impair, n doir tre pair
;
;  sortie :
;  a <- a + (-1)^s * b * 2^(q/2) mod (BASE^n + 1)
;  b <- a - (-1)^s * b * 2^(q/2) mod (BASE^n + 1)
;
;  remarque : 2^(1/2) = BASE^(3n/4)*(BASE^(n/2) + 1) mod (BASE^n + 1)

#ifdef assembly_sn_butterfly
#define L(x) .Lsn_butterfly_##x
#ifdef debug_butterfly
.globl _sn_butterfly_buggy
_sn_butterfly_buggy:
#else
.globl _sn_butterfly
_sn_butterfly:
#endif
	; variables locales
	#define _a_ r3
	#define _b_ r4
	#define _n_ r5
	#define _q_ r6
	#define _s_ r7

        ; force 2 <= a[n] <= BASE-3 pour absorber les retenues
	slwi   r8,   _n_,  2	; r8 <- 4n
	lwzx   r9,   _a_,  r8	; r9 <- a[n]
	addi   r10,  r9,   2	; a[n] >= BASE-2 ?
	cmplwi cr0,  r10,  2
	bge    2f
	subi   r9,   r9,   2	; si oui, retranche 2*(BASE^n + 1)
	stwx   r9,   _a_,  r8
	lwz    r9,   0(_a_)
	subic  r9,   r9,   2
	stw    r9,   0(_a_)
	subfe. r9,   r9,   r9	; propage la retenue
	beq    4f
	mr     r10,  _a_
1:
	lwzu   r9,   4(r10)
	and.   r9,   r9,   r9
	subi   r9,   r9,   1
	stw    r9,   0(r10)
	beq    1b
	b      4f
	
2:
	cmplwi cr0,  r9,   2	; a[n] >= 2 ?
	bge    4f
	stwx   r10,  _a_,  r8	; sinon, ajoute 2*(BASE^n + 1)
	lwz    r9,   0(_a_)
	addic  r9,   r9,   2
	stw    r9,   0(_a_)
	subfe. r9,   r9,   r9	; propage la retenue
	beq    4f
	mr     r10,  _a_
3:
	lwzu   r9,   4(r10)
	addze. r9,   r9
	stw    r9,   0(r10)
	beq    3b
4:

	andi.  r9,   _q_,  1
        srawi  _q_,  _q_,  1	; q <- q/2
        beq    L(sqrt_2_done)

        ; q est impair, il faut multiplier b par (BASE^(n/2) + 1)

	slwi   r8,   _n_,  1	; r8 <- 2n 
	add    r9,   r8,   _n_
	slwi   r9,   r9,   3
	add    _q_,  _q_,  r9	; q <- q/2 + 24*n
	addi   r2,   r8,   16	; rserve n/2 + 1 chiffres dans la pile
	clrrwi r2,   r2,   4	; arrondi  un multiple de 16 octets
	subf   r1,   r2,   r1

        ; x <- haut, haut <- haut + bas
	subi   r10,  _b_,  4	; r10 <- &b[-1]
	add    r11,  r10,  r8	; r11 <- &b[n/2-1]
	subi   r12,  r1,   4	; r12 <- &x[-1]
	srawi  r8,   _n_,  1	; r8 <- n/2, CA <- 0
	mtctr  r8		; ctr <- n/2
1:
	lwzu   r8,   4(r10)	; additionne les chiffres communs
	lwzu   r9,   4(r11)
	stwu   r9,   4(r12)
	adde   r8,   r8,   r9
	stw    r8,   0(r11)
	bdnz   1b
	lwz    r9,   4(r11)	; b[n] += retenue
	stw    r9,   4(r12)
	addze  r9,   r9
	subfe  r8,   r8,   r8	; CA <- 1 - CA
	srawi  r8,   r8,   1
	addme  r9,   r9		; b[n] <- retenue
	stw    r9,   4(r11)

	; bas <- bas - haut
	subi   r10,  _b_,  4	; r10 <- &b[-1]
	subi   r11,  r1,   4	; r11 <- &x[-1]
	srwi   r8,   _n_,  1	; r8 <- n/2+1
	addi   r8,   r8,   1
	mtctr  r8
1:
	lwzu   r8,   4(r10)	; retranche les chiffres communs
	lwzu   r9,   4(r11)
	subfe  r8,   r9,   r8
	stw    r8,   0(r10)
	bdnz   1b
	subfe. r8,   r8,   r8	; propage la retenue
	beq    3f
2:
	lwzu   r8,   4(r10)
	and.   r8,   r8,   r8
	subi   r8,   r8,   1
	stw    r8,   0(r10)
	beq    2b
3:
	add    r1,   r2,   r1	; nettoie la pile
L(sqrt_2_done):

	; dcompose le dcalage en nombre et fraction de chiffre
	andi.  r2,   _q_,  31
	srwi   _q_,  _q_,  5
1:
	subf.  _q_,  _n_,  _q_	; rduit modulo n
	xori   _s_,  _s_,  1
	bge    1b
	add    _q_,  _n_,  _q_

        ; b <- b*2^(q mod 32)  mod (BASE^n + 1), normalis
	and.   r2,   r2,   r2
	beq    L(norm)
	subi   r12,  r2,   32	; r12 <- 32-n
	neg    r12,  r12
	subi   r10,  _b_,  4	; r10 <- &b[-1]
	li     r8,   0		; r8 <- retenue
	mtctr  _n_
1:
	lwzu   r9,   4(r10)	; dcalage de r2 bits
	slw    r11,  r9,   r2
	or     r11,  r8,   r11
	srw    r8,   r9,   r12
	stw    r11,  0(r10)
	bdnz   1b
	lwz    r9,   4(r10)	; r9:r11 <- retenue  recycler
	slw    r11,  r9,   r2
	or     r11,  r8,   r11
	srw    r9,   r9,   r12
	addi   r10,  _b_,  4	; r10 <- &b[1]
	lwz    r8,   0(_b_)
	subfc  r8,   r11,  r8	; b[0] -= r11
	stw    r8,   0(_b_)
	subi   r8,   _n_,  2	; reste n-2 chiffres  traiter
	b      L(norm1)
	
        ; normalise b
L(norm):
	slwi   r8,   _n_,  2	; r8 <- 4n
	lwzx   r9,   _b_,  r8	; r9 <- b[n]
	mr     r10,  _b_
	subic  r8,   _n_,  1	; CA <- 1
L(norm1):
	mtctr  r8		; ctr <- n-1
	lwz    r8,   0(r10)	; b[0] -= b[n]
	subfe  r8,   r9,   r8
	stw    r8,   0(r10)
	subfe. r8,   r8,   r8	; propage la retenue
	beq    L(b_ok)
1:
	lwzu   r8,   4(r10)
	and.   r8,   r8,   r8
	subi   r8,   r8,   1
	stw    r8,   0(r10)
	bdnzt  eq,   1b
	bne    L(b_ok)
	subi   r10,  _b_,   4	; si elle ressort, ajoute BASE^n + 1
	mtctr  _n_
2:
	lwzu   r8,   4(r10)
	addi   r8,   r8,    1
	and.   r8,   r8,    r8
	stw    r8,   0(r10)
	bdnzt  eq,   2b
	bne    L(b_ok)

        ; cas b = BASE^n
	addi   r8,   _n_,   1	; b <- a
	mtctr  r8
	subi   r10,  _a_,   4
	subi   r11,  _b_,   4
1:
	lwzu   r8,   4(r10)
	stwu   r8,   4(r11)
	bdnz   1b
	subi   _q_,  _q_,   1
	slwi   _q_,  _q_,   2
	add    _a_,  _a_,   _q_	; a <- &a[q-1]
	add    _b_,  _b_,   _q_	; b <- &b[q-1]
	subi   r8,   _s_,   1
	xor    r9,   _a_,  _b_
	and    r9,   r8,   r9
	xor    r10,  r9,   _a_	; r10 <- (s) ? a : b
	xor    r11,  r9,   _b_	; r11 <- (s) ? b : a

1:
	lwzu   r8,   4(r10)	; a -= BASE^q
	and.   r8,   r8,   r8
	subi   r8,   r8,   1
	stw    r8,   0(r10)
	beq    1b

1:
	lwzu   r8,   4(r11)	; b += BASE^q
	addi   r8,   r8,   1
	and.   r8,   r8,   r8
	stw    r8,   0(r11)
	beq    1b
	blr

        ; ici on est sr que b tient sur n chiffres 
        ; x <- b*BASE^q mod (BASE^n + 1) 
L(b_ok):

	slwi   r2,   _n_,  2	; rserve n+1 chiffres dans la pile
	addi   r2,   r2,   16	; arrondi  un multiple de 16 octets
	clrrwi r2,   r2,   4
	subf   r1,   r2,   r1

	subfc  r8,   _q_,  _n_	; r8 <- n-q, CA <- 1
	subi   r10,  _b_,  4	; r10 <- &b[-1]
	slwi   r9,   _q_,  2
	subi   r9,   r9,   4
	add    r11,  r1,   r9	; r11 <- &x[q-1]
	
	mtctr  r8		; recopie le bas
1:
	lwzu   r9,   4(r10)
	stwu   r9,   4(r11)
	bdnz   1b
	li     r9,   0		; x[n] <- 0
	stw    r9,   4(r11)

	and.   _q_,  _q_,  _q_	; retranche le haut
	beq    L(x_ok)
	subi   r11,  r1,   4	; r11 <- &x[-1]
	mtctr  _q_
1:
	lwzu   r9,   4(r10)
	subfze r9,   r9
	stwu   r9,   4(r11)
	bdnz   1b
	
	subfe. r9,   r9,   r9	; propage la retenue
	beq    L(x_ok)
	mtctr  r8
1:
	lwzu   r9,   4(r11)
	and.   r9,   r9,   r9
	subi   r9,   r9,   1
	stw    r9,   0(r11)
	bdnzt  eq,   1b
	bne    L(x_ok)
	
	subi   r11,  r1,   4	; si elle ressort, ajoute 1
	mtctr  _n_
1:
	lwzu   r9,   4(r11)
	addi   r9,   r9,   1
	and.   r9,   r9,   r9
	stw    r9,   0(r11)
	beq    1b
L(x_ok):

	; prpare le droulement de l'addition et de la soustraction
        mflr   r0		; r0  <- adresse de retour
	addi   _n_,  _n_,  1
	neg    r8,   _n_
	addi   _n_,  _n_,  31
	srawi  _n_,  _n_,  5	; n <- ceil((n+1)/32)
        clrlslwi r8, r8,   27,2 ; r8 <- 4*((-n-1) % 32)
	subf   r10,  r8,   _a_  ; cadre a,b,x sur le multiple de 32 prcdent
	subf   r11,  r8,   r1
	subf   r12,  r8,   _b_
        slwi   _a_,  r8,   2	; r8 <- point d entre pour l addition
	bcl    20,31, L(here)	; r9 <- point d entre pour la soustraction 
L(here):
	mflr   r9
	add    r9,   r9,   _a_
	addi   r8,   r9, lo16(Lsn_addloop - L(here))
/*	addis  r8,   r8, ha16(Lsn_addloop - L(here)) */
	addi   r9,   r9, lo16(Lsn_subloop - L(here))
/*	addis  r9,   r9, ha16(Lsn_subloop - L(here)) */

	neg    _s_,  _s_	; permute les adresses si s = 1
	xor    _a_,  r8,   r9
	and    _a_,  _s_,  _a_
	xor    r8,   _a_,  r8
	xor    r9,   _a_,  r9

	mtctr  _n_
	addic  _s_,  _s_,  1	; init retenue
	mtlr   r8
	mr     r8,   r10	; sauve les adresses de a et x
	mr     _q_,  r11
	blrl			; b <- a + (-1)^s*x

	mtctr  _n_
	subic  _s_,  _s_,  1	; init retenue
	mtlr   r9
	mr     r10,  r8		; r10 <- &a
	mr     r11,  _q_	; r11 <- &x
	mr     r12,  r8		; r8 <- &a
	blrl			; a <- a - (-1)^s*x

	; termin
	add    r1,   r2,   r1	;  nettoie la pile
        mtlr   r0
	blr
	
	#undef  _a_
	#undef  _b_
	#undef  _n_
	#undef  _q_
	#undef  _s_

#undef L
#endif /* assembly_sn_butterfly */
