(* $Id: netbuffer.mli,v 1.3 2000/06/25 22:34:43 gerd Exp $
 * ----------------------------------------------------------------------
 *
 *)


(* A Netbuffer.t is a buffer that can grow and shrink dynamically. *)

type t

val create : int -> t
    (* Creates a netbuffer which allocates initially this number of bytes. 
     * The logical length is zero.
     *)

val contents : t -> string
    (* Returns the contents of the buffer as fresh string. *)

val sub : t -> pos:int -> len:int -> string
    (* sub nb k n: returns the n characters starting at position n from 
     * netbuffer nb as fresh string
     *)

val length : t -> int
    (* Returns the logical length of the buffer *)

val add_string : t -> string -> unit
    (* add_string nb s: Adds a copy of the string s to the logical end of
     * the netbuffer nb. If necessary, the nb grows.
     *)

val add_sub_string : t -> string -> pos:int -> len:int -> unit
    (* add_sub_string nb s k n: Adds the substring of s starting at position
     * k with length n to the logical end of the netbuffer nb. If necessary,
     * the nb grows.
     * This is semantically the same as
     * add_string nb (String.sub s k n), but the extra copy is avoided.
     *)

val delete : t -> pos:int -> len:int -> unit
    (* delete nb k n: Deletes the n bytes at position k of netbuffer nb
     * in-place.
     * The netbuffer does not shrink!
     *)

val clear : t -> unit
    (* deletes all contents from the buffer. As 'delete', the netbuffer does
     * not shrink.
     *)

val try_shrinking : t -> unit
    (* try_shrinking nb: If the length of the buffer is less than half of
     * the allocated space, the netbuffer is reallocated in order to save
     * memory.
     *)

val index_from : t -> int -> char -> int
    (* index_from nb k c: Searches the character c in the netbuffer beginning
     * at position k. If found, the position of the left-most occurence is
     * returned. Otherwise, Not_found is raised.
     *)

val unsafe_buffer : t -> string
    (* WARNING! This is a low-level function!
     * Returns the current string that internally holds the buffer.
     * The byte positions 0 to length - 1 actually store the contents of
     * the buffer. You can directly read and modify the buffer. Note that
     * there is no protection if you read or write positions beyond the
     * length of the buffer.
     *)

val print_buffer : t -> unit
    (* For the toploop *)


(* ======================================================================
 * History:
 * 
 * $Log: netbuffer.mli,v $
 * Revision 1.3  2000/06/25 22:34:43  gerd
 * 	Added labels to arguments.
 *
 * Revision 1.2  2000/06/24 20:20:33  gerd
 * 	Added the toploop printer.
 *
 * Revision 1.1  2000/04/15 13:07:48  gerd
 * 	Initial revision.
 *
 * 
 *)
