/*
 * user.c  -  user interface for makerom bootrom configuration utility
 *
 * Copyright (C) 1995-1998 Gero Kuhlmann <gero@gkminix.han.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#define NO_BINARY 1	/* No need to include binary support here */
#include "common.h"
#include "nblib.h"
#include "makerom.h"



/*
 * Local declarations
 */
#define DEFAULTINT	0x62
#define PROGNUM		(MAXDRIVERS - 1)



/*
 * Directory entry name list
 */
static struct direntry {
	struct direntry *next;
	char             fname[1];
} *dirlist = NULL;



/*
 * Elements of table with descriptions and directory pointers
 */
struct desctab {
	struct direntry *dp;
	char            *desc;
};



/*
 * Variables local to this module
 */
static struct bootdef bootd;
static char *kernelname = NULL;
static char *netdrvname = NULL;
static char *netdrvoptions = NULL;
static char *prognames[PROGNUM];
static char *progoptions[PROGNUM];
static int num_progs = 0;
static int is86 = 0;
static int isminimal = 0;
static int isdefint = 0;
static int useint18 = 0;
static int outtype = OUT_BINARY;



/*
 * Compare a description table entry for qsort routine
 */
static int descmp(entry1, entry2)
struct desctab *entry1;
struct desctab *entry2;
{
  return(strcmp(entry1->desc, entry2->desc));
}



/*
 * Compare a directory entry against a pattern.
 */
static int dircmp(pattern, name)
char *pattern;
char *name;
{
  while (*pattern && *name) {
	if (*pattern == '?' || *pattern == *name) {
		pattern++;
		name++;
	} else {
		if (*pattern != '*')
			break;
		pattern++;
		while (*name && *name != *pattern)
			name++;
	}
  }
  return(!*pattern && !*name);
}



/*
 * Get a list of directory entries with a specified name pattern.
 */
static int getdirlist(dirname, pattern)
char *dirname;
char *pattern;
{
  DIR *dir;
  struct dirent *ent;
  struct direntry *dp;
  int count;
  size_t len;
  char *src, *dest;

  /* First open the directory */
  if ((dir = opendir(dirname)) == NULL) {
	fprintf(stderr, "%s: cannot open directory %s\n", progname, dirname);
	exit(EXIT_OPENDIR);
  }

  /* Next step through all directory entries */
  count = 0;
  while ((ent = readdir(dir)) != NULL) {
	if (!dircmp(pattern, ent->d_name))
		continue;
	len = strlen(ent->d_name) + sizeof(struct direntry);
	dp = (struct direntry *)nbmalloc(len);
	for (src = ent->d_name, dest = dp->fname; *src; src++, dest++)
		*dest = tolower(*src);
	*dest = '\0';
	dp->next = dirlist;
	dirlist = dp;
	count++;
  }
  closedir(dir);
  return(count);
}



/*
 * Delete directory list
 */
static void deldirlist()
{
  struct direntry *dp1, *dp2;

  dp1 = dirlist;
  while (dp1 != NULL) {
	dp2 = dp1;
	dp1 = dp1->next;
	free(dp2);
  }
  dirlist = NULL;
}



/*
 * Create the bootrom configuration
 */
static void buildconfig()
{
  char *cp, *buf;
  int i;
  size_t len;

  memset(&bootd, 0, sizeof(bootd));

  bootd.name = strdup("default");
  bootd.kernelname = kernelname;
  bootd.use_int18 = useint18;

  cp = is86 ? "floppy86.bin" : "floppy.bin";
  len = strlen(kerneldir) + strlen(cp) + 2;
  buf = nbmalloc(len);
  sprintf(buf, "%s/%s", kerneldir, cp);
  bootd.loadernames[0] = buf;
  bootd.outtypes[0] = OUT_BINARY;
  bootd.outnames[0] = NULL;
  copystr(&bootd.outnames[0], "image.flo");

  cp = is86 ? "rom86.bin" : "rom.bin";
  len = strlen(kerneldir) + strlen(cp) + 2;
  buf = nbmalloc(len);
  sprintf(buf, "%s/%s", kerneldir, cp);
  bootd.loadernames[1] = buf;
  bootd.outtypes[1] = outtype;
  bootd.outnames[1] = NULL;
  switch (outtype) {
	case OUT_FLASH:
		copystr(&bootd.outnames[1], "image.flash");
		break;
	case OUT_IHEX:
	case OUT_MHEX:
	case OUT_THEX:
		copystr(&bootd.outnames[1], "image.hex");
		break;
	case OUT_BINARY:
	default:
		copystr(&bootd.outnames[1], "image.rom");
		break;
  }
  bootd.loadernum = 2;

  bootd.drivernames[0] = netdrvname;
  bootd.driverargs[0] = netdrvoptions;
  bootd.drivertypes[0] = TYPE_PD;
  for (i = 0; i < num_progs; i++) {
	bootd.drivernames[i+1] = prognames[i];
	bootd.driverargs[i+1] = progoptions[i];
	bootd.drivertypes[i+1] = TYPE_DOS;
  }
  bootd.drivernum = num_progs + 1;
}



/*
 * Find program description according to it's file name
 */
static struct desclist *findfname(dlist, fname)
struct desclist **dlist;
char *fname;
{
  struct desclist *ditem = *dlist;

  while (ditem) {
	if (!strcmp(ditem->filename, fname))
		return(ditem);
	ditem = ditem->next;
  }
  return(NULL);
}



/*
 * Get a string response from the user
 */
static char *getstring()
{
  static char buf[1024];
  char *cp;

  fflush(stdout);
  fgets(buf, sizeof(buf) - 1, stdin);
  for (cp = buf; *cp && *cp != '\n'; cp++) ;
  *cp = '\0';
  return(buf);
}



/*
 * Get a numeric response from the user
 */
static long getnum(prompt, min, max, hex)
char *prompt;
long min;
long max;
int hex;
{
  long j;
  char *cp;

  j = min - 1;
  while (j < min || j > max) {
	printf("  %s (%s): ", prompt, hex ? "hex" : "decimal");
	cp = getstring();
	if ((hex ? sscanf(cp, "%lx", &j) : sscanf(cp, "%ld", &j)) != 1)
		j = min - 1;
	if (j < min || j > max)
		if (hex)
			printf("  Allowed values are [0x%lx..0x%lx]\n",
								min, max);
		else
			printf("  Allowed values are [%ld..%ld]\n",
								min, max);
  }
  return(j);
}



/*
 * Get a list selection from the user
 */
static int getsel(prompt, minsel, maxsel, defsel)
char *prompt;
int minsel;
int maxsel;
int defsel;
{
  int sel = minsel - 1;
  char *cp;

  while (sel < minsel || sel > maxsel) {
	printf(prompt);
	if (defsel >= minsel && defsel <= maxsel) {
		printf(" [%d]: ", defsel);
		cp = getstring();
		if (!*cp)
			sel = defsel;
		else
			sscanf(cp, "%d", &sel);
	} else {
		printf(": ");
		cp = getstring();
		sscanf(cp, "%d", &sel);
	}
  }
  return(sel);
}



/*
 * Get a yes/no response from the user
 */
static int getyn(prompt, def)
char *prompt;
int def;
{
  char *cp;

  while (TRUE) {
	printf("%s (y/n) [%s] ? ", prompt, def ? "yes" : "no" );
	cp = getstring();
	if (*cp == 'y' || *cp == 'Y')
		return(TRUE);
	else if (*cp == 'n' || *cp == 'N')
		return(FALSE);
	else if (!*cp)
		return(def);
  }
  return(def);
}



/*
 * Get packet driver options
 */
static void getpktopt(descp)
struct desclist *descp;
{
  int pktopt = descp->options;
  char *cmdlp = descp->cmdline;
  long hwirq = 0L;
  long ioaddr = 0L;
  long basemem = 0L;
  long dmanum = 0L;
  long auitype = 0L;
  char *cp;

  /* Ask the user for required driver values */
  isdefint++;
  if (pktopt) {
	printf("Enter network driver command line options:\n");
	if (pktopt & HW_IRQ)
		hwirq = getnum("Hardware IRQ number", 2, 15, FALSE);
	if (pktopt & IO_ADDR)
		ioaddr = getnum("I/O address", 0, 0xffff, TRUE);
	if (pktopt & BASE_MEM)
		basemem = getnum("Shared memory address", 0, 0xffff, TRUE);
	if (pktopt & DMA_NUM)
		dmanum = getnum("DMA number", 0, 7, FALSE);
	if (pktopt & AUI_TYPE)
		auitype = getnum("AUI type (-1 for default)", -1, 1, FALSE);
  }

  /* Generate driver command line */
  cp = netdrvoptions = nbmalloc(1024);
  if (cmdlp == NULL)
	cmdlp = "0x%S";
  while (*cmdlp) {
	if (*cmdlp != '%') {
		*cp++ = *cmdlp++;
		continue;
	}
	if (!*(++cmdlp))
		break;
	switch (*cmdlp) {
		case 'S':
			sprintf(cp, "%x", DEFAULTINT);
			break;
		case 'H':
			sprintf(cp, "%ld", hwirq);
			break;
		case 'A':
			sprintf(cp, "%lx", ioaddr);
			break;
		case 'D':
			sprintf(cp, "%ld", dmanum);
			break;
		case 'T':
			if (auitype > 0)
				sprintf(cp, "%ld", auitype);
			break;
		case '%':
			cp[0] = '%';
			cp[1] = '\0';
			break;
		default:
			break;
	}
	while (*cp)
		cp++;
	cmdlp++;
  }
  *cp = '\0';
}



/*
 * Let the user select a packet driver.
 */
static void getnetdrv()
{
  struct direntry *dp;
  struct desclist *descp;
  struct desctab *dtab;
  int num, sel, i;
  size_t len;
  char *cp;

  /* Determine which drivers are available */
  num = getdirlist(netdrvdir, "*.com");
  num += getdirlist(netdrvdir, "*.exe");
  if (num == 0) {
	fprintf(stderr, "%s: Oops, no network drivers found in %s!\n",
							progname, netdrvdir);
	exit(EXIT_MAKEROM_NETFND);
  }

  /* Generate list of descriptions and sort it alphabetically */
  dtab = (struct desctab *)nbmalloc(sizeof(struct desctab) * num);
  for (dp = dirlist, i = 0; dp != NULL && i < num; dp = dp->next, i++) {
	if ((descp = findfname(&netdrvdesc, dp->fname)) == NULL)
		cp = dp->fname;
	else if (descp->descript == NULL)
		cp = descp->progname;
	else
		cp = descp->descript;
	dtab[i].dp = dp;
	dtab[i].desc = cp;
  }
  qsort(dtab, num, sizeof(struct desctab), &descmp);

  /* Print list of available network drivers and let user decide */
  printf("The following %d network drivers are available:\n", num);
  printf("(0)\tuser defined network driver\n");
  for (i = 0; i < num; i++)
	printf("(%d)\t%s\n", i + 1, dtab[i].desc);
  sel = getsel("Select the network driver you wish to use", 0, num, 0);
  if (sel == 0) {
	printf("Enter the full path name of the network driver: ");
	cp = getstring();
	if ((i = strcspn(cp, " \t")) < strlen(cp))
		cp[i] = '\0';
	copystr(&netdrvname, cp);
  } else {
	dp = dtab[sel - 1].dp;
	len = strlen(dp->fname) + strlen(netdrvdir) + 2;
	netdrvname = (char *)nbmalloc(len);
	sprintf(netdrvname, "%s/%s", netdrvdir, dp->fname);
	if ((descp = findfname(&netdrvdesc, dp->fname)) != NULL)
		getpktopt(descp);
  }
  if (netdrvoptions == NULL) {
	printf("Enter command line arguments for network driver: ");
	copystr(&netdrvoptions, getstring());
  }
  printf("\n\n\n");
  deldirlist();
  free(dtab);
}



/*
 * Let the user select the name of an additional program.
 */
static void getprogram()
{
  char *cp;
  long intnum = DEFAULTINT;
  long pkttype = 0;
  size_t len;
  int i;

  if (getdirlist(utilsdir, "ansidrv.com") == 1) {
	if (getyn("Do you want to use the ANSI display driver", TRUE)) {
		len = strlen(utilsdir) + strlen(dirlist->fname) + 6;
		cp = nbmalloc(len);
		sprintf(cp, "%s/%s", utilsdir, dirlist->fname);
		prognames[num_progs] = cp;
		progoptions[num_progs] = "";
		num_progs++;
	}
  }
  deldirlist();

  if (getdirlist(utilsdir, "pktwatch.com") == 1) {
	if (getyn("Do you want to use the packet driver debugger", FALSE)) {
		len = strlen(utilsdir) + strlen(dirlist->fname) + 6;
		cp = nbmalloc(len < 256 ? 256 : len);
		if (!isdefint)
			intnum = getnum("packet driver interrupt number",
							0x60, 0x7f, TRUE);
		pkttype = getnum("packet type (0 for default)", 0, 0xffff, TRUE);
		if (pkttype == 0)
			sprintf(cp, "%lx", intnum);
		else
			sprintf(cp, "%lx %lx", intnum, pkttype);
		progoptions[num_progs] = NULL;
		copystr(&(progoptions[num_progs]), cp);
		sprintf(cp, "%s/%s", utilsdir, dirlist->fname);
		prognames[num_progs] = cp;
		num_progs++;
	}
  }
  deldirlist();

  while(1) {
	if (getyn("Do you want to specify an additional program", FALSE)) {
		printf("Enter the full path name of the program: ");
		cp = getstring();
		if ((i = strcspn(cp, " \t")) < strlen(cp))
			cp[i] = '\0';
		prognames[num_progs] = NULL;
		copystr(&(prognames[num_progs]), cp);
		printf("Enter command line arguments for program: ");
		progoptions[num_progs] = NULL;
		copystr(&(progoptions[num_progs]), getstring());
		num_progs++;
	} else break;
  }
  printf("\n\n");
}



/*
 * Let the user select a bootrom kernel.
 */
static void getkernel()
{
  struct direntry *dp;
  struct desclist *descp;
  int num, sel, i;
  int defsel = 0;
  size_t len;
  char *cp;

  if ((num = getdirlist(kerneldir, "kernel*.bin")) == 0) {
	fprintf(stderr, "%s: Oops, no kernel binaries found!\n", progname);
	exit(EXIT_MAKEROM_KERNFND);
  }

  printf("The following %d kernels are available:\n", num);
  for (dp = dirlist, i = 1; dp != NULL; dp = dp->next, i++) {
	if ((descp = findfname(&kerneldesc, dp->fname)) == NULL)
		cp = dp->fname;
	else if (descp->descript == NULL)
		cp = descp->progname;
	else
		cp = descp->descript;
	printf("(%d)\t%s\n", i, cp);
	if (defsel == 0 && descp != NULL &&
	    !(descp->options & (K_MINIMAL | K_X86)))
		defsel = i;
  }
  sel = getsel("Select the kernel you wish to use", 1, num,
							defsel ? defsel : 1);

  for (dp = dirlist, i = 1; dp != NULL && sel != i; dp = dp->next, i++) ;
  if (dp != NULL) {
	len = strlen(dp->fname) + strlen(kerneldir) + 6;
	kernelname = nbmalloc(len);
	sprintf(kernelname, "%s/%s", kerneldir, dp->fname);
	if ((descp = findfname(&kerneldesc, dp->fname)) != NULL) {
		if (descp->options & K_MINIMAL)
			isminimal++;
		if (descp->options & K_X86)
			is86++;
	}
	useint18 = getyn("Do you want the bootrom to look for boot drives",
									FALSE);
  }
  printf("\n\n\n");
  deldirlist();
}



/*
 * Get type of output file
 */
static void getouttype()
{
  printf("Available output file types (floppy image gets always created):\n");
  printf("(%d)\tRaw binary\n", OUT_BINARY);
  printf("(%d)\tImage for programming a FlashCard across the network\n", OUT_FLASH);
  printf("(%d)\tIntel hex\n", OUT_IHEX);
  printf("(%d)\tMotorola hex\n", OUT_MHEX);
  printf("(%d)\tTektronix hex\n", OUT_THEX);
  outtype = getsel("Select the format you wish to use", OUT_MIN, OUT_MAX,
								OUT_BINARY);
}



/*
 * Main routine which handles all user input
 */
struct bootdef *getuser()
{
  getkernel();
  getnetdrv();
  if (!isminimal)
	getprogram();
  getouttype();
  buildconfig();
  return(&bootd);
}

