/*
 * getdb.c  -  Read system definition from database file
 *
 * Copyright (C) 1998 Gero Kuhlmann <gero@gkminix.han.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#define NO_BINARY 1	/* No need to include binary support here */
#include "common.h"
#include "nblib.h"
#include "makerom.h"



/* Variables private to this module */
static struct bootdef bootd;		/* boot definition record */



/* Available types for output file */
static char *ftypes[] = {
  "binary", "flash", "intel hex", "motorola hex", "tektronix hex", NULL
};



/* Parameters in each bootrom section of database file */
static struct paramdef dbparams[] = {
  { "kernel",	 par_string,	NULL,	{&bootd.kernelname}},
  { "useint18",	 par_bool,	NULL,	{(char **)&bootd.use_int18}},
#if MAXLOADERS < 3
#error Invalid number of loaders
#endif
  { "loader1",	 par_string,	NULL,	{&bootd.loadernames[0]}},
  { "loader2",	 par_string,	NULL,	{&bootd.loadernames[1]}},
  { "loader3",	 par_string,	NULL,	{&bootd.loadernames[2]}},
  { "outname1",	 par_string,	NULL,	{&bootd.outnames[0]}},
  { "outname2",	 par_string,	NULL,	{&bootd.outnames[1]}},
  { "outname3",	 par_string,	NULL,	{&bootd.outnames[2]}},
  { "outtype1",	 par_enum,	ftypes,	{(char **)&bootd.outtypes[0]}},
  { "outtype2",	 par_enum,	ftypes,	{(char **)&bootd.outtypes[1]}},
  { "outtype3",	 par_enum,	ftypes,	{(char **)&bootd.outtypes[2]}},
#if MAXDRIVERS < 8
#error Invalid number of drivers
#endif
  { "driver1",	 par_string,	NULL,	{&bootd.drivernames[0]}},
  { "driver2",	 par_string,	NULL,	{&bootd.drivernames[1]}},
  { "driver3",	 par_string,	NULL,	{&bootd.drivernames[2]}},
  { "driver4",	 par_string,	NULL,	{&bootd.drivernames[3]}},
  { "driver5",	 par_string,	NULL,	{&bootd.drivernames[4]}},
  { "driver6",	 par_string,	NULL,	{&bootd.drivernames[5]}},
  { "driver7",	 par_string,	NULL,	{&bootd.drivernames[6]}},
  { "driver8",	 par_string,	NULL,	{&bootd.drivernames[7]}},
  { "argument1", par_string,	NULL,	{&bootd.driverargs[0]}},
  { "argument2", par_string,	NULL,	{&bootd.driverargs[1]}},
  { "argument3", par_string,	NULL,	{&bootd.driverargs[2]}},
  { "argument4", par_string,	NULL,	{&bootd.driverargs[3]}},
  { "argument5", par_string,	NULL,	{&bootd.driverargs[4]}},
  { "argument6", par_string,	NULL,	{&bootd.driverargs[5]}},
  { "argument7", par_string,	NULL,	{&bootd.driverargs[6]}},
  { "argument8", par_string,	NULL,	{&bootd.driverargs[7]}},
  { "type1",	 par_enum,	drtype,	{(char **)&bootd.drivertypes[0]}},
  { "type2",	 par_enum,	drtype,	{(char **)&bootd.drivertypes[1]}},
  { "type3",	 par_enum,	drtype,	{(char **)&bootd.drivertypes[2]}},
  { "type4",	 par_enum,	drtype,	{(char **)&bootd.drivertypes[3]}},
  { "type5",	 par_enum,	drtype,	{(char **)&bootd.drivertypes[4]}},
  { "type6",	 par_enum,	drtype,	{(char **)&bootd.drivertypes[5]}},
  { "type7",	 par_enum,	drtype,	{(char **)&bootd.drivertypes[6]}},
  { "type8",	 par_enum,	drtype,	{(char **)&bootd.drivertypes[7]}},
  { NULL,	 par_null,	NULL,	{NULL}}
};



/*
 * Read one entry from the database file
 */
struct bootdef *getdb(name)
char *name;
{
  struct sectdef sect;
  char *namebuf;
  int i, j;
  size_t len;

  /* Generate section record */
  len = strlen(name) + 9;
  namebuf = nbmalloc(len);
  sprintf(namebuf, "%s:makerom", name);
  sect.name = namebuf;
  sect.params = dbparams;
  sect.startsect = NULL;
  sect.endsect = NULL;

  /* Get entry from database */
  memset(&bootd, 0, sizeof(bootd));
  readdb(&sect);

  /* Reorder the loader and driver lists, so that there are no holes */
  for (i = 0, j = -1; i < MAXLOADERS; i++) {
	if (bootd.loadernames[i] == NULL && j < 0)
		j = i;
	else if (bootd.loadernames[i] != NULL && j >= 0) {
		bootd.loadernames[j] = bootd.loadernames[i];
		bootd.outnames[j] = bootd.outnames[i];
		bootd.outtypes[j] = bootd.outtypes[i];
		bootd.loadernames[i] = NULL;
		i = j;
		j = -1;
	}
  }

  for (i = 0, j = -1; i < MAXDRIVERS; i++) {
	if (bootd.drivernames[i] == NULL && j < 0)
		j = i;
	else if (bootd.drivernames[i] != NULL && j >= 0) {
		bootd.drivernames[j] = bootd.drivernames[i];
		bootd.driverargs[j] = bootd.driverargs[i];
		bootd.drivertypes[j] = bootd.drivertypes[i];
		bootd.drivernames[i] = NULL;
		i = j;
		j = -1;
	}
  }

  /* Check all entries */
  if (!bootd.kernelname) {
	fprintf(stderr, "%s: no kernel file name given in section <%s>\n",
							progname, namebuf);
	exit(EXIT_DB);
  }

  bootd.loadernum = 0;
  for (i = 0; i < MAXLOADERS; i++)
	if (bootd.loadernames[i] != NULL) {
		if (bootd.outnames[i] == NULL) {
			fprintf(stderr, "%s: no output file given for loader %s in section <%s>\n",
				progname, bootd.loadernames[i], namebuf);
			exit(EXIT_DB);
		}
		if (!bootd.outtypes[i])
			bootd.outtypes[i] = OUT_BINARY;
		bootd.loadernum++;
	}
  if (!bootd.loadernum) {
	fprintf(stderr, "%s: no bootrom loader specified in section <%s>\n",
							progname, namebuf);
	exit(EXIT_DB);
  }

  bootd.drivernum = 0;
  for (i = 0; i < MAXDRIVERS; i++)
	if (bootd.drivernames[i] != NULL) {
		if (bootd.driverargs[i] == NULL)
			bootd.driverargs[i] = strdup("");
		if (!bootd.drivertypes[i])
			bootd.drivertypes[i] = TYPE_DOS;
		bootd.drivernum++;
	}
  if (!bootd.drivernum) {
	fprintf(stderr, "%s: no driver program specified in section <%s>\n",
							progname, namebuf);
	exit(EXIT_DB);
  }

  free(namebuf);
  return(&bootd);
}

