!**************************************************************************
!*
!* ANSIDRV - Display ANSI control sequences
!*
!* Module:  ansidrv.S
!* Purpose: Display ANSI control sequences onto the console
!* Entries: start
!*
!**************************************************************************
!*
!* Copyright (C) 1995-1998 Gero Kuhlmann <gero@gkminix.han.de>
!*
!*  This program is free software; you can redistribute it and/or modify
!*  it under the terms of the GNU General Public License as published by
!*  the Free Software Foundation; either version 2 of the License, or
!*  any later version.
!*
!*  This program is distributed in the hope that it will be useful,
!*  but WITHOUT ANY WARRANTY; without even the implied warranty of
!*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!*  GNU General Public License for more details.
!*
!*  You should have received a copy of the GNU General Public License
!*  along with this program; if not, write to the Free Software
!*  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
!*
!**************************************************************************
!*

!
!**************************************************************************
!
! This program links its resident section into interrupt $29 and provides
! ANSI control sequence interpretation. It can be used optionally for the
! bootrom.
!
! ANSI control sequences supported by this program:
!
!                       Display attributes
!
!    Code          Effect
!    <esc>[0m      normal text
!    <esc>[1m      high-intensity on
!    <esc>[21m     high-intensity off
!    <esc>[5m      blinking on
!    <esc>[25m     blinking off
!    <esc>[7m      reverse video on
!    <esc>[27m     reverse video off
!    <esc>[3xm     set foreground color:
!    <esc>[4xm     set background color. x can be:
!
!                       0 - black  4 - blue
!                       1 - red    5 - magenta
!                       2 - green  6 - cyan
!                       3 - yellow 7 - white
!
!
!                         Cursor control
!
!    Code          Effect
!    <esc>[r;cH    move cursor to row r col c (r and c are both numbers)
!    <esc>[rA      move cursor up r rows
!    <esc>[rB      move cursor down r rows
!    <esc>[cC      move cursor right c columns
!    <esc>[cD      move cursor left c columns
!    <esc>[?7l     turn off line wrap
!    <esc>[?$7     turn on line wrap
!    <esc>[J       clear screen and home cursor
!    <esc>[s       save the cursor position
!    <esc>[u       return cursor to saved position
!
!


!
!**************************************************************************
!
! Miscellaneous defines:
!
DEACTIVATE	equ	$0CF45		! magic key for deactivation

ARGMAX		equ	2		! maximum number of arguments
MAXLINES	equ	25		! default number of screen lines

CHR_BELL	equ	$07		! bell character
CHR_BS		equ	$08		! backspace character
CHR_TAB		equ	$09		! tab character
CHR_LF		equ	$0A		! linefeed character
CHR_CR		equ	$0D		! carriage return character
CHR_ESC		equ	$1B		! escape character
CHR_BRACKET	equ	$5B		! left bracket
CHR_COLON	equ	$21		! colon character
CHR_QUEST	equ	$3F		! question mark
CHR_EQUAL	equ	$3D		! equal sign

FLAG_NOWRAP	equ	%00000001	! flag indicating not to wrap
FLAG_INVERS	equ	%00000010	! flag for invers video
FLAG_BLINK	equ	%00000100	! flag for blinking
FLAG_INTENSE	equ	%00001000	! flag for high intensity
FLAG_NORMAL	equ	%11110001	! mask to set flags to normal mode

NORMAL_STATE	equ	0		! normal display state
ESC_STATE	equ	1		! found escape in input stream
BRACKET_STATE	equ	2		! found bracket in input stream
NUMBER_STATE	equ	3		! found numbers in input stream


!
!
!**************************************************************************
!
! There is no other than a code segment. All data will be placed in it
! as well. The resident section of the program starts here.
!
	.text

	org	$0100			! we have to start at $0100

	entry	start			! code begins here

start:	jmp	near start1


!
!**************************************************************************
!
! Data segment:
!
scnmode:	.byte	0		! current screen mode
scnpage:	.byte	0		! current screen page
scnattr:	.byte	$07		! attributes for characters to print
scncolumns:	.byte	0		! number of screen columns
curpos:		.word	0		! current cursor position
savedpos:	.word	0		! saved cursor position
flags:		.byte	0		! display flags
fg_color:	.byte	$07		! foreground color
bg_color:	.byte	0		! background color
dispstate:	.byte	NORMAL_STATE	! current display state
argbuf:		.space	ARGMAX		! argument number buffer
argnum:		.byte	0		! current argument number

old29_vect:	.long	0		! old $29 interrupt vector
old2f_vect:	.long	0		! old $2F interrupt vector


!
!**************************************************************************
!
! Interrupt routine to terminate ansidrv services. It will just restore the
! old interrupt vectors. Note that the interrupt vectors have to be restored
! first before calling the old interrupt handler, so that the deactivation
! routines in other such programs dont get confused.
! Input:  AX  -  Magic key $0CF45
! Output: none
! Registers used: none
!
new_int2f:

	cmp	ax,#DEACTIVATE		! check if we have to deactivate
	je	int2f1
int2f9:	seg	cs
	jmp	far ptr [old2f_vect]

int2f1:	push	ds
	push	ax
	xor	ax,ax			! restore all old interrupt vectors
	mov	ds,ax
	cli
	seg	cs
	mov	ax,word ptr [old29_vect + 0]
	mov	word ptr [$29 * 4 + 0],ax
	seg	cs
	mov	ax,word ptr [old29_vect + 2]
	mov	word ptr [$29 * 4 + 2],ax
	seg	cs
	mov	ax,word ptr [old2f_vect + 0]
	mov	word ptr [$2F * 4 + 0],ax
	seg	cs
	mov	ax,word ptr [old2f_vect + 2]
	mov	word ptr [$2F * 4 + 2],ax
	pop	ax
	pop	ds
	jmp	int2f9


!
!**************************************************************************
!
! Command table
!
macro	tab

	.byte	'?1				! put letter into table'
	.word	cmd_?2?1			! put offset to cmd routine

mend

cmdtab:

! Define commands for uppercase letters

	tab	(A,u)
	tab	(B,u)
	tab	(C,u)
	tab	(D,u)
	tab	(H,u)
	tab	(J,u)

! Define commands for lowercase letters

	tab	(h,l)
	tab	(l,l)
	tab	(m,l)
	tab	(s,l)
	tab	(u,l)

! Mark end of table

	.byte	0

TABSIZE	equ	3				! size of each table entry


!
!**************************************************************************
!
! Interrupt routine to display a character. It also interprets the ANSI
! control sequences mentioned above.
! Input:  AL  -  character to display
! Output: none
! Registers used: none
!
new_int29:

	sti
	push	ax
	push	bx
	push	cx
	push	dx
	push	ds
	mov	bx,cs
	mov	ds,bx

! Determine the current cursor position.

	push	ax
	mov	ah,#$03
	mov	bh,scnpage
	int	$10
	mov	curpos,dx
	pop	ax

! State switcher. Depending on the character received with this interrupt
! routine the state will be switched and the corresponding action executed.

	mov	ah,dispstate
	cmp	ah,#NORMAL_STATE		! first handle normal state
	jne	int292

	cmp	al,#CHR_ESC			! is it an escape character?
	jne	int291
	mov	byte ptr dispstate,#ESC_STATE	! yes, just switch the state and
	jmp	int29F				! terminate
int291:	call	putchar				! no, we have to print the
	jmp	int293				! character

int292:	cmp	ah,#ESC_STATE
	jne	int294				! are we in escape state?
	cmp	al,#CHR_BRACKET
	jne	int293				! a bracket has to follow
	mov	byte ptr dispstate,#BRACKET_STATE ! simply switch the state
	jmp	int29F				! and terminate
int293:	mov	byte ptr dispstate,#NORMAL_STATE  ! if no bracket, this is an
int29F:	jmp	int299				! error, so switch back to
						! normal mode
int294:	cmp	ah,#BRACKET_STATE		! check for bracket state
	jne	int295
	cmp	al,#CHR_QUEST			! skip any question marks
	je	int29F
	cmp	al,#CHR_EQUAL			! skip any equal signs
	je	int29F
	mov	byte ptr dispstate,#NUMBER_STATE ! continue with number state
	mov	byte ptr argnum,#0
if ARGMAX = 2
	mov	word ptr [argbuf],#0		! zero out argument buffer
else
	mov	bx,#ARGMAX
int29D:	mov	byte ptr argbuf[bx],#0		! zero out argument buffer
	dec	bx
	jnz	int29D
endif
	jmp	int296

int295:	cmp	ah,#NUMBER_STATE	! if we are not in number state its an
	jne	int293			! error, as we dont have anymore states
int296:	cmp	al,#CHR_COLON
	jne	int297			! a semicolon sign advances the argument
	mov	al,argnum		! counter
	inc	al
	cmp	al,#ARGMAX		! no more than ARGMAX arguments allowed,
	jae	int299			! simply skip the rest
	mov	argnum,al
	jmp	int299

int297:	cmp	al,#$30			! check if we got a number here
	jb	int298
	cmp	al,#$39
	ja	int298
	sub	al,#$30			! convert character into number
	mov	cl,al
	mov	bl,argnum
	xor	bh,bh
	mov	al,argbuf[bx]		! multiply the number in the buffer
	mov	ah,#10			! by 10
	mul	ah
	add	al,cl			! and add the new value
	mov	argbuf[bx],al		! set new value in argument buffer
	jmp	int299			! thats it

! Now execute the command associated with the last character of the escape
! sequence.

int298:	xor	bx,bx
int29A:	mov	ah,cmdtab[bx]
	or	ah,ah			! end of table reached
	je	int29C
	cmp	al,ah			! found character
	je	int29B
	add	bx,#TABSIZE		! continue with next table entry
	jmp	int29A

int29B:	mov	ax,word ptr cmdtab+1[bx]
	mov	dx,curpos
	call	ax				 ! call command handling routine
int29C:	mov	byte ptr dispstate,#NORMAL_STATE ! restore state counter

! Terminate the interrupt routine by restoring all registers

int299:	pop	ds
	pop	dx
	pop	cx
	pop	bx
	pop	ax
	iret


!
!**************************************************************************
!
! Actually display a character onto the console
! Input:  AL  -  character to display
!         DX  -  current cursor position
! Output: none
! Registers used: AX, BX, CX, DX
!
putchar:

	mov	bh,scnpage		! used throughout this routine

! The bell character is best handled by the BIOS

	cmp	al,#CHR_BELL
	jne	putch2
putch1:	mov	ah,#$0E			! print special character using the
	int	$10			! BIOS
	jmp	putch9

! TABs are not handled by the BIOS teletype driver, so we have to do it
! ourselves.

putch2:	cmp	al,#CHR_TAB		! check for tab character
	jne	putch4
	mov	cl,dl			! compute number of spaces to print
	or	cl,#$F8
	neg	cl
	xor	ch,ch
putch3:	push	cx
	mov	al,#$20			! print spaces
	call	putchar			! recursive call
	pop	cx
	loop	putch3
	jmp	putch9

! Carriage return simply involves setting the cursor position to zero

putch4:	cmp	al,#CHR_CR		! check for carriage return
	jne	putch5
	xor	dl,dl			! move cursor into column zero
	jmp	putcursor

! Linefeed involves incrementing the line number and scrolling the screen if
! necessary

putch5:	cmp	al,#CHR_LF		! check for linefeed
	je	putch8			! the linefeed code is below

! Backspace simply involves decrementing the current cursor position

	cmp	al,#CHR_BS		! check for backspace
	jne	putch6
	or	dl,dl			! dont decrement below zero
	jz	putcursor
	dec	dl			! decrement column number
	jmp	putcursor

! All other characters are printed at the cursor position

putch6:	mov	ah,#$09
	mov	bl,scnattr		! print the character using BIOS
	mov	cx,#1
	int	$10

	inc	dl			! increment cursor position
	cmp	dl,scncolumns		! check if at right end of line
	jb	putcursor
	dec	dl				! restore cursor to rightmost
	test	byte ptr flags,#FLAG_NOWRAP	! column and check if we should
	jnz	putcursor			! wrap at the end of the line
	xor	dl,dl				! put cursor into column 0
putch8:	inc	dh
	cmp	dh,#MAXLINES		! check if at bottom of screen
	jb	putcursor
	dec	dh			! restore cursor to lowest screen line
	push	dx
	mov	ax,#$0601
	xor	cx,cx			! define upper left corner
	mov	dl,scncolumns		! define lower right corner
	dec	dl
	mov	bh,scnattr
	int	$10			! use the BIOS to scroll the screen
	pop	dx

! Put cursor to position inidicated by the curpos variable

putcursor:

	mov	curpos,dx		! save new cursor position
	mov	ah,#$02
	mov	bh,scnpage		! set cursor using BIOS
	int	$10
putch9:	ret


!
!**************************************************************************
!
! Handle escape sequence 'A': move cursor up
! Input:  DX  -  current cursor position
! Output: none
! Registers used: AX, BX, CX, DX
!
cmd_uA:

	sub	dh,[argbuf+0]		! decrement line number
	jae	putcursor		! set new cursor position
	ret


!
!**************************************************************************
!
! Handle escape sequence 'B': move cursor down
! Input:  DX  -  current cursor position
! Output: none
! Registers used: AX, BX, CX, DX
!
cmd_uB:

	add	dh,[argbuf+0]		! increment line number
	cmp	dh,#MAXLINES
	jb	putcursor		! set new cursor position
	ret


!
!**************************************************************************
!
! Handle escape sequence 'C': move cursor right
! Input:  DX  -  current cursor position
! Output: none
! Registers used: AX, BX, CX, DX
!
cmd_uC:

	add	dl,[argbuf+0]		! increment column number
	cmp	dl,scncolumns
	jb	putcursor		! set new cursor position
	ret


!
!**************************************************************************
!
! Handle escape sequence 'D': move cursor left
! Input:  DX  -  current cursor position
! Output: none
! Registers used: AX, BX, CX, DX
!
cmd_uD:

	sub	dh,[argbuf+0]		! decrement column number
	jae	putcursor		! set new cursor position
	ret


!
!**************************************************************************
!
! Handle escape sequence 'H': move cursor to specified position
! Input:  none
! Output: none
! Registers used: AX, BX, CX, DX
!
cmd_uH:

	mov	dx,[argbuf+0]
	xchg	dh,dl			! get new cursor position
	jmp	putcursor


!
!**************************************************************************
!
! Handle escape sequence 'J': clear screen and move cursor into upper
! left corner of the screen
! Input:  none
! Output: none
! Registers used: AX, BX, CX, DX
!
cmd_uJ:

	mov	ax,#$0600
	mov	bh,scnattr
	xor	cx,cx
	mov	dh,#MAXLINES-1		! use the BIOS scrolling function
	mov	dl,scncolumns		! to clear the screen
	dec	dl
	int	$10
	xor	dx,dx			! put cursor into upper left corner
	jmp	putcursor


!
!**************************************************************************
!
! Handle escape sequence 'h': set display option, currently only option 7
! is supported, which turns line wrapping on.
! Input:  none
! Output: none
! Registers used: AX, BX, CX, DX
!
cmd_lh:

	mov	al,[argbuf+0]
	cmp	al,#7				! check for option 7
	jne	cmdlh9
	and	byte ptr flags,#~FLAG_NOWRAP	! turn wrapping off
cmdlh9:	ret


!
!**************************************************************************
!
! Handle escape sequence 'l': reset display option, currently only option 7
! is supported, which turns line wrapping off.
! Input:  none
! Output: none
! Registers used: AX, BX, CX, DX
!
cmd_ll:

	mov	al,[argbuf+0]
	cmp	al,#7				! check for option 7
	jne	cmdll9
	or	byte ptr flags,#FLAG_NOWRAP	! turn wrapping off
cmdll9:	ret


!
!**************************************************************************
!
! Handle escape sequence 'm': set screen attributes according to the
! following table:
!
!    <esc>[0m      normal text
!    <esc>[1m      high-intensity on
!    <esc>[21m     high-intensity off
!    <esc>[5m      blinking on
!    <esc>[25m     blinking off
!    <esc>[7m      reverse video on
!    <esc>[27m     reverse video off
!    <esc>[3xm     set foreground color
!    <esc>[4xm     set background color
!
! Input:  none
! Output: none
! Registers used: AX, BX, CX, DX
!
cmd_lm:

! First set the global variables according to the first parameter of the
! escape sequence

	mov	ah,flags
	mov	al,[argbuf+0]
	or	al,al
	jnz	cmdlm1
	and	ah,#FLAG_NORMAL		! set normal mode
	jmp	cmdlm9

cmdlm1:	mov	cl,al
	cmp	cl,#20			! check for reset command
	jb	cmdlm2
	sub	cl,#20
cmdlm2:	cmp	cl,#1
	jne	cmdlm3
	mov	ch,#FLAG_INTENSE	! set high-intensity mode
	jmp	cmdlm5
cmdlm3:	cmp	cl,#5
	jne	cmdlm4
	mov	ch,#FLAG_BLINK		! set blinking mode
	jmp	cmdlm5
cmdlm4:	cmp	cl,#7
	jne	cmdlm7
	mov	ch,#FLAG_INVERS		! set inverse mode
cmdlm5:	cmp	al,#20
	jae	cmdlm6
	or	ah,ch			! set the flag
	jmp	cmdlm9
cmdlm6:	not	ch
	and	ah,ch			! reset the flag
	jmp	cmdlm9

cmdlm7:	sub	al,#30
	jb	cmdlmF		! check for foreground color
	cmp	al,#7
	ja	cmdlm8
	mov	fg_color,al
	jmp	cmdlm9

cmdlm8:	sub	al,#10
	jb	cmdlmF		! check for background color
	cmp	al,#7
	ja	cmdlmF
	mov	bg_color,al

! Now set the actual attribute value according to the flags set previously

cmdlm9:	mov	flags,ah
	xor	ch,ch
	test	ah,#FLAG_BLINK
	jz	cmdlmA
	or	ch,#%10000000		! set blink attribute
cmdlmA:	test	ah,#FLAG_INTENSE
	jz	cmdlmB
	or	ch,#%00001000		! set high-intensity attribute
cmdlmB:	mov	cl,#4
	mov	bl,fg_color
	mov	bh,bg_color		! set colors
	test	ah,#FLAG_INVERS
	jz	cmdlmC
	shl	bl,cl			! shift foreground color into background
	jmp	cmdlmD			! for inverse display
cmdlmC:	shl	bh,cl			! shift background color
cmdlmD:	or	bl,bh
	and	bl,#%01110111		! compute color attribute value
	or	ch,bl			! and merge it into the final value
	mov	scnattr,ch		! save final screen attributes
cmdlmF:	ret


!
!**************************************************************************
!
! Handle escape sequence 's': save current cursor position
! Input:  none
! Output: none
! Registers used: AX, BX, CX, DX
!
cmd_ls:

	mov	ax,curpos
	mov	savedpos,ax		! save the current cursor position
	ret


!
!**************************************************************************
!
! Handle escape sequence 'u': restore cursor position
! Input:  none
! Output: none
! Registers used: AX, BX, CX, DX
!
cmd_lu:

	mov	dx,savedpos
	jmp	near putcursor		! place cursor to new position


!
!**************************************************************************
!
! Start the non-resident section here:
!
start1:	mov	ax,cs
	mov	ds,ax

	mov	dx,#cpymsg		! print copyright
	mov	ah,#$09
	int	$21

	mov	ah,#$0F			! determine current screen mode
	int	$10
	mov	scncolumns,ah		! set number of columns on screen
	mov	scnpage,bh		! set screen page
	mov	al,bh
	mov	ah,#$05			! select screen page
	int	$10

#ifdef ANSITEST
	mov	ax,#$3560
#else
	mov	ax,#$3529		! determine interrupt $29 vector
#endif
	int	$21
	mov	word ptr [old29_vect + 0],bx
	mov	word ptr [old29_vect + 2],es
	mov	dx,#new_int29
#ifdef ANSITEST
	mov	ax,#$2560
#else
	mov	ax,#$2529		! set new interrupt $29 interrupt
#endif
	int	$21
#ifndef ANSITEST
	mov	ax,#$352F		! determine $2F interrupt vector
	int	$21
	mov	word ptr [old2f_vect + 0],bx
	mov	word ptr [old2f_vect + 2],es
	mov	dx,#new_int2f
	mov	ax,#$252F		! set new interrupt $2F vector
	int	$21
#endif

	mov	dx,#start1
	mov	cl,#4			! determine number of resident
	shr	dx,cl			! paragraphs
	inc	dx
	mov	ax,#$3100		! terminate and stay resident
	int	$21
	ret

cpymsg:	.byte	$0D,$0A
	.ascii	"ansidrv - ANSI display driver"
	.byte	$0D,$0A
	.ascii	"Copyright (C) G. Kuhlmann, 1996, 1997"
	.byte	$0D,$0A
	.byte	$0D,$0A
	.ascii	"$"

	end

