// editor.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


#ifdef __GNUG__
#pragma implementation
#endif

#include <InterViews/button.h>
#include <InterViews/world.h>
#include <X11/keysym.h>
#include "localdefs.h"
#include <sys/time.h>
#include <assert.h>
#include "datafile.h"
#include "application.h"
#include "header.h"
#include "editor.h"
#include "cmdstate.h"
#include "comment.h"
#include "converter.h"
#include "controller.h"
#include "data.h"
#include "delay.h"
#include "filename.h"
#include "filecommand.h"
#include "sound.h"
#include "lpcdata.h"
#include "formantfilter.h"
#include "formantsynth.h"
#include "pchdata.h"
#include "fftdata.h"
#include "envelope.h"
#include "datamodifier.h"
#include "analysisfun.h"
#include "optionsetter.h"
#include "phraser.h"
#include "ellipfilt.h"
#include "fftfun.h"
#include "rescaler.h"
#include "interpolater.h"
#include "lowpassfilter.h"
#include "pulsegen.h"
#include "textwindow.h"
#include "pvoceditor.h"
#include "soundeditor.h"
#include "curvegen.h"
#include "crossfader.h"
#include "commenteditor.h"
#include "soundheader.h"
#include "fader.h"
#include "stack.h"

#undef debug

const char *
scat(const char *s1, const char *s2) {
    static char str[1024];
    sprintf(str, "%s%s", s1, s2);
    return str;
}

const char *
scat(const char *s1, const char *s2, const char *s3) {
    static char str[1024];
    sprintf(str, "%s%s%s", s1, s2, s3);
    return str;
}

Data *
DataEditor::_sInternalBuffer = nil;

DataEditor::C_Constructor 
DataEditor::ctor_table1[] = {
    SoundEditor::new_DataEditor1,
    LPCEditor::new_DataEditor1,
    PCHEditor::new_DataEditor1,
    FFTEditor::new_DataEditor1,
    EnvelopeEditor::new_DataEditor1,
    PvocEditor::new_DataEditor1
};

DataEditor::C_D_Constructor 
DataEditor::ctor_table2[] = {
    SoundEditor::new_DataEditor2,
    LPCEditor::new_DataEditor2,
    PCHEditor::new_DataEditor2,
    FFTEditor::new_DataEditor2,
    EnvelopeEditor::new_DataEditor2,
    PvocEditor::new_DataEditor2
};

DataEditor::DataEditor(Controller *c, const Data *d)
        : _controller(c), data(nil), selection(nil),
		  _commandState(new CommandState), _undoStack(nil), _redoStack(nil) {
    init((Data *) d);
	_commandState->ref();
	_commandState->Notify();	// To initialize menus
}

inline int typeToIndex(FileType type) {
    return powerOfTwoToLinearEnum(int(type)) - 2;
}

DataEditor *
DataEditor::create(Controller *c, const char* name) {
    int type_index = typeToIndex(FileName::fileType(name));
    static const int tableEnd = sizeof(ctor_table1) / sizeof(*ctor_table1);
    assert(0 < type_index && type_index < tableEnd);
    return (*DataEditor::ctor_table1[type_index])(c);
}

DataEditor *
DataEditor::create(Controller *c, const Data *d) {
    int type_index = typeToIndex(d->fileType());
    static const int tableEnd = sizeof(ctor_table2) / sizeof(*ctor_table2);
    assert(0 < type_index && type_index < tableEnd);
    return (*DataEditor::ctor_table2[type_index])(c, d);
}

void
DataEditor::init(Data* d) {
    reset();
	int undoDepth = 65535;
    const char *res = Application::getGlobalResource("UndoDepth");
	if (res) {
		undoDepth = atol(res);
		if (undoDepth < 0) undoDepth = 0;
		else if (undoDepth > 65535) undoDepth = 65535;
	}
	_undoStack = new ModifierStack(undoDepth);
	_redoStack = new ModifierStack(undoDepth);

    if (d)
        setData(d);
}

DataEditor::~DataEditor() {
	if (selectionMade())
		setInsertPoint(0, 0);	// unsets selection
	// Clear out all buffers
    setCopyBuffer(nil);		// XXX This is not really what we want to do.
    setSelection(nil);
    setData(nil);
	delete _redoStack;
	delete _undoStack;
	Resource::unref(_commandState);
}

void
DataEditor::setData(Data* d) {
    if (d != data) {
        Resource::unref(data);
        data = d;
        if (data != nil) {
            data->ref();
            location = data->frameRange();
            _channels = data->channelRange();
        }
    }
}

Data *
DataEditor::createModel(DataFile *f) {
    boolean status = false;
    Data *d = newModel();
    if (!f->readable())
        Application::error(scat("Unable to read file ", f->name()));
    else {
        Application::inform(scat("Reading ", f->name(), "..."));
        Header *header = d->createHeader(f, true);
        header->ref();
        if (!(status = d->read(f, header))) {
            if (d->length() == 0)
                /* non-read error */ ;
            else if (Application::globalResourceIsTrue("ReadRawFiles")) {
                if (header->configure(_controller)) {
                    f->clear();
					f->seek(0);
                    status = d->read(f, header);
                }
            }
        }
        Resource::unref(header);
    }
    f->failif (status != true);        // calling routine checks file state
    return d;
}

const char *
DataEditor::defaultDir() {
    const char* sfd = getenv("SFDIR");
    const char *a = 
        Application::getGlobalResource(data->defaultDirAttribute());
    return a ? a : sfd ? sfd : ".";
}

boolean
DataEditor::readFile(DataFile* file) {
    boolean status = true;
    file->ref();
    setData(createModel(file));
    status = file->good();
    Resource::unref(file);
    return status;
}

void
DataEditor::markEditRegion(int size) {
    _controller->showEditRegion(
        Range(currentInsert(), currentInsert() + size - 1),
        currentChannels()
    );
}

void
DataEditor::setInsertPoint(int point, int chan) {
	bool hadSelection = selectionMade();
    if (hadSelection || !insertPointSet() || chan < 0) {	// for first call after setting
        reset();
		commandState()->Set(Target_Region_Selected, false);
		commandState()->Set(Target_Selected, true);
		// Only notify if we have actually lost selection
		if (hadSelection)
			_controller->notifyOfSelection(false);
		_insertPointSet = true;
	}
    setLocation(Range(point, max(point, data->length() - 1)));
    addChannel(chan);
    setSelection(nil);
}

void
DataEditor::setEditRegion(const Range &region, int chan) {
    // when new region set in any window, it becomes default target rather
    // than any previously copied internal buffer.  Not so for insert point.
    setCopyBuffer(nil);

    if (!selectionMade()) {    // for first call after unsetting
        reset();
        _isSelected = true;
		commandState()->Set(Target_Selected | Target_Region_Selected, true);
		_controller->notifyOfSelection(true);
    }
    setLocation(region);
    addChannel(chan);
    setSelection(nil);
}

void
DataEditor::setLocation(const Range& newloc) {
    setTimeStamp();
    if (newloc != location)
        location = newloc;
}

void
DataEditor::addChannel(int chan) {
    if (chan >= 0) {
        int first = _channels.intMin();
        int last = _channels.intMax();
        int nfirst = (first < 0 || chan < first) ? chan : first;
        setChannels(nfirst, chan > last ? chan : last);
    }
}

Range
DataEditor::currentChannels() {
    // for now, either just one, or all channels
    // set to all if none chosen (-1 is flag for this)
    if (nchans() > 1 || _channels.includes(-1))
        setChannels(model()->channelRange());
    return _channels;
}

void
DataEditor::setTimeStamp() {
    struct timeval tp;
    struct timezone tzp;
    gettimeofday(&tp,&tzp);
    timestamp = tp.tv_sec;
}

void
DataEditor::setSelection(Data *s) {
    if (selection != s) {
        Resource::unref(selection);
        if ((selection = s) != nil)
            selection->ref();
    }
}

Data *
DataEditor::getSelection(long &timestamp) {
    timestamp = timeStamp();
    Data *sel = nil;
    if (selectionMade())
        sel = data->clone(currentRegion(), currentChannels());
    return sel;
}

Status
DataEditor::setCopyBuffer(Data *b) {
    if (_sInternalBuffer != b) {
        Resource::unref(_sInternalBuffer);
        if ((_sInternalBuffer = b) != nil) {
            _sInternalBuffer->ref();
		}
		bool copyAvail = (_sInternalBuffer != nil);
		_controller->notifyOfSelection(copyAvail);
    }
	return (_sInternalBuffer != nil);
}

Data *
DataEditor::currentSelection() {
    if (selection == nil)
        setSelection(data->clone(currentRegion(), currentChannels()));
    return selection;
}

Data *
DataEditor::currentSource() {
    Data *source = (copyBuffer() != nil) ? copyBuffer() : _controller->findSelection();
    if (source == nil) ;
//        Application::alert(
//           "You must select a source to use this operation", 
//            "or copy a selection into the internal buffer."
//       );
    else if (*source == *data) {
        // must make deep copy if source is same as target
        Data *copyOfSource = source->copyOf();
        Resource::unref(source);
        source = copyOfSource;
		if (source)
			source->ref();
    }
    else
        source->ref();
    return source;
}

void
DataEditor::freeSource(Data *sel) {
    if (sel != _sInternalBuffer)    // only unref source if it was not internal
        Resource::unref(sel);
}

// re-select edit region (after reverting to saved, for instance)
// unless that region is no longer within the data bounds

void
DataEditor::reselect() {
    if (selectionMade() && data->frameRange().includes(currentRegion()))
		setSelection(data->clone(currentRegion(), currentChannels()));
    else {
        _controller->unselectView();    // do to reset even if no selection made
        setSelection(nil);
    }
}

// static method -- used by "static constructor" methods
// isUndo: true if the operation being performed is undo of prev operation
// dontCreateUndo: true if undo operation is forced (due to cancel or error)

boolean
DataEditor::applyModifierUsing(DataEditor *editor, 
							   Modifier &mod,
							   Controller* controller,
							   bool isUndo,
							   bool dontCreateUndo) {
    bool status = false;
    if (mod.configure(controller)) {
		// Clear redo stack
		if (!isUndo)
			editor->invalidateRedo();
		Modifier *doMod = nil;
		if (!dontCreateUndo) {
			doMod = mod.createUndo();
#ifdef debug
			printf("modifier %s %s operation\n", 
				   doMod ? "generated" : "did not generate",
				   isUndo ? "a redo" : "an undo");
#endif
		}
		if (doMod) {
			if (isUndo)
				editor->pushRedo(doMod);
			else
				editor->pushUndo(doMod);
		}
        status = mod.apply();
		// On failure or cancel, automatic undo (but not redo!)
		if (status == false && doMod && !isUndo) {
			Modifier *undoMod = editor->popUndo();
			assert(undoMod == doMod);
			applyModifierUsing(editor, *undoMod, controller, true, true);
			delete undoMod;
		}
		else if (editor->model()->modified() && !isUndo)
			editor->commandState()->Set(File_Is_Modified, true);
	}
    return status;
}

boolean
DataEditor::applyModifier(Modifier &mod, bool isUndo) {
    return applyModifierUsing(this, mod, _controller, isUndo);
}

boolean
DataEditor::applyModifierToNew(Modifier &mod) {
    boolean status;
    if ((status = applyModifier(mod, false)) == true)
        displayInternalBuffer();
    setCopyBuffer(nil);
    return status;
}

void
DataEditor::pushUndo(Modifier *mod) {
	_undoStack->push(mod);
	commandState()->Set(Undo_Is_Available, !_undoStack->empty());
}

Modifier *
DataEditor::popUndo() {
	Modifier *undo = _undoStack->pop();
	commandState()->Set(Undo_Is_Available, !_undoStack->empty());
	return undo;
}

void
DataEditor::invalidateUndo() {
	_undoStack->clear();
	commandState()->Set(Undo_Is_Available, false);
}

void
DataEditor::pushRedo(Modifier *mod) {
	_redoStack->push(mod);
	commandState()->Set(Redo_Is_Available, !_redoStack->empty());
}

Modifier *
DataEditor::popRedo() {
	Modifier *redo = _redoStack->pop();
	commandState()->Set(Redo_Is_Available, !_redoStack->empty());
	return redo;
}

void
DataEditor::invalidateRedo() {
	_redoStack->clear();
	commandState()->Set(Redo_Is_Available, false);
}

// keysym is checked first by controller, then by view, then by editor
// subclass, and lastly by editor base class.  Checking stops when matched.

boolean
DataEditor::keyCommand(unsigned long sym) {
    boolean interested = true;
    switch (sym) {
    case XK_period:
        setDefaultDir();
        break;
    case XK_n:
        newFile();
        break;
    case XK_nobreakspace:
        newLPCFile();
        break;
    case XK_exclamdown:
        newSoundFile();
        break;
    case XK_cent:
        newEnvelopeFile();
        break;
    case XK_currency:
        newPvocFile();
        break;
    case XK_o:
        openNewFile();
        break;
    case XK_s:
        saveFile();
        break;
    case XK_S:
        saveToFile();
        break;
    case XK_periodcentered:
        saveSelectionToFile();
        break;
    case XK_threequarters:
        revertToSaved();
        break;
    case XK_quotedbl:
        changeComment();
        break;
    case XK_question:
        information();
        break;
    case XK_braceright:
        dataDump();
        break;
    case XK_I:
        displayInternalBuffer();
        break;
    case XK_c:
        copy();
        break;
    case XK_C:
        copyToNew();
        break;
    case XK_r:
        remove();
        break;
    case XK_y:
        removeToNew();
        break;
    case XK_e:
        erase();
        break;
    case XK_x:
        spliceOut();
        break;
    case XK_X:
        spliceOutToNew();
        break;
    case XK_d:
        zap();
        break;
    case XK_m:
        mix();
        break;
    case XK_R:
        replace();
        break;
    case XK_g:
        crossfade();
        break;
    case XK_v:
        spliceIn();
        break;
    case XK_P:
        scaleValues();
        break;
    case XK_b:
        reverse();
        break;
    case XK_braceleft:
        swapBytes();
        break;
    case XK_i:
        insertSpace();
        break;
    case XK_E:
        applyEnvelope();
        break;
	case XK_F1:
		fadeIn();
		break;
	case XK_F2:
		fadeOut();
		break;
    case XK_O:
        vertOffset();
        break;
    case XK_bracketright:
        rescale();
        break;
    case XK_D:
        delay();
        break;
    case XK_t:
        interpolate();
        break;
    case XK_L:
        lowPassFilter();
        break;
    case XK_a:
        autoCorrelate();
        break;
    case XK_bracketleft:
        normalize();
        break;
    case XK_l:
        changeLength();
        break;
    case XK_0:
        findZeroCrossing();
        break;
    case XK_1:
        findSlopeChange();
        break;
    case XK_2:
        showPeakFrame();
        break;
    case XK_3:
        extractEnvelope();
        break;
    case XK_degree:
        setDataOptions();
        break;
	case XK_questiondown:
		setEditorOptions();
		break;
    case XK_paragraph:
        setRawFileOptions();
        break;
	case XK_u:
		undo();
		break;
	case XK_U:
		redo();
		break;
    default:
        interested = false;
        break;
    }
    return interested;
}

Controller*
DataEditor::openNewFile(const char* windowtitle) {
    FileOpener fo(defaultDir(), windowtitle);
    applyModifier(fo);
    return fo.getNewController();
}

Modifier *
DataEditor::getFileSaver(const char *dir, Data *d, const char *filename) {
	return (filename != NULL) ?
    	new FileSaver(filename, dir, d) : new FileSaver(dir, d);
}

//
// The menu-called methods
//

// FILE methods

Status
DataEditor::openNewFile() {
    return (openNewFile("Select File to Open:") != nil);
}

Status
DataEditor::newLPCFile() {
    return (LPCEditor::new_File(_controller) != nil);
}

Status
DataEditor::newSoundFile() {
    return (SoundEditor::new_File(_controller) != nil);
}

Status
DataEditor::newEnvelopeFile() {
    return (EnvelopeEditor::new_File(_controller) != nil);
}

Status
DataEditor::newPvocFile() {
    return (PvocEditor::new_File(_controller) != nil);
}

Response
DataEditor::closeFile() {
    Response r = Yes;
    // warn if file modified and if this is the last open view of it
    if (data->modified() && data->currentViews() == 1) {
        r = Application::choice("Do you wish to save changes to",
            _controller->fileName(), "before closing?", Cancel);
        switch(r) {
        case Yes:
            saveFile();
            if (data->modified())
                r = Cancel;    // assumes cancel or error during save
            else
                r = No;    // indicates warning was issued
            break;
        case No:
        case Cancel:
            break;
          default:
            break;
        }
    }
    return r;    // defaults to Yes indicating no alert issued
}
    
Status
DataEditor::saveFile() {
    Modifier *fs = getFileSaver(defaultDir(), model(), _controller->fileName());
    Status status = applyModifier(*fs);
	if (status == Succeed && Application::globalResourceIsTrue("ClearUndoOnSave")) {
		invalidateUndo();
		invalidateRedo();
	}
	delete fs;
	return status;
}

Status
DataEditor::saveToFile() {
    Modifier *fs = getFileSaver(defaultDir(), model());
    Status status = applyModifier(*fs);
	delete fs;
	return status;
}

Status
DataEditor::saveSelectionToFile() {
    Status status;
    if ((status = copy()) == Succeed) {
		Data *sel = copyBuffer();
		Modifier *filesaver = getFileSaver(defaultDir(), sel);
		const char *oldName = _controller->fileName();
		char *savedFileName = new char[strlen(oldName) + 1];
		strcpy(savedFileName, oldName);
		status = applyModifier(*filesaver);
		_controller->setFileName(savedFileName);
		delete [] savedFileName;
		delete filesaver;
    }
    return status;
}

Status
DataEditor::revertToSaved() {
    const char* filename = _controller->fileName();
    Status status = Fail;
    if (FileName::isTempName(filename))
        Application::alert("This is a temporary file and does not",
            "exist yet on disk.  You must save it first.");
    else if (Application::confirm(
				data->modified() ?
            		"Are you sure you wish to revert to the saved version?" :
	    			"Re-read file from disk?"))
    {
        DataFile f(filename, "r");
		if (f.readable()) {
            Application::inform("Re-reading file from disk...");
            Header *header = data->createHeader(&f, true);
            header->ref();
            status = data->read(&f, header);
	    	if (status == Fail
	            && Application::globalResourceIsTrue("ReadRawFiles"))
			{
            	if (header->configure(_controller)) {
                	f.clear();
                	status = data->read(&f, header);
            	}
	    	}
			if (status == Succeed) {
				invalidateUndo();
				invalidateRedo();
				commandState()->Set(File_Is_Modified, false);
			}
            Resource::unref(header);
        }
		else
			Application::error(scat("Unable to re-read file ", f.name()));

        _controller->resetScaleTimes();
        reselect();
    }
    return status;
}

Status
DataEditor::displayInternalBuffer() {
    Status status = Fail;
    if (copyBuffer() != nil) {
        if (copyBuffer()->length() > 3) {
            Controller *newCtlr = new Controller(copyBuffer());
            newCtlr->display(_controller->world());
            copyBuffer()->Notify();    // to assure rescan of amps
            status = Succeed;
        }
        else
            Application::alert("Cannot display data with length < 4 frames.");
        // once displayed, not available internally to avoid confusion
        setCopyBuffer(nil);
    }
    else Application::alert("Internal buffer is currently empty.");
    return status;
}

Status
DataEditor::changeComment() {
    TextWindow* commentWindow = new TextWindow(
        scat(_controller->fileName(), ": Comment"),
        new CommentEditor(new ButtonState, model())
    ); 
    commentWindow->display(_controller->world());
    return Succeed;
}

Status
DataEditor::dataDump() {
    DataDumper dumper(defaultDir(), currentSelection());
    return applyModifier(dumper);
}

Status
DataEditor::information() {
    data->information(_controller);
    return Succeed;
}

// EDIT methods

Status
DataEditor::undo() {
	Status status = Fail;
	Modifier *mod = popUndo();
	if (mod)
		status = applyModifier(*mod, /* isUndo = */ true);
	else
		World::current()->RingBell(75);	// beep to show no undo
	delete mod;
	return status;
}

Status
DataEditor::redo() {
	Status status = Fail;
	Modifier *mod = popRedo();
	if (mod)
		status = applyModifier(*mod);
	else
		World::current()->RingBell(75);	// beep to show no redo
	delete mod;
	return status;
}

// previous internal buffer is kept until actual copy of new buffer takes place

Status
DataEditor::copy() {
    Application::inform("Copying...");
    boolean was = Data::deferRescan(true);
    Status status = setCopyBuffer(currentSelection()->copyOf());
    Data::deferRescan(was);
    return status;
}

Status
DataEditor::copyToNew() {
    return copy() && displayInternalBuffer();
}

Status
DataEditor::remove() {
    return copy() && erase();
}

Status
DataEditor::removeToNew() {
    return remove() && displayInternalBuffer();
}

Status
DataEditor::erase() {
    Eraser e(currentSelection());
    return applyModifier(e);
}

Status
DataEditor::spliceOut() {
    return copy() && zap();
}

Status
DataEditor::spliceOutToNew() {
    return spliceOut() && displayInternalBuffer();
}

Status
DataEditor::zap() {
    Status status = Fail;
    if (selectionMade()
        || Application::confirm("Splice out everything from this point",
            "to the end of the file?")) {
        OutSplicer splicer(model(), currentRegion());
        if ((status = applyModifier(splicer)))
            _controller->showInsertPoint(currentInsert(), currentChannels());
    }
    return status;
}

Status
DataEditor::mix() {
    // for now this mixes at unity gain
    Data* src = nil;
    Status status = Fail;
    if ((src = currentSource()) != nil) {
        Mixer mixer(currentSelection(), src);
        if ((status = applyModifier(mixer)))
            markEditRegion(min(currentSelection()->length(), src->length()));
    }
    freeSource(src);
    return status;
}

Status
DataEditor::replace() {
    Data* src = nil;
    Status status = Fail;
    if ((src = currentSource()) != nil) {
        Replacer replacer(currentSelection(), src);
        if ((status = applyModifier(replacer)))
            markEditRegion(min(currentSelection()->length(), src->length()));
    }
    freeSource(src);
    return status;
}

Status
DataEditor::crossfade() {
    Data* src = nil;
    Status status = Fail;
    if ((src = currentSource()) != nil) {
        Crossfader crossfader(currentSelection(), src);
        if ((status = applyModifier(crossfader)))
            markEditRegion(min(currentSelection()->length(), src->length()));
    }
    freeSource(src);
    return status;
}

Status
DataEditor::spliceIn() {
    Data* src = nil;
    Status status = Fail;
    if ((src = currentSource()) != nil) {
        Splicer splicer(currentSelection(), src);
        if ((status = applyModifier(splicer))) {
            Range edit = src->frameRange();
            edit += currentInsert();        // offset for highlight display
            _controller->showEditRegion(edit, data->channelRange());
        }
    }
    freeSource(src);
    return status;
}

Status
DataEditor::scaleValues() {
    Scaler s(currentSelection());
    return applyModifier(s);
}

Status
DataEditor::insertSpace() {
    SpaceInsert s(currentSelection());
    return applyModifier(s);
}

Status
DataEditor::reverse() {
    Reverser r(currentSelection());
    return applyModifier(r);
}

Status
DataEditor::swapBytes() {
    ByteSwapper s(currentSelection());
    return applyModifier(s);
}

Status
DataEditor::vertOffset() {
    Offset o(currentSelection());
    return applyModifier(o);
}

Status
DataEditor::rescale() {
    Rescaler r(currentSelection());
    return applyModifier(r);
}

Status
DataEditor::interpolate() {
    // create new buffer which will be resized by Interpolater
    setCopyBuffer(currentSelection()->newData(1));
    Interpolater i(currentSelection(), copyBuffer());
    return applyModifierToNew(i);
}

Status
DataEditor::changeLength() {
    LengthChanger l(model());
    Status status = applyModifier(l);
    reselect();
    return status;
}

Status
DataEditor::delay() {
    Delay d(currentSelection());
    return applyModifier(d);
}

Status
DataEditor::lowPassFilter() {
    LowPassFilter l(currentSelection(), 0.5);
    return applyModifier(l);
}

Status
DataEditor::autoCorrelate() {
    AutoCorrelator a(currentSelection());
    return applyModifier(a);
}

Status
DataEditor::applyEnvelope() {
    Envelope *envelope = nil;
    Data *source = nil;
    Controller* ctlr = nil;
    if ((source = currentSource()) != nil) {
        ;
    }
    else if ((ctlr = openNewFile("Select Envelope File to Apply:")) != nil) {
        source = ctlr->model();
        source->ref();
    }
    Status status = Fail;
    if (source) {
        // make copy of source so that it may be normalized, etc.
        envelope = new Envelope(source->length());
        envelope->copyFrom(source);
        envelope->ref();
        Application::inform("Scaling envelope between -1 and 1...");
		Scaler s(envelope, 1.0/envelope->maxValue());
        s.apply();
        Phraser p(currentSelection(), envelope);
		status = applyModifier(p);
        Resource::unref(envelope);
    }
    freeSource(source);
    return status;
}

Status
DataEditor::fadeIn() {
	Fader f(currentSelection(), Fader::FadeIn);
	return applyModifier(f);
}

Status
DataEditor::fadeOut() {
	Fader f(currentSelection(), Fader::FadeOut);
	return applyModifier(f);
}

Status
DataEditor::normalize() {
    Scaler s(currentSelection(), 1.0/currentSelection()->maxValue());
    return applyModifier(s);
}

Status
DataEditor::changeSampleRate() {
    SampRateChanger s(model());	// apply to entire model
    Status status;
    if ((status = applyModifier(s))) {
        _controller->resetScaleTimes();
        reselect();
    }
    return status;
}

Status
DataEditor::showPeakFrame() {
    int peakchan, peakloc;
    currentSelection()->maxValue(&peakchan, &peakloc);
    Range chan(peakchan, peakchan);
	chan += _channels.intMin();	// offset if selected lowest channel not 0
    _controller->showInsertPoint(peakloc+currentInsert(), chan, true);
    return Succeed;
}

Status
DataEditor::findZeroCrossing() {
    int offset = currentSelection()->zeroCrossing();
    Status status = Succeed;
    if (offset > 0) {
        Range chan(currentChannels().intMin(), currentChannels().intMin());
        _controller->showInsertPoint(currentInsert() + offset, chan);
    }
    else {
        Application::alert("No zero crossings found.");
        status = Fail;
    }
    return status;
}

// for now, this just displays the location of the first discontinuity found, though the
// analysis data is an array of offsets to all such locations within the current selection

Status
DataEditor::findSlopeChange() {
    SlopeChangeDetecter s(currentSelection());
    Status status = Fail;
    if (s.configure(_controller)) {
        if ((status = s.apply())) {
            int offset = (int) s.getAnalysis()->get(0);
            Range chan(currentChannels().intMin(), currentChannels().intMin());
            _controller->showInsertPoint(currentInsert() + offset, chan);
	}
        else {
            Application::alert("No slope changes of that magnitude found.");
            status = Fail;
        }
    }
    return status;
}

Status
DataEditor::extractEnvelope() {
    EnvelopeExtracter e(currentSelection());
    Status status;
    if ((status = applyModifier(e))) {
        Controller *newctlr = new Controller(e.getAnalysis());
        newctlr->display(_controller->world());
    }
    return status;
}

Status
DataEditor::setEditorOptions() {
	EditorOptionSetter e;
	return applyModifier(e);
}

//**************

Data *
LPCEditor::newModel() { return new LPCData(); }

Range
LPCEditor::currentChannels() {
    if (nchans() == 1 && !_channels.includes(-1))
        ;
    else if (_channels.includesZero() || _channels.includes(-1))  // default
        setChannels(model()->channelRange());
    else if (_channels.intMin() >= 4)        // coefficient display
        setChannels(4, model()->channels() - 1);
    return _channels;
}

boolean
LPCEditor::keyCommand(unsigned long sym) {
    boolean interested = true;
    switch (sym) {
    case XK_asterisk:
        stabilizeFrames();
        break;
    case XK_A:
        displayFrameAmplitudes();
        break;
    case XK_F:
        displayFrameFormants();
        break;
    case XK_k:
        changeSampleRate();
        break;
    case XK_M:
        mergePitchData();
        break;
    case XK_dollar:
        adjustPitchDeviation();
        break;
    case XK_p:
        audition();
        break;
    default:
        interested = DataEditor::keyCommand(sym);
        break;
    }
    return interested;
}

Status
LPCEditor::newFile() {
    return (LPCEditor::new_File(_controller) != nil);
}

Modifier *
LPCEditor::getFileSaver(const char *dir, Data *d, const char *filename) {
	return (filename != NULL) ?
    	new LPCFileSaver(filename, dir, d) : new LPCFileSaver(dir, d);
}

// static constructor

Controller *
LPCEditor::new_File(Controller* controller) {
    LPCFileCreator lfc;
    applyModifierUsing(controller->editor(), lfc, controller, false);
    return lfc.getNewController();
}

Status
LPCEditor::stabilizeFrames() {
    FrameStabilizer fs(model());
    return applyModifier(fs);
}

Status
LPCEditor::displayFrameAmplitudes() {
    const int pulseFrameSize = 1000;
    const int pulsePerFrame = 10;
    LPCData* selected = (LPCData *) currentSelection();
    Application::inform("Creating test pattern...");
    int lpcLen = selected->length();
    int srate = selected->sRate();
    Sound* pulses = new Sound(lpcLen * pulseFrameSize, srate, 1, FloatData);
    pulses->ref();
    // add pulsePerFrame pulses per original LPC frame
    PulseGenerator pgen(pulses, pulseFrameSize/pulsePerFrame);
    pgen.apply();
    FormantFilter filter(pulses, pulses, selected, 1.0);
    if (!filter.apply())
        return Fail;
    Envelope* amplitudes = new Envelope(lpcLen * pulsePerFrame);
    amplitudes->setFrameRangeLabel("LPC Analysis Frames");
    amplitudes->setRangeFactor(1.0/pulsePerFrame);
    Application::inform("Extracting amplitudes...");
    pulses->getEnvelope(amplitudes, 0, AbsoluteMagnitude);
    Resource::unref(pulses);
    Controller* ampDisplay = new Controller(amplitudes, 
        scat(_controller->windowName(), ":  Frame Amplitudes"));
    ampDisplay->display(_controller->world());
    return Succeed;
}

Status
LPCEditor::displayFrameFormants() {
    const int pulseFrameSize = 1024;
    LPCData* selected = (LPCData *) currentSelection();
    int lpcLen = selected->length();
    int srate = selected->sRate();
    Sound* pulses = new Sound(lpcLen * pulseFrameSize, srate, 1, FloatData);
    pulses->ref();
    PulseGenerator pgen(pulses, pulseFrameSize);
    Application::inform("Creating test pattern...");
    pgen.apply();
    FormantFilter filter(pulses, pulses, selected, 1.0);
    if (!filter.apply())
        return Fail;
    Application::inform("Analyzing formants...");
    // fft size depends on npoles
    FFT_Function analyzer(
        pulses, selected->nPoles() > 32 ? 128 : 64, pulseFrameSize
    );
    Status status;
    if ((status = analyzer.apply())) {
        Controller* fftDisplay = new Controller(
            analyzer.getAnalysis(), 
            scat(_controller->windowName(), ":  Formant Frequencies"));
        fftDisplay->display(_controller->world());
    }
    Resource::unref(pulses);
    return status;
}

Status
LPCEditor::audition() {
    LPCData* selected = (LPCData *) currentSelection();
    int lpcLen = selected->length();
    int srate = selected->sRate();
	double framerate = selected->frameRate();
	bool wasSelected = selectionMade();
	Range theRegion = currentRegion();
    Sound* toPlay = new Sound(lpcLen/framerate, srate, 1, FloatData);
    toPlay->ref();
    Application::inform("Resynthesizing selected region...");
	FormantSynthesizer synth(toPlay, selected, 200.0, 50, 1.0);
	Status status;
    if (synth.apply())
	{
		Converter* cvtr = Converter::getInstance();
		class ProgressAction* paction = _controller->getConverterProgressAction();
    	Application::inform("Playing synthesized portion. <control>-backspace to stop");
		status = toPlay->play(cvtr, paction);
	}
	else status = Fail;
	// This puts us back at the beginning of the play region, 
	// regardless of whether we have a selection or not.
	_controller->showInsertPoint(theRegion.intMin(), _channels, false);
	if (wasSelected)
		_controller->showEditRegion(theRegion, _channels, false);
    Resource::unref(toPlay);
    return status;
}

Status
LPCEditor::mergePitchData() {
    // regardless of selection, select only channel 3 for merge
    const int pitchChannel = 3;
    setChannels(pitchChannel, pitchChannel);
    LPCData *lpc = (LPCData *) currentSelection();
    Data *pitches = nil;
    Controller* newController = nil;
    Status status = Succeed;
    if ((pitches = currentSource()) != nil) {
        ;
    }
    else if ((newController = openNewFile("Select Pitch File for Merge:")) != nil)
        pitches = newController->model();
    if (pitches && Application::confirm("Please confirm merge operation.")) {
        Application::inform("Merging pitch data...");
        lpc->mergePitchData(pitches);    
        markEditRegion(lpc->length());
    }
    else status = Fail;
    freeSource(pitches);
    return status;
}

Status
LPCEditor::setDataOptions() {
    LPCOptionSetter setter;
    return applyModifier(setter);
}

// these static functions have their addresses loaded into a ctor array
// in the DataEditor base class

DataEditor *
LPCEditor::new_DataEditor1(Controller *c) {
    return new LPCEditor(c);
}

DataEditor *
LPCEditor::new_DataEditor2(Controller *c, const Data *d) {
    return new LPCEditor(c, d);
}

//**************

boolean
PCHEditor::keyCommand(unsigned long sym) {
    boolean interested = true;
    switch (sym) {
    case XK_underscore:
        interpolate();
        break;
    default:
        interested = DataEditor::keyCommand(sym);
        break;
    }
    return interested;
}

Data *
PCHEditor::newModel() { return new PCHData(); }

// these static functions have their addresses loaded into a ctor array
// in the DataEditor base class

DataEditor *
PCHEditor::new_DataEditor1(Controller *c) {
    return new PCHEditor(c);
}

DataEditor *
PCHEditor::new_DataEditor2(Controller *c, const Data *d) {
    return new PCHEditor(c, d);
}

Status
PCHEditor::interpolate() {
    Data *sel = currentSelection();
    LinearCurveGenerator l(sel, sel->get(0), sel->get(sel->length() - 1));
    return applyModifier(l);
}

//**************

Data *
FFTEditor::newModel() { return new FFTData(); }

Status
FFTEditor::setDataOptions() { return Succeed; }

// these static functions have their addresses loaded into a ctor array
// in the DataEditor base class

DataEditor *
FFTEditor::new_DataEditor1(Controller *c) {
    return new FFTEditor(c);
}

DataEditor *
FFTEditor::new_DataEditor2(Controller *c, const Data *d) {
    return new FFTEditor(c, d);
}

//**************

Data *
EnvelopeEditor::newModel() { return new Envelope(); }

boolean
EnvelopeEditor::keyCommand(unsigned long sym) {
    boolean interested = true;
    switch (sym) {
    case XK_underscore:
        createLinearCurve();
        break;
    case XK_asterisk:
        createExponentialCurve();
        break;
    default:
        interested = DataEditor::keyCommand(sym);
        break;
    }
    return interested;
}

Status
EnvelopeEditor::newFile() {
    return (EnvelopeEditor::new_File(_controller) != nil);
}

Status
EnvelopeEditor::setDataOptions() {
    EnvelopeOptionSetter setter;
    return applyModifier(setter);
}

// static constructor

Controller *
EnvelopeEditor::new_File(Controller* controller) {
    EnvelopeFileCreator efc;
    applyModifierUsing(controller->editor(), efc, controller, false);
    return efc.getNewController();
}

Modifier *
EnvelopeEditor::getFileSaver(const char *dir, Data *d, const char *filename) {
	return (filename != NULL) ?
    	new EnvelopeFileSaver(filename, dir, d) : new EnvelopeFileSaver(dir, d);
}

Status
EnvelopeEditor::createLinearCurve() {
    LinearCurveGenerator l(currentSelection());
    return applyModifier(l);
}

Status
EnvelopeEditor::createExponentialCurve() {
    ExponentialCurveGenerator e(currentSelection());
    return applyModifier(e);
}

// these static functions have their addresses loaded into a ctor array
// in the DataEditor base class

DataEditor *
EnvelopeEditor::new_DataEditor1(Controller *c) {
    return new EnvelopeEditor(c);
}

DataEditor *
EnvelopeEditor::new_DataEditor2(Controller *c, const Data *d) {
    return new EnvelopeEditor(c, d);
}
