// curvegen.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


#ifdef __GNUG__
#pragma implementation
#endif

#include "curvegen.h"
#include "data.h"
#include "localdefs.h"
#include "request.h"
#include "requester.h"

class CurveGenerateRequester : public TitledRequester {
	friend class LinearCurveGenerator;
protected:
	CurveGenerateRequester(const char* title, CurveGenerator* c)
		: TitledRequester(title), client(c) {}
	redefined void configureRequest(Request *);
protected:
	CurveGenerator* client;
};

void
CurveGenerateRequester::configureRequest(Request* request) {
	request->appendValue("Starting Amplitude:", &client->baseValue);
	request->appendValue("Ending Amplitude: ", &client->endValue);
}

//********
	
CurveGenerator::CurveGenerator(Data* data)
	: OutputFunction(data),
	  counter(0), totalLength(data->length()),
	  baseValue(data->get(0)),
	  endValue(data->get(data->length() - 1)) {}

CurveGenerator::CurveGenerator(Data* data, double base, double ending)
		: OutputFunction(data),
		  counter(0), totalLength(data->length()),
		  baseValue(base), endValue(ending) {
	// initialize NOT called here because derived classes call it
}

int
CurveGenerator::operator () (double *signal, int len, RunStatus *status) {
	RunStatus st = Run;
	int n;
	for (n=0; n < len && st == Run; ) {
		st = (counter < totalLength) ? Run : Stop;
		signal[n] = baseValue + height * curvePoint(double(counter)/totalLength);
		++counter;
		++n;
	}
	*status = st;
	return n;
}

void
CurveGenerator::initialize() {
	Super::initialize();
	height = endValue - baseValue;
}

//********

LinearCurveGenerator::LinearCurveGenerator(Data* data, double base,
		double ending)
		: CurveGenerator(data, base, ending) {
	initialize();
}

Requester *
LinearCurveGenerator::createRequester() {
	return new CurveGenerateRequester(
		"Create Linear Segment over Selected Region:",
		this
	);
}

//********

class ExponentialCurveGenerateRequester : public CurveGenerateRequester {
	friend class ExponentialCurveGenerator;
protected:
	ExponentialCurveGenerateRequester(ExponentialCurveGenerator* e)
		: CurveGenerateRequester(
			"Create Exponential Curve over Selected Region:", e) {}
	redefined void configureRequest(Request *);
};

void
ExponentialCurveGenerateRequester::configureRequest(Request* request) {
	CurveGenerateRequester::configureRequest(request);
	ExponentialCurveGenerator* ecg = (ExponentialCurveGenerator *) client;
	request->appendValue("Exponent for Curve:", &ecg->_exponent,
			    NonNegativeNumbers);
}

//********

double ExponentialCurveGenerator::_savedExponent = 2.0;

ExponentialCurveGenerator::ExponentialCurveGenerator(Data* data, double base,
		double ending, double expon)
		: CurveGenerator(data, base, ending), _exponent(expon) {
	initialize();
}

ExponentialCurveGenerator::ExponentialCurveGenerator(Data* data)
	: CurveGenerator(data), _exponent(_savedExponent) {}

Requester *
ExponentialCurveGenerator::createRequester() {
	return new ExponentialCurveGenerateRequester(this);
}

void
ExponentialCurveGenerator::saveConfig() {
    _savedExponent = _exponent;
}

double
ExponentialCurveGenerator::curvePoint(double fraction) {
	return pow(fraction, _exponent);
}
