// This may look like C code, but it is really -*- C++ -*-
/* 
Copyright (C) 1988 Free Software Foundation
    written by Doug Lea (dl@rocky.oswego.edu)

This file is part of the GNU C++ Library.  This library is free
software; you can redistribute it and/or modify it under the terms of
the GNU Library General Public License as published by the Free
Software Foundation; either version 2 of the License, or (at your
option) any later version.  This library is distributed in the hope
that it will be useful, but WITHOUT ANY WARRANTY; without even the
implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU Library General Public License for more details.
You should have received a copy of the GNU Library General Public
License along with this library; if not, write to the Free Software
Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifndef _MyComplex_h
#ifdef __GNUG__
#pragma interface
#endif
#define _MyComplex_h 1

#include <math.h>

class MyComplex
{
#ifdef __ATT_complex__
public:
#else
protected:
#endif

  double           re;
  double           im;

public:

  double           real() const;
  double           imag() const;

                   MyComplex();
                   MyComplex(const MyComplex& y);
                   MyComplex(double r, double i=0);

                  ~MyComplex();

  MyComplex&         operator =  (const MyComplex& y);

  MyComplex&         operator += (const MyComplex& y);
  MyComplex&         operator += (double y);
  MyComplex&         operator -= (const MyComplex& y);
  MyComplex&         operator -= (double y);
  MyComplex&         operator *= (const MyComplex& y);
  MyComplex&         operator *= (double y);

  MyComplex&         operator /= (const MyComplex& y); 
  MyComplex&         operator /= (double y); 

  void             error(const char* msg) const;
};


// non-inline functions

MyComplex   operator /  (const MyComplex& x, const MyComplex& y);
MyComplex   operator /  (const MyComplex& x, double y);
MyComplex   operator /  (double   x, const MyComplex& y);

MyComplex   cos(const MyComplex& x);
MyComplex   sin(const MyComplex& x);

MyComplex   cosh(const MyComplex& x);
MyComplex   sinh(const MyComplex& x);

MyComplex   exp(const MyComplex& x);
MyComplex   log(const MyComplex& x);

MyComplex   pow(const MyComplex& x, int p);
MyComplex   pow(const MyComplex& x, const MyComplex& p);
MyComplex   pow(const MyComplex& x, double y);
MyComplex   sqrt(const MyComplex& x);
   
// other functions defined as inlines

int  operator == (const MyComplex& x, const MyComplex& y);
int  operator == (const MyComplex& x, double y);
int  operator != (const MyComplex& x, const MyComplex& y);
int  operator != (const MyComplex& x, double y);

MyComplex  operator - (const MyComplex& x);
MyComplex  conj(const MyComplex& x);
MyComplex  operator + (const MyComplex& x, const MyComplex& y);
MyComplex  operator + (const MyComplex& x, double y);
MyComplex  operator + (double x, const MyComplex& y);
MyComplex  operator - (const MyComplex& x, const MyComplex& y);
MyComplex  operator - (const MyComplex& x, double y);
MyComplex  operator - (double x, const MyComplex& y);
MyComplex  operator * (const MyComplex& x, const MyComplex& y);
MyComplex  operator * (const MyComplex& x, double y);
MyComplex  operator * (double x, const MyComplex& y);

double  real(const MyComplex& x);
double  imag(const MyComplex& x);
double  abs(const MyComplex& x);
double  norm(const MyComplex& x);
double  arg(const MyComplex& x);

MyComplex  polar(double r, double t = 0.0);


// inline members

inline double  MyComplex::real() const { return re; }
inline double  MyComplex::imag() const { return im; }

inline MyComplex::MyComplex() {}
inline MyComplex::MyComplex(const MyComplex& y) :re(y.real()), im(y.imag()) {}
inline MyComplex::MyComplex(double r, double i) :re(r), im(i) {}

inline MyComplex::~MyComplex() {}

inline MyComplex&  MyComplex::operator =  (const MyComplex& y) 
{ 
  re = y.real(); im = y.imag(); return *this; 
} 

inline MyComplex&  MyComplex::operator += (const MyComplex& y)
{ 
  re += y.real();  im += y.imag(); return *this; 
}

inline MyComplex&  MyComplex::operator += (double y)
{ 
  re += y; return *this; 
}

inline MyComplex&  MyComplex::operator -= (const MyComplex& y)
{ 
  re -= y.real();  im -= y.imag(); return *this; 
}

inline MyComplex&  MyComplex::operator -= (double y)
{ 
  re -= y; return *this; 
}

inline MyComplex&  MyComplex::operator *= (const MyComplex& y)
{  
  double r = re * y.real() - im * y.imag();
  im = re * y.imag() + im * y.real(); 
  re = r; 
  return *this; 
}

inline MyComplex&  MyComplex::operator *= (double y)
{  
  re *=  y; im *=  y; return *this; 
}


//  functions

inline int  operator == (const MyComplex& x, const MyComplex& y)
{
  return x.real() == y.real() && x.imag() == y.imag();
}

inline int  operator == (const MyComplex& x, double y)
{
  return x.imag() == 0.0 && x.real() == y;
}

inline int  operator != (const MyComplex& x, const MyComplex& y)
{
  return x.real() != y.real() || x.imag() != y.imag();
}

inline int  operator != (const MyComplex& x, double y)
{
  return x.imag() != 0.0 || x.real() != y;
}

inline MyComplex  operator - (const MyComplex& x)
{
  return MyComplex(-x.real(), -x.imag());
}

inline MyComplex  conj(const MyComplex& x)
{
  return MyComplex(x.real(), -x.imag());
}

inline MyComplex  operator + (const MyComplex& x, const MyComplex& y)
{
  return MyComplex(x.real() + y.real(), x.imag() + y.imag());
}

inline MyComplex  operator + (const MyComplex& x, double y)
{
  return MyComplex(x.real() + y, x.imag());
}

inline MyComplex  operator + (double x, const MyComplex& y)
{
  return MyComplex(x + y.real(), y.imag());
}

inline MyComplex  operator - (const MyComplex& x, const MyComplex& y)
{
  return MyComplex(x.real() - y.real(), x.imag() - y.imag());
}

inline MyComplex  operator - (const MyComplex& x, double y)
{
  return MyComplex(x.real() - y, x.imag());
}

inline MyComplex  operator - (double x, const MyComplex& y)
{
  return MyComplex(x - y.real(), -y.imag());
}

inline MyComplex  operator * (const MyComplex& x, const MyComplex& y)
{
  return MyComplex(x.real() * y.real() - x.imag() * y.imag(), 
                 x.real() * y.imag() + x.imag() * y.real());
}

inline MyComplex  operator * (const MyComplex& x, double y)
{
  return MyComplex(x.real() * y, x.imag() * y);
}

inline MyComplex  operator * (double x, const MyComplex& y)
{
  return MyComplex(x * y.real(), x * y.imag());
}

inline double  real(const MyComplex& x)
{
  return x.real();
}

inline double  imag(const MyComplex& x)
{
  return x.imag();
}

inline double  abs(const MyComplex& x)
{
  return hypot(x.real(), x.imag());
}

inline double  norm(const MyComplex& x)
{
  return (x.real() * x.real() + x.imag() * x.imag());
}

inline double  arg(const MyComplex& x)
{
  return atan2(x.imag(), x.real());
}

inline MyComplex  polar(double r, double t)
{
  return MyComplex(r * cos(t), r * sin(t));
}

#endif
