/*
 *   dspispos.c -- Functions that activate tasks and frame managers
 *
 *  Written By: Mike Sullivan IBM Corporation
 *
 *  Copyright (C) 1999 IBM Corporation
 *
 * This program is free software; you can redistribute it and/or modify      
 * it under the terms of the GNU General Public License as published by      
 * the Free Software Foundation; either version 2 of the License, or         
 * (at your option) any later version.                                       
 *                                                                           
 * This program is distributed in the hope that it will be useful,           
 * but WITHOUT ANY WARRANTY; without even the implied warranty of            
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             
 * GNU General Public License for more details.                              
 *                                                                           
 * NO WARRANTY                                                               
 * THE PROGRAM IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OR        
 * CONDITIONS OF ANY KIND, EITHER EXPRESS OR IMPLIED INCLUDING, WITHOUT      
 * LIMITATION, ANY WARRANTIES OR CONDITIONS OF TITLE, NON-INFRINGEMENT,      
 * MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. Each Recipient is    
 * solely responsible for determining the appropriateness of using and       
 * distributing the Program and assumes all risks associated with its        
 * exercise of rights under this Agreement, including but not limited to     
 * the risks and costs of program errors, damage to or loss of data,         
 * programs or equipment, and unavailability or interruption of operations.  
 *                                                                           
 * DISCLAIMER OF LIABILITY                                                   
 * NEITHER RECIPIENT NOR ANY CONTRIBUTORS SHALL HAVE ANY LIABILITY FOR ANY   
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL        
 * DAMAGES (INCLUDING WITHOUT LIMITATION LOST PROFITS), HOWEVER CAUSED AND   
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR     
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE    
 * USE OR DISTRIBUTION OF THE PROGRAM OR THE EXERCISE OF ANY RIGHTS GRANTED  
 * HEREUNDER, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGES             
 *                                                                           
 * You should have received a copy of the GNU General Public License         
 * along with this program; if not, write to the Free Software               
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 *                                                                           
 * 
 *  10/23/2000 - Alpha Release 0.1.0
 *            First release to the public
 *
 */

#include <stdio.h>
#include <unistd.h>
#include "dspmgr.h"

#include "dspbios.h"
#include "dspfxutl.h"
#include "dspgpc.h"
#include "dspispos.h"  /* CH02 */
#define  hiword(x)     (USHORT)(x>>16)     /* CH01 move upper word of ULONG */
                                           /* to lower word & convert to US */
#define GetOverlaySem(prdsp)  DSP_NOERROR
#define ReleaseOverlaySem(prdsp)

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: InsertFrameMgr                                          */
/*                                                                          */
/* FUNCTION: This function builds an image of the frame mgr control block,  */
/*           writes the image to DSP data memory, then inserts the frame    */
/*           manager into the FM Activation list.                           */
/*                                                                          */
/* INPUT:  ptr to memory structure for FMCB                                 */
/*         ptr to segment structure                                         */
/*                                                                          */
/* OUTPUT:  error code                                                      */
/*                                                                          */
/* SIDE EFFECTS:                                                            */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/************************** END OF SPECIFICATIONS ***************************/

RC InsertFrameMgr(PRTSK prtsk,PRFM prfm)

{
   ULONG      ulDspAddrACTLIST;        /* DSP address of Activation list    */
   ULONG      ulDspAddrACTIVATE;       /* DSP address of Activation flag    */
   RC         ulRC;                    /* error code                        */
   PRFMCB     prfmcb;                  /* ptr to FMCB template              */
   PRDSP      prdsp;                   /* ptr to DSP structure              */
   ULONG      ulAddr;                  /* temp ptr                          */
   RFMCB      rfmcb;                   /* FMCB template                     */
   ULONG      ulFrameTime;
   USHORT     usBuf;                   /* CH02 buffer used for BIOS read and*/
                                       /* write                             */
   static char aszOS_Labels[4][9] =  {
      "SCHEDLER",
      "FM",
      "ACTIVATE",
      "ACTLIST"
   };

   static char aszHW_Labels[4][8] =
   {
      "CDINT",
      "AICINT1",
      "AICINT2",
      "CDINT"
   };

   /* Initialize ptrs                                                          */

   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspispos::InsertFrameMgr entry prtsk %x\n",(int)prtsk);
   prfmcb = &rfmcb;
   prdsp = prtsk->TSK_prdsp;

   do {

      if ((ulRC = GetOverlaySem(prdsp)) != DSP_NOERROR)
         break;

   /*build the FMCB image. Building the image consists of initializing the     */
   /*following parameters:                                                     */
   /*      INTFLG, IntLink, Framesize, FrameTime, OSRtn, FirstTCB, ISAdrr,     */
   /***********************  IntLink  *************************************    */

      if ((ulRC = IsposLabelToAddress(prdsp, aszHW_Labels[prfm->FM_hwid], /*   */

         &ulAddr)) != DSP_NOERROR)
      {
	MW_SYSLOG_3(TRACE_MANAGER_CORE,"dspispos::InsertFrameManager error %lx Mwave/OS label: %s",ulRC,
                  aszHW_Labels[prfm->FM_hwid]);  
         break;
      }
      prfmcb->FMCB_usIntLink = (USHORT)(ulAddr);

   /***********************  INTFLG  *******************************************/

      prfmcb->FMCB_usMoreZeros[4] = 0;

   /***********************  FrameSize and FrameTime  **************************/

      prfmcb->FMCB_usFrameSize = prfm->FM_usSamplesPerFrame;
      ulFrameTime = (ULONG)prfm->FM_usSamplesPerFrame;

      if (((prfm->FM_hwid == DSPFILE_CDDAC) ||
         (prfm->FM_hwid == DSPFILE_ADC)) &&
         (prfm->FM_ulSampleRate == 44100))
         {
         if ((ulFrameTime&0x1F) == 0)
            ulFrameTime = ulFrameTime >> 5;
         else
         {
            ulRC = DSP_INV_FM;
            break;
         }
         ulFrameTime *= 0x829CBC;
      }
      else
         if ((prfm->FM_hwid == DSPFILE_AIC2) &&

            (prfm->FM_ulSampleRate == 8000)) {
            ulFrameTime *= 0x168000;
         }
         else
            if ((prfm->FM_hwid == DSPFILE_AIC1) &&

               (prfm->FM_ulSampleRate == 9600)) {
               ulFrameTime *= 0x12C000;
            }
            else
               {
               ulRC = DSP_INV_FM;
	       MW_SYSLOG_5(TRACE_MANAGER_CORE,"dspispos::InsertFrameManager error ulRC %lx FM id: %d, Frame: %ld, SampleRate %ld",ulRC,
                         prfm->FM_hwid, ulFrameTime, prfm->FM_ulSampleRate);
               break;
            }

      if ((ulFrameTime > 0x3FFFFFFF) || (ulFrameTime == 0))
         return (DSP_INV_FM);
      prfmcb->FMCB_usFrameTime = hiword(ulFrameTime); 

   /***********************  OSRtn  *****************************************  */

      if ((ulRC = IsposLabelToAddress(prdsp, aszOS_Labels[0], &ulAddr)) !=
         DSP_NOERROR) {

	 MW_SYSLOG_3(TRACE_MANAGER_CORE,"dspispos::InsertFrameManager error ulRC %lx Mwave/OS Label: %s\n",ulRC, aszOS_Labels[0]);
         break;
      }
      prfmcb->FMCB_usOSRtn = (USHORT)(ulAddr);

   /***********************  FirstTCB  **************************************  */

      prfmcb->FMCB_usFirstTCB = 0;

   /***********************  ISAddr  ****************************************  */

      if ((ulRC = IsposLabelToAddress(prdsp, aszOS_Labels[1], &ulAddr)) !=
         DSP_NOERROR) {

	MW_SYSLOG_3(TRACE_MANAGER_CORE,"dspispos::InsertFrameManager error %lx Mwave/OS Label: %s\n",ulRC, aszOS_Labels[1]);
         break;
      }
      prfmcb->FMCB_usISAddr = (USHORT)(ulAddr);

   /* write the framemgr control block image to dsp memory                     */

      PRINT_DEBUG_FOPEN(EMBEDFILEPTR, EMBEDFILENAME, "a", "dspispos::InsertFrameMgr");
      PRINT_DEBUG(EMBEDFILEPTR, "  Sample rate (0x%X)", prfm->FM_ulSampleRate);
      PRINT_DEBUG(EMBEDFILEPTR, "  Frame time (0x%X)", ulFrameTime);
      PRINT_DEBUG(EMBEDFILEPTR, "  FM_hwid label (0x%X)\n",
           aszHW_Labels[prfm->FM_hwid]);
      PRINT_DEBUG_BIOS_WRITE(EMBEDFILEPTR, "WRITE01: FMCB to dsp",
           EOF, (sizeof(rfmcb)/2),
           prdsp, prfm->FM_prmemFMCB->MEM_ulSegAddr, (sizeof(rfmcb)/2), prfmcb);

      if ((ulRC = DspBIOS_D_Write(prdsp, prfm->FM_prmemFMCB->MEM_ulSegAddr,
         (ULONG)(sizeof(rfmcb)/2), prfmcb)) != DSP_NOERROR)
         break;

   /* activate the frame manager                                               */


      if ((ulRC = IsposLabelToAddress(prdsp, aszOS_Labels[2], &ulDspAddrACTIVATE)
         ) != DSP_NOERROR) {
	MW_SYSLOG_3(TRACE_MANAGER_CORE,"dspispos::InsertFrameManager error ulRC %lx Mwave/OS Label: %s\n",ulRC, aszOS_Labels[2]);
         break;
      }

      if ((ulRC = IsposLabelToAddress(prdsp, aszOS_Labels[3], &ulDspAddrACTLIST))
         != DSP_NOERROR)
         {

	   MW_SYSLOG_3(TRACE_MANAGER_CORE,"dspispos::InsertFrameManager error ulRC %lx Mwave/OS Label: %s\n",ulRC, aszOS_Labels[3]);
         break;
      }

   /* check ISPOS 'ACTIVATE' flag...if not zero, assume ISPOS has died         */

      if ((ulRC = DspBIOS_D_Read(prdsp, ulDspAddrACTIVATE, (ULONG)1, &usBuf)) !=
         DSP_NOERROR)
         break;

   #ifndef  NODSP
      if (usBuf != 0)   /* CH02 */
      {
         ulRC = DSP_MWAVEOS_ERROR;
         break;
      }
   #endif

      PRINT_DEBUG_FOPEN(EMBEDFILEPTR, EMBEDFILENAME, "a", "dspispos::InsertFrameMgr");
      PRINT_DEBUG_BIOS_WRITE(EMBEDFILEPTR,
           "  READ02:   Check ISPOS 'ACTIVATE' flag.  Must be zero or ISPOS died",
           "\n",(ULONG)1, prdsp, ulDspAddrACTIVATE, (ULONG)1, &usBuf);
      PRINT_DEBUG_BIOS_WRITE(EMBEDFILEPTR,
           "  WRITE03:  Write addr of FMCB to ISPOS 'ACTivationLIST'",
           "\n",(ULONG)1, prdsp, ulDspAddrACTLIST, (ULONG)1, &usBuf);
      PRINT_DEBUG_BIOS_WRITE(EMBEDFILEPTR,
           "  WRITE04:  Write ISPOS activation flag to true",
           EOF, (ULONG)1, prdsp, ulDspAddrACTIVATE, (ULONG)1, &usBuf);

   /* write address of FMCB to ISPOS 'ACTivationLIST'                          */

      usBuf = (USHORT)(prfm->FM_prmemFMCB->MEM_ulSegAddr); /* CH02 */
      if ((ulRC = DspBIOS_D_Write(prdsp, ulDspAddrACTLIST,
         (ULONG)1, &usBuf)) != DSP_NOERROR)
         break;

   /* set ISPOS activation flag to true                                        */

      usBuf = 1;  /* CH02 */
      if ((ulRC = DspBIOS_D_Write(prdsp, ulDspAddrACTIVATE,
         (ULONG)1, &usBuf)) != DSP_NOERROR)
         break;

      ulRC = WaitISPOSLocZero( prdsp, ulDspAddrACTIVATE); /* CH01 */

   } while (0);

   ReleaseOverlaySem(prdsp);

   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspispos::InsertFrameMgr exit ulRC %lx\n",ulRC);
   return (ulRC);
}                                      /* end of InsertFrameMgr             */

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: InsertTaskInOS                                          */
/*                                                                          */
/* FUNCTION: This function contains the dsp manager code that inserts a     */
/*           task into the frame manager's linked list of tasks.            */
/*                                                                          */
/* INPUT:  ptr to FMCB memory structure                                     */
/*         ptr to DDS  memory structure                                     */
/*                                                                          */
/* OUTPUT:  error code                                                      */
/*                                                                          */
/* SIDE EFFECTS:                                                            */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/************************** END OF SPECIFICATIONS ***************************/

RC InsertTaskInOS(PRTSK prtsk,PRMEM prmemFMCB,PRMEM prmemDDS)
{
   ULONG      ulDspAddrFirstTCB;       /* DSP address of first TCB          */
   ULONG      ulDspAddrNextTCB;        /* DSP address of next TCB           */
   RC         ulRC;                    /* error code                        */
   PRFMCB     prfmcb;                  /* ptr to FMCB template              */
   PRTCB      prtcb;                   /* ptr to TCB template               */
   PRDSP      prdsp;                   /* ptr to DSP structure              */
   RTCB       rtcb;                    /* TCB template                      */
   RFMCB      rfmcb;                   /* FMCB template                     */
   USHORT     usBuf;                   /* CH01 buffer used for BIOS read and*/
                                       /* write                             */

/* initialize ptrs                                                          */

   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspispos::InsertTaskInOS prtsk %x\n",(int)prtsk);
   prtcb = &rtcb;
   prfmcb = &rfmcb;
   prdsp = prtsk->TSK_prdsp;

   do {

      if ((ulRC = GetOverlaySem(prdsp)) != DSP_NOERROR)
         break;

   /* compute address of FrameMgr's firstTCB                                   */

      if ((prtsk->TSK_usflgType&TSK_TYPEMASK) == TSK_NRT1) {
         if ((ulRC = IsposLabelToAddress(prdsp, "NRT1FTCB", &ulDspAddrFirstTCB))
            != DSP_NOERROR)
         {

	   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspispos::InsertTaskInOS error ulRC %lx Mwave/OS Label: NRT1FTCB\n",ulRC);
            break;
         }
      }
      else
         if ((prtsk->TSK_usflgType&TSK_TYPEMASK) == TSK_NRT2) {
            if ((ulRC = IsposLabelToAddress(prdsp, "NRT2FTCB", &ulDspAddrFirstTCB
               )) != DSP_NOERROR)
            {

	      MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspispos::InsertTaskInOS error ulRC %lx Mwave/OS Label: NRT2FTCB\n",ulRC);
               break;
            }
         }
         else {
            ulDspAddrFirstTCB = prmemFMCB->MEM_ulSegAddr;

         /* add offset of label FirstTCB                                       */

            ulDspAddrFirstTCB += (ULONG)&(prfmcb->FMCB_usFirstTCB)-(ULONG)prfmcb;
         }

   /* compute address of next TCB                                              */

      ulDspAddrNextTCB = prmemDDS->MEM_ulSegAddr-(ULONG)sizeof(rtcb);
      ulDspAddrNextTCB += (ULONG)&(prtcb->TCB_usNextTCB)-(ULONG)prtcb;

   /* read the frame manager's FirstTCB                                        */

      if ((ulRC = DspBIOS_D_Read(prdsp, ulDspAddrFirstTCB, (ULONG)1, &usBuf)) !=
         DSP_NOERROR)
         break;

      PRINT_DEBUG_FOPEN(EMBEDFILEPTR, EMBEDFILENAME, "a", "dspispos::InsertTaskInOS");
      PRINT_DEBUG(EMBEDFILEPTR, "  prtsk (0x%X)", (ULONG) prtsk);
      PRINT_DEBUG(EMBEDFILEPTR, "  prmemFMCB (0x%X)", (ULONG) prmemFMCB);
      PRINT_DEBUG(EMBEDFILEPTR, "  prmemDDS (0x%X)\n", (ULONG) prmemDDS);
      PRINT_DEBUG_BIOS_WRITE(EMBEDFILEPTR,
           "  READ05:   frame manager's FirstTCB", EOF, (ULONG)1,
           prdsp, ulDspAddrFirstTCB, (ULONG)1, &usBuf);

   /* write value of FirstTCB into task's NextTCB                              */

      PRINT_DEBUG_FOPEN(EMBEDFILEPTR, EMBEDFILENAME, "a", "dspispos::InsertTaskInOS");
      PRINT_DEBUG_BIOS_WRITE(EMBEDFILEPTR,
           "  WRITE06:  into task's NextTCB addr with FirstTCB", EOF, (ULONG)1,
           prdsp, ulDspAddrNextTCB, (ULONG)1, &usBuf);

      if ((ulRC = DspBIOS_D_Write(prdsp, ulDspAddrNextTCB, (ULONG)1, &usBuf)) !=
         DSP_NOERROR)
         break;

   /* write ptr to task's TCB into frame manager's FirstTCB                    */
      usBuf = (USHORT)prmemDDS->MEM_ulSegAddr;  /* CH01 */

      PRINT_DEBUG_FOPEN(EMBEDFILEPTR, EMBEDFILENAME, "a", "dspispos::InsertTaskInOS");
      PRINT_DEBUG_BIOS_WRITE(EMBEDFILEPTR,
           "  WRITE07:  ptr to task's TCB into FM's FirstTCB addr",
           EOF, (ULONG)1, prdsp, ulDspAddrFirstTCB, (ULONG)1, &usBuf);

      if ((ulRC = DspBIOS_D_Write(prdsp, ulDspAddrFirstTCB, (ULONG)1, &usBuf)) !=
         DSP_NOERROR)
         break;

   } while (0);

   ReleaseOverlaySem(prdsp);

   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspispos::InsertTaskInOS exit ulRC %lx\n",ulRC);
   return (ulRC);

}                                      /* end of InsertTaskInOS             */

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: ChangeActivationState                                   */
/*                                                                          */
/* FUNCTION: This function contains the dsp manager code that switches a    */
/*           task from standby to active mode, or active to standby mode.   */
/*                                                                          */
/* INPUT:  ptr to task structure                                            */
/*         USHORT flags                                                     */
/*                                                                          */
/* OUTPUT:  error code                                                      */
/*          DSP_NOERRROR         - if no error occured                      */
/*          DSP_NO_DATA          - if no data segment exists                */
/*          DSP_CHNG_STATE_ERROR - general change state error               */
/*                                                                          */
/* SIDE EFFECTS:                                                            */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/************************** END OF SPECIFICATIONS ***************************/

RC ChangeActivationState(PRTSK prtsk,USHORT usFlags)
{
   ULONG      ulDspAddrStandby;        /* DSP address of Standby            */
   ULONG      ulDspAddrCodeptr;        /* DSP address of Code Ptr           */
   ULONG      ulDspAddrActive;         /* DSP address of Active             */
   RC         ulRC;                    /* error code                        */
   PRDSP      prdsp;                   /* ptr to dsp structure              */
   PRTCB      prtcb;                   /* ptr to task control block template*/
   RTCB       rtcb;                    /* tcb template                      */
   PRSEG      prseg;                   /* ptr to segment structure          */
   PRMEM      prmem;                   /* ptr to memory structure           */
   USHORT     usBuf;                   /* CH01 buffer used for BIOS read and*/
                                       /* write                             */

/* initialize pointer                                                       */

   MW_SYSLOG_3(TRACE_MANAGER_CORE,"dspispos::ChangeActivationState entry prtsk %x usFlags %x\n",(int)prtsk,usFlags);
   prtcb = &rtcb;
   prdsp = prtsk->TSK_prdsp;
   prseg = prtsk->TSK_prsegDTail;

/* check to see if data segment exist                                       */

   if (prseg == NULL)
      return (DSP_NO_DATA);
   prmem = prseg->SEG_prsegNext->SEG_prmem;

/* check to see if status of task is inactive                               */

   if (prtsk->TSK_usflgState == TSK_INACTIVE)
      return (DSP_CHNG_STATE_ERROR);

   if ((prtsk->TSK_usflgType&TSK_TYPEMASK) == TSK_OS)
      return (DSP_NOERROR);

   if ((usFlags&DSP_ACTIVATE_MASK) == DSP_ACTIVATE_STANDBY) {

      /* compute address of Standby and Codeptr                             */

      ulDspAddrStandby = prmem->MEM_ulSegAddr-(ULONG)sizeof(rtcb);
      if (ulDspAddrStandby < prmem->MEM_ulDspAddr)
         return (DSP_CHNG_STATE_ERROR);
      ulDspAddrCodeptr = ulDspAddrStandby;
      ulDspAddrStandby += (ULONG)&(prtcb->TCB_usStandbyAddr)-(ULONG)prtcb;
      ulDspAddrCodeptr += (ULONG)&(prtcb->TCB_usCodePtr)-(ULONG)prtcb;

      /* read Standby                                                       */

      if ((ulRC = DspBIOS_D_Read(prdsp, ulDspAddrStandby, (ULONG)1, &usBuf))
         != DSP_NOERROR)
         return (ulRC);

      PRINT_DEBUG_FOPEN(EMBEDFILEPTR, EMBEDFILENAME, "a", \
          "dspispos::ChangeActivationState");
      PRINT_DEBUG(EMBEDFILEPTR, "  prtsk (0x%X)", (ULONG) prtsk);
      PRINT_DEBUG(EMBEDFILEPTR, "  usFlags (0x%X)\n",(ULONG) usFlags);
      PRINT_DEBUG_BIOS_WRITE(EMBEDFILEPTR,
           "  READ08:   Standby at DSP Addr shown",
           EOF, (ULONG)1, prdsp, ulDspAddrStandby, (ULONG)1, &usBuf);

      /* write contents of standby to codeptr                               */

      PRINT_DEBUG_FOPEN(EMBEDFILEPTR, EMBEDFILENAME,"a","dspispos::ChangeActivationState");
      PRINT_DEBUG(EMBEDFILEPTR, "  prtsk (0x%X)", (ULONG) prtsk);
      PRINT_DEBUG(EMBEDFILEPTR, "  usFlags (0x%X)\n", (ULONG) usFlags);
      PRINT_DEBUG_BIOS_WRITE(EMBEDFILEPTR,
           "  WRITE09:  contents of standby to Codeptr addr shown",
           EOF, (ULONG)1, prdsp, ulDspAddrCodeptr, (ULONG)1, &usBuf);

      if ((ulRC = DspBIOS_D_Write(prdsp, ulDspAddrCodeptr, (ULONG)1, &usBuf))
         != DSP_NOERROR)
         return (ulRC);
   }
   else
      if ((usFlags&DSP_ACTIVATE_MASK) == DSP_ACTIVATE_ACTIVE) {

      /* compute address of Active and Codeptr                              */

         ulDspAddrActive = prmem->MEM_ulSegAddr-(ULONG)sizeof(rtcb);
         if (ulDspAddrActive < prmem->MEM_ulDspAddr)
            return (DSP_CHNG_STATE_ERROR);
         ulDspAddrCodeptr = ulDspAddrActive;
         ulDspAddrActive += (ULONG)&(prtcb->TCB_usMainAddr)-(ULONG)prtcb;
         ulDspAddrCodeptr += (ULONG)&(prtcb->TCB_usCodePtr)-(ULONG)prtcb;

      /* read Active                                                        */

         PRINT_DEBUG_FOPEN(EMBEDFILEPTR, EMBEDFILENAME, "a", \
             "dspispos::ChangeActivationState");
         PRINT_DEBUG(EMBEDFILEPTR, "  prtsk (0x%X)", (ULONG) prtsk);
         PRINT_DEBUG(EMBEDFILEPTR, "  usFlags (0x%X)\n", (ULONG) usFlags);
         PRINT_DEBUG_BIOS_WRITE(EMBEDFILEPTR,
              "  READ10:   Active at DSP Addr shown",
              EOF, (ULONG)1, prdsp, ulDspAddrActive, (ULONG)1, &usBuf);

         if ((ulRC = DspBIOS_D_Read(prdsp, ulDspAddrActive, (ULONG)1, &usBuf))
            != DSP_NOERROR)
            return (ulRC);

      /* write contents of Active to codeptr                                */

         PRINT_DEBUG_FOPEN(EMBEDFILEPTR, EMBEDFILENAME,"a","dspispos::ChangeActivationState");
         PRINT_DEBUG(EMBEDFILEPTR, "  prtsk (0x%X)", (ULONG) prtsk);
         PRINT_DEBUG(EMBEDFILEPTR, "  usFlags (0x%X)\n", (ULONG) usFlags);
         PRINT_DEBUG_BIOS_WRITE(EMBEDFILEPTR,
              "  WRITE11:  contents of active to Codeptr addr shown",
              EOF, (ULONG)1, prdsp, ulDspAddrCodeptr, (ULONG)1,
              (ULONG) &usBuf);

         if ((ulRC = DspBIOS_D_Write(prdsp, ulDspAddrCodeptr, (ULONG)1, &usBuf
            )) != DSP_NOERROR)
            return (ulRC);
      }
      else
         return (DSP_CHNG_STATE_ERROR);

   MW_SYSLOG_1(TRACE_MANAGER_CORE,"dspispos::ChangeActivationState exit\n");
   return (Wait_DONERT(prdsp));   /* CH02 MTS #955 Wait for safe period */

}                                      /* end of ChangeActivationState      */

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: RemoveFrameMgr                                          */
/*                                                                          */
/* FUNCTION: This function causes a Frame Manager to be removed from        */
/*           the operating system by modifying its instruction store        */
/*           address to point to ISPOS's frame mgr deactvation routine      */
/*                                                                          */
/* INPUT:  ptr to task structure                                            */
/*         ptr to memory structure for FMCB (Frame Manager Control Block)   */
/*                                                                          */
/* OUTPUT:  error code                                                      */
/*                                                                          */
/* SIDE EFFECTS:                                                            */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/************************** END OF SPECIFICATIONS ***************************/

RC RemoveFrameMgr(PRTSK prtsk,PRFM prfm)

{
   RC         ulRC;                    /* error code                        */
   ULONG      ulAddrDEACT;             /* DSP address of deactivation       */
                                       /* routine                           */
   ULONG      ulFMISADDR;              /* CH01 Addr of FMCB ISADDR.         */
   PRDSP      prdsp;                   /* CH01 Temp storage for HDSP.       */
   USHORT     usAddr;                  /* ushort version of DEACTIVATE addr */

   MW_SYSLOG_3(TRACE_MANAGER_CORE,"dspispos::RemoveFrameManager entry prtsk %x prfm %x\n",(int)prtsk,(int)prfm);
   prdsp = prtsk->TSK_prdsp;           /* CH01 */

   do {

      if ((ulRC = GetOverlaySem(prdsp)) != DSP_NOERROR)
         break;

   /* find the address of ISPOS frameMgr deactivation routine                  */

      if ((ulRC = IsposLabelToAddress(prdsp, "DEACTVAT", &ulAddrDEACT)
         ) != DSP_NOERROR) {

	MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspispos::RemoveFrameManager error ulRC %lx Mwave/OS Label: DEACTVAT\n",ulRC);
         break;
      }
      usAddr = (USHORT)ulAddrDEACT;

   /* write the address of deactivation routine to framemgr label: ISADDR      */
   /* CH01 Begin changes for MTS #982 */
      ulFMISADDR = prfm->FM_prmemFMCB->MEM_ulSegAddr+FMCB_ISADDRESS; /* CH01 */

      PRINT_DEBUG_FOPEN(EMBEDFILEPTR, EMBEDFILENAME,"a","dspispos::RemoveFrameMgr");
      PRINT_DEBUG(EMBEDFILEPTR, "  prtsk (0x%X)", (ULONG) prtsk);
      PRINT_DEBUG(EMBEDFILEPTR, "  prfm (0x%X)\n", (ULONG) prfm);
      PRINT_DEBUG_BIOS_WRITE(EMBEDFILEPTR,
           "  WRITE12:  addr of deact to framemgr label ISADDR at addr shown",
           EOF, (ULONG)1, prdsp, ulFMISADDR, (ULONG)1, &usAddr);

      ulRC = DspBIOS_D_Write(prdsp, ulFMISADDR, (ULONG)1, &usAddr);
      if (DSP_NOERROR != ulRC)
         break;

   /****************************************************************************/
   /* CH01  Wait/test  for ISPOS to complete removal of FrameManager.          */
   /****************************************************************************/
      if (!fDSPINIT_FLAG)  /* CH02 Only wait if not a DSPINIT function */
         ulRC = WaitISPOSLocZero( prdsp, ulFMISADDR); /* CH01 */


   } while (0);

   ReleaseOverlaySem(prdsp);
   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspispos::RemoveFrameManager exit ulRC %lx\n",ulRC);
   return(ulRC);

}                                      /* end of RemoveFrameMgr             */

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: Wait_DONERT                                             */
/*                                                                          */
/* FUNCTION: This function waits for the MWaveOS variable DONERT to change. */
/*           IF DONERT (done real time) does not change after a reasonable  */
/*           amount of time, the function returns DSP_MWAVEOS_ERROR.        */
/*                                                                          */
/* INPUT:  ptr to DSP structure                                             */
/*                                                                          */
/* OUTPUT:  error code                                                      */
/*                                                                          */
/* SIDE EFFECTS:                                                            */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/************************** END OF SPECIFICATIONS ***************************/

RC Wait_DONERT(PRDSP prdsp)
{
   RC         ulRC;                    /* error code                        */
   ULONG      ulAddrDONERT;
   USHORT     usWaitCnt,usTmp1,usTmp2;


   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspispos::Wait_DONERT entry prdsp %x\n",(int)prdsp);
   if (fDSPINIT_FLAG)
      return (DSP_NOERROR);            /* if INIT then dont wait       */
   ulRC = IsposLabelToAddress(prdsp, "DONERT", &ulAddrDONERT);
   if (ulRC != DSP_NOERROR)
      return (ulRC);

   ulRC = DspBIOS_D_Read(prdsp, ulAddrDONERT, (ULONG)1, &usTmp1);
   if (ulRC != DSP_NOERROR)
      return (ulRC);

   PRINT_DEBUG_FOPEN(EMBEDFILEPTR, EMBEDFILENAME, "a", "dspispos::Wait_DONERT");
   PRINT_DEBUG_BIOS_WRITE(EMBEDFILEPTR,
        "  READ13:   DONERT value at addr shown",
        EOF, (ULONG)1, prdsp, ulAddrDONERT, (ULONG)1, &usTmp1);

/****************************************************************************/
/* CH01  Wait/test  for ISPOS to complete real-time domain processing:      */
/* This is a two-step delay loop.  The first 10 reads do not have a sleep   */
/* call.  The last group of reads are slowed by the sleep.  This permits */
/* a faster response because MWAVEOS usually responds in a few millisecs    */
/* and the sleep is never needed.  Timed delay is around 1 second:       */
/* 10 loops w/o delay + 32 loops w/ delay = total of 42 times thru loop.    */
/****************************************************************************/
   usWaitCnt = 0;                      /* CH01 Init to 0 for up counter     */
   do {
      if (++usWaitCnt>10) usleep(32); /* CH01 wait 32ms after 10 reads   */
      ulRC = DspBIOS_D_Read(prdsp, ulAddrDONERT, (ULONG)1, &usTmp2);
   }  while ((usTmp1 == usTmp2) && (usWaitCnt <= 42) && (ulRC == DSP_NOERROR));

   if (usTmp1 == usTmp2) { /* CH01 before and after values should differ */
      ulRC = DSP_MWAVEOS_ERROR;
      MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspispos::Wait_DONERT error ulRC %lx Mwave/OS is stopped. DONERT is static.\n",ulRC);
   }

   PRINT_DEBUG_FOPEN(EMBEDFILEPTR, EMBEDFILENAME, "a", "dspispos::Wait_DONERT");
   PRINT_DEBUG_BIOS_WRITE(EMBEDFILEPTR,
        "  READ14:   after while-loop, DONERT value at addr shown",
        EOF, (ULONG)1, prdsp, ulAddrDONERT, (ULONG)1, &usTmp2);

   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspispos::Wait_DONERT exit ulRC %lx\n",ulRC);
   return (ulRC);
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: RemoveTaskFromOS                                        */
/*                                                                          */
/* FUNCTION: This function removes a task from the operating system by      */
/*           deleting its TCB from the frame manager's linked list of TCBs  */
/*           Starting at the head of list (FMCB), the list (each TCB)       */
/*           is walked until the TCB of the task to be removed is found     */
/*                                                                          */
/* INPUT:  ptr to task structure                                            */
/*         ptr to FMCB memory structure                                     */
/*         ptr to DDS  memory structure                                     */
/*                                                                          */
/* OUTPUT:  error code                                                      */
/*                                                                          */
/* SIDE EFFECTS:                                                            */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/************************** END OF SPECIFICATIONS ***************************/

RC RemoveTaskFromOS(PRTSK prtsk)
{
   ULONG      ulDspAddrCurrTCB;        /* DSP address of current TCB's      */ 
                                       /* NextTCB                           */
   ULONG      ulDspAddrPrevTCB=0;      /* DSP address of next TCB's NextTCB */
   ULONG      ulFMCB;                  /* CH01 Holds FM control block addr  */
   ULONG      ulDSPAddr;               /* CH01 Temp DSP address.            */ 
   PSZ        pszTemp, pszFMTemp;      /* CH01 FMTemp used for NRT label    */
   PRDSP      prdsp;                   /* CH01                              */
   RC         ulRC;                    /* error code                        */
   USHORT     usContentCurrTCB = 0;    /* content of current TCB's NextTCB  */
   USHORT     usFMCB;                  /* CH01 FMCB used for Bios_Write, etc*/
   USHORT     usTCBAddr;               /* CH01 DSP address of task TCB.     */
   PRMEM      prmemDDS;
   PRFM       prfm;


   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspispos::RemoveTaskFromOS entry prtsk %x\n",(int)prtsk);
   if ((prtsk->TSK_prsegDTail == NULL) ||
       (prtsk->TSK_prsegDTail->SEG_prsegNext == NULL) ||
       (prtsk->TSK_prsegDTail->SEG_prsegNext->SEG_prmem == NULL)) {
      return (DSP_INTERNAL_ERROR);
   }
   prmemDDS = prtsk->TSK_prsegDTail->SEG_prsegNext->SEG_prmem;
   prfm = prtsk->TSK_prfm;
   prdsp = prtsk->TSK_prdsp;           /* CH01 Init prdsp                   */

   do {

      if ((ulRC = GetOverlaySem(prdsp)) != DSP_NOERROR)
         break;

   /* setup dsp address of head of list (framemgr control block)               */

      if (prfm != NULL)
         {
         ulFMCB = prfm->FM_prmemFMCB->MEM_ulSegAddr;  /* CH01 Addr of FMCB */
         ulDspAddrCurrTCB = ulFMCB+FMCB_FIRSTTCB;     /* CH01 Addr of 1st TCB */
      }

      else {
         switch (prtsk->TSK_usflgType&TSK_TYPEMASK) {
            case (TSK_NRT1) :
               pszFMTemp = "NRT";         /* CH01 Label for NRT1 Control blk */
               pszTemp = "NRT1FTCB";
               break;
            case (TSK_NRT2) :
               pszFMTemp = "NRT2";        /* CH01 Label for NRT2 Control blk */
               pszTemp = "NRT2FTCB";
               break;
            default  :
               return (DSP_INV_TASK_TYPE);
         }                                /* endswitch                         */

         ulRC = IsposLabelToAddress(prdsp, pszTemp, &ulDspAddrCurrTCB);
         if (ulRC != DSP_NOERROR) {
	   MW_SYSLOG_3(TRACE_MANAGER_CORE,"dspispos::RemoveTaskFromOS error ulRC %lx Mwave/OS Label: %s\n",ulRC, pszTemp);
            break;
         }

         /* CH01 Get the NRT FM control block address */
         ulRC = IsposLabelToAddress(prdsp, pszFMTemp, &ulFMCB);
         if (ulRC != DSP_NOERROR) {
	   MW_SYSLOG_3(TRACE_MANAGER_CORE,"dspispos::RemoveTaskFromOS error ulRC %lx Mwave/OS Label: %s\n",ulRC, pszFMTemp);
            break;
         }
      }

   /* walk the TCB list until we find the TCB that points to the               */
   /* TCB of the task to be removed                                            */

      usTCBAddr = (USHORT) prmemDDS->MEM_ulSegAddr;  /* CH01 Save TCB for later */
      while (usContentCurrTCB != usTCBAddr) {

      /* read the contents of current TCB's nextTCB                            */

         if ((ulRC = DspBIOS_D_Read(prdsp, ulDspAddrCurrTCB, (ULONG)1,
            &usContentCurrTCB)) != DSP_NOERROR)
            break;

         PRINT_DEBUG_FOPEN(EMBEDFILEPTR, EMBEDFILENAME, "a", "dspispos::RemoveTaskFromOS");
         PRINT_DEBUG(EMBEDFILEPTR, "  prtsk (0x%X)\n", (ULONG) prtsk);
         PRINT_DEBUG_BIOS_WRITE(EMBEDFILEPTR,
            "  READ15:  contents of current TCB's nextTCB at addr shown",
            EOF, (ULONG)1, prdsp, ulDspAddrCurrTCB, (ULONG)1, &usContentCurrTCB);

      /* Have we reached the end of the list                                   */

         if (usContentCurrTCB == 0)
         {
            ulRC = DSP_INTERNAL_ERROR;
            break;
         }

      /* move to next TCB in chain                                             */

         ulDspAddrPrevTCB = ulDspAddrCurrTCB;
         ulDspAddrCurrTCB = (ULONG)usContentCurrTCB+TCB_NEXTTCB;
      }

   /* read the contents of targetTCB's nextTCB and write to previous TCB       */

      PRINT_DEBUG_FOPEN(EMBEDFILEPTR, EMBEDFILENAME, "a", "dspispos::RemoveTaskFromOS");
      PRINT_DEBUG(EMBEDFILEPTR, "  prtsk (0x%X)\n", (ULONG) prtsk);
      PRINT_DEBUG_BIOS_WRITE(EMBEDFILEPTR,
         "  READ16:  contents of targetTCB's nextTCB at addr shown",
         EOF, (ULONG)1, prdsp, ulDspAddrCurrTCB, (ULONG)1, &usContentCurrTCB);

      if ((ulRC = DspBIOS_D_Read(prdsp, ulDspAddrCurrTCB, (ULONG)1,
         &usContentCurrTCB)) != DSP_NOERROR)
         break;

      PRINT_DEBUG_FOPEN(EMBEDFILEPTR, EMBEDFILENAME, "a", "dspispos::RemoveTaskFromOS");
      PRINT_DEBUG(EMBEDFILEPTR, "  prtsk (0x%X)\n", (ULONG) prtsk);
      PRINT_DEBUG_BIOS_WRITE(EMBEDFILEPTR,
         "  WRITE17:  contents of previousTCB at addr shown",
         EOF, (ULONG)1, prdsp, ulDspAddrPrevTCB, (ULONG)1, &usContentCurrTCB);

      if ((ulRC = DspBIOS_D_Write(prdsp, ulDspAddrPrevTCB,
         (ULONG)1, &usContentCurrTCB)) != DSP_NOERROR)
         break;

   /* CH01 Begin changes major changes */
   /****************************************************************************/
   /* CH01 Use MWAVEOS facility to verify removal from the execution queue.    */
   /* The following steps must be taken: (MTS #868)                            */
   /*   1.) Unlink task TCB from linked list (see code above)                  */
   /*   2.) Write location MWAVEOS.REMTFM to the FMCB that owns the task.      */
   /*   3.) Write location MWAVEOS.REMTASK to the SYSDSPTR of the task.        */
   /*   4.) Wait for MWAVEOS.REMTASK to go to 0.                               */
   /****************************************************************************/
      ulRC = IsposLabelToAddress(prdsp, "REMTFM", &ulDSPAddr);
      if (ulRC != DSP_NOERROR) {
         break;
      }

      /* CH02 Begin changes for MTS # 2124 */
      usFMCB = (USHORT)ulFMCB;
      /* CH02 End changes for MTS # 2124   */

      /* Write FMCB to location REMTFM */

      PRINT_DEBUG_FOPEN(EMBEDFILEPTR, EMBEDFILENAME, "a", "dspispos::RemoveTaskFromOS");
      PRINT_DEBUG(EMBEDFILEPTR, "  prtsk (0x%X)\n", (ULONG) prtsk);
      PRINT_DEBUG_BIOS_WRITE(EMBEDFILEPTR,
         "  WRITE18:  FMCB to location REMTFM at addr shown",
         EOF, (ULONG)1, prdsp, ulDSPAddr, (ULONG)1, &usFMCB);

      if ((ulRC = DspBIOS_D_Write(prdsp, ulDSPAddr,
         (ULONG)1, &usFMCB)) != DSP_NOERROR)
         break;

      ulRC = IsposLabelToAddress(prdsp, "REMTASK", &ulDSPAddr);
      if (ulRC != DSP_NOERROR) {
         break;
      }

      /* Write Task TCB to location REMTASK */

      PRINT_DEBUG_FOPEN(EMBEDFILEPTR, EMBEDFILENAME, "a", "dspispos::RemoveTaskFromOS");
      PRINT_DEBUG(EMBEDFILEPTR, "  prtsk (0x%X)\n", (ULONG) prtsk);
      PRINT_DEBUG_BIOS_WRITE(EMBEDFILEPTR,
         "  WRITE19:  Task TCB to location REMTASK at addr shown",
         EOF, (ULONG)1, prdsp, ulDSPAddr, (ULONG)1, &usTCBAddr);

      if ((ulRC = DspBIOS_D_Write(prdsp, ulDSPAddr,
         (ULONG)1, &usTCBAddr)) != DSP_NOERROR)
         break;

   /****************************************************************************/
   /* CH01  Wait/test  for ISPOS to complete removal of Task: REMTASK=0.       */
   /****************************************************************************/
      if (!fDSPINIT_FLAG)  /* CH03 Only wait if not a DSPINIT function */
         ulRC = WaitISPOSLocZero( prdsp, ulDSPAddr); /* CH01 */

   /* CH01 End major changes */

   } while (0);

   ReleaseOverlaySem(prdsp);

   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspispos::RemoveTaskFromOS exit ulRC %lx\n",ulRC);
   return (ulRC);
}                                      /* end of RemoveTaskFromOS           */

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: WaitISPOSLocZero                                        */
/*                                                                          */
/* FUNCTION: This function waits for a DSP location to be reset to zero.    */
/*           It assumes that MWAVEOS is the code that is to write the zero. */
/*           Hence, it sets ulRC=DSP_MWAVEOS_ERROR if a timeout occurs.     */
/*           The timeout is set to 1 second using a two step delay.         */
/*                                                                          */
/* INPUT:  prdsp     : ptr to dsp structure                                 */
/*         ulDSPAddr : DSP address to read and wait for zero.               */
/*                                                                          */
/* OUTPUT: ulRC                                                             */
/*                                                                          */
/* SIDE EFFECTS:  time delay                                                */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/************************** END OF SPECIFICATIONS ***************************/

RC WaitISPOSLocZero(PRDSP prdsp, ULONG ulDSPAddr)
{
   RC     ulRC;                        /* function return code              */
   USHORT usCount;                     /* Used for the delay loop counter   */
   USHORT usValue;                     /* Used to store value read from DSP */

/****************************************************************************/
/* Wait for MWAVOS to complete some activity:                               */
/* This is a two-step delay loop.  The first 10 reads do not have a sleep   */
/* call.  The last group of reads are slowed by the sleep.  This permits */
/* a faster response because MWAVEOS usually responds in a few millisecs    */
/* and the sleep is never needed.  Timed delay is around 1 second:       */
/* 10 loops w/o delay + 32 loops w/ delay = total of 42 times thru loop.    */
/****************************************************************************/

   MW_SYSLOG_3(TRACE_MANAGER_CORE,"dspispos::WaitISPOSLocZero entry prdsp %x ulDSPAddr %lx\n",(int)prdsp,ulDSPAddr);
   usCount = 0;
   do {
      if (++usCount>10) usleep(32); /* wait 32 ms after 10 reads.   */
      if ((ulRC = DspBIOS_D_Read(prdsp, ulDSPAddr, (ULONG)1, &usValue))
         != DSP_NOERROR)  /* Read REMTASK location */
         return (ulRC);
   }  while ((usValue != 0) && (usCount <= 42));

   PRINT_DEBUG_FOPEN(EMBEDFILEPTR, EMBEDFILENAME, "a", "dspispos::WaitISPOSLocZero");
   PRINT_DEBUG_BIOS_WRITE(EMBEDFILEPTR,
      "  READ20:   after while-loop on addr to wait till 0",
      EOF, (ULONG)1, prdsp, ulDSPAddr, (ULONG)1, &usValue);

   if (usValue != 0) {  /* must have timed out */
      ulRC = DSP_MWAVEOS_ERROR;
      MW_SYSLOG_3(TRACE_MANAGER_CORE,"dspispos::WaitISPOSLocZero error ulRC %lx Mwave/OS not responding. Addr 0x%04lX not reset\n", ulRC,ulDSPAddr);
   }

   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspispos::WaitISPOSLocZero exit ulRC %lx\n",ulRC);
   return (ulRC);
}
