/***************************************************************************
                          gnuupload.cpp  -  description
                             -------------------
    begin                : Tue May 29 2001
    copyright            : (C) 2001 by
    email                : maksik@gmx.co.uk
 ***************************************************************************/

// the original version of this file was taken from Gnucleus (http://gnucleus.sourceforge.net)

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "mutella.h"
#include "controller.h"

#include "gnushare.h"
#include "gnuhash.h"
#include "gnudirector.h"
#include "preferences.h"

#include "gnuupload.h"
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <sys/ioctl.h>

#include "asyncfile.h"

#define DEF_UPLOAD_BUFFER 0x00004000

/////////////////////////////////////////////////////////////////////////////
// service: we need this set for thread-safe callbacks

static MMutex s_mutexUploadSet(true); // recursive to enable 'delete this' from callbacks
static set<MGnuUpload*> s_setUploads;
//
static inline void RegisterUpload(MGnuUpload* p)
{
	s_mutexUploadSet.lock();
	ASSERT(s_setUploads.find(p)==s_setUploads.end());
	s_setUploads.insert(p);
	s_mutexUploadSet.unlock();
}
static inline void UnregisterUpload(MGnuUpload* p)
{
	s_mutexUploadSet.lock();
	ASSERT(s_setUploads.find(p)!=s_setUploads.end());
	s_setUploads.erase(p);
	s_mutexUploadSet.unlock();
}
static inline bool IsRegistred(MGnuUpload* p)
{
	//MLock lock(s_mutexUploadSet); // it is always called with mutex locked
	return s_setUploads.find(p)!=s_setUploads.end();
}

//////////////////////////////////////////////////////////////////////////////
class MUploadFile : public MAsyncFile
{
public:
	MUploadFile(MGnuUpload* pUpload);
	virtual void OnSuccess();
	virtual void OnError();
	void GiveAway(){
		s_mutexUploadSet.lock();
		m_pBuddy = NULL;
		s_mutexUploadSet.unlock();
	}
protected:
	MGnuUpload* m_pBuddy;
private:
	virtual ~MUploadFile();
};

MUploadFile::MUploadFile(MGnuUpload* pBuddy) : MAsyncFile(AFM_READ, DEF_UPLOAD_BUFFER), m_pBuddy(pBuddy)
{
}

MUploadFile::~MUploadFile()
{
}

void MUploadFile::OnSuccess()
{
	//TRACE2("MUploadFile::OnSuccess; request=", m_pRequest->type);
	s_mutexUploadSet.lock();
	//ASSERT(m_pThread==s_context.pAFThreadRead || m_pThread==s_context.pAFThreadWrite);
	//
	switch (m_pRequest->type_compl)
	{
		case AFR_OPEN:
			if (m_pBuddy && IsRegistred(m_pBuddy))
				m_pBuddy->OnFileOpen();
		break;
		case AFR_READ:
		case AFR_PREREAD:
			if (m_pBuddy && IsRegistred(m_pBuddy))
				m_pBuddy->OnFileRead(m_pRequest->size, m_pRequest->nReturn, m_pRequest->nErrNo);
		break;
		case AFR_NONE:
			TRACE("MUploadFile::OnSuccess: AFR_NONE - should not happen");
		break;
	}
	//
	s_mutexUploadSet.unlock();
}

void MUploadFile::OnError()
{
	//TRACE2("MUploadFile: on error; request=",m_pRequest->type_compl);
	s_mutexUploadSet.lock();
	//ASSERT(m_pThread==s_context.pAFThreadRead || m_pThread==s_context.pAFThreadWrite);
	//
	if (m_pBuddy && IsRegistred(m_pBuddy))
		m_pBuddy->OnFileError(m_pRequest->nErrNo);
	if (m_pRequest->type_compl == AFR_CLOSE)
		delete this;
	s_mutexUploadSet.unlock();
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////

MGnuUpload::MGnuUpload(MGnuDirector* pComm)
{
	RegisterUpload(this);
	//
	MLock lock(m_mutex);
	//
	m_pDirector  = pComm;
	m_pShare = m_pDirector->GetShare();
	m_pPrefs = m_pDirector->GetPrefs();

	m_sFileName = "Unknown";
	m_nFileLength = 0;
	m_pAFile = new MUploadFile(this);

	m_nBytesCompleted = 0;
	m_nFileIndex = -1;
	m_nFileIndexPush = -1;

	// Bandwidth
	for(int i = 0; i < 60; i++)
		m_dwAvgBytes[i] = 0;

	m_nSecsUnderLimit = 0;
	m_nSecsDead       = 0;
	m_dwBytes60       = 0;
	m_dwSecBytes      = 0;
    m_dRate           = 0;
	m_dMaxRate        = -1;
	m_bBelowRateLimit = true;
	m_bBufferReady    = false;
	
	if(m_pPrefs->m_dBandwidthTotal > 0)
		m_dMaxRate = 0;
	else
		m_dMaxRate = -1;
	
	m_nSecNum = 0;
	m_nSecPos = 0;

	//
	m_dwID = 0;

	m_nError = UPLERR_NONE;
	StatusUpdate(TRANSFER_CONNECTING);
}

MGnuUpload::~MGnuUpload()
{
	//TRACE("~MGnuUpload");
	UnregisterUpload(this);
	//
	m_mutex.lock();
	
	ForceDisconnect();
	
	if( m_pAFile )
	{
		((MUploadFile*)m_pAFile)->GiveAway();
		m_pAFile->Close();
		m_pAFile->Destroy(); // will call 'delete this'
		m_pAFile = NULL;
	}
	m_mutex.unlock();
}

/////////////////////////////////////////////////////////////////////////////
// MGnuUpload member functions

LPCSTR SGnuUpload::GetErrorString(int nError)
{
	switch (nError)
	{
		case UPLERR_NONE:
			return "";
		case UPLERR_NODATA:
			return "No Data";
		case UPLERR_SOCKET_EROR:
			return "Socket Error";
		case UPLERR_BAD_PUSH:
			return "Bad Push";
		case UPLERR_BUSY:
			return "Busy";
		case UPLERR_NOT_FOUND:
			return "No File";
		case UPLERR_REM_CANCELED:
			return "Rem. Canceled";
		case UPLERR_CONNECT_FAIL:
			return "Connect Fail";
		case UPLERR_BAD_REQUEST:
			return "Bad Request";
		case UPLERR_NO_RESPONSE:
			return "No Response";
		case UPLERR_TOO_SLOW:
			return "Too Slow";
	}
	return "Unknown";
}

void MGnuUpload::OnConnect(int nErrorCode) 
{
	MLock lock(m_mutex);
	
	if (TRANSFER_PUSH_CONNECTING != m_nStatus)
	{
		MAsyncSocket::OnConnect(nErrorCode);
		return;
	}

	BYTE* ClientID = (BYTE*) m_pDirector->GetClientID();
	char szGuid[40];
	sprintf(szGuid,"%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x",
				ClientID[0],  ClientID[1],  ClientID[2],  ClientID[3],  ClientID[4],
				ClientID[5],  ClientID[6],  ClientID[7],  ClientID[8],  ClientID[9],
				ClientID[10], ClientID[11], ClientID[12], ClientID[13], ClientID[14],
				ClientID[15]);
	MakeUpper(szGuid);

	CString HttpGiv = "GIV " + DWrdtoStr(m_nFileIndex) + ":" + szGuid + "/" + m_sFileName + "\n\n";

	Send(HttpGiv.c_str(), HttpGiv.length());
	StatusUpdate(TRANSFER_PUSH_CONNECTED);

	m_sHandshake += HttpGiv;
	
	MAsyncSocket::OnConnect(nErrorCode);
}

void MGnuUpload::OnReceive(int nErrorCode)
{
	//TRACE("MGnuUpload::OnReceive");
	MLock lock(m_mutex);
	
	BYTE* pBuff = new BYTE[6000];
	DWORD dwBuffLength = Receive(pBuff, 4096);
	
	switch (dwBuffLength)
	{
	case 0:
		delete [] pBuff;
		m_nError = UPLERR_NODATA;
		ForceDisconnect();
		return;
	case SOCKET_ERROR:
		delete [] pBuff;
		m_nError = UPLERR_SOCKET_EROR;
		ForceDisconnect();
		return;
	}
	
	if (m_nStatus != TRANSFER_PUSH_CONNECTED)
	{
		delete [] pBuff;
		MAsyncSocket::OnReceive(nErrorCode);
		return; // ignore everything
	}

	pBuff[dwBuffLength] = 0;
	CString Header((char*)pBuff);
	m_sHandshake += Header;

	// New Upload
	if(Header.find("\r\n\r\n") >= 0 )
	{
		CString Handshake = Header;

		if(Handshake.find("GET /get/") == 0)
		{
			// Get Node info
			CString Host;
			UINT    nPort;
			GetPeerName(Host, nPort);
			
			// Set Variables
			m_ipHost       = StrtoIP(Host.c_str());
			m_nPort        = 0;
			// unlock/lock here doesnt look elegant, but we heed it
			UploadFile(Handshake,false);
		}
		else
		{
			m_nError = UPLERR_BAD_PUSH;
			ForceDisconnect();
		}
	}

	delete [] pBuff;

	MAsyncSocket::OnReceive(nErrorCode);
}

void MGnuUpload::Send_HttpOK()
{	
	CString sLength = DWrdtoStr(m_nFileLength);

	if(m_nBytesCompleted)
		sLength = "BYTEs=" + DWrdtoStr(m_nBytesCompleted) + "-" + DWrdtoStr(m_nFileLength - 1) + "/" + DWrdtoStr(m_nFileLength);

	CString HttpOK =  "HTTP 200 OK\r\n";
			HttpOK += "Server: Mutella\r\n";
			HttpOK += "Content-type:application/binary\r\n";
			if(m_nBytesCompleted)
				HttpOK += "Accept-Ranges: BYTEs\r\nContent-range: " + sLength + "\r\n";
			else
				HttpOK += "Content-length: " + sLength + "\r\n";
			HttpOK += "\r\n";
	Send(HttpOK.c_str(), HttpOK.length());

	StatusUpdate(TRANSFER_SENDING);
	m_sHandshake += HttpOK;
}

void MGnuUpload::Send_HttpBusy()
{
	CString Http503 =  "HTTP 503\r\n";
			Http503 += "Server: Mutella\r\n";
			Http503 += "Content-type:text/html\r\n\r\n";
			Http503 += "<HTML>\r\n";
			Http503 += "<HEAD><TITLE>503 Server Busy</TITLE></HEAD>\r\n";
			Http503 += "<BODY>\r\n";
			Http503 += "<H1>Server Busy</H1>\r\n";
			Http503 += "This server's upload max has been met, try again later.\r\n";
			Http503 += "<HR NOSHADE SIZE=1>\r\n";
			Http503 += "Powered by <I><B><A HREF=\"http://mutella.sourceforge.net\">Mutella</A></B></I>.\r\n";
			Http503 += "</BODY>\r\n";
			Http503 += "</HTML>\r\n\r\n";
	Send(Http503.c_str(), Http503.length());

	m_nError = UPLERR_BUSY;
	Close();
	StatusUpdate(TRANSFER_CLOSED);
	m_sHandshake += Http503;
}

void MGnuUpload::Send_HttpNotFound()
{
	CString Http404 =  "HTTP 404\r\n";
			Http404 += "Server: Mutella\r\n";
			Http404 += "Content-type:text/html\r\n\r\n";
			Http404 += "<HTML>\r\n";
			Http404 += "<HEAD><TITLE>404 Not Found</TITLE></HEAD>\r\n";
			Http404 += "<BODY>\r\n";
			Http404 += "<H1>Not Found</H1>\r\n";
			Http404 += "The requested file \"<I>" + m_sFileName + "\"</I> was not found on this server.\r\n";
			Http404 += "<HR NOSHADE SIZE=1>\r\n";
			Http404 += "Powered by <I><B><A HREF=\"http://mutella.sourceforge.net\">Mutella</A></B></I>.\r\n";
			Http404 += "</BODY>\r\n";
			Http404 += "</HTML>\r\n\r\n";
	Send(Http404.c_str(), Http404.length());

	m_nError = UPLERR_NOT_FOUND;
	Close();
	StatusUpdate(TRANSFER_CLOSED);
	m_sHandshake += Http404;
}

void MGnuUpload::StatusUpdate(DWORD Status)
{
	ASSERT(m_mutex.locked());
	m_nSecsDead = 0;

	m_nStatus = Status;
	m_nChangeTime = xtime();
	
	m_pDirector->TransferMessage(UPLOAD_UPDATE, (WPARAM) this);
}

void MGnuUpload::ForceDisconnect()
{
	ASSERT(m_mutex.locked());
	
	if(m_hSocket != INVALID_SOCKET)
	{
		AsyncSelect(FD_CLOSE);
		ShutDown(2);
	}
	
	StatusUpdate(TRANSFER_CLOSED);
	Close();
}

void MGnuUpload::OnClose(int nErrorCode)
{
	MLock lock(m_mutex);
	
	if(m_nError == UPLERR_NONE)
		m_nError = UPLERR_REM_CANCELED;
	
	StatusUpdate(TRANSFER_CLOSED);
		
	MAsyncSocket::OnClose(nErrorCode);
}

void MGnuUpload::PushFile()
{
	MLock lock(m_mutex);
	
	StatusUpdate(TRANSFER_PUSH);
	
	// Set file name and size
	if(!m_pShare->GetFile(m_nFileIndex, m_sFileName, m_sFilePath, false))
	{
		m_nError = UPLERR_NOT_FOUND;
		ForceDisconnect();
		return;
	}
	
	ASSERT(m_pAFile);
	ASSERT(!m_pAFile->IsOpen());
	m_pAFile->Open(m_sFilePath.c_str());
}

void MGnuUpload::UploadFile(CString lowHandshake, bool bLock)
{
	//TRACE("MGnuUpload::UploadFile");
	MLock lock(m_mutex, bLock);
	ASSERT(m_mutex.locked());

	lowHandshake.make_lower();
	// Web browser compatibility
	ReplaceSubStr(lowHandshake,"%20", " ");
	
	// Index
	m_nFileIndex = -1;
	CString FirstLine = lowHandshake.substr(0, lowHandshake.find("\r\n") );
	
	sscanf( FirstLine.c_str(), "get /get/%ld/", &m_nFileIndex);

	// Filename
	int Begin  = FirstLine.find("/", 9) + 1;
	int	End    = FirstLine.find(" http/", Begin);
	if (End < 0)
		End = FirstLine.length();
	m_sFileName = FirstLine.substr(Begin, End - Begin);
	
	// If there's an upload max
	if(m_pPrefs->m_nMaxUploads >= 0 && m_pDirector->CountUploading() >= m_pPrefs->m_nMaxUploads)
	{
		Send_HttpBusy();
		return;
	}

	// If we are already uploading to this host, give a busy signal
	if ( m_pPrefs->m_nMaxPerHostUploads >= 0 && m_pDirector->GetUploadsCount(m_ipHost, this) >= m_pPrefs->m_nMaxPerHostUploads )
	{
		Send_HttpBusy();
		return;
	}
	
	// Range
	m_nBytesCompleted = 0;
	Begin = lowHandshake.find("\r\nrange: bytes=");
	if(Begin > 0)
	{	
		CString sRange = lowHandshake.substr(Begin);
		sscanf( sRange.c_str(), "\r\nrange: bytes=%ld-", &m_nBytesCompleted);
	}
	
	// here compex part starts: if it is a push we have file already open, but it would make sence to verify if it is the right file.
	ASSERT(m_pAFile);
	if (m_nStatus != TRANSFER_PUSH_CONNECTING )
	{
		// normal transfer
		if(m_nFileIndex == -1 || !m_pShare->GetFile(m_nFileIndex, m_sFileName, m_sFilePath, true))
		{
			m_nBytesCompleted = -1;
			Send_HttpNotFound();
			return;
		}
		StatusUpdate(TRANSFER_CONNECTED);
		m_pAFile->Open(m_sFilePath.c_str());
		return;
	}
	// so the file is already open, lets check if it is the right one
	if (m_nFileIndexPush != m_nFileIndex || !m_pAFile->IsOpen())
	{
		TRACE("wrong push");
		m_nBytesCompleted = -1;
		Send_HttpNotFound();
		return;
	}
	//
	StatusUpdate(TRANSFER_CONNECTED);
	// need to call OnFileOpen or do what is done there
	OnFileOpen(false);
	
	m_pDirector->TransferMessage(UPLOAD_UPDATE, 0);
}

void MGnuUpload::OnFileOpen(bool bLock)
{
	//TRACE("OnFileOpen");	
	MLock lock(m_mutex, bLock);
	ASSERT(m_mutex.locked());
	
	// called in two cases: push request and actual upload request.
	// we keep file open after push request.
	
	if (TRANSFER_PUSH == m_nStatus)
	{
		StatusUpdate(TRANSFER_PUSH_CONNECTING);
		m_nFileIndexPush = m_nFileIndex;
		m_nFileLength = m_pAFile->GetSize();
		if(!Connect(IPtoStr(m_ipHost).c_str(), m_nPort))
			if(GetLastError() != EWOULDBLOCK && GetLastError() != EINPROGRESS)
			{
				m_nError = UPLERR_CONNECT_FAIL;
				StatusUpdate(TRANSFER_CLOSED);
				Close();
			}
		return;
	}

	m_nFileLength = m_pAFile->GetSize();
	
	if (m_nBytesCompleted < 0 || m_nBytesCompleted>=m_nFileLength)
	{
		m_nError = UPLERR_BAD_REQUEST;
		StatusUpdate(TRANSFER_CLOSED);
		Close();
		return;
	}
	
	Send_HttpOK();
	
	//TRACE("MGnuUpload::OnFileOpen -- calling PreFetch");
	if(m_nBytesCompleted)
		VERIFY(m_pAFile->PrefetchSeek(m_nBytesCompleted,SEEK_SET));
	else
		VERIFY(m_pAFile->Prefetch());
}

void MGnuUpload::OnFileRead(int Requested, int nRead, int nError)
{
	//TRACE("MGnuUpload::OnFileRead");
	
	MLock lock(m_mutex);
	
	m_bBufferReady = true;
	UpdateSelectFlags();
}

void MGnuUpload::OnFileError(int nError)
{
	//TRACE("OnFileError");
	MLock lock(m_mutex);
	
	if ( m_nStatus == TRANSFER_CONNECTED )
		Send_HttpNotFound();
	// TODO: set m_nError for other cases correctly
	ForceDisconnect();
}

void MGnuUpload::UpdateSelectFlags()
{
	if (m_bBelowRateLimit && m_bBufferReady)
		ModifySelectFlags(FD_WRITE,0);
	else
		ModifySelectFlags(0,FD_WRITE);
}

void MGnuUpload::BandwidthTimer()
{
	//TRACE("BandwidthTimer");
	MLock lock(m_mutex);
	
	m_nSecNum++;
	if(m_nSecPos >= 60)
		m_nSecPos = 0;

	// Bytes
	m_dwBytes60 -= m_dwAvgBytes[m_nSecPos];
	m_dwAvgBytes[m_nSecPos] = m_dwSecBytes;
	m_dwBytes60 += m_dwSecBytes;
	if (m_nSecNum)
		if (m_nSecNum >= 60)
			m_dRate = m_dwBytes60/60.0;
		else
			m_dRate = m_dwBytes60/m_nSecNum;

	// Mininum Trasfer Speed Verification
	if(TRANSFER_CONNECTING == m_nStatus ||
	   TRANSFER_PUSH_CONNECTING == m_nStatus ||
	   TRANSFER_CONNECTED  == m_nStatus)
	{
		m_nSecsDead++;

		if(m_nSecsDead > 15)
		{
			m_nError = UPLERR_NO_RESPONSE;
			StatusUpdate(TRANSFER_CLOSED);
			Close();
		}
	}
	else if(TRANSFER_PUSH == m_nStatus)
	{
		m_nSecsDead++;

		if(m_nSecsDead > 15)
		{
			m_nError = UPLERR_CONNECT_FAIL;
			StatusUpdate(TRANSFER_CLOSED);
			Close();
		}
	}
	else if(TRANSFER_SENDING == m_nStatus)
	{
		// Check for dead transfer
		if(m_dwBytes60 == 0)
		{
			m_nSecsDead++;

			if(m_nSecsDead > 30)
			{
				m_nError = UPLERR_NO_RESPONSE;
				Close();
				StatusUpdate(TRANSFER_CLOSED);
			}
		}
		else
			m_nSecsDead = 0;

		if(m_pPrefs->m_dMinUpSpeed > 0 && m_nSecNum > 30)
		{
			// Check if its under the bandwidth limit
			if(m_dRate < m_pPrefs->m_dMinUpSpeed)
				m_nSecsUnderLimit++;
			else
				m_nSecsUnderLimit = 0;

			if(m_nSecsUnderLimit > 30)
			{	
				m_nError = UPLERR_TOO_SLOW;
				Close();
				StatusUpdate(TRANSFER_CLOSED);
			}
		}
		
		// bandwidth limits
		if (TRANSFER_SENDING == m_nStatus && !m_bBelowRateLimit)
		{
			if (m_dMaxRate<0 || m_dRate<m_dMaxRate)
			{
				m_bBelowRateLimit = true;
				UpdateSelectFlags();
			}
		}
	}
	else if(TRANSFER_CLOSED == m_nStatus)
	{
	}

	m_dwSecBytes = 0;
	m_nSecPos++;

	// Check for completion
	if(TRANSFER_SENDING == m_nStatus && ( m_nFileLength <= m_nBytesCompleted ))
	{
		ASSERT(m_nFileLength);
		//m_pDirector->m_pController->SetHaveUploaded(true);
		Close();
		StatusUpdate(TRANSFER_COMPLETED);
	}
}

void MGnuUpload::OnSend(int nErrorCode) 
{
	//TRACE("OnSend");
	MLock lock(m_mutex);
	
	ASSERT(m_pAFile);
	if (!m_pAFile->BytesAvail())
	{
		TRACE("WARNING: MGnuUpload::OnSend occured when no data is available");
		return ; // quick and dirty
	}
	if (m_nStatus != TRANSFER_SENDING)
		return ; // quick and dirty
	if(m_nBytesCompleted >= m_nFileLength)
		return ; // quick and dirty
		
	// we have what to send -- do it
	int nBytesAvail = m_pAFile->BytesAvail();
	int nBytesSentNow = Send(m_pAFile->GetBuffer(), nBytesAvail);
	if (nBytesSentNow <= 0)
	{
		m_pAFile->ReleaseBuffer(0);
		ForceDisconnect();
		m_nStatus = TRANSFER_CLOSED;
		m_nError = UPLERR_SOCKET_EROR;
		return;
	}
	VERIFY(m_pAFile->ReleaseBuffer(nBytesSentNow));
	// total count, statistics, etc
	m_dwSecBytes += nBytesSentNow;
	m_nBytesCompleted += nBytesSentNow;
	ASSERT(m_pAFile->GetPos()==m_nBytesCompleted);
	// limit bandwidth
	if (m_dMaxRate>0 && m_dwSecBytes>m_dMaxRate && m_bBelowRateLimit)
	{
		m_bBelowRateLimit = false;
		UpdateSelectFlags();
	}
	// check for completion
	if (m_nBytesCompleted == m_nFileLength)
	{
		 ForceDisconnect();
		 m_nStatus = TRANSFER_COMPLETED;
	}
	//
	if (!m_pAFile->BytesAvail())
	{
		m_bBufferReady = false;
		UpdateSelectFlags();
		VERIFY(m_pAFile->Prefetch());
	}
	MAsyncSocket::OnSend(nErrorCode);
}

