/*
** Modular Logfile Analyzer
** Copyright 2000 Jan Kneschke <jan@kneschke.de>
**
** Homepage: http://www.kneschke.de/projekte/modlogan
**

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version, and provided that the above
    copyright and permission notice is included with all distributed
    copies of this or derived software.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA

**
** $Id: mplugins.c,v 1.31 2002/01/03 12:15:19 ostborn Exp $
*/

#include <stdio.h>
#include "ltdl.h"
#include <stdlib.h>
#include <assert.h>
#include <string.h>

#include "mconfig.h"
#include "mrecord.h"
#include "mplugins.h"

mplugin *mplugins_create() {
	mplugin *plug = malloc(sizeof(mplugin));
	memset(plug, 0, sizeof(mplugin));
	
	return plug;
}

void mplugins_destroy(mplugin *plug) {
	if (plug) free(plug);
}

int mplugins_load_plugins(mconfig *conf) {
	const char *error;
	char *filename = NULL;
	mlist *l;
	mplugin *func;
	
	int (*initfunc)(mplugin *func);
	
/* reset all elements */
	
	if (conf->loadplugins == NULL) return -1;
	
	l = conf->loadplugins;
	
	while (l) {
		char *fn, *cfg;
		if (l->data == NULL) {
			l = l->next;
			continue;
		}
		
		if ((cfg = strchr(l->data->key, ','))) {
			*cfg++ = '\0';
			while (*cfg == ' ') cfg++;
			fn = l->data->key;
		} else {
			cfg = fn = l->data->key;
			
		}
		
		if (conf->debug_level > 0)
			fprintf(stderr, "%s.%d: loading plugin %s\n", __FILE__, __LINE__, fn);
		
		filename = malloc(strlen("/libmla_.la") + strlen(fn) + strlen(LIBDIR) + 1);
		assert(filename);

		sprintf(filename, "%s/libmla_%s.la", LIBDIR, l->data->key);
		
		func = mplugins_create();
	
		func->handle = lt_dlopen (filename);
		if (conf->debug_level > 2)
			fprintf(stderr, "%s.%d: %s = %p\n", __FILE__, __LINE__, l->data->key, func->handle);
	
		if (func->handle == NULL) {
			sprintf(filename, "%s/libmla_%s.so", LIBDIR, l->data->key);
			func->handle = lt_dlopen (filename);
			if (func->handle == NULL && (error = lt_dlerror()) != NULL) {
				fprintf(stderr, "%s.%d: %s -> %s\n", __FILE__, __LINE__, error, filename);
				
				return(-1);
			}
		}
	
		if ((initfunc = lt_dlsym(func->handle, "mplugins_init")) == NULL) {
			fprintf(stderr, "%s.%d: plugin %s (%s) - no init function found\n", __FILE__, __LINE__, l->data->key, filename);
			return -1;
		}
		
		initfunc(func);
		
		if (func->dlinit)
			if (func->dlinit(conf)) {
				fprintf(stderr, "%s.%d: plugin %s (%s) - dlinit failed\n", __FILE__, __LINE__, l->data->key, filename);
				return -1;
			}
		
		if (func->parse_config)
			if (func->parse_config(conf, NULL, cfg)) {
				fprintf(stderr, "%s.%d: plugin %s (%s) - configfile parser failed\n", __FILE__, __LINE__, l->data->key, filename);
				return -1;
			}
		
		if (conf->show_options == 0 && func->set_defaults) 
			if (func->set_defaults(conf)) {
				fprintf(stderr, "%s.%d: plugin %s (%s) - setting defaults failed\n", __FILE__, __LINE__, l->data->key, filename);
				return -1;
			}
		
		func->config = conf->plugin_conf;
		((mplugin **)(conf->plugins))[conf->plugin_count] = func;
		if (conf->debug_level > 1)
			fprintf(stderr, "%s.%d: plugin %s (%s) loaded\n", __FILE__, __LINE__, l->data->key, filename);
		free(filename);
		
		l = l->next;
		conf->plugin_count++;
	}
	
	if (conf->plugin_count != mlist_count(conf->loadplugins)) {
		fprintf(stderr, "%s.%d: estimated plugin count and real plugin count differ: %d != %d\n", __FILE__, __LINE__, 
			conf->plugin_count, mlist_count(conf->loadplugins));
		
		return -1;
	}
	
	return 0;
}

int mplugins_setup(mconfig *conf) {
	if (lt_dlinit () != 0) {
		return -1;
	}
	
	conf->plugins = malloc(mlist_count(conf->loadplugins) * sizeof(mplugin));
	conf->plugin_count = 0;
	
	return 0;
}

int mplugins_free(mconfig *conf) {
	mplugin *func;
	int i;

	for (i = 0; i < conf->plugin_count; i++) {

		func = ((mplugin **)(conf->plugins))[i];
		conf->plugin_conf = func->config;

		if (func->dlclose)
			func->dlclose(conf);
#ifndef WITH_DMALLOC
		/* if we are compiling with -ldmalloc we shouldn't remove the 
		 * shared libs. dmalloc will crash otherwise.
		 */
		lt_dlclose(func->handle);
#endif	
		mplugins_destroy(func);
	}
	free(conf->plugins);
#ifndef WITH_DMALLOC
	lt_dlexit();
#endif	
	return 0;
}

