/*
** Modular Logfile Analyzer
** Copyright 2000 Jan Kneschke <jan@kneschke.de>
**
** Homepage: http://www.kneschke.de/projekte/modlogan
**

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version, and provided that the above
    copyright and permission notice is included with all distributed
    copies of this or derived software.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA

**
** $Id: main.c,v 1.73 2002/01/03 12:13:08 ostborn Exp $
*/

/**
 * \mainpage ModLogAn API Documenation
 * 
 * \section Introduction
 * Add docs.
 *
 */

#include <libintl.h>
#include <locale.h>
#include <stdlib.h>
#include <stdio.h>
#include <time.h>
#include <sys/time.h>
#include <unistd.h>
#include <dirent.h>
#include <errno.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <assert.h>

#include <libxml/parser.h>
#include <libxml/xmlerror.h>
#include <zlib.h>

#include "config.h"

#ifdef HAVE_GETOPT_H
#include <getopt.h>
#endif

#include "mrecord.h"
#include "mstate.h"
#include "mlocale.h"
#include "mconfig.h"
#include "mplugins.h"
#include "mdatatypes.h"
#include "misc.h"
#include "datatypes/webhist/datatype.h"
#include "datatypes/state/datatype.h"

#define HISTORY_FILE_VERSION	"0.2"

#define MSTATE_WRITE 1

int history_write(mconfig *conf, mlist *l, char *subpath) {
	char filename[255];
	gzFile *fd;
	
	sprintf(filename, "%s%s%s/mla.history.xml", 
		conf->outputdir ? conf->outputdir : ".", 
		subpath ? "/" : "",
		subpath ? subpath : "");
	
	if ((fd = gzopen(filename, "wb")) == NULL) {
		fprintf(stderr, "%s.%d: can't open %s: %s\n", __FILE__, __LINE__, filename, strerror(errno));
		return -1;
	}
	
	gzprintf(fd, "<?xml version=\"1.0\"?>\n");
	gzprintf(fd, "<!DOCTYPE history PUBLIC \"-//MLA//history\" \"mla.history.dtd\">\n");
	gzprintf(fd, "<history version=\"%s\" package=\"%s\">\n",
		HISTORY_FILE_VERSION, PACKAGE);
	
	mlist_write(fd, l);
	
	gzprintf(fd, "</history>\n");
	gzclose(fd);
	
	return 0;
}

static xmlSAXHandler mlaSAXHandler = {
	NULL,
		NULL,
		NULL,
		NULL,
		NULL,
		NULL,
		NULL,
		NULL,
		NULL,
		NULL,
		NULL,
		NULL,
		NULL,
		NULL,
		NULL,
		NULL,
		NULL,
		NULL,
		NULL,
		NULL,
		NULL,
		NULL,
		NULL,
		NULL,
		NULL,
		NULL,
		NULL
};

void history_startElement(void *user_data, const xmlChar *name, const xmlChar **attrs) {
	int i;
	mstate_stack *m = user_data;
	
	/* level 0 <state> */
	m->st_depth = 0;
	
	/* setup the array if we are here the first time */
	if (m->st[0].id == -1) {
		for (i = 0; i < M_STATE_ST_ELEM; i++) {
			m->st[i].id = 0;
			m->st[i].data = NULL;
		}
	}
#if 0
	fprintf(stderr, "--> %s\n", name);
#endif 
#if 0
	fprintf(stderr, "o->");
	for (i = 0; i < m->st_depth_max + 2; i++) {
		fprintf(stderr, " %d", m->st[i].id);
	}
	fprintf(stderr, "\n");
#endif	
	/* the first element */
	if (m->st[0].id == 0) {
		/* should be state */
		if (0 == strcmp(name, "history")) {
			m->st[0].id = 1;
			/*m->st_depth_max++;*/
		} else {
			exit(-1);
		}
	} else {
		/* level 1 <web>, <global> */
		m->st_depth = 1;
		if (m->st[m->st_depth].id == 0) {
			m->st_depth_max++;
			m->st[m->st_depth].id = 1;
		}
		if (mdata_insert_value(m, M_TAG_BEGIN, &(m->state), M_DATA_FIELDTYPE_LIST, name, attrs)) {
			M_DEBUG0(M_DEBUG_LEVEL_ERRORS, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_ERRORS,
				 "read failed - going down hard\n");
			exit(-1);
		}
	}
#if 0
	fprintf(stderr, "o-<");
	for (i = 0; i < m->st_depth_max + 2; i++) {
		fprintf(stderr, " %d", m->st[i].id);
	}
	fprintf(stderr, "\n");
#endif
}

void history_endElement(void *user_data, const xmlChar *name) {
	int i;
	mstate_stack *m = user_data;
	
	/* level 0 <state> */
	m->st_depth = 0;
	
	/* setup the array if we are here the first time */
	if (m->st[0].id == -1) {
		for (i = 0; i < M_STATE_ST_ELEM; i++) {
			m->st[i].id = 0;
			m->st[i].data = NULL;
		}
	}
	
#if 0
	fprintf(stderr, "<-- %s\n", name);
#endif
	
	/* the first element */
	if (m->st[0].id != 1) {
		/* should be history */
		exit(-1);
	} else {
		/* level 1 <web>, <global> */
		m->st_depth++;
		if (m->st[m->st_depth].id != 0) {
			m->st_depth++;
			if (mdata_insert_value(m, M_TAG_END, &(m->state), M_DATA_FIELDTYPE_LIST, name, NULL)) {
				M_DEBUG0(M_DEBUG_LEVEL_ERRORS, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_ERRORS,
					 "read failed - going down hard\n");
				exit(-1);
			}
			if (0) {
				m->st[m->st_depth].id = 0;
				m->st_depth_max--;
			}
		}
	}
}

void history_characters(void *user_data, const xmlChar *name, int len) {
	int i;
	mstate_stack *m = user_data;
	
	xmlChar *s;
	
	/* level 0 <state> */
	m->st_depth = 0;
	
	/* setup the array if we are here the first time */
	if (m->st[0].id == -1) {
		for (i = 0; i < M_STATE_ST_ELEM; i++) {
			m->st[i].id = 0;
			m->st[i].data = NULL;
		}
	}
	
#if 0	
	fprintf(stderr, "--| %s\n", s);
#endif	
	/* the first element */
	if (m->st[0].id != 1) {
		/* should be history */
		exit(-1);
	} else {
		s = malloc((len + 1) * sizeof(xmlChar));
		strncpy(s, name, len);
		s[len] = '\0';
		/* level 1 <web>, <global> */
		m->st_depth++;
		if (m->st[m->st_depth].id != 0) {
			m->st_depth++;
			if (mdata_insert_value(m, M_TAG_TEXT, &(m->state), M_DATA_FIELDTYPE_LIST, s, NULL)) {
				M_DEBUG0(M_DEBUG_LEVEL_ERRORS, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_ERRORS,
					 "read failed - going down hard\n");
				exit(-1);
			}
		}
	}
	
	free(s);
}


int history_read(mconfig *conf, mlist *l, char *subpath) {
	char filename[255];
	mstate_stack m;
	struct stat s;

	sprintf(filename, "%s%s%s/mla.history.xml", 
		conf->outputdir ? conf->outputdir : ".", 
		subpath ? "/" : "",
		subpath ? subpath : "");
	
	mlaSAXHandler.startElement = history_startElement;
	mlaSAXHandler.endElement = history_endElement;
	mlaSAXHandler.characters = history_characters;
	mlaSAXHandler.warning = xmlParserWarning;
	mlaSAXHandler.error = xmlParserError;
	mlaSAXHandler.fatalError = xmlParserError;
	
	m.ext_conf = conf;
	m.st[0].id = -1;
	m.state = l;
	m.st_depth_max = 0;
	m.st_depth = 0;
	
	if (0 == stat(filename, &s)) {
		xmlSAXUserParseFile(&mlaSAXHandler, &m, filename);
	} else {
		M_DEBUG1(M_DEBUG_LEVEL_ERRORS, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_ERRORS,
			 "stating history-file '%s' failed\n", filename);
	}
	
	return 0;
}

void show_version() {
	printf("%s %s\n", PACKAGE, VERSION);
	fflush(stdout);
}

void show_header() {
	printf("%s %s\n", PACKAGE, VERSION);
	fflush(stdout);
}

void show_usage() {
	show_header();
	

#ifdef HAVE_GETOPT_LONG
	printf("Options:\n");
	printf(" -c <configfile>    filename of the configfile\n");
	printf(" -h --help          this help screen\n");
	printf(" -v --version       display version\n");
#else
	printf("Options:\n");
	printf(" -c <configfile>    filename of the configfile\n");
	printf(" -h                 this help screen\n");
	printf(" -v                 display version\n");
#endif
};

int restore_internal_state (mconfig *conf, mlist *state_list) {
	char *fn;
	char buf[255];
	FILE *split_file;
	
	if (conf->outputdir == NULL) {
		return -1;
	}
	
	fn = malloc(strlen(conf->outputdir) + strlen("/modlogan.statefiles") + 1);
	strcpy(fn, conf->outputdir);
	strcat(fn, "/modlogan.statefiles");
	
	if ((split_file = fopen(fn, "r")) == NULL) {
		free(fn);
		return -1;
	}

	while (fgets(buf, sizeof(buf), split_file)) {
		mstate *state = NULL;
		mlist *hist = mlist_init();
		
		/* remove the \n */
		buf[strlen(buf)-1] = '\0';
		
		M_DEBUG2(conf->debug_level, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_VERBOSE,
			"reading statefile from %s/%s/mla.state.xml\n", conf->outputdir, buf);

		if (history_read(conf, hist, buf)) {
			fprintf(stderr, "%s.%d: hist failed ?\n", __FILE__, __LINE__);
			mlist_free(hist);
			hist = NULL;
		}

		if (conf->incremental) {
			state = mstate_init();
			
			if (mstate_read(conf, state, 0, 0, buf)) {
				mstate_free(state);
				state = NULL;
			}
		}
				
		if (!hist || (conf->incremental && !state)) {
			if (hist)	mlist_free(hist);
			else fprintf(stderr, "%s.%d: reading history failed\n", __FILE__, __LINE__);
			
			if (state)	mstate_free(state);
			else if (conf->incremental) fprintf(stderr, "%s\n", _("Reading State file failed"));
			
			state = NULL;
			hist = NULL;
		} else {
			mdata *data = mdata_State_create(buf, state, hist);
			mlist_insert(state_list, data);
					
			M_DEBUG1(conf->debug_level, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_VERBOSE,
				"state restored for '%s'\n", buf);
		} 
	}

	fclose(split_file);
	free(fn);
	
	return 0;
}

typedef struct {
	mlogrec *record;
	int rec_state;
} mgetrec;


int get_next_record (mconfig *conf, mlogrec *rec) {
	mplugin *func;
	int i;
	int state = M_RECORD_EOF;
	static mgetrec **recs = NULL;
	time_t t = -1;
	int ndx = -1;
	
	if (recs == NULL) {
		/* a small mem leak. fix it if you like */
		recs = malloc(sizeof(mgetrec *) * conf->plugin_count);
		for (i = 0; i < conf->plugin_count; i++) {
			/* a leak - but we don't care about this one either */
			recs[i] = malloc(sizeof(mgetrec));
			recs[i]->record = mrecord_init();
			recs[i]->rec_state = M_RECORD_EOF;
		}
	}
	
	for (i = 0; i < conf->plugin_count; i++) {
		func = ((mplugin **)(conf->plugins))[i];
		conf->plugin_conf = func->config;
		
		if (func->get_next_record) {
			if (recs[i]->record->ext_type == M_RECORD_TYPE_UNSET) {
				recs[i]->rec_state = func->get_next_record(conf, recs[i]->record);
			}
		
			if (t == -1 || recs[i]->record->timestamp < t) {
				t = recs[i]->record->timestamp;
				ndx = i;
			}
		}
	}
	
	if (ndx != -1) {
		mrecord_move(rec, recs[ndx]->record);
		state = recs[ndx]->rec_state;
	}

	return state;
}

int generate_monthly_output(mconfig *conf, mstate *state, const char *subpath) {
	int i;
	mplugin *func;
	for (i = 0; i < conf->plugin_count; i++) {
		func = ((mplugin **)(conf->plugins))[i];
		conf->plugin_conf = func->config;
		
		if (func->gen_report &&	func->gen_report(conf, state, subpath)) return -1;
	}

	return 0;
}
int generate_history_output(mconfig *conf, mlist *history, const char *subpath) {
	int i;
	mplugin *func;
	for (i = 0; i < conf->plugin_count; i++) {
		func = ((mplugin **)(conf->plugins))[i];
		conf->plugin_conf = func->config;

		if (func->gen_history && func->gen_history(conf, history, subpath)) return -1;
	}

	return 0;
}

void dump (mlist *state_list) {
	mlist *s = state_list;
	int i;
	
	while (s) {
		mstate_web *w;
		if (!s->data) return;
		
		w = ((mstate *)(s->data->data.state.state))->ext;
	
		for (i = 0; i < 31; i++) {
			fprintf(stderr, "dump_list (%p)->'%s': %d: %ld\n", s, s->data->key, i, w->days[i].hits);
		}
		
		s = s->next;
	}
}

int insert_record(mconfig *conf, mlist *state_list, mlogrec *record) {
	int i = 0;
	mplugin *func;
	
	for (i = 0; i < conf->plugin_count; i++) {
		func = ((mplugin **)(conf->plugins))[i];
		conf->plugin_conf = func->config;

		if (func->insert_record)
			if (func->insert_record(conf, state_list, record))
				return -1;
	}

	return 0;
}

int main (int argc, char **argv) {
	mconfig *conf = NULL;
	mlogrec *rec = NULL;
	mlist *state_list = NULL;
	mlist *l;
	
	int ret, l_month = -1, l_year = -1, l_hour = -1, l_day = -1, i;
	char *conf_filename = NULL;
	long lines = 0, lines_corrupt = 0, lines_skipped = 0, lines_ignored = 0;
	int first_valid_line = 0;
	int root_check = 1;

	struct tm *tm;
	
	time_t l_stamp = -1;
	mtimer process_timer, post_process_timer, parse_timer, setup_timer;
	
#ifdef HAVE_GETOPT_LONG	
	int option_index = 0;
	static struct option long_options[] = {
		{"help", 0, NULL, 0},
		{"version", 0, NULL, 0},
		{"root", 0, NULL, 0},
		{NULL, 0, NULL, 0}
	};
#endif
	
	MTIMER_RESET(process_timer);
	MTIMER_RESET(post_process_timer);
	MTIMER_RESET(parse_timer);
	MTIMER_RESET(setup_timer);
	
#ifdef HAVE_GETOPT_LONG	
	while ((i = getopt_long(argc,argv,"c:hvr", 
				long_options, &option_index)) != -1) {
		
		/* map the long options to the short ones */
		if (i == 0) {
			switch(option_index) {
			case 0: i = 'h'; break;
			case 1: i = 'v'; break;
			case 2: i = 'r'; break;
			default:
				show_usage();
				exit( EXIT_FAILURE);
			}
		}
#if 0
		/* for jed and it formating problems */
	}
#endif
#else
	while ((i = getopt(argc,argv,"c:hvr")) != EOF) { 
#endif		
		switch(i) {
		case 'c': 
			conf_filename = optarg;
			break;
		case 'h':
			show_usage();
			exit( EXIT_SUCCESS);
		case 'v':
			show_version();
			exit( EXIT_SUCCESS);
		case 'r':
			root_check = 0;
			break;
		default:
			show_usage();
			exit( EXIT_FAILURE);
		}
	}
		
	show_header();
	
	if (root_check) {
		if (getuid() == 0 || geteuid() == 0) {
			fprintf(stderr, "ModLogAn detected that it is running with root permissions.\n"
				"As root permissions are not directly required to run modlogan, \n"
				"modlogan decided do stop working with root permission.\n\n"
				"The reason for this decision is quite simple:\n"
				"ModLogAn doesn't want to be announced at BugTraq.\n"
				"Although this might be a little bit too defensive it is still better\n"
				"then nothing.\n\n"
				"If you still have to run modlogan with root permissions \n"
				"use the -r/--root switch which will disable this test\n");
			exit(-1);
		}
	}
	
	init_locale(conf);
	
	MTIMER_START(setup_timer);
	
	if (!(conf = mconfig_init(conf_filename))) {
		fprintf(stderr, "%s.%d: reading configfile failed - going down hard\n", __FILE__, __LINE__);
		return -1;
	}
	
	if (conf->show_options) {
		/* we only have to show some options */
		
		mconfig_free(conf);
		
		return 0;
	}
	
	state_list = mlist_init();
	rec = mrecord_init();

	restore_internal_state(conf, state_list);
	
	if (conf->incremental) {
		l = state_list;
		
		while (l) {
			mdata *data = l->data;
			mstate *int_state;
			if (!data) break;
			
			int_state = data->data.state.state;
			if (int_state->timestamp > l_stamp) {
				l_stamp = int_state->timestamp;
				l_year = int_state->year-1900;
				l_month = int_state->month-1;
			}
			
			l = l->next;
		}
	}
		
	if (l_stamp != -1) {
		M_DEBUG1(conf->debug_level, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_VERBOSE,
			 "Ignoring all records before %s", ctime(&l_stamp));
	}

	M_DEBUG0(conf->debug_level, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_NONE,
		"startup - finished\n");
	
	MTIMER_STOP(setup_timer);
	MTIMER_CALC(setup_timer);
	
	printf("[");
	
	MTIMER_START(parse_timer);
	while ((ret = get_next_record(conf, rec)) != M_RECORD_EOF) {
		MTIMER_STOP(parse_timer);
		MTIMER_CALC(parse_timer);

		MTIMER_START(process_timer);
		
		lines++;
		if (ret == M_RECORD_NO_ERROR) {
			/* HACK */
			if (rec->timestamp == 0) {
				fprintf(stderr, "%s.%d: line %ld returned no timestamp !! something strange will happen. Going down hard\n", 
					__FILE__, __LINE__,
					lines);
				return -1;
			} 
			
			/* set the first timestamp */
			if (l_month == -1) {
				tm = localtime(&(rec->timestamp));
				l_month = tm->tm_mon;
				l_year = tm->tm_year;
				l_stamp = rec->timestamp;
				l_hour = tm->tm_hour;
			}

			if (l_stamp > rec->timestamp) {
				if (first_valid_line == 0 && conf->incremental == 1) {
					lines_skipped++;
					MTIMER_STOP(process_timer);
					MTIMER_CALC(process_timer);
					MTIMER_START(parse_timer);
					continue;
				}
				M_DEBUG2(conf->debug_level, M_DEBUG_SECTION_MAINLOOP, M_DEBUG_LEVEL_VERBOSE,
					"Out of Sequence - ignoring timestamp: %ld > %ld\n", l_stamp, rec->timestamp);
				
				rec->timestamp = l_stamp;
			}
			
			tm = localtime(&(rec->timestamp));
			
			/* set last day of month */
			l_day = tm->tm_mday;

			/* generate report if we have reached a threshold (semi-online mode) */
			if (conf->gen_report_threshold > 0 && 
				(lines - lines_corrupt + 1) % conf->gen_report_threshold == 0) {
				
				MTIMER_STOP(process_timer);
				MTIMER_CALC(process_timer);
				
				printf("]\n");
				
				MTIMER_START(post_process_timer);
				M_DEBUG2(conf->debug_level, M_DEBUG_SECTION_MAINLOOP, M_DEBUG_LEVEL_WARNINGS,
					"o(t) writing month %02i - %04i\n", l_month+1, l_year+1900);
				
				l = state_list;
				
				while (l) {
					mdata *data = l->data;
					mdata *histdata = NULL;
					
					if (!data) break;
					
					histdata = mdata_WebHist_create_by_state(data->data.state.state);
				
					mlist_insert_replace(data->data.state.history, histdata);
#if MSTATE_WRITE
					if (0 != mstate_write(conf, data->data.state.state, 
							      M_STATE_WRITE_BY_MONTH, 
							      *(data->key) ? data->key : NULL)) {
						fprintf(stderr, "%s.%d: writing state file failed\n", __FILE__, __LINE__);
						return -1;
					}
					history_write(conf, data->data.state.history, *(data->key) ? data->key : NULL);
#endif
					if (0 != generate_monthly_output(conf, data->data.state.state, *(data->key) ? data->key : NULL)) {
						fprintf(stderr, "%s.%d: output generation failed - no data this month?\n", __FILE__, __LINE__);
						/* return -1; */
					}
					if (0 != generate_history_output(conf, data->data.state.history, *(data->key) ? data->key : NULL)) {
						fprintf(stderr, "%s.%d: history generation failed - going down hard\n", __FILE__, __LINE__);
						return -1;
					}
	
					l = l->next;
				}
				MTIMER_STOP(post_process_timer);
				MTIMER_CALC(post_process_timer);
				
				
				printf("[");
				MTIMER_START(process_timer);
			}
			
			/* we have left a month, generate the reports */
			if (tm->tm_mon != l_month || tm->tm_year != l_year) {
				
				MTIMER_STOP(process_timer);
				MTIMER_CALC(process_timer);
				
				printf("]\n");
				
				MTIMER_START(post_process_timer);
				
				M_DEBUG2(conf->debug_level, M_DEBUG_SECTION_MAINLOOP, M_DEBUG_LEVEL_WARNINGS,
					"o writing month %02i - %04i\n", l_month+1, l_year+1900);
				
				
				l = state_list;
				
				while (l) {
					mdata *data = l->data;
					mdata *histdata = NULL;
					
					if (!data) break;
					
					histdata = mdata_WebHist_create_by_state(data->data.state.state);
				
					mlist_insert_replace(data->data.state.history, histdata);
					
					if (conf->debug_level > 3)
						fprintf(stderr, "o(t) -- state written: (...)/%s/\n", *(data->key) ? data->key : ".");
#if MSTATE_WRITE
					mstate_write(conf, data->data.state.state, M_STATE_WRITE_BY_MONTH, *(data->key) ? data->key : NULL);
#endif
					if (0 != generate_monthly_output(conf, data->data.state.state, *(data->key) ? data->key : NULL)) {
						fprintf(stderr, "%s.%d: output generation failed - no data this month?\n", __FILE__, __LINE__);
						/* return -1; */
					}
					if (0 != generate_history_output(conf, data->data.state.history, *(data->key) ? data->key : NULL)) {
						fprintf(stderr, "%s.%d: history generation failed - going down hard\n", __FILE__, __LINE__);
						return -1;
					}
					
					mstate_free(data->data.state.state);
					data->data.state.state = mstate_init();
					
					l = l->next;
				}
				
				tm = localtime(&(rec->timestamp));
			
				l_month = tm->tm_mon;
				l_year = tm->tm_year;
				
				MTIMER_STOP(post_process_timer);
				MTIMER_CALC(post_process_timer);
				
				for (i = 0; i < ((lines % 50000) / 1000); i++) printf(" ");
				printf("[");
				
				MTIMER_START(process_timer);
			}

			if (insert_record(conf, state_list, rec) != 0) {
				fprintf(stderr, "%s.%d: inserting record failed - going down hard\n", __FILE__, __LINE__);
				return -1;
			}
			
			first_valid_line = 1;
			
			l_stamp = rec->timestamp;
		} else if (ret == M_RECORD_SKIPPED) {
			lines_skipped++;
		} else if (ret == M_RECORD_IGNORED) {
			lines_ignored++;
		} else if (ret == M_RECORD_HARD_ERROR) {
			M_DEBUG1(conf->debug_level, M_DEBUG_SECTION_MAINLOOP, M_DEBUG_LEVEL_ERRORS,
				 " a hard error occured in line %ld - going down hard\n", lines);
			
			exit(-1);
		} else {
			lines_corrupt++;
			
			M_DEBUG1(conf->debug_level, M_DEBUG_SECTION_MAINLOOP, M_DEBUG_LEVEL_ERRORS,
				 " parser reported an error in line %ld\n", lines);
		}
		mrecord_reset(rec);
		
		/* cosmetics */
		if (lines % 1000 == 0) {
			printf("."); 
			if (lines % (1000 * 50)== 0) {
				printf(" %8ld", lines);
				if (conf->debug_level > 1) {
					printf(" - %10.2f (%4ld, %4ld)",
					       (lines)/((MTIMER_GET_USER_MSEC(parse_timer)+MTIMER_GET_USER_MSEC(process_timer))/1000.0), 
					       lines_corrupt,
					       lines_skipped);
				}
				printf("\n ");
			}
			fflush(stdout); 
		}
		
		MTIMER_STOP(process_timer);
		MTIMER_CALC(process_timer);
		
		MTIMER_START(parse_timer);
	}

	MTIMER_STOP(parse_timer);
	MTIMER_CALC(parse_timer);

	printf("]\n");

	MTIMER_START(post_process_timer);

	if (l_month != -1) {
		int create_bp = 1;
		FILE *split_file;
		char *fn;
		printf(" ");

		if (conf->debug_level > 0) 
			printf("writing month %02i - %04i\n",l_month+1, l_year+1900);
		
		if (conf->debug_level > 1)
			fprintf(stderr, "%s.%d: writing the last month\n", __FILE__, __LINE__ );
		
		/* create filename */
		fn = malloc(strlen(conf->outputdir) + strlen("/modlogan.statefiles") + 1);
		strcpy(fn, conf->outputdir);
		strcat(fn, "/modlogan.statefiles");
		split_file = fopen(fn, "w");
			
		l = state_list;
				
		while (l) {
			mdata *data = l->data;
			mdata *histdata = NULL;
			
			if (!data) break;
			
			if (conf->debug_level > 1)
				fprintf(stderr, "%s.%d: subpath = '%s'\n", __FILE__, __LINE__, data->key);
			
			histdata = mdata_WebHist_create_by_state(data->data.state.state);
				
			mlist_insert_replace(data->data.state.history, histdata);
#if MSTATE_WRITE
			mstate_write(conf, data->data.state.state, M_STATE_WRITE_DEFAULT, *(data->key) ? data->key : NULL);
			history_write(conf, data->data.state.history, *(data->key) ? data->key : NULL);
#endif
			if (0 != generate_monthly_output(conf, data->data.state.state, *(data->key) ? data->key : NULL)) {
				fprintf(stderr, "%s.%d: output generation failed - no data this month?\n", __FILE__, __LINE__);
				/* return -1; */
			}
			if (0 != generate_history_output(conf, data->data.state.history, *(data->key) ? data->key : NULL)) {
				fprintf(stderr, "%s.%d: history generation failed - going down hard\n", __FILE__, __LINE__);
				return -1;
			}

			l = l->next;
			
			/* disable the bigpicture if we aren't splitting the logs */
			if (*(data->key) == '\0') create_bp = 0;
			
			/* write the current directoy to the split file */
			fprintf(split_file, "%s\n", data->key);
			
		}
		
		fclose(split_file);
		free(fn);
	}
	
	MTIMER_STOP(post_process_timer);
	MTIMER_CALC(post_process_timer);

	mlist_free(state_list);
	mrecord_free(rec);

	if (conf->debug_level > 0) {
		printf(" --> Setup       : Wall %10.2fs, User %10.2fs, System %10.2fs <--\n",
			MTIMER_GET_WALL_MSEC(setup_timer)/1000.0,
			MTIMER_GET_USER_MSEC(setup_timer)/1000.0,
			MTIMER_GET_SYSTEM_MSEC(setup_timer)/1000.0);
		printf(" --> Parse       : Wall %10.2fs, User %10.2fs, System %10.2fs <--\n",
			MTIMER_GET_WALL_MSEC(parse_timer)/1000.0,
			MTIMER_GET_USER_MSEC(parse_timer)/1000.0,
			MTIMER_GET_SYSTEM_MSEC(parse_timer)/1000.0);
		printf(" --> Process     : Wall %10.2fs, User %10.2fs, System %10.2fs <--\n",
			MTIMER_GET_WALL_MSEC(process_timer)/1000.0,
			MTIMER_GET_USER_MSEC(process_timer)/1000.0,
			MTIMER_GET_SYSTEM_MSEC(process_timer)/1000.0);
		printf(" --> Post-Process: Wall %10.2fs, User %10.2fs, System %10.2fs <--\n",
			MTIMER_GET_WALL_MSEC(post_process_timer)/1000.0,
			MTIMER_GET_USER_MSEC(post_process_timer)/1000.0,
			MTIMER_GET_SYSTEM_MSEC(post_process_timer)/1000.0);

		printf("%s: %.2f %s (%ld %s, %ld %s, %ld %s, %ld %s)\n",
		       "Throughput", (lines)/((MTIMER_GET_USER_MSEC(parse_timer)+MTIMER_GET_USER_MSEC(process_timer))/1000.0), 
		       "rec/s",
		       lines, "records", 
		       lines_corrupt, "corrupt records",
		       lines_skipped, "skipped records",
		       lines_ignored, "ignored records"
		       );
	}
	
	mconfig_free(conf);
	return 0;
}
