/*
** Modular Logfile Analyzer
** Copyright 2000 Jan Kneschke <jan@kneschke.de>
**
** Homepage: http://www.kneschke.de/projekte/modlogan
**

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version, and provided that the above
    copyright and permission notice is included with all distributed
    copies of this or derived software.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA

**
** $Id: datatype.c,v 1.9 2001/10/06 16:32:17 ostborn Exp $
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>

#include <zlib.h>
#include <libxml/tree.h>

#include "config.h"
#include "mdatatypes.h"
#include "datatype.h"

/* begin of WebHist */

int mdata_WebHist_to_xml(gzFile *fd, mdata *data) {
#define WR(x,f) \
	gzprintf(fd, "<%s>%"#f"</%s>\n", #x, data->data.webhist.x, #x);
	
	WR(hits, ld);
	WR(files, ld);
	WR(pages, ld);
	WR(visits, ld);
	WR(hosts, ld);
	
	WR(year, d);
	WR(month, d);
	WR(week, d);
	WR(days_passed, d);
	
	WR(xfersize, .0f);
#undef WR
	
	return 0;
}

int mdata_WebHist_free(mdata *data) {
	if (!data) return -1;
	if (data->type != M_DATA_TYPE_WEBHIST) return -1;
	
	return 0;
}

int mdata_WebHist_setdata(mdata *data, const char *str, 
	unsigned int hits, unsigned int files,
	unsigned int pages, unsigned int visits,
	unsigned int hosts, unsigned int year,
	unsigned int month, unsigned int week,
	unsigned int days_passed, double xfersize ) {
	
	data->key	= malloc(strlen(str)+1);
	assert(data->key);
	strcpy(data->key, str);
	
	if (data->type == M_DATA_TYPE_UNSET) 
		data->type		= M_DATA_TYPE_WEBHIST;
	
	data->data.webhist.hits		= hits;
	data->data.webhist.files	= files;
	data->data.webhist.pages	= pages;
	data->data.webhist.visits	= visits;
	data->data.webhist.hosts	= hosts;
	
	data->data.webhist.year		= year;
	data->data.webhist.month	= month;
	data->data.webhist.week		= week;
	
	data->data.webhist.days_passed	= days_passed;
	data->data.webhist.xfersize	= xfersize;
	
	return 0;
}

int mdata_WebHist_from_xml(mstate_stack *m, int tagtype, mdata *data, int type, const xmlChar *value, const xmlChar **attrs) {
	int i;
	
	const mdata_values data_values[] = {
		{ "hits", M_DATA_FIELDTYPE_LONG, &(data->data.webhist.hits) },
		{ "files", M_DATA_FIELDTYPE_LONG, &(data->data.webhist.files) },
		{ "pages", M_DATA_FIELDTYPE_LONG, &(data->data.webhist.pages) },
		{ "visits", M_DATA_FIELDTYPE_LONG, &(data->data.webhist.visits) },
		{ "hosts", M_DATA_FIELDTYPE_LONG, &(data->data.webhist.hosts) },
		{ "year", M_DATA_FIELDTYPE_LONG, &(data->data.webhist.year) },
		{ "month", M_DATA_FIELDTYPE_LONG, &(data->data.webhist.month) },
		{ "week", M_DATA_FIELDTYPE_LONG, &(data->data.webhist.week) },
		{ "days_passed", M_DATA_FIELDTYPE_LONG, &(data->data.webhist.days_passed) },
		{ "xfersize", M_DATA_FIELDTYPE_DOUBLE, &(data->data.webhist.xfersize) },
		
		{ NULL, M_DATA_FIELDTYPE_UNSET, NULL}
	};
	
#if 0	
	M_DEBUG2(M_DEBUG_LEVEL_ERRORS, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_ERRORS,
		 "|--> %s - %d\n", 
		 value, tagtype);
#endif 
	switch(tagtype) {
	case M_TAG_BEGIN:
		if (m->st_depth != m->st_depth_max) {
			M_DEBUG3(M_DEBUG_LEVEL_ERRORS, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_ERRORS,
				 "cur(depth) != max(depth) [%d - %d]- not my job (%s)\n",
				 m->st_depth, 
				 m->st_depth_max,
				 value);
		} else {
			for (i = 0; data_values[i].string != NULL && 0 != strcmp(data_values[i].string, value); i++)
				;
			if (data_values[i].string != NULL) {
				m->st_depth_max++;
				m->st[++m->st_depth].id = i + 1;
			} else {
				M_DEBUG1(M_DEBUG_LEVEL_ERRORS, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_ERRORS,
					 "unknown tag '%s'\n", 
					 value);
				return -1;
			}
		}
		break;
	case M_TAG_END:
		if (m->st_depth != m->st_depth_max) {
			M_DEBUG3(M_DEBUG_LEVEL_ERRORS, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_ERRORS,
				 "cur(depth) != max(depth) [%d - %d]- not my job (%s)\n",
				 m->st_depth, 
				 m->st_depth_max,
				 value);
			return -1;
		} else {
			
			
			for (i = 0; data_values[i].string != NULL && 0 != strcmp(data_values[i].string, value); i++)
				;
			if (data_values[i].string != NULL) {
				/* set data */
				if (data->type == M_DATA_TYPE_UNSET) 
					data->type		= M_DATA_TYPE_WEBHIST;
#if 0				
				for (i = 0; data_values[i].string != NULL; i++) {
					fprintf(stderr, "--p %s ", data_values[i].string);
					switch(data_values[i].type) {
					case M_DATA_FIELDTYPE_LONG:
						fprintf(stderr, "%ld", *(long *)(data_values[i].dest));
						break;
					case M_DATA_FIELDTYPE_DOUBLE:
						fprintf(stderr, "%f", *(double *)(data_values[i].dest));
						break;
					}
					fprintf(stderr, "\n");
				}
#endif
				m->st[m->st_depth].id = 0;
				m->st_depth_max--;
			} else {
				M_DEBUG1(M_DEBUG_LEVEL_ERRORS, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_ERRORS,
					 "unknown tag '%s'\n", 
					 value);
				return -1;
			}
		}
		break;
	case M_TAG_TEXT:
		if (m->st_depth != m->st_depth_max) {
			M_DEBUG3(M_DEBUG_LEVEL_ERRORS, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_ERRORS,
				 "cur(depth) != max(depth) [%d - %d]- not my job (%s)\n",
				 m->st_depth, 
				 m->st_depth_max,
				 value);
			return -1;
		} else {
			if (mdata_insert_value(m, tagtype, 
					       data_values[m->st[m->st_depth].id - 1].dest, 
					       data_values[m->st[m->st_depth].id - 1].type, 
					       value, attrs)) {
				M_DEBUG1(M_DEBUG_LEVEL_ERRORS, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_ERRORS,
					 "insert failed for '%s'\n", 
					 value);
				return -1;
			} else {
#if 0
				M_DEBUG2(M_DEBUG_LEVEL_ERRORS, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_ERRORS,
					 "inserting '%s' for '%s'\n", 
					 value, data_values[m->st[m->st_depth] - 1].string);
#endif
			}
		}
		break;
	default:
		M_DEBUG1(M_DEBUG_LEVEL_ERRORS, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_ERRORS,
			 "can't handle tagtype '%d'\n", 
			 type);
		return -1;
	}
	
	return 0;
}

int mdata_WebHist_append(mdata *dst, mdata *src) {

#define IS_EQ(x) (dst->data.webhist.x == src->data.webhist.x)
#define SET(x) dst->data.webhist.x += src->data.webhist.x
	if (IS_EQ(year) && IS_EQ(month) /* && IS_EQ(days_passed) && IS_EQ(week)*/) {
		fprintf(stderr, "%s.%d: combining webhist. i hope this is intended here.\n", __FILE__, __LINE__ );
		SET(hits);
		SET(files);
		SET(pages);
		SET(visits);
		SET(hosts);
	
		SET(xfersize);
	} else {
		fprintf(stderr, "%s.%d: combining webhist. -- not done.\n", __FILE__, __LINE__ );
	}
#undef SET
#undef IS_EQ
	return 0;
}

mdata *mdata_WebHist_init() {
	mdata *data = mdata_init();
	assert(data);
	
	data->key		= NULL;
	data->type		= M_DATA_TYPE_WEBHIST;

#ifdef DEBUG_DATATYPES
	fprintf(stderr, "%s.%d: got %ld (%d)\n", __FILE__, __LINE__, data->id, data->type);
#endif

/* init specifics */	
	data->data.webhist.hits		= 0;
	data->data.webhist.files	= 0;
	data->data.webhist.pages	= 0;
	data->data.webhist.visits	= 0;
	data->data.webhist.hosts	= 0;
	
	data->data.webhist.year		= 0;
	data->data.webhist.month	= 0;
	data->data.webhist.week		= 0;
	
	data->data.webhist.days_passed	= 0;
	data->data.webhist.xfersize	= 0;
	
	return data;
}

mdata *mdata_WebHist_copy(mdata *src) {
	mdata *data = mdata_WebHist_init();
	assert(data);
	
	mdata_WebHist_setdata(data, src->key, 
		src->data.webhist.hits, src->data.webhist.files,
		src->data.webhist.pages, src->data.webhist.visits,
		src->data.webhist.hosts, src->data.webhist.year,
		src->data.webhist.month, src->data.webhist.week,
		src->data.webhist.days_passed, src->data.webhist.xfersize
		);
	
	return data;
}

mdata *mdata_WebHist_create(const char *str, 
	unsigned int hits, unsigned int files,
	unsigned int pages, unsigned int visits,
	unsigned int hosts, unsigned int year,
	unsigned int month, unsigned int week,
	unsigned int days_passed, double xfersize ) {
	
	mdata *data = mdata_WebHist_init();
	assert(data);
	mdata_WebHist_setdata(data, str, 
		hits, files,
		pages, visits,
		hosts, year,
		month, week,
		days_passed, xfersize
	);
	
	return data;
}

mdata  *mdata_WebHist_create_by_state(mstate *state) {
	char buf[7];
	long i, pages, hits, files, visits, hosts, last_day = 0;
	double xfersize;
	struct tm tm;
	mstate_web *staweb;
	time_t timestamp;
	
	mdata *data = mdata_WebHist_init();
	assert(data);
	
	if (state == NULL) return NULL;
	
	timestamp = state->timestamp;
	
	tm = *localtime(&(state->timestamp));
	
	sprintf(buf, "%04i%02i", state->year, state->month);
	
	pages	= 0;
	hits	= 0;
	files	= 0;
	visits	= 0;
	hosts	= 0;
	xfersize = 0;
	
	if (state->ext && state->ext_type == M_STATE_TYPE_WEB) {
		staweb = state->ext;
		
		for ( i = 0; i < 31; i++) {
			files		+= staweb->days[i].files;
			xfersize	+= staweb->days[i].xfersize;
			hits		+= staweb->days[i].hits;
			hosts		+= staweb->days[i].hosts;
			pages		+= staweb->days[i].pages;
			visits		+= staweb->days[i].visits;
			if (staweb->days[i].hits) last_day = i+1;
		}
	}
	mdata_WebHist_setdata(data, buf, 
		hits, files,
		pages, visits,
		hosts, state->year,
		state->month, 0,
		last_day, xfersize
	);
	
	return data;
}

/* end of WebHist */
