/*
** Modular Logfile Analyzer
** Copyright 2000 Jan Kneschke <jan@kneschke.de>
**
** Homepage: http://www.kneschke.de/projekte/modlogan
**

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version, and provided that the above
    copyright and permission notice is included with all distributed
    copies of this or derived software.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA

**
** $Id: datatype.c,v 1.16 2001/12/31 14:14:50 ostborn Exp $
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>

#include <zlib.h>

#include "config.h"
#include "mdatatypes.h"
#include "datatype.h"
#include "datatypes/brokenlink/datatype.h"
#include "mlist.h"
#include "misc.h"

/* begin of Visit */

int mdata_Visit_to_xml(gzFile *fd, mdata *data) {
#define WR(x,f) \
	gzprintf(fd, "<%s>%"#f"</%s>\n", #x, data->data.visit.x, #x);
#define WRS(x,f) \
	if (data->data.visit.x != NULL) { \
		char *s = url_encode(data->data.visit.x); \
		gzprintf(fd, "<%s>%"#f"</%s>\n", #x, s, #x); \
		free(s); \
	} else { \
		gzprintf(fd, "<%s />", #x); \
	}
	
	WR(count, d);
	WR(type, d);
	WR(timestamp, ld);
	WR(timediff, ld);
	WRS(useragent, s);
	
	gzprintf(fd, "<hits>\n");
	mlist_write(fd, data->data.visit.hits);
	gzprintf(fd, "</hits>\n");
#undef WRS
#undef WR
	return 0;
}

int mdata_Visit_free(mdata *data) {
	if (!data) return -1;
	if (data->type != M_DATA_TYPE_VISIT) return -1;
	
	if (data->data.visit.useragent) free(data->data.visit.useragent);
	if (data->data.visit.hits) mlist_free(data->data.visit.hits);
	
	return 0;
}

int mdata_Visit_setdata(mdata *data, const char *str, 
	const char *url, const char *useragent,
	int count, time_t timestamp,
	time_t timediff, int type) {
	
	data->key	= malloc(strlen(str)+1);
	assert(data->key);
	strcpy(data->key, str);
	
	if (data->type == M_DATA_TYPE_UNSET) 
		data->type		= M_DATA_TYPE_VISIT;	
	
	data->data.visit.count		= count;
	data->data.visit.type		= type;
	data->data.visit.timestamp	= timestamp;
	data->data.visit.timediff	= timediff;
	
	if (useragent) {
		data->data.visit.useragent	= malloc(strlen(useragent)+1);;
		assert(data->data.visit.useragent);
		strcpy(data->data.visit.useragent, useragent);
	} else {
		data->data.visit.useragent = malloc(strlen("") + 1);
		assert(data->data.visit.useragent);
		strcpy(data->data.visit.useragent, "");
	}
	
	data->data.visit.hits = mlist_init();
#if 0	
	fprintf(stderr, "++ %p: %s.%d.%s\n", data->data.visit.hits, __FILE__, __LINE__, __FUNCTION__);
#endif	
	if (url) {
		mdata *hit;
		hit = mdata_BrokenLink_create(url, 1, M_DATA_STATE_PLAIN, timestamp, NULL);
	
		mlist_append(data->data.visit.hits, hit);
	}
	
	return 0;
}

int mdata_Visit_from_xml(mstate_stack *m, int tagtype, mdata *data, int type, const xmlChar *value, const xmlChar **attrs) {
	char *useragent = NULL, *lasturl = NULL;
	int i;
	
	/* add your definitions here */
	static mdata_values data_values[] = {
		{ "count", M_DATA_FIELDTYPE_LONG, 	NULL },
		{ "type", M_DATA_FIELDTYPE_LONG, 	NULL },
		{ "timestamp", M_DATA_FIELDTYPE_LONG, 	NULL },
		{ "timediff", M_DATA_FIELDTYPE_LONG, 	NULL },
		{ "useragent", M_DATA_FIELDTYPE_STRING, NULL },
		{ "lasturl", M_DATA_FIELDTYPE_STRING, 	NULL }, /* unused */
		{ "hits", M_DATA_FIELDTYPE_LIST,        NULL },
		
		{ NULL, M_DATA_FIELDTYPE_UNSET,  NULL}
	};
	
#if 0	
	M_DEBUG2(M_DEBUG_LEVEL_ERRORS, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_ERRORS,
		 "|--> %s - %d\n", 
		 value, tagtype);
#endif 
	switch(tagtype) {
	case M_TAG_BEGIN:
		if (m->st_depth != m->st_depth_max) {
#if 0
			M_DEBUG3(M_DEBUG_LEVEL_ERRORS, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_ERRORS,
				 "cur(depth) != max(depth) [%d - %d]- not my job (%s)\n",
				 m->st_depth, 
				 m->st_depth_max,
				 value);
#endif
			/* hits is a list !! */
			
			m->st_depth++;
			if (mdata_insert_value(m, tagtype, 
					       data_values[m->st[m->st_depth].id - 1].dest, 
					       data_values[m->st[m->st_depth].id - 1].type, 
					       value, attrs)) {
				M_DEBUG1(M_DEBUG_LEVEL_ERRORS, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_ERRORS,
					 "mdata_read failed: tag %s\n",
					 value);
				return -1;
			}
		} else {
			for (i = 0; data_values[i].string != NULL && 0 != strcmp(data_values[i].string, value); i++)
				;
			
			if (data_values[i].string != NULL) {
				if (data_values[6].dest == NULL) {
					data->data.visit.hits = mlist_init();
					data_values[0].dest = &(data->data.visit.count);
					data_values[1].dest = &(data->data.visit.type);
					data_values[2].dest = &(data->data.visit.timestamp);
					data_values[3].dest = &(data->data.visit.timediff);
					data_values[4].dest = &(useragent);
					data_values[5].dest = &(lasturl);
					data_values[6].dest = &(data->data.visit.hits);
				}
				
				m->st_depth_max++;
				m->st_depth++;
				m->st[m->st_depth].id = i + 1;
				m->st[m->st_depth].data = data_values[i].dest;
			} else {
				M_DEBUG1(M_DEBUG_LEVEL_ERRORS, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_ERRORS,
					 "unknown tag '%s'\n", 
					 value);
				return -1;
			}
		}
		break;
	case M_TAG_END:
		if (m->st_depth != m->st_depth_max) {
#if 0
			M_DEBUG3(M_DEBUG_LEVEL_ERRORS, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_ERRORS,
				 "cur(depth) != max(depth) [%d - %d]- not my job (%s)\n",
				 m->st_depth, 
				 m->st_depth_max,
				 value);
#endif
			m->st_depth++;
			if (mdata_insert_value(m, tagtype, 
					       data_values[m->st[m->st_depth - 1].id - 1].dest, 
					       data_values[m->st[m->st_depth - 1].id - 1].type, 
					       value, attrs)) {
				M_DEBUG1(M_DEBUG_LEVEL_ERRORS, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_ERRORS,
					 "mdata_read failed: tag %s\n",
					 value);
				return -1;
			}
		} else {
			for (i = 0; data_values[i].string != NULL && 0 != strcmp(data_values[i].string, value); i++)
				;
			if (data_values[i].string != NULL) {
				if (lasturl) free(lasturl);
				lasturl = NULL;
	
				data->data.visit.useragent = url_decode(useragent);
				
				data_values[6].dest = NULL;
				
				/* set data */
				if (data->type == M_DATA_TYPE_UNSET) 
					data->type		= M_DATA_TYPE_VISIT;
				
				m->st[m->st_depth].id = 0;
				m->st_depth_max--;
			} else {
				M_DEBUG1(M_DEBUG_LEVEL_ERRORS, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_ERRORS,
					 "unknown tag '%s'\n", 
					 value);
				return -1;
			}
		}
		break;
	case M_TAG_TEXT:
		if (m->st_depth != m->st_depth_max) {
#if 0
			M_DEBUG3(M_DEBUG_LEVEL_ERRORS, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_ERRORS,
				 "cur(depth) != max(depth) [%d - %d]- not my job (%s)\n",
				 m->st_depth, 
				 m->st_depth_max,
				 value);
#endif
			m->st_depth++;
			if (mdata_insert_value(m, tagtype, 
					       data_values[m->st[m->st_depth - 1].id - 1].dest, 
					       data_values[m->st[m->st_depth - 1].id - 1].type, 
					       value, attrs)) {
				M_DEBUG1(M_DEBUG_LEVEL_ERRORS, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_ERRORS,
					 "mdata_read failed: tag %s\n",
					 value);
				return -1;
			}
		} else {
			if (mdata_insert_value(m, tagtype, 
					       data_values[m->st[m->st_depth].id - 1].dest, 
					       data_values[m->st[m->st_depth].id - 1].type, 
					       value, attrs)) {
				M_DEBUG1(M_DEBUG_LEVEL_ERRORS, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_ERRORS,
					 "insert failed for '%s'\n", 
					 value);
				return -1;
			} else {
#if 0
				M_DEBUG2(M_DEBUG_LEVEL_ERRORS, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_ERRORS,
					 "inserting '%s' for '%s'\n", 
					 value, data_values[m->st[m->st_depth] - 1].string);
#endif
			}
		}
		break;
	default:
		M_DEBUG1(M_DEBUG_LEVEL_ERRORS, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_ERRORS,
			 "can't handle tagtype '%d'\n", 
			 type);
		return -1;
	}

	return 0;
}

int mdata_Visit_append(mdata *dst, mdata *src) {
	return 0;
}

mdata *mdata_Visit_init() {
	mdata *data = mdata_init();
	assert(data);
	
	data->key		= NULL;
	data->type		= M_DATA_TYPE_VISIT;
	
#ifdef DEBUG_DATATYPES
	fprintf(stderr, "%s.%d: got %ld (%d)\n", __FILE__, __LINE__, data->id, data->type);
#endif
/* init specifics */	
	data->data.visit.count		= 0;
	data->data.visit.type		= M_DATA_VISIT_USER;
	data->data.visit.timestamp	= 0;
	data->data.visit.timediff	= 0;
	data->data.visit.useragent	= NULL;
	data->data.visit.hits           = NULL;
	
	return data;
}

mdata *mdata_Visit_copy(mdata *src) {
	mdata *data = mdata_Visit_init();
	assert(data);
	mdata_Visit_setdata(data, src->key, 
		NULL, 
		src->data.visit.useragent,
		src->data.visit.count,
		src->data.visit.timestamp,
		src->data.visit.timediff,
		src->data.visit.type
	);
	
	return data;
}

mdata *mdata_Visit_create(const char *str, 
	const char *url, const char *useragent,
	int count, time_t timestamp,
	time_t timediff, int type
	) {
	
	mdata *data = mdata_Visit_init();
	assert(data);

	mdata_Visit_setdata(data, str, url,
		useragent, count, timestamp, 
		timediff, type);
	
	return data;
}

/* end of Visit */
