/* 
 * validate.c
 *
 * Validate a username/pw combination, using the /etc/shadow
 * file.
 *
 * Two lines are read from stdin.  The first is the user, and 
 * the second is the (unencrypted) password.
 *
 * We exit with 0 if they match, 1 otherwise.
 * Errors are written to either stderr or the error log, or both.
 */

#include <stdio.h>
#include <shadow.h>
#include <string.h>
#include <time.h>
#include "validate.h"
#ifdef LOG_FAILED_ATTEMPTS
#include <sys/syslog.h>
#include <sys/sysmacros.h>
#endif

/* USE_XOPEN makes crypt available from unistd.h. */
#define __USE_XOPEN 
#include <unistd.h>
static void stripreturn(char *str)
{
    if (strlen(str)==0)
        return;
    str[(strlen(str)-1)] = '\0';
    return;
}

int 
main(int argc, char **argv)
{
    int fd;
    FILE* fp;
    FILE* fptemp;
    struct spwd *ret = NULL;
    char user[MAX_USERNAME_LENGTH+2];
    char pw[MAX_PW_LENGTH+2];
    char *cryptpw;
    char *progname;

    progname = argv[0];
    
    if (argc!=1) 
        exit(1);

    /* Read user,pw from fd */
    fp = stdin;
    if (!fp) {
        fprintf(stderr,"%s: Unable to open file descriptor %d.\n",progname, fd);
        exit (1);
    }
    fgets(user, MAX_USERNAME_LENGTH, fp);
    fgets(pw, MAX_PW_LENGTH, fp);

    /* Strip carriage return */
    stripreturn(user);
    stripreturn(pw);

    ret = getspnam(user);
    if (!ret) {
        /* Did the user not exist?  Or can we not read /etc/shadow?
         * Print a helpful message if the latter...
         */
        fptemp = fopen("/etc/shadow","r");
        if (!fptemp) {
            fprintf(stderr,
                "%s: No read access to /etc/shadow.  This program must be suid or sgid.\n",
                progname);
            exit(1);
        }
        fclose(fptemp);
         
        fprintf(stderr,"%s: Couldn't find user '%s'\n",progname, user);
#ifdef LOG_FAILED_ATTEMPTS
        syslog(LOG_ERR,"FAILED VALIDATE: unknown user: %s",user);
#endif
        if ( (SLEEP_SECONDS>0) && (sleep(SLEEP_SECONDS)!=0))
            fprintf(stderr,"%s: Error sleeping for 3 seconds.\n", progname);
        exit(1);
        }
    cryptpw = crypt(pw,ret->sp_pwdp);
    if (strcmp(cryptpw,ret->sp_pwdp)!=0){
#ifdef LOG_FAILED_ATTEMPTS
        syslog(LOG_ERR,"VALIDATE: user: %s, Authentication failure",user);
#endif
        if ( (SLEEP_SECONDS>0) && (sleep(SLEEP_SECONDS)!=0)) 
            fprintf(stderr,"%s: Error sleeping for %d seconds.\n", progname,SLEEP_SECONDS);
        fprintf(stderr,"%s: User %s: authentication failure\n",progname,user);
        exit(1);
    }
    exit(0);    
}


