#ifndef XMLELEMENTS_H
#define XMLELEMENTS_H

#include <string>

#include "Tools.h"


//----------------------------------------------------------------------------
class XMLVisitor;
class XMLConstVisitor;


//----------------------------------------------------------------------------
#define DECLARE_XML_VISITOR_API() \
    void do_accept(XMLVisitor &v); \
    void do_accept(XMLConstVisitor &v) const

//----------------------------------------------------------------------------
#define DECLARE_XML_VISITOR_API_BODY(C) \
void C::do_accept(XMLVisitor &v) { v.visit(this); } \
void C::do_accept(XMLConstVisitor &v) const { v.visit(this); }


//----------------------------------------------------------------------------
/**
 * This is a base class for all XML elements.
 */
class XMLElement
{
  protected:
    //------------------------------------------------------------------------
    XMLElement() {}
    XMLElement(const std::string &name) : m_name(name) {}

  public:
    //------------------------------------------------------------------------
    virtual ~XMLElement() = 0;

    //------------------------------------------------------------------------
    inline const std::string &getName() const { return m_name; }

    //------------------------------------------------------------------------
    inline void accept(XMLVisitor &v) { do_accept(v); }
    inline void accept(XMLConstVisitor &v) const  { do_accept(v); }

  private:

    //------------------------------------------------------------------------
    virtual void do_accept(XMLVisitor &v) = 0;
    virtual void do_accept(XMLConstVisitor &v) const = 0;

    //------------------------------------------------------------------------
    std::string m_name;
};


//----------------------------------------------------------------------------
/**
 * This class represents a property of an XML node.
 */
class XMLProperty : public XMLElement
{
  public:
    //------------------------------------------------------------------------
    XMLProperty(const std::string &name, const bool value);
    XMLProperty(const std::string &name, const int value);
    XMLProperty(const std::string &name, const unsigned value);
    XMLProperty(const std::string &name, const char *value);
    XMLProperty(const std::string &name, const std::string &value);
    ~XMLProperty();

    //------------------------------------------------------------------------
    XMLProperty *clone() const;

    //------------------------------------------------------------------------
    void setValue(const bool value);
    void setValue(const int value);
    void setValue(const unsigned value);
    void setValue(const char *value);
    void setValue(const std::string &value);

    //------------------------------------------------------------------------
    inline const std::string &getValue() const { return m_value; }

    //------------------------------------------------------------------------
    /**
     * @return The value of this property as bool.
     * @throw XMLException if the value cannot be converted to a bool.
     */
    bool getAsBool() const;

    /**
     * @return The value of this property as int.
     * @throw XMLException if the value cannot be converted to an int.
     */
    int getAsInt() const;

    /**
     * @return The value of this property as unsigned.
     * @throw XMLException if the value cannot be converted to an unsigned.
     */
    unsigned getAsUnsigned() const;

  private:

    //------------------------------------------------------------------------
    DECLARE_XML_VISITOR_API();

    //------------------------------------------------------------------------
    std::string m_value;
};


//----------------------------------------------------------------------------
/**
 * This class represents a text within an XML node.
 */
class XMLText : public XMLElement
{
  public:
    //------------------------------------------------------------------------
    XMLText(const std::string &text);
    ~XMLText();

    //------------------------------------------------------------------------
    inline void setText(const std::string &text) { m_text = text; }
    inline const std::string &getText() const { return m_text; }

  private:

    //------------------------------------------------------------------------
    DECLARE_XML_VISITOR_API();

    //------------------------------------------------------------------------
    std::string m_text;
};


//----------------------------------------------------------------------------
/**
 * This class represents an XML node.
 * It can contain XML properties and further XML nodes.
 */
class XMLNode : public XMLElement
{
  public:
    //------------------------------------------------------------------------
    XMLNode();
    XMLNode(const std::string &name);
    ~XMLNode();

    //------------------------------------------------------------------------
    void clear();

    //------------------------------------------------------------------------
    bool empty() const;


    //------------------------------------------------------------------------
    /**
     * @param p The property to add to this node.
     */
    void addProperty(XMLProperty *p);

    /**
     * @param p The property to remove from this node.
     */
    void removeProperty(const XMLProperty *p);

    /**
     * @param name The name of the property to get.
     * @return A pointer to the property with the given name.
     * @throw XMLException if no such property exists.
     */
    XMLProperty *getMandatoryProperty(const std::string &name);

    /**
     * @param name The name of the property to get.
     * @return A const pointer to the property with the given name.
     * @throw XMLException if no such property exists.
     */
    const XMLProperty *getMandatoryProperty(const std::string &name) const;

    /**
     * @param name The name of the property to get.
     * @return A pointer to the property with the given name,
     *         or NULL, if no such property exists.
     */
    XMLProperty *getProperty(const std::string &name);

    /**
     * @param name The name of the property to get.
     * @return A const pointer to the property with the given name,
     *         or NULL, if no such property exists.
     */
    const XMLProperty *getProperty(const std::string &name) const;

    /**
     * @param name The name of the property to check for.
     * @return true, if the property exists, else false.
     */
    bool hasProperty(const std::string &name) const;

    /**
     * @return true, if this node has properties.
     */
    bool hasProperties() const;

    /**
     * @return The number of properties in this node.
     */
    size_t getNumberOfProperties() const;

    //------------------------------------------------------------------------
    /**
     * @param t The text node to add to this node.
     */
    void addText(XMLText *t);

    /**
     * @param t The text node to remove from this node.
     */
    void removeText(const XMLText *t);

    /**
     * @return true, if a text node exists, else false.
     */
    bool hasText() const;

    /**
     * @return A pointer to the text node.
     * @throw XMLException if no text node exists.
     */
    const std::string &getText() const;


    //------------------------------------------------------------------------
    /**
     * @param n The node to add to this node.
     */
    void addNode(XMLNode *n);

    /**
     * @param n The node to remove from this node.
     */
    void removeNode(const XMLNode *n);

    /**
     * @param name The name of the node to get.
     * @return A pointer to the node with the given name.
     * @throw XMLException if no such node exists.
     */
    XMLNode *getMandatoryNode(const std::string &name);

    /**
     * @param name The name of the node to get.
     * @return A const pointer to the node with the given name.
     * @throw XMLException if no such node exists.
     */
    const XMLNode *getMandatoryNode(const std::string &name) const;

    /**
     * @param name The name of the node to get.
     * @return A pointer to the node with the given name,
     *         or NULL, if no such node exists.
     */
    XMLNode *getNode(const std::string &name);

    /**
     * @param name The name of the node to get.
     * @return A const pointer to the node with the given name,
     *         or NULL, if no such node exists.
     */
    const XMLNode *getNode(const std::string &name) const;

    /**
     * @return A pointer to the first node or NULL, if no child nodes exist.
     */
    XMLNode *getFirstNode();

    /**
     * @return A pointer to the first node,
     *         or NULL, if no child nodes exist.
     */
    const XMLNode *getFirstNode() const;

    /**
     * @param name The name of the node to pop.
     * @return A pointer to the node with the given name,
     *         or NULL, if no such node exists.
     */
    XMLNode *popNode(const std::string &name);

    /**
     * @return A pointer to the first node,
     *         or NULL, if no child nodes exist.
     */
    XMLNode *popFirstNode();

    /**
     * @param name The name of the node to check for.
     * @return true, if the node exists, else false.
     */
    bool hasNode(const std::string &name) const;

    /**
     * @return true, if this node has child nodes.
     */
    bool hasNodes() const;

    /**
     * @return The number of nodes in this node.
     */
    size_t getNumberOfNodes() const;

    //------------------------------------------------------------------------
    /**
     * @param name The name of the property to search for.
     * @return The string value of the found property.
     * @throw XMLException if no such property exists.
     */
    const std::string &getStringProperty(const std::string &name) const;

    /**
     * @param name The name of the property to search for.
     * @param defaultValue The default value to return,
     *                     if the property doesn't exist.
     * @return The string value of the found property,
     *         or the given default value, if the property doesn't exist.
     */
    const std::string &getStringProperty(const std::string &name,
                                         const std::string &defaultValue) const;

    /**
     * @param name The name of the property to search for.
     * @return The bool value of the found property.
     * @throw XMLException if no such property exists, or the propertie's
     *                     value cannot be converted to a bool.
     */
    bool getBoolProperty(const std::string &name) const;

    /**
     * @param name The name of the property to search for.
     * @param defaultValue The default value to return,
     *                     if the property doesn't exist.
     * @return The bool value of the found property,
     *         or the given default value, if the property doesn't exist.
     * @throw XMLException if the propertie's value cannot be converted
     *                     to a bool.
     */
    bool getBoolProperty(const std::string &name,
                         const bool defaultValue) const;

    /**
     * @param name The name of the property to search for.
     * @return The int value of the found property.
     * @throw XMLException if no such property exists, or the propertie's
     *                     value cannot be converted to an int.
     */
    int getIntProperty(const std::string &name) const;

    /**
     * @param name The name of the property to search for.
     * @param defaultValue The default value to return,
     *                     if the property doesn't exist.
     * @return The int value of the found property,
     *         or the given default value, if the property doesn't exist.
     * @throw XMLException if the propertie's value cannot be converted
     *                     to an int.
     */
    int getIntProperty(const std::string &name,
                       const int defaultValue) const;

    /**
     * @param name The name of the property to search for.
     * @return The unsigned value of the found property.
     * @throw XMLException if no such property exists, or the propertie's
     *                     value cannot be converted to an unsigned.
     */
    unsigned getUnsignedProperty(const std::string &name) const;

    /**
     * @param name The name of the property to search for.
     * @param defaultValue The default value to return,
     *                     if the property doesn't exist.
     * @return The unsigned value of the found property,
     *         or the given default value, if the property doesn't exist.
     * @throw XMLException if the propertie's value cannot be converted
     *                     to an unsigned.
     */
    unsigned getUnsignedProperty(const std::string &name,
                                 const unsigned defaultValue) const;

    //------------------------------------------------------------------------
    void acceptAllProperties(XMLVisitor &v);
    void acceptAllProperties(XMLConstVisitor &v) const;

    //------------------------------------------------------------------------
    void acceptAllNodes(XMLVisitor &v);
    void acceptAllNodes(XMLConstVisitor &v) const;

    //------------------------------------------------------------------------
    void acceptAllChilds(XMLVisitor &v);
    void acceptAllChilds(XMLConstVisitor &v) const;

  private:
    //------------------------------------------------------------------------
    DECLARE_XML_VISITOR_API();

    //------------------------------------------------------------------------
    DECLARE_PIMPL;
};

#endif //XMLELEMENTS_H
