/* Copyright (C) 2000-2006 Lavtech.com corp. All rights reserved.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
*/

#include "udm_config.h"

#ifdef HAVE_SQL


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#include <ctype.h>
#include <time.h>
#include <assert.h>

#ifdef WIN32
#include <time.h>
#endif

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#endif

#include "udm_common.h"
#include "udm_utils.h"
#include "udm_spell.h"
#include "udm_robots.h"
#include "udm_mutex.h"
#include "udm_db.h"
#include "udm_unicode.h"
#include "udm_unidata.h"
#include "udm_url.h"
#include "udm_log.h"
#include "udm_proto.h"
#include "udm_conf.h"
#include "udm_hash.h"
#include "udm_xmalloc.h"
#include "udm_boolean.h"
#include "udm_searchtool.h"
#include "udm_searchcache.h"
#include "udm_server.h"
#include "udm_stopwords.h"
#include "udm_doc.h"
#include "udm_result.h"
#include "udm_vars.h"
#include "udm_agent.h"
#include "udm_store.h"
#include "udm_hrefs.h"
#include "udm_word.h"
#include "udm_db_int.h"
#include "udm_sqldbms.h"
#include "udm_match.h"
#include "udm_indexer.h"
#include "udm_textlist.h"
#include "udm_parsehtml.h"

#ifdef HAVE_ZLIB
#include <zlib.h>
#endif

/* Macros for DBMode=single */


#define OLD_SINGLE_FORMAT 0

#if OLD_SINGLE_FORMAT
#define UDM_WRDCOORD(p,w)  ((((unsigned int)(p))<<16)+(((unsigned int)(w))<<8))
#define UDM_WRDSEC(c)      ((((unsigned int)(c))>>8)&0xFF)
#define UDM_WRDPOS(c)      (((unsigned int)(c))>>16)
#else
#define UDM_WRDCOORD(p,s)  ((((unsigned int)(s))<<24)+(unsigned int)(p))
#define UDM_WRDSEC(c)      (((unsigned int)(c))>>24)
#define UDM_WRDPOS(c)      (((unsigned int)(c))&0x001FFFFF)
#endif

static int UdmRemoveWordsBlob(UDM_AGENT *Indexer, UDM_DOCUMENT *Doc, UDM_DB *db);
static int UdmDeleteURL(UDM_AGENT *Indexer, UDM_DOCUMENT *Doc,UDM_DB *db);
static char udm_hex_digits[]= "0123456789ABCDEF";

/*
  Escape a byte for PgSQL bytea encoding
*/
static inline size_t
UdmSQLBinEscCharForPg(char *dst, unsigned int ch)
{
  if (ch  >= 0x20 && ch <= 0x7F && ch != '\'' && ch != '\\')
  {
    *dst= ch;
    return 1;
  }
  
  dst[4]= udm_hex_digits[ch & 0x07]; ch /= 8;
  dst[3]= udm_hex_digits[ch & 0x07]; ch /= 8;
  dst[2]= udm_hex_digits[ch & 0x07];
  dst[1]= '\\';
  dst[0]= '\\';
  return 5;
}


static size_t
UdmSQLBinEscStr(UDM_DB *db, char *dst, const char *src0, size_t len)
{
  const unsigned char *src= (const unsigned char*) src0;
  if (db->DBType == UDM_DB_PGSQL)
  {
    char *dst0;
    if (!dst)
      dst= UdmMalloc(len * 5 + 1);
    dst0= dst;
    for ( ; len > 0 ; len--, src++)
      dst+= UdmSQLBinEscCharForPg(dst, *src);
    *dst= '\0';
    return dst - dst0;
  }
  UdmSQLEscStr(db, dst, src0, len);
  return 0;
}


typedef struct {
  char empty;
  char exclude;
  urlid_t *urls;
  size_t nurls;
} UDM_URL_TMP;


static int cmpaurls (urlid_t *s1, urlid_t *s2)
{
  if (*s1 > *s2) return(1);
  if (*s1 < *s2) return(-1);
  return(0);
}


static const char *BuildWhere(UDM_ENV * Conf,UDM_DB *db)
{
  size_t  i;
  char    *urlstr;
  char    *tagstr;
  char    *statusstr;
  char    *catstr;
  char    *langstr;
  char    *typestr;
  char    *fromstr;
  char    *serverstr;
  char    *sitestr;
  char    *timestr;
  char    *urlinfostr;
  int     fromserver = 1, fromurlinfo_lang = 1, fromurlinfo_type = 1, fromurlinfo = 1;
  int     dt = UDM_DT_UNKNOWN, dx = 1, dm = 0, dy = 1970, dd = 1;
  time_t  dp = 0, DB = 0, DE = time(NULL), dstmp= 0;
  struct tm tm;
  const char      *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  UDM_DSTR ue;
  
  if(db->where)return(db->where);
  
  bzero((void*)&tm, sizeof(struct tm));
  UdmDSTRInit(&ue, 64);
  urlstr = (char*)UdmStrdup("");
  tagstr = (char*)UdmStrdup("");
  statusstr = (char*)UdmStrdup("");
  catstr = (char*)UdmStrdup("");
  langstr = (char*)UdmStrdup("");
  typestr = (char*)UdmStrdup("");
  fromstr = (char*)UdmStrdup("");
  serverstr = (char*)UdmStrdup("");
  sitestr = (char*)UdmStrdup("");
  timestr = (char*)UdmStrdup("");
  urlinfostr = (char*)UdmStrdup("");
  
  
  for(i=0;i<Conf->Vars.nvars;i++)
  {
    const char *var=Conf->Vars.Var[i].name?Conf->Vars.Var[i].name:"";
    const char *val=Conf->Vars.Var[i].val?Conf->Vars.Var[i].val:"";
    int intval=atoi(val);
    int longval= atol(val);
    
    if(!val[0])continue;
    
    if(!strcmp(var,"tag") || !strcmp(var,"t"))
    {
      tagstr=(char*)UdmRealloc(tagstr,strlen(tagstr)+strlen(val)+50);
      if(tagstr[0])strcpy(UDM_STREND(tagstr)-1," OR ");
      else  strcat(tagstr,"(");

      sprintf(UDM_STREND(tagstr),"s.tag LIKE '%s')",val);
      if (fromserver)
      {
        fromserver = 0;
        fromstr = (char*)UdmRealloc(fromstr, strlen(fromstr) + 12);
        sprintf(UDM_STREND(fromstr), ", server s");
        serverstr = (char*)UdmRealloc(serverstr, strlen(serverstr) + 64);
        sprintf(UDM_STREND(serverstr), " AND s.rec_id=url.server_id");
      }
    }
    
    if(!strcmp(var,"status"))
    {
      const char *range= strchr(val, '-');
      statusstr=(char*)UdmRealloc(statusstr,strlen(statusstr)+strlen(val)+50);
      
      if(db->DBSQL_IN && !range)
      {
        if(statusstr[0])sprintf(UDM_STREND(statusstr)-1,",%d)",intval);
        else  sprintf(statusstr," url.status IN (%d)",intval);
      }
      else
      {
        int first, second;
        if(statusstr[0])strcpy(UDM_STREND(statusstr)-1," OR ");
        else  strcat(statusstr,"(");
        if (range && 2 == sscanf(val, "%d-%d", &first, &second))
        {
          sprintf(UDM_STREND(statusstr),
                  "url.status>=%d AND url.status<=%d)", first, second);
        }
        else
          sprintf(UDM_STREND(statusstr),"url.status=%d)",intval);
      }
    }

    if(!strcmp(var,"ul"))
    {
      UDM_URL  URL;
      const char *first = "%";

      UdmURLInit(&URL);
      UdmURLParse(&URL,val);
      urlstr=(char*)UdmRealloc(urlstr,strlen(urlstr)+strlen(val)+50);

      if((URL.schema != NULL) && (URL.hostinfo != NULL))
      {
        first = "";
      }
      if(urlstr[0])strcpy(UDM_STREND(urlstr)-1," OR ");
      else  strcat(urlstr,"(");
      sprintf(UDM_STREND(urlstr),"url.url LIKE '%s%s%%')", first, val);
      UdmURLFree(&URL);
    }
    
    if(!strcmp(var,"ue"))
    {
      UDM_URL URL;
      const char *first;
      UdmURLInit(&URL);
      UdmURLParse(&URL, val);
      first= ((URL.schema != NULL) && (URL.hostinfo != NULL)) ? "" : "%";
      UdmURLFree(&URL);
      if (ue.size_data)
      {
        ue.size_data--; /* Remove right parenthesis */
        UdmDSTRAppend(&ue, " AND ", 5);
      }
      else
      {
        UdmDSTRAppend(&ue, "(", 1);
      }
      UdmDSTRAppendf(&ue, "url.url NOT LIKE '%s%s%%')", first, val);
    }
    
    if(!strcmp(var,"u"))
    {
      urlstr=(char*)UdmRealloc(urlstr,strlen(urlstr)+strlen(val)+50);
      if(urlstr[0])strcpy(UDM_STREND(urlstr)-1," OR ");
      else  strcat(urlstr,"(");
      sprintf(UDM_STREND(urlstr),"url.url LIKE '%s')", val);
    }
    
    if(!strcmp(var,"lang") || !strcmp(var,"g"))
    {
      langstr=(char*)UdmRealloc(langstr,strlen(langstr)+strlen(val)+50);
      if(langstr[0])strcpy(UDM_STREND(langstr)-1," OR ");
      else  strcat(langstr,"(");
      sprintf(UDM_STREND(langstr),"il.sval LIKE '%s')",val);
      if (fromurlinfo_lang)
      {
        fromurlinfo_lang = 0;
        fromstr = (char*)UdmRealloc(fromstr, strlen(fromstr) + 16);
        sprintf(UDM_STREND(fromstr), ", urlinfo il");
        serverstr = (char*)UdmRealloc(serverstr, strlen(serverstr) + 64);
        sprintf(UDM_STREND(serverstr), " AND il.url_id=url.rec_id AND il.sname='Content-Language'");
      }
    }

    if(!strncmp(var, "sl.", 3))
    {
      urlinfostr=(char*)UdmRealloc(urlinfostr, strlen(urlinfostr)+strlen(var)+strlen(val)+50);
      if(urlinfostr[0])strcpy(UDM_STREND(urlinfostr)-1," AND ");
      else  strcat(urlinfostr,"(");
      sprintf(UDM_STREND(urlinfostr),"isl%d.sname='%s' AND isl%d.sval LIKE '%s')",fromurlinfo,var+3,fromurlinfo,val);

      fromstr = (char*)UdmRealloc(fromstr, strlen(fromstr) + 32);
      sprintf(UDM_STREND(fromstr), ", urlinfo isl%d", fromurlinfo);
      serverstr = (char*)UdmRealloc(serverstr, strlen(serverstr) + 64);
      sprintf(UDM_STREND(serverstr), " AND isl%d.url_id=url.rec_id", fromurlinfo);
      fromurlinfo++;
    }
    
    if(!strcmp(var,"cat") && val[0])
    {
      catstr=(char*)UdmRealloc(catstr,strlen(catstr)+strlen(val)+50);
      if(catstr[0])strcpy(UDM_STREND(catstr)-1," OR ");
      else  strcat(catstr,"(");
      sprintf(UDM_STREND(catstr),"c.path LIKE '%s%%')",val);
      if (fromserver)
      {
        fromserver = 0;
        fromstr = (char*)UdmRealloc(fromstr, strlen(fromstr) + 32);
        sprintf(UDM_STREND(fromstr), ", server s, categories c");
        serverstr = (char*)UdmRealloc(serverstr, strlen(serverstr) + 64);
        sprintf(UDM_STREND(serverstr), " AND s.rec_id=url.server_id AND s.category=c.rec_id");
      }
    }
    if(!strcmp(var,"type") || !strcmp(var, "typ"))
    {
      /* 
         "type" is a reserved word in ASP,
         so "typ" is added as a workaround
      */
      typestr = (char*)UdmRealloc(typestr, strlen(typestr) + strlen(val) + 50);
      if(typestr[0]) strcpy(UDM_STREND(typestr) - 1, " OR ");
      else  strcat(typestr,"(");

      sprintf(UDM_STREND(typestr),"it.sval LIKE '%s')",val);
      if (fromurlinfo_type)
      {
        fromurlinfo_type = 0;
        fromstr = (char*)UdmRealloc(fromstr, strlen(fromstr) + 16);
        sprintf(UDM_STREND(fromstr), ", urlinfo it");
        serverstr = (char*)UdmRealloc(serverstr, strlen(serverstr) + 64);
        sprintf(UDM_STREND(serverstr), " AND it.url_id=url.rec_id AND it.sname='Content-Type'");
      }
    }
    if(!strcmp(var,"site") && intval != 0)
    {
      sitestr=(char*)UdmRealloc(sitestr, strlen(sitestr) + strlen(val) + 50);
      
      if(db->DBSQL_IN)
      {
        if (sitestr[0]) sprintf(UDM_STREND(sitestr) - 1, ",%s%i%s)", qu, intval, qu);
        else  sprintf(sitestr, " url.site_id IN (%s%i%s)", qu, intval, qu);
      }else{
        if (sitestr[0]) strcpy(UDM_STREND(sitestr) - 1, " OR ");
        else  strcat(sitestr, "(");
        sprintf(UDM_STREND(sitestr), "url.site_id=%s%d%s)", qu, intval, qu);
      }
    }
    if (!strcmp(var, "dt"))
    {
      if(!strcasecmp(val, "back")) dt = UDM_DT_BACK;
      else if (!strcasecmp(val, "er")) dt = UDM_DT_ER;
      else if (!strcasecmp(val, "range")) dt = UDM_DT_RANGE;
    }
    if (!strcmp(var, "dx"))
    {
      if (intval == 1 || intval == -1) dx = intval;
      else dx = 1;
    }
    if (!strcmp(var, "dm"))
    {
      dm = (intval) ? intval : 1;
    }
    if (!strcmp(var, "dy"))
    {
      dy = (intval) ? intval : 1970;
    }
    if (!strcmp(var, "dd"))
    {
      dd = (intval) ? intval : 1;
    }
    if (!strcmp(var, "dstmp"))
    {
      dstmp= longval ? longval : 0;
    }
    if (!strcmp(var, "dp"))
    {
      dp = Udm_dp2time_t(val);
    }
    if (!strcmp(var, "db"))
    {
      sscanf(val, "%d/%d/%d", &tm.tm_mday, &tm.tm_mon, &tm.tm_year);
      tm.tm_year -= 1900; tm.tm_mon--;
      DB = mktime(&tm);
    }
    if (!strcmp(var, "de"))
    {
      sscanf(val, "%d/%d/%d", &tm.tm_mday, &tm.tm_mon, &tm.tm_year);
      tm.tm_year -= 1900; tm.tm_mon--;
      DE= mktime(&tm) + 86400; /* Including the given date */
    }
  }
  
  switch(dt)
  {
    case UDM_DT_BACK:
      timestr= (char*)UdmRealloc(timestr, 128);
      if (dp)
        sprintf(timestr, "url.last_mod_time >= %li",
                (long int) time(NULL) - dp);
      break;
    case UDM_DT_ER:
      timestr= (char*)UdmRealloc(timestr, 128);
      tm.tm_mday= dd; tm.tm_mon= dm, tm.tm_year= dy - 1900;
      sprintf(timestr, "url.last_mod_time %s %li",
              (dx == 1) ? ">=" : "<=", (long int) (dstmp ? dstmp : mktime(&tm)));
      break;
    case UDM_DT_RANGE:
      timestr= (char*)UdmRealloc(timestr, 128);
      sprintf(timestr, "url.last_mod_time >= %li AND url.last_mod_time <= %li",
              (long int) DB, (long int) DE);
      break;
    case UDM_DT_UNKNOWN:
    default:
      break;
  }


  if(!urlstr[0] && !tagstr[0] && !statusstr[0] && !catstr[0] && !langstr[0] &&
     !typestr[0] && !serverstr[0] && !fromstr[0] && !sitestr[0] && !timestr[0] &&
     !urlinfostr[0] && !ue.size_data)
  {
    db->where = (char*)UdmStrdup("");
    db->from = (char*)UdmStrdup("");
    goto ret;
  }
  i= strlen(urlstr) + strlen(tagstr) + strlen(statusstr) + strlen(catstr) + 
     strlen(langstr) + strlen(typestr) + strlen(serverstr) + strlen(sitestr) +
     strlen(timestr) + strlen(urlinfostr) + ue.size_data;
  db->where=(char*)UdmMalloc(i+100);
  db->where[0] = '\0';
  UDM_FREE(db->from);
  db->from = (char*)UdmStrdup(fromstr);
  
  if(urlstr[0])
  {
    strcat(db->where,urlstr);
  }
  if (ue.size_data)
  {
    if(db->where[0])strcat(db->where," AND ");
    strcat(db->where,ue.data);
  }
  if(tagstr[0])
  {
    if(db->where[0])strcat(db->where," AND ");
    strcat(db->where,tagstr);
  }
  if(statusstr[0])
  {
    if(db->where[0])strcat(db->where," AND ");
    strcat(db->where,statusstr);
  }
  if(catstr[0])
  {
    if(db->where[0])strcat(db->where," AND ");
    strcat(db->where,catstr);
  }
  if(langstr[0])
  {
    if(db->where[0])strcat(db->where," AND ");
    strcat(db->where,langstr);
  }
  if(typestr[0])
  {
    if(db->where[0]) strcat(db->where, " AND ");
    strcat(db->where, typestr);
  }
  if(sitestr[0])
  {
    if(db->where[0]) strcat(db->where, " AND ");
    strcat(db->where, sitestr);
  }
  if(serverstr[0])
  {
    if(!db->where[0]) strcat(db->where, " 1=1 ");
    strcat(db->where, serverstr);
  }
  if(timestr[0])
  {
    if(db->where[0]) strcat(db->where, " AND ");
    strcat(db->where, timestr);
  }
  if (urlinfostr[0])
  {
    if(db->where[0]) strcat(db->where, " AND ");
    strcat(db->where, urlinfostr);
  }
ret:
  UDM_FREE(urlstr);
  UDM_FREE(tagstr);
  UDM_FREE(statusstr);
  UDM_FREE(catstr);
  UDM_FREE(langstr);
  UDM_FREE(typestr);
  UDM_FREE(fromstr);
  UDM_FREE(serverstr);
  UDM_FREE(sitestr);
  UDM_FREE(timestr);
  UDM_FREE(urlinfostr);
  UdmDSTRFree(&ue);
  return db->where;
}


/********************************************************************/
/* --------------------------------------------------------------- */
/* Almost Unified code for all supported SQL backends              */
/* --------------------------------------------------------------- */
/*******************************************************************/


static int
UdmSQLTableTruncateOrDelete(UDM_DB *db, const char *name)
{
  char qbuf[128];
  if (db->flags & UDM_SQL_HAVE_TRUNCATE)
    udm_snprintf(qbuf, sizeof(qbuf), "TRUNCATE TABLE %s", name);
  else
    udm_snprintf(qbuf, sizeof(qbuf), "DELETE FROM %s", name);
  return UdmSQLQuery(db,NULL,qbuf);
}



/************* Servers ******************************************/
static int UdmLoadServerTable(UDM_AGENT * Indexer, UDM_SERVERLIST *S, UDM_DB *db)
{
  size_t    rows, i, j, jrows;
  UDM_SQLRES  SQLRes, SRes;
  UDM_HREF  Href;
  char    qbuf[1024];
  const char  *filename= UdmVarListFindStr(&db->Vars, "filename", NULL);
  const char  *name = (filename && filename[0]) ? filename : "server";
  const char      *infoname = UdmVarListFindStr(&db->Vars, "srvinfo", "srvinfo");
  int    res;
  const char      *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  
  udm_snprintf(qbuf,sizeof(qbuf)-1,"\
SELECT rec_id,url,tag,category,command,weight,ordre \
FROM %s WHERE enabled=1 AND parent=%s0%s ORDER BY ordre", name, qu, qu);
  
  if(UDM_OK!=(res=UdmSQLQuery(db,&SQLRes,qbuf)))
    return res;
  
  bzero((void*)&Href, sizeof(Href));
  
  rows=UdmSQLNumRows(&SQLRes);
  for(i=0;i<rows;i++)
  {
    const char  *val;
    UDM_SERVER  *Server = Indexer->Conf->Cfg_Srv;
    
    Server->site_id    = UDM_ATOI(UdmSQLValue(&SQLRes, i, 0));
    val= UdmSQLValue(&SQLRes, i, 1);
    Server->Match.pattern= UdmStrdup(val ? val : "");
    Server->ordre    = UDM_ATOI(UdmSQLValue(&SQLRes, i, 6));
    Server->command    = *UdmSQLValue(&SQLRes, i, 4);
    Server->weight    = UDM_ATOF(UdmSQLValue(&SQLRes, i, 5));
    
    if((val=UdmSQLValue(&SQLRes,i,2)) && val[0])
      UdmVarListReplaceStr(&Server->Vars, "Tag", val);
    
    if((val=UdmSQLValue(&SQLRes,i,3)) && val[0])
      UdmVarListReplaceStr(&Server->Vars, "Category", val);
    

    sprintf(qbuf,"SELECT sname,sval FROM %s WHERE srv_id=%s%i%s", infoname, qu, Server->site_id, qu);
    if(UDM_OK != (res = UdmSQLQuery(db, &SRes, qbuf)))
      return res;
    jrows = UdmSQLNumRows(&SRes);
    for(j = 0; j < jrows; j++)
    {
      const char *sname = UdmSQLValue(&SRes, j, 0);
      const char *sval = UdmSQLValue(&SRes, j, 1);
      UdmVarListReplaceStr(&Server->Vars, sname, sval);
    }
    UdmSQLFree(&SRes);

    Server->Match.match_type  = UdmVarListFindInt(&Server->Vars, "match_type", UDM_MATCH_BEGIN);
    Server->Match.case_sense= UdmVarListFindInt(&Server->Vars, "case_sense", UDM_CASE_INSENSITIVE);
    Server->Match.nomatch  = UdmVarListFindInt(&Server->Vars, "nomatch", 0);
    Server->Match.arg  = (char *)UdmStrdup(UdmVarListFindStr(&Server->Vars, "Arg", "Disallow"));

    if (Server->command == 'S')
    {
      UdmServerAdd(Indexer, Server, 0);
      if((Server->Match.match_type==UDM_MATCH_BEGIN) &&
         (Indexer->flags & UDM_FLAG_ADD_SERVURL))
      {
        Href.url = Server->Match.pattern;
        Href.method=UDM_METHOD_GET;
        Href.site_id = Server->site_id;
        Href.server_id = Server->site_id;
        Href.hops= (uint4) UdmVarListFindInt(&Server->Vars, "StartHops", 0);
        UdmHrefListAdd(&Indexer->Conf->Hrefs, &Href);
      }
    }
    else
    {
      char errstr[128];
      UdmMatchListAdd(Indexer, &Indexer->Conf->Filters, &Server->Match, errstr, sizeof(errstr), Server->ordre);
    }
    UDM_FREE(Server->Match.pattern);
  }
  UdmSQLFree(&SQLRes);
  return(UDM_OK);
}

static int UdmServerTableFlush(UDM_DB *db)
{
  int rc;
  const char      *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  char str[128];
  
  udm_snprintf(str, sizeof(str),  "UPDATE server SET enabled=0 WHERE parent=%s0%s", qu, qu);
  rc = UdmSQLQuery(db, NULL, str);
  return rc;
}

static int UdmServerTableAdd(UDM_SERVERLIST *S, UDM_DB *db)
{
  UDM_SQLRES  SQLRes;
  int    res = UDM_OK, done = 1;
  const char  *alias=UdmVarListFindStr(&S->Server->Vars,"Alias",NULL);
  const char      *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  size_t    i, len= 0;
  char    *buf, *arg;
  udmhash32_t     rec_id = UdmStrHash32(S->Server->Match.pattern);
  UDM_VARLIST  *Vars= &S->Server->Vars;
  
  for (i=0; i < Vars->nvars; i++)
    len= udm_max(len, Vars->Var[i].curlen);
  
  len+= S->Server->Match.pattern ? strlen(S->Server->Match.pattern) : 0;
  len+= alias ? strlen(alias) : 0;
  len+= 2048;
  
  buf = (char*)UdmMalloc(len);
  arg = (char*)UdmMalloc(len);
  if (buf == NULL || arg == NULL)
  {
    UDM_FREE(buf);
    UDM_FREE(arg);
    strcpy(db->errstr, "Out of memory");
    db->errcode = 1;
    return UDM_ERROR;
  }
  
  while(done)
  {
    udm_snprintf(buf, len, "SELECT rec_id, url FROM server WHERE rec_id=%s%d%s", qu, rec_id, qu);
    if (UDM_OK != (res = UdmSQLQuery(db, &SQLRes, buf)))
      goto ex;
  
    if (UdmSQLNumRows(&SQLRes) && (strcasecmp(S->Server->Match.pattern,UdmSQLValue(&SQLRes, 0, 1)) != 0))
    {
      rec_id++;
    } else done = 0;
    UdmSQLFree(&SQLRes);
  }

  UdmVarListReplaceInt(&S->Server->Vars, "match_type",  S->Server->Match.match_type);
  UdmVarListReplaceInt(&S->Server->Vars, "case_sense",  S->Server->Match.case_sense);
  UdmVarListReplaceInt(&S->Server->Vars, "nomatch",  S->Server->Match.nomatch);
  if (S->Server->command == 'F' && S->Server->Match.arg != NULL)
  {
    UdmVarListReplaceStr(&S->Server->Vars, "Arg", S->Server->Match.arg);
  } 
  
  udm_snprintf(buf, len, "\
INSERT INTO server (rec_id, enabled, tag, category, \
command, parent, ordre, weight, url, pop_weight \
) VALUES (%s%d%s, 1, '%s', %s, '%c', %s%d%s, %d, %f, '%s', 0\
)",
         qu, rec_id, qu,
         UdmVarListFindStr(&S->Server->Vars, "Tag", ""),
         UdmVarListFindStr(&S->Server->Vars, "Category", "0"),
         S->Server->command,
         qu, S->Server->parent, qu,
         S->Server->ordre,
         S->Server->weight,
         UdmSQLEscStr(db, arg, UDM_NULL2EMPTY(S->Server->Match.pattern), strlen(UDM_NULL2EMPTY(S->Server->Match.pattern)))
     );
  
  if(UDM_OK != (res = UdmSQLQuery(db, NULL, buf))) goto ex;
  
  udm_snprintf(buf, len, "\
UPDATE server SET enabled=1, tag='%s', category=%s, \
command='%c', parent=%s%i%s, ordre=%d, weight=%f \
WHERE rec_id=%s%d%s",
         UdmVarListFindStr(&S->Server->Vars, "Tag", ""),
         UdmVarListFindStr(&S->Server->Vars, "Category", "0"),
         S->Server->command,
         qu, S->Server->parent, qu,
         S->Server->ordre,
         S->Server->weight,
         qu, rec_id, qu
     );
  
  if(UDM_OK != (res = UdmSQLQuery(db, NULL, buf))) goto ex;
  
  S->Server->site_id = rec_id;

  sprintf(buf, "DELETE FROM srvinfo WHERE srv_id=%s%i%s", qu, S->Server->site_id, qu);
  if(UDM_OK != (res = UdmSQLQuery(db, NULL, buf))) goto ex;

  for(i = 0; i < S->Server->Vars.nvars; i++)
  {
    UDM_VAR *Sec = &S->Server->Vars.Var[i];
    if(Sec->val && Sec->name && 
       strcasecmp(Sec->name, "Category") &&
       strcasecmp(Sec->name, "Tag"))
    {
      arg = UdmSQLEscStr(db, arg, Sec->val,strlen(Sec->val));
      udm_snprintf(buf, len, "INSERT INTO srvinfo (srv_id,sname,sval) VALUES (%s%i%s,'%s','%s')",
        qu, S->Server->site_id, qu, Sec->name, arg);
      if(UDM_OK != (res = UdmSQLQuery(db, NULL, buf)))break;
    }
  }

ex:
  UDM_FREE(buf);
  UDM_FREE(arg);
  return res;
}

static int UdmServerTableGetId(UDM_AGENT *Indexer, UDM_SERVERLIST *S, UDM_DB *db)
{
  UDM_SQLRES  SQLRes;
  size_t len = ((S->Server->Match.pattern)?strlen(S->Server->Match.pattern):0) + 1024;
  char *buf = (char*)UdmMalloc(len);
  char *arg = (char*)UdmMalloc(len);
  int res, id = 0, i;
  const char      *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  

  if (buf == NULL)
  {
    UdmLog(Indexer, UDM_LOG_ERROR, "Out of memory");
    return UDM_ERROR;
  }
  if (arg == NULL)
  {
    UDM_FREE(buf);
    UdmLog(Indexer, UDM_LOG_ERROR, "Out of memory");
    return UDM_ERROR;
  }

  for (i = 0; i < UDM_SERVERID_CACHE_SIZE; i++)
  {
    if (Indexer->ServerIdCacheCommand[i] == S->Server->command)
      if (!strcmp(Indexer->ServerIdCache[i], S->Server->Match.pattern))
      {
        S->Server->site_id = id = Indexer->ServerIdCacheId[i];
        break;
      }
  }

  if (id == 0)
  {
    udmhash32_t     rec_id;
    int done = 1;
  
    udm_snprintf(buf, len, "SELECT rec_id FROM server WHERE command='%c' AND url='%s'",
           S->Server->command,
           UDM_NULL2EMPTY(S->Server->Match.pattern)
     );
    res = UdmSQLQuery(db, &SQLRes, buf);
    if ((res == UDM_OK) && UdmSQLNumRows(&SQLRes))
    {
      id = S->Server->site_id = UDM_ATOI(UdmSQLValue(&SQLRes, 0, 0));
      UDM_FREE(Indexer->ServerIdCache[Indexer->pServerIdCache]);
      Indexer->ServerIdCache[Indexer->pServerIdCache] = (char*)UdmStrdup(S->Server->Match.pattern);
      Indexer->ServerIdCacheCommand[Indexer->pServerIdCache] = S->Server->command;
      Indexer->ServerIdCacheId[Indexer->pServerIdCache] = id;
      Indexer->pServerIdCache = (Indexer->pServerIdCache + 1) % UDM_SERVERID_CACHE_SIZE;
      UDM_FREE(buf); UDM_FREE(arg);
      UdmSQLFree(&SQLRes);
      return UDM_OK;
    }
    UdmSQLFree(&SQLRes);
    rec_id = UdmStrHash32(S->Server->Match.pattern);
    while(done) 
    {
      udm_snprintf(buf, len, "SELECT rec_id, url FROM server WHERE rec_id=%s%i%s", qu, rec_id, qu);
      if (UDM_OK != (res = UdmSQLQuery(db, &SQLRes, buf)))
        return res;
      
      if (UdmSQLNumRows(&SQLRes) && (strcasecmp(S->Server->Match.pattern,UdmSQLValue(&SQLRes, 0, 1)) != 0))
      {
        rec_id++;
      } else done = 0;
      UdmSQLFree(&SQLRes);
    }
    udm_snprintf(buf, len, "SELECT enabled,tag,category,ordre FROM server WHERE rec_id=%s%i%s", qu, S->Server->parent, qu);
    res = UdmSQLQuery(db, &SQLRes, buf);
    if (res != UDM_OK)
    {
      UDM_FREE(buf); UDM_FREE(arg);
      UdmSQLFree(&SQLRes);
      return res;
    }

    udm_snprintf(buf, len, "\
INSERT INTO server (rec_id, enabled, tag, category, command, parent, ordre, weight, url) \
VALUES (%s%d%s, %d, '%s', %s, '%c', %s%d%s, %d, %f, '%s')\
",
           qu, rec_id, qu,
           UDM_ATOI(UdmSQLValue(&SQLRes, 0, 0)),
           UdmSQLValue(&SQLRes, 0, 1),
           UdmSQLValue(&SQLRes, 0, 2),
           S->Server->command,
           qu, S->Server->parent, qu,
           UDM_ATOI(UdmSQLValue(&SQLRes, 0, 3)),
           S->Server->weight,
           UdmSQLEscStr(db, arg, UDM_NULL2EMPTY(S->Server->Match.pattern), strlen(UDM_NULL2EMPTY(S->Server->Match.pattern)) )
     );
    res = UdmSQLQuery(db, NULL, buf);
    UdmSQLFree(&SQLRes);

    S->Server->site_id = id = rec_id;
    UDM_FREE(Indexer->ServerIdCache[Indexer->pServerIdCache]);
    Indexer->ServerIdCache[Indexer->pServerIdCache] = (char*)UdmStrdup(S->Server->Match.pattern);
    Indexer->ServerIdCacheCommand[Indexer->pServerIdCache] = S->Server->command;
    Indexer->ServerIdCacheId[Indexer->pServerIdCache] = id;
    Indexer->pServerIdCache = (Indexer->pServerIdCache + 1) % UDM_SERVERID_CACHE_SIZE;
  }
  UDM_FREE(buf); UDM_FREE(arg);
  return UDM_OK;
}

/************************* find url ********************************/

static int UdmFindURL(UDM_AGENT *Indexer, UDM_DOCUMENT * Doc, UDM_DB *db)
{
  UDM_SQLRES  SQLRes;
  const char  *url=UdmVarListFindStr(&Doc->Sections,"URL","");
  udmhash32_t  id = 0;
  int    use_crc32_url_id;
  int    rc=UDM_OK;
  
  use_crc32_url_id = !strcasecmp(UdmVarListFindStr(&Indexer->Conf->Vars, "UseCRC32URLId", "no"), "yes");

  if(use_crc32_url_id)
  {
    /* Auto generation of rec_id */
    /* using CRC32 algorythm     */
    id = UdmStrHash32(url);
  }else{
    const char *o;
    char *qbuf, *e_url;
    size_t i, l, url_length= strlen(url);
    
    /* Escape URL string */
    if ((e_url = (char*)UdmMalloc(l = (8 * url_length + 1))) == NULL ||
        (qbuf = (char*)UdmMalloc( l + 100 )) == NULL)
    {
      UDM_FREE(e_url);
      UdmLog(Indexer, UDM_LOG_ERROR, "Out of memory");
      return UDM_ERROR;
    }
    UdmSQLEscStr(db,e_url,url,url_length);


    for(i = 0; i < UDM_FINDURL_CACHE_SIZE; i++)
    {
      if (Indexer->UdmFindURLCache[i])
        if (!strcmp(e_url, Indexer->UdmFindURLCache[i]))
        {
          id = Indexer->UdmFindURLCacheId[i];
          break;
        }
    }

    if (id == 0)
    {
      udm_snprintf(qbuf, l + 100, "SELECT rec_id FROM url WHERE url='%s'",e_url);
      if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLRes,qbuf)))
      {
        UDM_FREE(e_url);
        UDM_FREE(qbuf);
        return rc;
      }
      for(i=0;i<UdmSQLNumRows(&SQLRes);i++)
      {
        if((o=UdmSQLValue(&SQLRes,i,0)))
        {
          id=atoi(o);
          break;
        }
      }
      UdmSQLFree(&SQLRes);
      UDM_FREE(Indexer->UdmFindURLCache[Indexer->pURLCache]);
      Indexer->UdmFindURLCache[Indexer->pURLCache] = (char*)UdmStrdup(e_url);
      Indexer->UdmFindURLCacheId[Indexer->pURLCache] = id;
      Indexer->pURLCache = (Indexer->pURLCache + 1) % UDM_FINDURL_CACHE_SIZE;
    }
    UDM_FREE(e_url);
    UDM_FREE(qbuf);
  }
  UdmVarListReplaceInt(&Doc->Sections, "ID", id);
  return  rc;
}


static int UdmFindMessage(UDM_AGENT *Indexer, UDM_DOCUMENT * Doc, UDM_DB *db)
{
  size_t     i, len;
  char     *qbuf;
  char     *eid;
  UDM_SQLRES  SQLRes;
  const char  *message_id=UdmVarListFindStr(&Doc->Sections,"Header.Message-ID",NULL);
  int    rc;
  
  if(!message_id)
    return UDM_OK;
  
  len = strlen(message_id);
  eid = (char*)UdmMalloc(4 * len + 1);
  if (eid == NULL) return UDM_ERROR;
  qbuf = (char*)UdmMalloc(4 * len + 128);
  if (qbuf == NULL)
  { 
    UDM_FREE(eid);
    return UDM_ERROR;
  }

  /* Escape URL string */
  UdmSQLEscStr(db, eid, message_id, len);
  
  udm_snprintf(qbuf, 4 * len + 128, 
     "SELECT rec_id FROM url u, urlinfo i WHERE u.rec_id=i.url_id AND i.sname='Message-ID' AND i.sval='%s'", eid);
  rc = UdmSQLQuery(db,&SQLRes,qbuf);
  UDM_FREE(qbuf);
  UDM_FREE(eid);
  if (UDM_OK != rc)
    return rc;
  
  for(i=0;i<UdmSQLNumRows(&SQLRes);i++)
  {
    const char * o;
    if((o=UdmSQLValue(&SQLRes,i,0)))
    {
      UdmVarListReplaceInt(&Doc->Sections,"ID", UDM_ATOI(o));
      break;
    }
  }
  UdmSQLFree(&SQLRes);
  return(UDM_OK);
}


/*********** Coord tools *******************/

/*
  Scan one coord.
  Return the number of bytes scanned.
*/
static inline size_t
udm_coord_get(size_t *pwc, const unsigned char *s, const unsigned char *e)
{
  unsigned char c;
  
  if (s >= e)
    return 0;
  
  c= s[0];
  if (c < 0x80) 
  {
    *pwc = c;
    return 1;
  } 
  else if (c < 0xc2) 
    return UDM_ERROR;
  else if (c < 0xe0) 
  {
    if (s+2 > e) /* We need 2 characters */ 
      return 0;
    
    if (!((s[1] ^ 0x80) < 0x40))
      return 0;
    
    *pwc = ((size_t) (c & 0x1f) << 6) | (size_t) (s[1] ^ 0x80);
    return 2;
  } 
  else if (c < 0xf0) 
  {
    if (s+3 > e) /* We need 3 characters */
      return 0;
    
    if (!((s[1] ^ 0x80) < 0x40 && (s[2] ^ 0x80) < 0x40 && (c >= 0xe1 || s[1] >= 0xa0)))
      return 0;
    
    *pwc = ((size_t) (c & 0x0f) << 12)   | 
           ((size_t) (s[1] ^ 0x80) << 6) | 
            (size_t) (s[2] ^ 0x80);
    
    return 3;
  } 
  else if (c < 0xf8)
  {
    if (s + 4 > e) /* Need 4 bytes */
      return 0;
    if (!((s[1] ^ 0x80) < 0x40 && (s[2] ^ 0x80) < 0x40 && (s[3] ^ 0x80) < 0x40 && (c >= 0xf1 || s[1] >= 0x90)))
      return 0;
    
    *pwc= ((size_t) (c & 0x07) << 18)    |
          ((size_t) (s[1] ^ 0x80) << 12) |
          ((size_t) (s[2] ^ 0x80) << 6)  |
          (size_t)  (s[3] ^ 0x80);
    return 4;
  }
  return 0;
}


/*
  A faster version, without range checking
  The incoming string must have enough space
  to scan full miltibyte sequence.
*/
static inline size_t
udm_coord_get_quick(size_t *pwc, const unsigned char *s)
{
  unsigned char c;
  
  c= s[0];
  if (c < 0x80) 
  {
    *pwc = c;
    return 1;
  } 
  else if (c < 0xc2) 
    return UDM_ERROR; /* The caller will skip one byte */
  else if (c < 0xe0) 
  {
    size_t s1;
    if (!((s1= (s[1] ^ 0x80)) < 0x40))
      return 0;
    *pwc= (((size_t) (c & 0x1f)) << 6) | s1;
    return 2;
  } 
  else if (c < 0xf0) 
  {
    if (!((s[1] ^ 0x80) < 0x40 && (s[2] ^ 0x80) < 0x40 && (c >= 0xe1 || s[1] >= 0xa0)))
      return 0;
    
    *pwc = ((size_t) (c & 0x0f) << 12)   | 
           ((size_t) (s[1] ^ 0x80) << 6) | 
            (size_t) (s[2] ^ 0x80);
    
    return 3;
  } 
  else if (c < 0xf8)
  {
    if (!((s[1] ^ 0x80) < 0x40 && (s[2] ^ 0x80) < 0x40 && (s[3] ^ 0x80) < 0x40 && (c >= 0xf1 || s[1] >= 0x90)))
      return 0;
    
    *pwc= ((size_t) (c & 0x07) << 18)    |
          ((size_t) (s[1] ^ 0x80) << 12) |
          ((size_t) (s[2] ^ 0x80) << 6)  |
          (size_t)  (s[3] ^ 0x80);
    return 4;
  }

  return 0;
}


/*
  Skip the given amount of coords, accumulating sum.
*/

static inline const unsigned char *
udm_coord_sum(size_t *sump,
              const unsigned char *s,
              const unsigned char *e,
              size_t n)
{
  size_t sum= 0, nbytes, tmp;
  
  /* String must have at least "n" bytes */
  if (s + (n * 4) > e)
    goto mb;
  
  /* Quickly sum leading 7bit numbers */
  for (; *s < 128 && n; n--)
    sum+= *s++;

  for ( ; n; n--)
  {
    if (!(nbytes= udm_coord_get_quick(&tmp, s)))
      return e;
    s+= nbytes;
    sum+= tmp;
  }
  *sump= sum;
  return s;
  
mb:
  for ( ; n; n--)
  {
    if (!(nbytes= udm_coord_get(&tmp, s, e)))
      return e;
    s+= nbytes;
    sum+= tmp;
  }
  *sump= sum;
  return s;
}


/*
  Skip the given anount of coords in the given string
  s - beginning of the string
  e - end of the string (last byte + 1)
  n - how many characters to skip
  RETURNS
    position after n characters, or end of the string (if shorter)
*/
static inline const unsigned char *
udm_coord_skip(const unsigned char *s, const unsigned char *e, size_t n)
{
#if defined(__i386__) && !defined(_WIN64)
  {
    const unsigned char *e4= e - 4;
    for ( ; n > 4 && s < e4; n-= 4, s+= 4)
    {
      if (*((const uint4*)s) & 0x80808080)
        break;
    }
  }
#endif

  for ( ; n && s < e; n--)
  {
    if (*s++ < 0x80)
    {
      /* Single byte character, nothing to do */
    }
    else
    {
      unsigned char c= s[-1];
      if (c < 0xc2) /* Error */
        return e;
      else if (c < 0xe0) /* Two-byte sequence */
      {
        s++;
      }
      else if (c < 0xf0) /* Three-byte sequence */
      {
        s+= 2;
      }
      else if (c < 0xf8) /* Four-byte sequence */
      {
        s+= 3;
      }
      else
      {
        /* TODO: longer sequences */
        return e;
      }
      if (s > e)
        return e;
    }
  }
  return s;
}


static inline size_t
udm_coord_len(const char *str)
{
  size_t len, nbytes, lintag, crd;
  const char *s, *e;

  if (!str)
    return 0;
  lintag= strlen(str);
  for (len= 0, s= str, e = str + lintag; e > s; s += nbytes, len++)
  {
    if (!(nbytes= udm_coord_get(&crd,
                                (const unsigned char *)s,
                                (const unsigned char *)e)))
      break;
  }
  return len;
}


static inline size_t
udm_coord_put(size_t wc, unsigned char *r, unsigned char *e)
{
  int count;
  
  if (r >= e)
    return 0;
  
  if (wc < 0x80) 
    count= 1;
  else if (wc < 0x800) 
    count= 2;
  else if (wc < 0x10000) 
    count= 3;
  else if (wc < 0x200000)
    count= 4;
  else return 0;
  
  /* 
    e is a character after the string r, not the last character of it.
    Because of it (r+count > e), not (r+count-1 >e )
   */
  if (r + count > e) 
    return 0;
  
  switch (count)
  { 
    /* Fall through all cases */
    case 4: r[3]= (unsigned char) (0x80 | (wc & 0x3f)); wc = wc >> 6; wc |= 0x10000;
    case 3: r[2]= (unsigned char) (0x80 | (wc & 0x3f)); wc = wc >> 6; wc |= 0x800;
    case 2: r[1]= (unsigned char) (0x80 | (wc & 0x3f)); wc = wc >> 6; wc |= 0xc0;
    case 1: r[0]= (unsigned char) wc;
  }
  return count;
}

/********************** Words ***********************************/

static int UdmDeleteWordFromURL(UDM_AGENT *Indexer, UDM_DOCUMENT *Doc,
                                UDM_DB *db)
{
  char qbuf[512];
  int i= 0, rc= UDM_OK;
  urlid_t url_id= UdmVarListFindInt(&Doc->Sections, "ID", 0);
  const char *qu= (db->DBType == UDM_DB_PGSQL) ? "'" : "";

  if (! UdmVarListFindInt(&Doc->Sections, "PrevStatus", 0))
    return UDM_OK;

  switch (db->DBMode)
  {
    case UDM_DBMODE_BLOB:
      rc= UdmRemoveWordsBlob(Indexer, Doc, db);
      break;
    case UDM_DBMODE_MULTI:
      for(i= 0; i <= MULTI_DICTS; i++)
      {
        udm_snprintf(qbuf, sizeof(qbuf),
                     "DELETE FROM dict%02X WHERE url_id=%s%i%s",
                     i, qu, url_id, qu);
        if (UDM_OK != (rc= UdmSQLQuery(db, NULL, qbuf)))
          break;
      }
      break;
    case UDM_DBMODE_SINGLE:
      udm_snprintf(qbuf, sizeof(qbuf), "DELETE FROM dict WHERE url_id=%s%d%s",
                   qu, url_id, qu);
      rc= UdmSQLQuery(db, NULL, qbuf);
      break;
    default:
      rc= UDM_ERROR;
  }
  return rc;
}


#define COORD_MAX_LEN 4

/*
  Encode intag:
  0 - hex encoding
  1 - hex encoding with '0x' prefix
  2 - PgSQL bytea encoding
*/
static char *UdmMultiCachePutIntag (UDM_MULTI_CACHE_WORD *cache, char type)
{
  size_t c;
  size_t len;
  size_t i;
  char *_;
  unsigned char buf[COORD_MAX_LEN];
  unsigned char *bufend = buf + sizeof(buf);
  size_t nbytes;
  size_t last;
  size_t multiply= (type == 2) ? 3*5 : 3*2;

  if (! cache->nintags) return(NULL);

  /* Allocate maximum possible length. */
  _ = UdmMalloc(cache->nintags * multiply + 3);
  if (! _) return(NULL);

  if (type == 1)
  {
    strcpy(_, "0x");
    len= 2;
  }
  else
  {
    *_ = 0;
    len = 0;
  }

  last = 0;
  for (c = 0; c < cache->nintags; c++)
  {
    if (cache->intags[c] < last)
    {
      UdmFree(_);
      return(NULL);
    }

    nbytes = udm_coord_put(cache->intags[c] - last, buf, bufend);
    if (! nbytes)
    {
      UdmFree(_);
      return(NULL);
    }

    if (type == 2)
    {
      for (i= 0; i < nbytes; i++)
        len+= UdmSQLBinEscCharForPg(_ + len, buf[i]);
      _[len]= '\0';
    }
    else
    {
      for (i= 0; i < nbytes; i++)
      {
        udm_snprintf(_ + len, 3, "%02X", buf[i]);
        len += 2;
      }
    }

    last = cache->intags[c];
  }

  return(_);
}


static char *UdmMultiCachePutIntag1 (UDM_MULTI_CACHE_WORD *cache)
{
  size_t c;
  size_t len = 0;
  char *_;
  unsigned char buf[COORD_MAX_LEN];
  unsigned char *bufend = buf + sizeof(buf);
  size_t nbytes;
  size_t last;

  if (! cache->nintags) return(NULL);

  /* Allocate maximum possible length. */
  _ = UdmMalloc(cache->nintags * COORD_MAX_LEN + 1);
  if (! _) return(NULL);

  last = 0;
  for (c = 0; c < cache->nintags; c++)
  {
    if (cache->intags[c] < last)
    {
      UdmFree(_);
      return(NULL);
    }

    nbytes = udm_coord_put(cache->intags[c] - last, buf, bufend);
    if (! nbytes)
    {
      UdmFree(_);
      return(NULL);
    }

    memcpy(_ + len, buf, nbytes);
    len += nbytes;

    last = cache->intags[c];
  }
  _[len] = 0;

  return(_);
}

#define UDM_THREADINFO(A,s,m)	if(A->Conf->ThreadInfo)A->Conf->ThreadInfo(A,s,m)

int UdmWordCacheWrite (UDM_AGENT *Indexer, UDM_DB *db, size_t limit)
{
  size_t i;
  size_t in_num, in_limit= UdmVarListFindInt(&Indexer->Conf->Vars, "URLSelectCacheSize", URL_DELETE_CACHE);
  size_t LastLocked = 0;
  int rc;
  UDM_WORD_CACHE *cache = &db->WordCache;
  UDM_DSTR buf, qbuf;
  UDM_MULTI_CACHE_WORD mintag;
  size_t aintags = 0;
  char arg[128];

  if (cache->nbytes <= limit) return(UDM_OK);
  UdmLog(Indexer, UDM_LOG_ERROR, "Writing words (%d words, %d bytes%s).", cache->nwords, cache->nbytes, limit ? "" : ", final");

  UDM_THREADINFO(Indexer, "Starting tnx", "");
  if(UDM_OK!=(rc=UdmSQLBegin(db)))
  {
    UdmWordCacheFree(cache);
    return(rc);
  }

  UdmDSTRInit(&buf, 8192);
  UdmDSTRInit(&qbuf, 8192);
  mintag.intags = NULL;
  mintag.word = NULL;
  
  /* Sort IN list */
  if (cache->nurls)
    UdmSort(cache->urls, cache->nurls, sizeof(urlid_t), (udm_qsort_cmp)cmpaurls);

  /* Remove expired words */
  for (i= 0, in_num= 0; i < cache->nurls; i++)
  {
    if (buf.size_data) UdmDSTRAppend(&buf, ",", 1);
    UdmDSTRAppendf(&buf, "'%d'", cache->urls[i]);

    /* Don't allow too long IN list */
    if (++in_num >= in_limit || (i + 1) == cache->nurls)
    {
      size_t dictno;
      for (dictno= 0; dictno <= MULTI_DICTS; dictno++)
      {
        udm_snprintf(arg, sizeof(arg), "dict%02X", dictno);
        UDM_THREADINFO(Indexer, "Deleting", arg);
        UdmDSTRReset(&qbuf);
        UdmDSTRAppendf(&qbuf, "DELETE FROM dict%02X WHERE url_id IN (%s)", dictno, buf.data);
        if(UDM_OK != (rc= UdmSQLQuery(db,NULL,qbuf.data)))
          goto unlock_UdmStoreWordsMulti;
      }
      UdmDSTRReset(&buf);
      UdmDSTRReset(&qbuf);
      in_num= 0;
    }
  }

  UdmDSTRReset(&buf);
  UdmDSTRReset(&qbuf);

  if (cache->nwords)
    UdmWordCacheSort(cache);

  for (i = 0; i < cache->nwords;)
  {
    UDM_WORD_CACHE_WORD *cword = &cache->words[i];
    unsigned char seed = cword->seed;
    char *word = cword->word;
    urlid_t url_id = cword->url_id;
    unsigned char secno = cword->secno;
    char *intag;

    mintag.nintags = 0;

    do
    {
      if (mintag.nintags == aintags)
      {
        uint4 *itmp;
        itmp = UdmRealloc(mintag.intags, (aintags + 256) * sizeof(uint4));
	if (! itmp)
	{
	  goto unlock_UdmStoreWordsMulti;
	}
	mintag.intags = itmp;
	aintags += 256;
      }
      mintag.intags[mintag.nintags]= cword->pos;
      mintag.nintags++;

      i++;
      if (i == cache->nwords) break;
      cword = &cache->words[i];
    }
    while (seed == cword->seed &&
	url_id == cword->url_id &&
        secno == cword->secno &&
	! strcmp(word, cword->word));

    udm_snprintf(arg, sizeof(arg), "dict%02X", seed);
    UDM_THREADINFO(Indexer, "Writting", arg);

    if (db->DBType == UDM_DB_MYSQL)
    {
      intag = UdmMultiCachePutIntag(&mintag, 1);
      if (! intag) continue;

      if (buf.size_data)
      {
        UdmDSTRAppendf(&buf, ",(%d, %d, '%s', %s)",
	               url_id, secno, word, intag);
      } else {
        UdmDSTRAppendf(&buf, "INSERT INTO dict%02X (url_id,secno,word,intag) VALUES(%d,%d,'%s',%s)",
                       seed, url_id, secno, word, intag);
      }
      UdmFree(intag);
      if (seed != cword->seed || i == cache->nwords)
      {
        if (LastLocked <= seed)
	{
          if (LastLocked) UdmSQLQuery(db, NULL, "UNLOCK TABLES");
	  LastLocked = seed;
	  UdmDSTRAppendf(&qbuf, "LOCK TABLES dict%02X WRITE", LastLocked);
          for (LastLocked++; LastLocked <= MULTI_DICTS; LastLocked++)
          {
	    if (LastLocked - seed == 0x10) break;
            UdmDSTRAppendf(&qbuf, ",dict%02X WRITE", LastLocked);
          }
          UdmSQLQuery(db, NULL, qbuf.data);
	  UdmDSTRReset(&qbuf);
	}

        if (buf.size_data)
        {
          if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,buf.data)))
          {
            goto unlock_UdmStoreWordsMulti;
          }
	  UdmDSTRReset(&buf);
        }
      }
    }
    else
    {
      const char *quot;
      const char *x;
      const char *castb;
      const char *caste;

      if (db->DBType == UDM_DB_ORACLE8 || db->DBType == UDM_DB_DB2) 
        intag = UdmMultiCachePutIntag(&mintag, 0);
      else if (db->DBType == UDM_DB_MSSQL ||
               db->DBType == UDM_DB_SYBASE ||
               db->DBType == UDM_DB_ACCESS)
        intag = UdmMultiCachePutIntag(&mintag, 1);
      else if (db->DBType == UDM_DB_PGSQL)
        intag= UdmMultiCachePutIntag(&mintag, 2);
      else
      {
        if ((intag= UdmMultiCachePutIntag1(&mintag)))
        {
          char *tmp= UdmSQLEscStr(db, NULL, intag, strlen(intag));
          UdmFree(intag);
          intag= tmp;
        }
      }
      if (! intag) continue;

      if (db->DBType == UDM_DB_MSSQL ||
          db->DBType == UDM_DB_SYBASE ||
          db->DBType == UDM_DB_ACCESS)
        quot="";
      else
        quot="'";

      if (db->DBType == UDM_DB_DB2)
      {
        x="X";
        castb="CAST(";
        caste=" AS BLOB)";
      }
      else
      {
        x="";
        castb="";
        caste="";
      }

      UdmDSTRAppendf(&buf,
        "INSERT INTO dict%02X (url_id,secno,word,intag) VALUES(%d,%d,'%s',%s%s%s%s%s%s)",
        seed, url_id, secno, word, castb,x,quot,intag,quot,caste);
      UdmFree(intag);
      if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,buf.data))) goto unlock_UdmStoreWordsMulti;
      UdmDSTRReset(&buf);
    }
  }

unlock_UdmStoreWordsMulti:
  UDM_FREE(mintag.intags);
  UdmDSTRFree(&buf);
  UdmDSTRFree(&qbuf);
  UDM_THREADINFO(Indexer, "Committing tnx", "");

  if (LastLocked && rc == UDM_OK)
    rc = UdmSQLQuery(db, NULL, "UNLOCK TABLES");

  if(rc==UDM_OK)
    rc=UdmSQLCommit(db);

  UdmWordCacheFree(&db->WordCache);
  UdmLog(Indexer, UDM_LOG_ERROR, "The words are written successfully.%s", limit ? "" : " (final)");
  return(rc);
}

static int UdmStoreWordsMulti (UDM_AGENT *Indexer, UDM_DOCUMENT *Doc, UDM_DB *db)
{
  size_t i;
  int rc = UDM_OK;
  urlid_t  url_id = UdmVarListFindInt(&Doc->Sections, "ID", 0);
  unsigned char PrevStatus = UdmVarListFindInt(&Doc->Sections, "PrevStatus", 0) ? 1 : 0;
  int WordCacheSize = UdmVarListFindInt(&Indexer->Conf->Vars, "WordCacheSize", 0);

  if (UdmVarListFindInt(&Indexer->Conf->Vars, "SaveSectionSize", 1))
  {
    if (UDM_OK!= (rc= UdmWordListSaveSectionSize(Doc)))
      return rc;
  }
  

  if (WordCacheSize <= 0) WordCacheSize = 0x800000;

  if (PrevStatus) UdmWordCacheAddURL(&db->WordCache, url_id);

  for (i = 0; i < Doc->Words.nwords; i++)
  {
    if (! Doc->Words.Word[i].secno) continue;
    UdmWordCacheAdd(&db->WordCache, url_id, &Doc->Words.Word[i]);
  }

  rc = UdmWordCacheWrite(Indexer, db, WordCacheSize);
  return(rc);
}


static int swbcmp(UDM_WORD *w1, UDM_WORD *w2)
{
  register int _;
  /*
    We don't need to compare W->hash, because
    the words already distributed into different 32 lists,
    according to hash
    if ((_= (int) w1->hash - (int) w2->hash))
      return _;
  */
  if ((_= strcmp(w1->word, w2->word)))
    return _;
  if ((_= (int) w1->secno - (int) w2->secno))
    return _;
  return (int) w1->pos - (int) w2->pos;
}


static int UdmRemoveWordsBlob(UDM_AGENT *Indexer, UDM_DOCUMENT *Doc, UDM_DB *db)
{
  urlid_t url_id= UdmVarListFindInt(&Doc->Sections, "ID", 0);
  char buf[64];
  /*
   * Remove document if it is not in searching table (state = 1).
   * Mark document for deletion if it is in searching table
   * (state = 2).
   */
  udm_snprintf(buf, sizeof(buf),
               "DELETE FROM bdicti WHERE state=1 AND url_id=%d", url_id);
  if (UDM_OK != UdmSQLQuery(db, NULL, buf))
    return UDM_ERROR;
  udm_snprintf(buf, sizeof(buf),
               "UPDATE bdicti SET state=0 WHERE state=2 AND url_id=%d", url_id);
  if (UDM_OK != UdmSQLQuery(db, NULL, buf))
    return UDM_ERROR;
  return UDM_OK;
}


static int
UdmStoreWordBlobUsingHex(UDM_DB *db, urlid_t url_id,
                         UDM_DSTR *dbuf, size_t *chunks)
{
  size_t i;
  int rc;
  UDM_DSTR qbuf;
  const char *prefix= ",0x"; /* 0xAABBCC syntax by default */
  const char *suffix= "";
  size_t prefix_length= 3;
  size_t suffix_length= 0;

  if (db->flags & UDM_SQL_HAVE_STDHEX) /* X'AABBCC' syntax */
  {
    prefix= ",X'";
    suffix= "'";
    suffix_length= 1;
  }
  
  UdmDSTRInit(&qbuf, dbuf->size_data * 2 + 256);
  UdmDSTRAppendf(&qbuf, "INSERT INTO bdicti VALUES(%d,1", url_id);
  for (i= 0; i < 32; i++)
  {
    size_t length= chunks[i + 1] - chunks[i];
    char *chunk= dbuf->data + chunks[i];
    if (length)
    {
      UdmDSTRAppend(&qbuf, prefix, prefix_length);
      UdmDSTRAppendHex(&qbuf, chunk, length);
      if (suffix_length)
        UdmDSTRAppend(&qbuf, suffix, suffix_length);
    }
    else
      UdmDSTRAppend(&qbuf, ",''", 3);
  }
  UdmDSTRAppend(&qbuf, ")", 1);
  rc= UdmSQLQuery(db, NULL, qbuf.data);
  UdmDSTRFree(&qbuf);
  return rc;
}

/*   
   bdicti table has `state` field, that signals bdicti record state.
   There are three possible states:
   0 - deleted document (record is in bdicti and bdict)
   1 - new document (record is in bdicti only)
   2 - converted document (record is in bdicti and bdict)

   New document is added to a collection
   -------------------------------------
   New record is added to bdicti table with `state` set to 1.

   Convert to bdict (full)
   -----------------------
   Read records from bdicti with `state` > 0;
   Convert these records and write converted records to bdict;
   Set `state` to 2 for records with `state` = 1;
   Delete from bdicti records with `state` = 0;
   (afair no parallel bdicti updates assumed)

   Convert to bdict (partial)
   --------------------------
   Read records from bdicti with `state` set to either 0 (deleted) or 1 (new);
   Convert these records and merge converted records with those that are in bdict;
   Set `state` to 2 for records with `state` = 1;
   Delete from bdicti records with `state` = 0;
   (same as above - no parallel updates)

   Document is removed from collection
   -----------------------------------
   Either delete from bdicti document record with `state` = 1,
   Or set `state` to 0 if record `state` is 2.

   All documents are removed (truncation)
   --------------------------------------
   This is simple - truncate both bdict and bdicti tables.

   Document is updated
   -------------------
   See:
     Document is removed from collection;
     Document is added to a collection.

   Searching bdicti
   ----------------
   Only records with `state` 1 (new) or 2 (converted) matter.

   Searching bdicti + bdict
   ------------------------
   Only records with `state` 0 (deleted) or 1 (new) matter.
 */

static int UdmStoreWordsBlob(UDM_AGENT *Indexer, UDM_DOCUMENT *Doc, UDM_DB *db)
{
  size_t i, j, nwords, chunks[33];
  urlid_t url_id= UdmVarListFindInt(&Doc->Sections, "ID", 0);
  UDM_DSTR dbuf;
  int rc= UDM_OK;
  int save_section_size= UdmVarListFindInt(&Indexer->Conf->Vars, "SaveSectionSize", 1);
  UDM_WORDLIST WordList[32];

  if (UdmVarListFindInt(&Doc->Sections, "PrevStatus", 0))
    if (UDM_OK != UdmRemoveWordsBlob(Indexer, Doc, db))
      return UDM_ERROR;
  /*
   * Set hash to UdmStrHash32(word) & 31 for further sorting.
   * If this is stopword, set hash to 0xff (will be skipped while
   * inserting).
   */
  if (!(nwords= Doc->Words.nwords))
    return UDM_OK;
  bzero((void*) &WordList, sizeof(WordList));

  /* Count number of words in each WordList */
  for (i= 0; i < nwords; i++)
  {
    UDM_WORD *W= &Doc->Words.Word[i];
    W->hash= W->secno ? (UdmStrHash32(W->word) & 0x1f) : 0xff;
    if (W->hash < 32)
      WordList[W->hash].nwords++;
  }
  
  /* Allocate memory for each WordList */
  for (i= 0; i < 32; i++)
  {
    if (WordList[i].nwords)
    {
      WordList[i].Word= (UDM_WORD*) UdmMalloc(WordList[i].nwords * sizeof(UDM_WORD));
      WordList[i].nwords= 0;
    }
  }
  
  /* Copy distribute words into 32 lists */
  for (i= 0; i < nwords; i++)
  {
    UDM_WORD *W= &Doc->Words.Word[i];
    if (W->hash < 32)
    {
      UDM_WORDLIST *WL= &WordList[W->hash];
      assert(WL->Word != NULL);
      WL->Word[WL->nwords++]= *W;
    }
  }
  
  /*
    Sort 32 small lists,
    this is faster than sorting a single big word list.
  */
  for (i= 0; i < 32; i++)
  {
    UDM_WORDLIST *WL= &WordList[i];
    if (WL->nwords)
    {
      UdmSort(WL->Word, WL->nwords, sizeof(UDM_WORD), (udm_qsort_cmp) swbcmp);
    }
  }
  
  UdmDSTRInit(&dbuf, 1024*64);
  for (j= 0, i= 0; i < 32; i++)
  {
    UDM_WORDLIST *WL= &WordList[i];
    UDM_WORD *WLWord= WL->Word;
    int prev_pos= 0;
    unsigned char prev_secno= 0;
    const char *prev_word= "";
    chunks[i]= dbuf.size_data;
    
    for (j= 0; j < WL->nwords; )
    {
      unsigned char buf[COORD_MAX_LEN];
      size_t nbytes;
      UDM_WORD *W= &WLWord[j];
      int pos= W->pos;
      unsigned char secno= W->secno;
      if (strcmp(W->word, prev_word))
      {
        if (*prev_word)
          UdmDSTRAppend(&dbuf, "\0\0", 2);
        UdmDSTRAppendSTR(&dbuf, W->word);
        prev_secno= 0;
        prev_pos= 0;
        prev_word= W->word;
      }
      if (secno != prev_secno)
      {
        UdmDSTRAppend(&dbuf, "", 1);
        UdmDSTRAppend(&dbuf, (char*) &secno, 1);
        prev_pos= 0;
        prev_secno= 0;
      }
      nbytes= udm_coord_put((int) pos - prev_pos, buf, buf + sizeof(buf));
      UdmDSTRAppend(&dbuf, (char *)buf, nbytes);
      prev_pos= pos;
      prev_secno= secno;
      j++;
      
      /* Detect end of section, and put section length */
      if (save_section_size &&
          (j >= WL->nwords || strcmp(W->word, W[1].word)))
      {
        int seclen= (Doc->CrossWords.wordpos[prev_secno] + 1) - prev_pos;
        nbytes= udm_coord_put(seclen, buf, buf + sizeof(buf));
        UdmDSTRAppend(&dbuf, (char*) buf, nbytes);
      }
    }
  }
  chunks[32]= dbuf.size_data;
  if (db->flags & UDM_SQL_HAVE_0xHEX)
  {  
    rc= UdmStoreWordBlobUsingHex(db, url_id, &dbuf, chunks);
  }
  else if (db->DBType == UDM_DB_PGSQL)
  {
    UDM_DSTR qbuf;
    UdmDSTRInit(&qbuf, dbuf.size_data * 5 + 256);
    UdmDSTRAppendf(&qbuf, "INSERT INTO bdicti VALUES(%d,1", url_id);
    for (i= 0; i < 32; i++)
    {
      size_t srclen= chunks[i + 1] - chunks[i];
      UdmDSTRAppend(&qbuf, ",'", 2);
      if (srclen)
      {
        size_t slen;
        slen= UdmSQLBinEscStr(db, qbuf.data + qbuf.size_data, dbuf.data + chunks[i], srclen);
        qbuf.size_data+= slen;
      }
      UdmDSTRAppend(&qbuf, "'", 1);
    }
    UdmDSTRAppend(&qbuf, ")", 1);
    rc= UdmSQLQuery(db, NULL, qbuf.data);
    UdmDSTRFree(&qbuf);
  }
  else if (db->flags & UDM_SQL_HAVE_BIND)
  {
    char qbuf[512];
    if (db->DBDriver == UDM_DB_ORACLE8)
    {
      udm_snprintf(qbuf, sizeof(qbuf),
                 "INSERT INTO bdicti VALUES(%d,1,:1,:2,:3,:4,:5,:6,:7,:8,:9,:10,:11,:12,:13,:14,:15,:16,:17,:18,:19,:20,:21,:22,:23,:24,:25,:26,:27,:28,:29,:30,:31,:32)", url_id);
    }
    else
    {
      udm_snprintf(qbuf, sizeof(qbuf),
                  "INSERT INTO bdicti VALUES(%d,1,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?)", url_id);
    }

    if (UDM_OK != (rc= db->sql->SQLPrepare(db, qbuf)))
      goto ret;
    for (i= 0; i < 32; i++)
    {
      int length= chunks[i + 1] - chunks[i];
      char *data= dbuf.data + chunks[i];
      if (length == 0 &&
          db->DBDriver == UDM_DB_ODBC &&
          db->DBType == UDM_DB_ORACLE8)
        length= UDM_SQL_NULL_DATA;
      if (UDM_OK != (rc= db->sql->SQLBind(db, i + 1, data, length, UDM_SQLTYPE_LONGVARBINARY)))
        goto ret;
    }
    if(UDM_OK != (rc= db->sql->SQLExec(db)))
      goto ret;
  }
  else if (db->flags & UDM_SQL_HAVE_STDHEX) /* For SQLite3 */
  {
    rc= UdmStoreWordBlobUsingHex(db, url_id, &dbuf, chunks);
  }

ret:
  /*
    Just free Word pointers
    Don't need UdmWordListFree here,
    because it also frees Word[x].word.
  */
  for (i= 0; i < 32; i++)
    UDM_FREE(WordList[i].Word);
  UdmDSTRFree(&dbuf);
  return rc;
}


/*
static int
cmpurlid(UDM_URL_CRD *s1, UDM_URL_CRD *s2)
{
  if (s1->url_id > s2->url_id) return(1);
  if (s1->url_id < s2->url_id) return(-1);
  return s1->pos - s2->pos;
}
*/


static void
UdmSectionListPrint(UDM_SECTIONLIST *SectionList)
{
  UDM_SECTIONLIST *L= SectionList;
  size_t section;
  fprintf(stderr, "ncoords=%d nsec=%d\n",
          SectionList->ncoords, SectionList->nsections);
  for (section= 0; section < L->nsections; section++)
  {
    UDM_SECTION *S= &L->Section[section];
    size_t coord;
    /*
    fprintf(stderr,
            "       [%d:%d] ncoords=%d seclen=%d secno=%d min=%d max=%d offs=%d\n",
            S->url_id, S->secno, S->ncoords, S->seclen, S->secno,
            S->minpos, S->maxpos,
            S->Coord - SectionList->Coord);
    */
    if (!S->Coord)
      continue;
    for (coord= 0; coord < S->ncoords; coord++)
    {
      UDM_COORD2 *C= &S->Coord[coord];
      fprintf(stderr, "[%d]secno=%d pos=%d seclen=%d num=%d order=%d ncoords=%d min=%d max=%d\n",
                       S->url_id, S->secno, C->pos, S->seclen, S->wordnum, S->order,
                       S->ncoords, S->minpos, S->maxpos);
    }
  }
}


static int
UdmSectionListAlloc(UDM_SECTIONLIST *List, size_t ncoords, size_t nsections)
{
  bzero((void*)List, sizeof(*List));
  if (ncoords)
  {
    List->Coord= (UDM_COORD2*) UdmMalloc(ncoords * sizeof(UDM_COORD2));
    List->Section= (UDM_SECTION*) UdmMalloc(nsections * sizeof(UDM_SECTION));
    List->mcoords= ncoords;
    List->msections= nsections;
    UDM_ASSERT(nsections > 0);
  }
  return UDM_OK;
}


static void
UdmSectionListFree(UDM_SECTIONLIST *List)
{
  UDM_FREE(List->Coord);
  UDM_FREE(List->Section);
}


static int
UdmSectionListListAdd(UDM_SECTIONLISTLIST *List, UDM_SECTIONLIST *Item)
{
  if (List->nitems >= List->mitems)
  {
    List->mitems+= 256;
    List->Item= (UDM_SECTIONLIST*) UdmRealloc(List->Item, List->mitems * sizeof(*Item));
  }
  List->Item[List->nitems]= *Item;
  List->nitems++;
  return UDM_OK;
}


static void
UdmSectionListListInit(UDM_SECTIONLISTLIST *List)
{
  bzero((void*) List, sizeof(*List));
}


static void
UdmSectionListListFree(UDM_SECTIONLISTLIST *List)
{
  size_t i;
  for (i= 0; i < List->nitems; i++)
    UdmSectionListFree(&List->Item[i]);
  UDM_FREE(List->Item);
}


/*******************************************/
typedef struct
{
  urlid_t       url_id;
  udm_pos_t     seclen;
  udm_pos_t     pos;
  udm_wordnum_t num;
  udm_secno_t   secno;
} UDM_URL_CRD;


typedef struct
{
  size_t       acoords;
  size_t       ncoords;
  size_t       order;
  char	       *word;
  UDM_URL_CRD  *Coords;
} UDM_URLCRDLIST;


typedef struct
{
  int                freeme;
  size_t             nlists;
  UDM_URLCRDLIST     *List;
} UDM_URLCRDLISTLIST;


static void
UdmURLCRDListListInit(UDM_URLCRDLISTLIST *Lst)
{
  bzero((void*)Lst, sizeof(*Lst));
}

static int
UdmURLCRDListListAdd(UDM_URLCRDLISTLIST *LstLst, UDM_URLCRDLIST *List)
{
  size_t nbytes= (LstLst->nlists+1) * sizeof(UDM_URLCRDLIST);
  if (!(LstLst->List= (UDM_URLCRDLIST*) UdmRealloc(LstLst->List, nbytes)))
    return UDM_ERROR;
  LstLst->List[LstLst->nlists]= *List;
  LstLst->nlists++;
  return UDM_OK;
}


static void
UdmURLCRDListListFree(UDM_URLCRDLISTLIST *Lst)
{
  size_t i;
  
  for(i=0;i<Lst->nlists;i++)
  {
    UDM_FREE(Lst->List[i].word);
    UDM_FREE(Lst->List[i].Coords);
  }
  UDM_FREE(Lst->List);
  if(Lst->freeme)UDM_FREE(Lst);
}


static int
cmpurlid(UDM_URL_CRD *s1, UDM_URL_CRD *s2)
{
  if (s1->url_id > s2->url_id) return(1);
  if (s1->url_id < s2->url_id) return(-1);
  if (s1->secno > s2->secno) return 1;
  if (s1->secno < s2->secno) return -1;
  return (int) s1->pos - (int) s2->pos;
}


static void
UdmSortSearchWordsByURL(UDM_URL_CRD *wrd,size_t num)
{
  if(wrd && num)UdmSort((void*)wrd,num,sizeof(*wrd),(udm_qsort_cmp)cmpurlid);
}


/*****************************************/
typedef struct st_udm_findword_args
{
  UDM_AGENT *Agent;
  UDM_DB *db;
  UDM_URLCRDLISTLIST CoordListList;
  UDM_WIDEWORDLIST *WWList;
  UDM_URL_TMP urls; /* Combination of WHERE limit, fl, and live_deleted */
  UDM_URL_TMP live_update_active_urls;  /* Combination of WHERE and fl */
  UDM_URL_TMP live_update_deleted_urls; /* The list of deleted */
  UDM_WIDEWORDLIST CollationMatches;
  UDM_SECTIONLISTLIST SectionListList;
  const char *cmparg;
  const char *where;
  const char *word;
  char *wf;
  size_t wordnum;
  size_t count;
  size_t secno;
  int word_match;
  int save_section_size;
  int live_updates;
} UDM_FINDWORD_ARGS;


static int
cmp_data_urls(UDM_URLDATA *d1, UDM_URLDATA *d2)
{
  if (d1->url_id > d2->url_id) return 1;
  if (d1->url_id < d2->url_id) return -1;
  return 0;
}

static int UdmURLSearchForConv(UDM_URLDATALIST *List, urlid_t id)
{
  UDM_URLDATA d;
  void *found;
  if (!List->nitems)
    return 0;
  d.url_id= id;
  found= UdmBSearch(&d, List->Item, List->nitems, sizeof(UDM_URLDATA),
                    (udm_qsort_cmp) cmp_data_urls);
  return found ? 1 : 0;
}


static int
UdmLoadSlowLimit(UDM_DB *db, UDM_URL_TMP *list, const char *q)
{
  size_t i;
  int rc;
  UDM_SQLRES SQLRes;
  int exclude= list->exclude;
  bzero((void*) list, sizeof(UDM_URL_TMP));
  list->exclude= exclude;
  if (UDM_OK != (rc= UdmSQLQuery(db, &SQLRes, q)))
    goto ret;

  if (!(list->nurls= UdmSQLNumRows(&SQLRes)))
    goto sqlfree;

  if (!(list->urls= UdmMalloc(sizeof(urlid_t) * list->nurls)))
  {
    rc= UDM_ERROR;
    list->nurls= 0;
    goto ret;
  }
  for (i= 0; i < list->nurls; i++)
  {
    list->urls[i]= atoi(UdmSQLValue(&SQLRes, i, 0));
  }
  UdmSort(list->urls, list->nurls, sizeof(urlid_t), (udm_qsort_cmp)cmpaurls);

sqlfree:
  UdmSQLFree(&SQLRes);
ret:
  return rc;
}


static int
UdmLoadURLDataFromURLForConv(UDM_AGENT *A, UDM_DB *db, UDM_URLDATALIST *C)
{
  int rc= UDM_OK;
  char qbuf[4*1024];
  UDM_SQLRES  SQLres;
  size_t nbytes, i, j;
  const char *fl= UdmVarListFindStr(&A->Conf->Vars, "fl", NULL);
  const char *url= (db->from && db->from[0]) ? "url." : "";
  UDM_URL_TMP fl_urls;

  bzero((void*)&fl_urls, sizeof(fl_urls));
  bzero((void*)C, sizeof(*C));
  
  if (fl)
  {
    char name[64];
    const char *q;
    if ((fl_urls.exclude= (fl[0] == '-')))
      fl++;
    udm_snprintf(name, sizeof(name), "Limit.%s", fl);
    q= UdmVarListFindStr(&A->Conf->Vars, name, NULL);
    if (!q)
    {
      UdmLog(A, UDM_LOG_ERROR, "Limit '%s' not specified", fl);
      return UDM_ERROR;
    }
    if (UDM_OK != (rc= UdmLoadSlowLimit(db, &fl_urls, q)))
      return rc;
    UdmLog(A, UDM_LOG_DEBUG, "Limit '%s' loaded%s, %d records",
           fl, fl_urls.exclude ? " type=excluding" : "", fl_urls.nurls);
  }
  
  UdmLog(A, UDM_LOG_DEBUG, "Loading URL list");
  
  udm_snprintf(qbuf, sizeof(qbuf),
              "SELECT %srec_id, site_id, pop_rank, last_mod_time FROM url%s%s%s",
              url, db->from, db->where[0] ? " WHERE " : "", db->where);
  if (UDM_OK != (rc = UdmSQLQuery(db, &SQLres, qbuf)))
    goto fin;
  
  C->nitems= UdmSQLNumRows(&SQLres);
  nbytes= C->nitems * sizeof(UDM_URLDATA);
  C->Item= (UDM_URLDATA*) UdmMalloc(nbytes);

  for (i= 0, j= 0; i < C->nitems; i++)
  {
    urlid_t url_id= UDM_ATOI(UdmSQLValue(&SQLres, i, 0));
    if (fl_urls.nurls)
    {
      void *found= UdmBSearch(&url_id, fl_urls.urls, fl_urls.nurls,
                              sizeof(urlid_t), (udm_qsort_cmp)cmpaurls);
      if (found && fl_urls.exclude)
        continue;
      if (!found && !fl_urls.exclude)
        continue;
    }
    C->Item[j].url_id= url_id;
    C->Item[j].coord= 0;
    C->Item[j].per_site= 0;
    C->Item[j].site_id= UDM_ATOI(UdmSQLValue(&SQLres, i, 1));
    C->Item[j].pop_rank= UDM_ATOF(UdmSQLValue(&SQLres, i, 2));
    C->Item[j].last_mod_time= UDM_ATOI(UdmSQLValue(&SQLres, i, 3));
    C->Item[j].url= NULL;
    C->Item[j].section= NULL;
    j++;
  }
  C->nitems= j;
  UdmSQLFree(&SQLres);

  if (C->nitems)
    UdmSort(C->Item, C->nitems, sizeof(UDM_URLDATA), (udm_qsort_cmp) cmp_data_urls);

fin:
  UdmLog(A, UDM_LOG_DEBUG, "URL list loaded: %d urls", C->nitems);
  UDM_FREE(fl_urls.urls);
  return rc;
}


/*
  Convert URLCRDList to SectionList.
  Only single word is supported.
*/
static int
UdmURLCRDListToSectionList(UDM_FINDWORD_ARGS *args,
                           UDM_SECTIONLIST *SectionList,
                           UDM_URLCRDLIST *CoordList)
{
  size_t ncoords= CoordList->ncoords;
  UDM_URL_CRD *CrdFrom, *CrdCurr;
  UDM_URL_CRD *CrdLast= CoordList->Coords + ncoords;
  UDM_COORD2 *Coord;
  UDM_SECTION *Section;
  
  UdmSectionListAlloc(SectionList, CoordList->ncoords, CoordList->ncoords);

#if 0
  UdmCoordListPrint(CoordList);
#endif
  
  if (!CoordList->ncoords)
    return UDM_OK;
  Coord= SectionList->Coord;
  Section= SectionList->Section;
  
  for (CrdFrom= CoordList->Coords; CrdFrom < CrdLast; CrdFrom= CrdCurr)
  {
    Section->Coord= Coord;
    Section->secno= CrdFrom->secno;
    Section->wordnum= CrdFrom->num;
    Section->order= args->WWList->Word[CrdFrom->num].order;
    for (CrdCurr= CrdFrom;
         CrdCurr < CrdLast &&
         CrdCurr->url_id == CrdFrom->url_id &&
         CrdCurr->secno == CrdFrom->secno ;
         CrdCurr++)
    {
      Coord->pos= CrdCurr->pos;
      Section->maxpos= CrdCurr->pos;
      Coord++;
    }
    Section->url_id= CrdFrom->url_id;
    Section->ncoords= CrdCurr - CrdFrom;
    Section->seclen= CrdFrom->seclen;
    Section->minpos= Section->Coord->pos;
    Section++;
  }
  SectionList->ncoords= Coord - SectionList->Coord;
  SectionList->nsections= Section - SectionList->Section;
  
#if 0
  UdmSectionListPrint(SectionList);
#endif
  
  return UDM_OK;
}


static int
UdmAddCollationMatch(UDM_FINDWORD_ARGS *args, const char *word, size_t count)
{
  /*
    If match is not full, then we don't know whether
    the word is a substring or a collation match.
    Let's assume it is a substring, to avoid long 
    word lists in $(WE).
  */
  if (args->word_match == UDM_MATCH_FULL)
  {
    UDM_WIDEWORD WW= args->WWList->Word[args->wordnum];
    WW.origin= UDM_WORD_ORIGIN_COLLATION;
    WW.count= count;
    UdmWideWordListAddLike(&args->CollationMatches, &WW, word);
  }
  return UDM_OK;
}


static int
UdmBlobAddCoords2(UDM_FINDWORD_ARGS *args,
                  UDM_SQLRES *SQLRes)
{
  size_t numrows= UdmSQLNumRows(SQLRes);
  size_t i;
  char *wf= args->wf;
  UDM_URL_TMP *urls= &args->urls;
  size_t nurls= args->urls.nurls;
  UDM_COORD2 C;
  int need_coords= (args->WWList->nwords > 1);
  int save_section_size= args->save_section_size;
  unsigned char wordnum= args->wordnum & 0xFF;
  unsigned char order= args->WWList->Word[wordnum].order;

  for (i= 0; i < numrows; i++)
  {
    const unsigned char *s, *e, *last_urlid_start;
    size_t length= UdmSQLLen(SQLRes, i, 1);
    unsigned char secno= UDM_ATOI(UdmSQLValue(SQLRes, i, 0));
    const char *cmatch= UdmSQLValue(SQLRes, i, 2);
    UDM_SECTIONLIST SectionList;
    UDM_COORD2 *Coord;
    UDM_SECTION *Section;
    size_t ncoords= 0;
    
    /*
      Shortest section is 6 bytes:
      - 4 bytes for URL id
      - 1 byte for "ncoords"
      - 1 byte for coord
    */

    UdmSectionListAlloc(&SectionList, length, length / 6);
    Coord= need_coords ? SectionList.Coord : NULL;
    Section= SectionList.Section;

    if (!wf[secno])
      continue;
    
    s= (const unsigned char *)UdmSQLValue(SQLRes, i, 1);
    
    /*
      A chunk consists of:
      - sizeof(urlid_t)
      - at least one byte for length
    */
    for (e= s + length, last_urlid_start= e - sizeof(urlid_t) - 1;
         s < last_urlid_start; )
    {
      int active= 1;
      size_t nrecs;

      Section->Coord= Coord;
      Section->secno= secno;
      Section->url_id= (urlid_t)udm_get_int4(s);
      Section->wordnum= wordnum;
      Section->order= order;
      s+= 4;

      if (nurls)
      {
        void *found= UdmBSearch(&Section->url_id, urls->urls, urls->nurls,
                                    sizeof(urlid_t), (udm_qsort_cmp)cmpaurls);
        if (found && urls->exclude)
          active= 0;
        if (!found && !urls->exclude)
          active= 0;
      }
      
      /* Get number of coords */
      if (*s < 128)
      {
        nrecs= *s++;
      }
      else
      {
        size_t nbytes= udm_coord_get(&nrecs, s, e);
        if (!nbytes) break;
        s+= nbytes;
      }
      
      if (!nrecs) /* extra safety */
        break;
      
      if (!active)
      {
        s= udm_coord_skip(s, e, nrecs);
        continue;
      }

      ncoords+= nrecs;
      
      if (save_section_size && nrecs > 1)
        ncoords--;
      
      /* Get first coord and put into S->minpos */
      if (*s < 128)
      {
        C.pos= *s++;
      }
      else
      {
        size_t crd;
        size_t nbytes= udm_coord_get(&crd, s, e);
        if (!nbytes) break;
        s+= nbytes;
        C.pos= crd;
      }
      
      Section->minpos= C.pos;
      
      /*
        If no coords anymore.
        Maybe the "section length" record didn't fit
        into "64K words per section" limit.
        Add section with seclen=0.
        QQ: shouldn't seclen be set to pos instead?
      */
      if (!--nrecs)
      {
        Section->seclen= 0;
        Section->ncoords= 1;
        Section->maxpos= C.pos;
        if (need_coords)
          *Coord++= C;
        Section++;
        continue;
      }
      
      if (!need_coords)
      {
        size_t crd, nbytes;
        if (save_section_size)
        {
          if (nrecs > 1)
          {
            s= udm_coord_sum(&crd, s, e, nrecs - 1); /* Sum middle coords */
            C.pos+= crd;
            Section->maxpos= C.pos;
          }
          else
            Section->maxpos= C.pos; /* One coord, minpos=maxpos */
          
          if (!(nbytes= udm_coord_get(&crd, s, e))) /* Get seclen */
            break;
          s+= nbytes;
          C.pos+= crd;
          Section->seclen= C.pos;
          Section->ncoords= nrecs;
        }
        else
        {
          s= udm_coord_sum(&crd, s, e, nrecs);      /* Sum middle coords  */
          C.pos+= crd;
          Section->maxpos= C.pos;
          Section->seclen= 0;
          Section->ncoords= nrecs + 1;
        }
        Section++;
        continue;
      }
      
      
      *Coord++= C; /* Add first coord */
      
      /* Get all other coords */
      if (s + nrecs * COORD_MAX_LEN < e)
      {
        /* Not the last chunk: unpack without range check */
        for ( ; nrecs > 0 ; nrecs--)
        {
          if (*s < 128)
          {
            C.pos+= *s++;
          }
          else
          {
            size_t crd, nbytes;
            nbytes= udm_coord_get_quick(&crd, s);
            if (!nbytes) break;
            s+= nbytes;
            C.pos+= crd;
          }
          *Coord++= C;
        }
      }
      else
      {
        /* Possibly last chunk: unpack with range check */
        for ( ; nrecs > 0 ; nrecs--)
        {
          if (s < e && *s < 128)
          {
            C.pos+= *s++;
          }
          else
          {
            size_t crd, nbytes;
            nbytes= udm_coord_get(&crd, s, e);
            if (!nbytes) break;
            s+= nbytes;
            C.pos+= crd;
          }
          *Coord++= C;
        }
      }
      
      
      /* Set section length */
      nrecs= Coord - Section->Coord;
      if (save_section_size)
      {
        /*
          We need to check whether Coord > Coord0 in the above
          condition: URL could be skipped because of limit
        */
        Section->seclen= ((--Coord)->pos);
        Section->ncoords= nrecs - 1;
        Section->maxpos= Coord[-1].pos;;
      }
      else
      {
        Section->seclen= 0;
        Section->ncoords= nrecs;
        Section->maxpos= C.pos;
      }
      Section++;
    }
    

    SectionList.ncoords= ncoords; /*Coord - SectionList.Coord;*/
    SectionList.nsections= Section - SectionList.Section;
    /*UdmSectionListPrint(&SectionList);*/
    if (SectionList.nsections && SectionList.ncoords)
    {
      UdmSectionListListAdd(&args->SectionListList, &SectionList);
      args->count+= SectionList.ncoords;
      UdmAddCollationMatch(args, cmatch, SectionList.ncoords);
    }
  }

  return UDM_OK;
}


static int
UdmApplyFastLimit(UDM_URLCRDLIST *Coord, UDM_URL_TMP *urls)
{
  UDM_URL_CRD *dst= Coord->Coords;
  UDM_URL_CRD *src= Coord->Coords;
  UDM_URL_CRD *srcend= Coord->Coords + Coord->ncoords;
  
  if (urls->exclude)
  {
    for ( ; src < srcend; src++)
    {
      if (!UdmBSearch(&src->url_id,
                      urls->urls, urls->nurls, sizeof(urlid_t),
                      (udm_qsort_cmp)cmpaurls))
      {
        *dst= *src;
        dst++;
      }
    }
  }
  else
  {
    for ( ; src < srcend; src++)
    {
      if (UdmBSearch(&src->url_id,
                     urls->urls, urls->nurls, sizeof(urlid_t),
                     (udm_qsort_cmp)cmpaurls))
      {
        *dst= *src;
        dst++;
      }
    }
  }
  Coord->ncoords= dst- Coord->Coords;
  return UDM_OK;
}


static int
UdmMultiAddCoordStr(UDM_FINDWORD_ARGS *args, UDM_URLCRDLIST *CoordList,
                    urlid_t url_id, size_t secno,
                    const unsigned char *intag, size_t lintag)
{
  const unsigned char *s, *e;
  size_t last= 0;
  size_t ncoords0= CoordList->ncoords;
  
  for (s= intag, e= intag + lintag; e > s; )
  {
    UDM_URL_CRD *Crd= &CoordList->Coords[CoordList->ncoords];
    size_t crd;
    size_t nbytes= udm_coord_get(&crd, s, e);
    if (!nbytes) break;
    last+= crd;
    Crd->url_id= url_id;
    Crd->pos= last;
    Crd->secno= secno;
    Crd->num= args->wordnum;
    Crd->seclen= 0;
    CoordList->ncoords++;
    s+= nbytes;
  }
  
  if (args->save_section_size)
  {
    uint4 seclen= CoordList->Coords[--CoordList->ncoords].pos;
    for ( ; ncoords0 < CoordList->ncoords; ncoords0++)
      CoordList->Coords[ncoords0].seclen= seclen;
  }
  return UDM_OK;
}


static int
UdmURLCRDListListAddWithSort2(UDM_FINDWORD_ARGS *args,
                              UDM_URLCRDLISTLIST *List,
                              UDM_URLCRDLIST *CoordList)
{
  UDM_SECTIONLIST SectionList;
  UdmURLCRDListToSectionList(args, &SectionList, CoordList);
  UdmSectionListListAdd(&args->SectionListList, &SectionList);
  UDM_FREE(CoordList->Coords);
  return UDM_OK;
}


static int
UdmMultiAddCoords(UDM_FINDWORD_ARGS *args, UDM_SQLRES *SQLRes)
{
  size_t i;
  size_t numrows;
  UDM_URLCRDLIST CoordList;

  bzero((void*)&CoordList, sizeof(CoordList));
  numrows= UdmSQLNumRows(SQLRes);

  for (i= 0; i < numrows; i++)
  {
    size_t tmp = UdmSQLLen(SQLRes, i, 2);
    if (tmp) CoordList.acoords += tmp;
    else CoordList.acoords += strlen(UdmSQLValue(SQLRes, i, 2));
  }
  CoordList.Coords= (UDM_URL_CRD*)UdmMalloc((CoordList.acoords) * sizeof(UDM_URL_CRD));

  for (i = 0; i < numrows; i++)
  {
    urlid_t url_id= UDM_ATOI(UdmSQLValue(SQLRes, i, 0));
    unsigned char secno= UDM_ATOI(UdmSQLValue(SQLRes, i, 1));
    size_t lintag= UdmSQLLen(SQLRes, i, 2);
    const unsigned char *intag= (const unsigned char *)UdmSQLValue(SQLRes, i, 2);
    uint4 weight= args->wf[secno];

    if (! weight) continue;
    
    /* FIXME: Check UdmSQLLen */
    if (! lintag) lintag = strlen((const char *)intag);
    UdmMultiAddCoordStr(args, &CoordList, url_id, secno, intag, lintag);
  }
  if (args->urls.nurls)
    UdmApplyFastLimit(&CoordList, &args->urls);
  if (CoordList.ncoords)
  {
    /*
      We have to sort here, because DBMode=multi
      returns data unsorted.
    */
    UdmSortSearchWordsByURL(CoordList.Coords, CoordList.ncoords);
    UdmURLCRDListListAddWithSort2(args, &args->CoordListList, &CoordList);
  }
  args->count= CoordList.ncoords;
  return UDM_OK;
}


static int UdmBlobGetTable (UDM_DB *db)
{
  UDM_SQLRES SQLRes;
  int rc;
  const char *val;

  return(1);

  rc = UdmSQLQuery(db, &SQLRes, "SELECT n FROM bdictsw");
  if (rc != UDM_OK) return(1);

  if (! UdmSQLNumRows(&SQLRes) || ! (val = UdmSQLValue(&SQLRes, 0, 0))) rc = 2;
  else if (*val != '1') rc = 3;
  else rc = 4;

  UdmSQLFree(&SQLRes);
  return(rc);
}

static const char *UdmBlobGetRTable (UDM_DB *db)
{
  if (db->DBType == UDM_DB_MYSQL)
    return "bdict";
  if (UdmBlobGetTable(db) == 3) return("bdict00");
  return("bdict");
}


static int UdmSQLDropTableIfExists(UDM_DB *db, const char *name)
{
  char qbuf[128];
  int rc;
  db->flags|= UDM_SQL_IGNORE_ERROR;
  udm_snprintf(qbuf, sizeof(qbuf), "DROP TABLE %s", name);
  rc= UdmSQLQuery(db, NULL, qbuf);
  db->flags^= UDM_SQL_IGNORE_ERROR;
  return rc;
}


static int UdmBlobGetWTable (UDM_DB *db, const char **name)
{
  int rc;
  *name= "bdict";
  if (db->DBType == UDM_DB_MYSQL)
  {
    if ((UDM_OK != (rc= UdmSQLQuery(db, NULL, "DROP TABLE IF EXISTS bdict_tmp"))) ||
        (UDM_OK != (rc= UdmSQLQuery(db, NULL, "CREATE TABLE bdict_tmp MAX_ROWS=300000000 AVG_ROW_LENGTH=512 SELECT * FROM bdict LIMIT 0"))) ||
        (UDM_OK != (rc= UdmSQLQuery(db, NULL, "ALTER TABLE bdict_tmp ADD KEY (word)"))))
      return rc;
    *name= "bdict_tmp";
  }
  else if (db->DBType == UDM_DB_DB2)
  {
    /* Note, DB2 'CREATE TABLE LIKE' does not copy indexes */
    if (UDM_OK != (rc= UdmSQLDropTableIfExists(db, "bdict_tmp")) ||
        UDM_OK != (rc= UdmSQLQuery(db, NULL, "CREATE TABLE bdict_tmp LIKE bdict")))
      return rc;
    *name= "bdict_tmp";
  }
  if (UdmBlobGetTable(db) == 4)
    *name= "bdict00";
  return UDM_OK;
}

static int UdmBlobSetTable (UDM_DB *db)
{
  char qbuf[64];
  int rc, t, n;

  if (db->DBType == UDM_DB_MYSQL)
  {
    if (UDM_OK == (rc= UdmSQLQuery(db, NULL, "DROP TABLE IF EXISTS bdict")))
      rc= UdmSQLQuery(db, NULL, "ALTER TABLE bdict_tmp RENAME bdict");
    return rc;
  }
  else if (db->DBType == UDM_DB_DB2)
  {
    if (UDM_OK != (rc= UdmSQLDropTableIfExists(db, "bdict")) ||
        UDM_OK != (rc= UdmSQLQuery(db, NULL, "RENAME TABLE bdict_tmp TO bdict")) ||
        UDM_OK != (rc= UdmSQLQuery(db, NULL, "CREATE INDEX bdict_word ON bdict (word)")))
      return rc;
    return UDM_OK;
  }
  
  t= UdmBlobGetTable(db);
  if (t == 1) return(UDM_OK);
  else if (t == 4) n = 0;
  else n = 1;

  rc = UdmSQLQuery(db, NULL, "DELETE FROM bdictsw");
  if (rc != UDM_OK) return(UDM_OK);
  udm_snprintf(qbuf, sizeof(qbuf), "INSERT INTO bdictsw VALUES(%d)", n);
  rc = UdmSQLQuery(db, NULL, qbuf);
  if (rc != UDM_OK) return(UDM_OK);
  return(UDM_OK);
}



static int UdmBlobWriteWord(UDM_DB *db, const char *table,
                            const char *word, size_t secno,
                            char *data, size_t len, UDM_DSTR *buf)
{
  const char *param= db->DBDriver == UDM_DB_ORACLE8 ? ":1" : "?";
  int rc;
  int use_bind= db->flags & UDM_SQL_HAVE_BIND;
  
  UdmDSTRReset(buf);
  
  if (use_bind)
  {
    char qbuf[128];
    udm_snprintf(qbuf, sizeof(qbuf), "INSERT INTO %s VALUES('%s', %d, %s)",
                 table, word, secno, param);
    if (UDM_OK != (rc= db->sql->SQLPrepare(db, qbuf)) ||
        UDM_OK != (rc= db->sql->SQLBind(db, 1, data, (int) len, UDM_SQLTYPE_LONGVARBINARY)) ||
        UDM_OK != (rc= db->sql->SQLExec(db)))
      return rc;
  }
  else
  {
    size_t escape_factor= db->DBType == UDM_DB_PGSQL ? 5 : 2;
    const char *pf= db->DBType == UDM_DB_PGSQL ? "'" : "0x";
    const char *sf= db->DBType == UDM_DB_PGSQL ? "'" : "";
    char *s;
    size_t nbytes= 100 + len * escape_factor + 1;
    
    if (db->flags & UDM_SQL_HAVE_STDHEX) /* X'AABBCC' syntax */
    {
      pf= "X'";
      sf= "'";
    }
    if (UdmDSTRAlloc(buf, nbytes))
    {
      fprintf(stderr, "BlobWriteWord: DSTRAlloc(%d) failed: word='%s' secno=%d len=%d",
              nbytes, word, secno, len);
      return UDM_OK; /* Skip this word - try to continue */
    }
    UdmDSTRAppendf(buf, "INSERT INTO %s VALUES('%s', %d, %s",
                   table, word, secno, pf);
    s= buf->data + buf->size_data;
    if (db->DBType == UDM_DB_PGSQL)
    {
      size_t slen= UdmSQLBinEscStr(db, s, (const char *) data, len);
      buf->size_data+= slen;
    }
    else
    {
      size_t i;
      for (i=0; i < len; i++)
      {
        unsigned int ch= (unsigned int) (unsigned char) data[i];
        *s++= udm_hex_digits[(ch >> 4) & 0x0F];
        *s++= udm_hex_digits[ch & 0x0F];
      }
      *s= '\0';
      buf->size_data+= len * 2;
    }
    UdmDSTRAppendf(buf, "%s)", sf);
    if (UDM_OK != (rc= UdmSQLQuery(db, NULL, buf->data)))
      return rc;
  }
  return UDM_OK;
}


static void UdmDSTRAppendINT4(UDM_DSTR *s, int i)
{
  char buf[4];
  udm_put_int4(i, buf);
  UdmDSTRAppend(s, buf, 4);
}


#ifdef HAVE_ZLIB
static size_t UdmInflate(char *dst, size_t dstlen,
                         char *src, size_t srclen)
{
  z_stream z;
  z.zalloc= Z_NULL;
  z.zfree= Z_NULL;
  z.opaque= Z_NULL;
  z.next_in= (Byte *) src;
  z.next_out= (Byte *) dst;
  z.avail_in= srclen;
  z.avail_out= dstlen;
  
  if (inflateInit2(&z, 15) != Z_OK)
    return 0;
  
  inflate(&z, Z_FINISH);
  inflateEnd(&z);
  return z.total_out;
}


static size_t UdmDeflate(char *dst, size_t dstlen, 
                         char *src, size_t srclen)
{
  z_stream z;
  z.zalloc= Z_NULL;
  z.zfree= Z_NULL;
  z.opaque= Z_NULL;
  z.next_in= (Byte*) src;
  z.next_out= (Byte*) dst;
  z.avail_in= srclen;
  z.avail_out= dstlen;
  
  if (deflateInit2(&z, 9, Z_DEFLATED, 15, 9, Z_DEFAULT_STRATEGY) != Z_OK)
    return 0;
  
  deflate(&z, Z_FINISH);
  deflateEnd(&z);
  return (size_t) z.total_out;
}
#endif

static int UdmBlobWriteWordCmpr(UDM_DB *db, const char *table,
                                const char *word, size_t secno,
                                char *data, size_t len,
                                UDM_DSTR *buf, UDM_DSTR *z,
                                int use_zint4)
{
#ifdef HAVE_ZLIB
  if (z && len > 256)
  {
    UdmDSTRReset(z);
    UdmDSTRRealloc(z, len + 8 + 1); /* 8 for two INTs */
    /* Append Format version */
    UdmDSTRAppendINT4(z, 0xFFFFFFFF);
    if (use_zint4)
    {
      UdmDSTRAppendINT4(z, 0x00000003);
      z->size_data+= UdmDeflate(z->data + z->size_data,
                                z->size_total - z->size_data, data + 8, len - 8);
    }
    else
    {
      UdmDSTRAppendINT4(z, 0x00000001);
      z->size_data+= UdmDeflate(z->data + z->size_data,
                                z->size_total - z->size_data, data, len);
    }
    if (z->size_data < len)
    {
      data= z->data;
      len= z->size_data;
    }
  }
#endif
  return UdmBlobWriteWord(db, table, word, secno, data, len, buf);
}


static int UdmBlobCacheWrite (UDM_DB *db, UDM_BLOB_CACHE *cache, const char *table, int use_deflate)
{
  size_t w, w1;
  int rc= UDM_OK;
  unsigned char ubuf[COORD_MAX_LEN];
  unsigned char *ubufend= ubuf + sizeof(ubuf);
  size_t nbytes;
  UDM_DSTR buf, qbuf, zbuf;
  UDM_BLOB_CACHE_WORD *word;
  UDM_BLOB_CACHE_WORD *word1;
  char utmp[4];

  if (!cache->nwords)
    return(UDM_OK);

  UdmDSTRInit(&buf, 8192);
  UdmDSTRInit(&qbuf, 8192);
  UdmDSTRInit(&zbuf, 8192);
  
  for (w= 0; w < cache->nwords; w++)
  {
    word = &cache->words[w];
    for (w1= w; w1 < cache->nwords; w1++)
    {
      word1= &cache->words[w1];
      if (word->secno != word1->secno || strcmp(word->word, word1->word)) break;
      udm_put_int4(word1->url_id, utmp);
      nbytes= udm_coord_put(word1->nintags, ubuf, ubufend);
      if (!nbytes)
        continue;
      if (!UdmDSTRAppend(&buf, utmp, sizeof(urlid_t)) ||
          !UdmDSTRAppend(&buf, (char *)ubuf, nbytes) ||
          !UdmDSTRAppend(&buf, word1->intags, word1->ntaglen))
      {
        fprintf(stderr, "BlobCacheWrite: DSTRAppend() failed: word='%s' secno=%d len=%d",
                word1->word, word1->secno, word1->ntaglen);
      }
    }
    
    if (UDM_OK != (rc= UdmBlobWriteWordCmpr(db, table,
                                        word->word, word->secno,
                                        buf.data, buf.size_data, &qbuf,
                                        use_deflate ? &zbuf : NULL, 0)))
      goto end;

    UdmDSTRReset(&buf); 
    w= w1 - 1;
  }

end:
  UdmDSTRFree(&zbuf);
  UdmDSTRFree(&qbuf);
  UdmDSTRFree(&buf);
  return rc;
}


static int
UdmBlobWriteTimestamp(UDM_AGENT *A, UDM_DB *db, const char *table)
{
  UDM_DSTR buf;
  int rc= UDM_OK;
  size_t size_data;
  char lname[64]= "#ts";
  char vname[64]= "#version";
  char data[64];
  char qbuf[64];

  UdmLog(A, UDM_LOG_DEBUG, "Writting '%s'", lname);
  UdmDSTRInit(&buf, 128);
  size_data= udm_snprintf(data, sizeof(data), "%d", (int) time(0));
  udm_snprintf(qbuf, sizeof(qbuf),
              "DELETE FROM %s WHERE word='%s'", table, lname);
  if (UDM_OK != (rc= UdmSQLQuery(db, NULL, qbuf)) ||
      UDM_OK != (rc= UdmBlobWriteWord(db, table, lname, 0,
                                       data, size_data, &buf)))
    goto ex;

  size_data= udm_snprintf(data, sizeof(data), "%d", UDM_VERSION_ID);
  udm_snprintf(qbuf, sizeof(qbuf),
               "DELETE FROM %s WHERE word='%s'", table, vname);
  if (UDM_OK != (rc= UdmSQLQuery(db, NULL, qbuf)) ||
      UDM_OK != (rc= UdmBlobWriteWord(db, table, vname, 0,
                                      data, size_data, &buf)))
    goto ex;
ex:
  UdmDSTRFree(&buf);
  return rc;
}


static int
UdmBlobReadTimestamp(UDM_AGENT *A, UDM_DB *db, int *ts, int def)
{
  int rc;
  char lname[]= "#ts";
  char qbuf[64];
  UDM_SQLRES SQLRes;

  udm_snprintf(qbuf, sizeof(qbuf), "SELECT intag FROM bdict WHERE word='%s'", lname);
  if (UDM_OK == (rc= UdmSQLQuery(db, &SQLRes, qbuf)) &&
      UdmSQLNumRows(&SQLRes) > 0)
    *ts= atoi(UdmSQLValue(&SQLRes, 0,0));
  else
    *ts= def;
  UdmSQLFree(&SQLRes);
  return rc;
}


typedef struct udm_url_int4_st
{
  urlid_t url_id;
  int4  param;
} UDM_URL_INT4;

typedef struct udm_url_int4_list_st
{
  size_t nitems;
  UDM_URL_INT4 *Item;
} UDM_URL_INT4_LIST;


static int
UdmLoadUserScoreList(UDM_DB *db, UDM_URL_INT4_LIST *List, const char *q)
{
  size_t i;
  int rc;
  UDM_SQLRES SQLRes;
  bzero((void*) List, sizeof(UDM_URL_INT4_LIST));

  if (UDM_OK != (rc= UdmSQLQuery(db, &SQLRes, q)))
    goto ret;

  if (!(List->nitems= UdmSQLNumRows(&SQLRes)))
    goto sqlfree;

  if (2 != UdmSQLNumCols(&SQLRes))
    goto sqlfree;

  if (!(List->Item= UdmMalloc(sizeof(UDM_URL_INT4) * List->nitems)))
  {
    rc= UDM_ERROR;
    List->nitems= 0;
    goto sqlfree;
  }
  for (i= 0; i < List->nitems; i++)
  {
    List->Item[i].url_id= atoi(UdmSQLValue(&SQLRes, i, 0));
    List->Item[i].param= atoi(UdmSQLValue(&SQLRes, i, 1));
  }
  UdmSort(List->Item, List->nitems, sizeof(UDM_URL_INT4_LIST), (udm_qsort_cmp)cmpaurls);

sqlfree:
  UdmSQLFree(&SQLRes);
ret:
  return rc;
}


/*
  Write limits, but don't COMMIT and don't write timestamp
*/
static int
UdmBlobWriteLimitsInternal(UDM_AGENT *A, UDM_DB *db,
                           const char *table, int use_deflate)
{
  UDM_VARLIST *Vars= &A->Conf->Vars;
  UDM_VAR *Var;
  UDM_DSTR l, buf;
  int rc= UDM_OK;
  
  UdmDSTRInit(&l, 8192);
  UdmDSTRInit(&buf, 8192);
  for (Var= Vars->Var; Var < Vars->Var + Vars->nvars; Var++)
  {
    size_t i;
    char qbuf[128];
    char lname[64];
    UDM_URL_TMP list;
    
    if (strncmp(Var->name, "Limit.", 6))
      continue;
    udm_snprintf(lname, sizeof(lname), "#limit#%s", Var->name + 6);
    UdmLog(A, UDM_LOG_DEBUG, "Writting '%s'", lname);

    if (UDM_OK != (rc= UdmLoadSlowLimit(db, &list, Var->val)))
      goto ret;
    
    UdmDSTRReset(&buf);
    UdmDSTRReset(&l);
    for (i= 0; i < list.nurls; i++)
    {
      UdmDSTRAppendINT4(&l, list.urls[i]);
    }

    udm_snprintf(qbuf, sizeof(qbuf), "DELETE FROM %s WHERE word=('%s')", table, lname);
    if(UDM_OK != (rc= UdmSQLQuery(db, NULL, qbuf)))
      goto ret;

    if (l.size_data)
    {
      if (UDM_OK != (rc= UdmBlobWriteWordCmpr(db, table, lname, 0, l.data,
                                              l.size_data, &buf, NULL, 0)))
        goto ret;
    }

    UDM_FREE(list.urls);
    UdmLog(A, UDM_LOG_DEBUG, "%d documents written to '%s'", list.nurls, lname);
  }
ret:
  UdmDSTRFree(&l);
  UdmDSTRFree(&buf);
  return rc;
}


/*
  Write limits with COMMIT and timestamp
*/
int
UdmBlobWriteLimits(UDM_AGENT *A, UDM_DB *db, const char *table, int use_deflate)
{
  int rc;
  if (UDM_OK != (rc= UdmSQLBegin(db)) ||
      UDM_OK != (rc= UdmBlobWriteLimitsInternal(A, db, table, use_deflate)) ||
      UDM_OK != (rc= UdmBlobWriteTimestamp(A, db, table)) ||
      UDM_OK != (rc= UdmSQLCommit(db)))
    return rc;
  return UDM_OK;
}

#ifdef WIN32
#define UDM_DEFAULT_ZINT4   1
#define UDM_DEFAULT_DEFLATE 1
#else
#define UDM_DEFAULT_ZINT4   0
#define UDM_DEFAULT_DEFLATE 0
#endif

int UdmBlobWriteURL(UDM_AGENT *A, UDM_DB *db, const char *table, int use_deflate)
{
  int use_zint4= UdmVarListFindBool(&db->Vars, "zint4", UDM_DEFAULT_ZINT4);
  UDM_DSTR buf;
  UDM_DSTR r, s, l, p, z, *pz= use_deflate ? &z : NULL;
  UDM_SQLRES SQLRes;
  int rc= UDM_OK;
  UDM_PSTR row[4];
  int odbcbind= db->DBDriver == UDM_DB_ODBC &&
                db->DBType != UDM_DB_MSSQL  &&
                db->DBType != UDM_DB_SYBASE;

  UdmDSTRInit(&buf, 8192);
  UdmDSTRInit(&r, 8192);
  UdmDSTRInit(&s, 8192);
  UdmDSTRInit(&l, 8192);
  UdmDSTRInit(&p, 8192);
  UdmDSTRInit(&z, 8192);

  rc= UdmSQLExecDirect(db, &SQLRes, "SELECT rec_id, site_id, last_mod_time, pop_rank FROM url ORDER BY rec_id");
  if (rc != UDM_OK) goto ex;

  while (db->sql->SQLFetchRow(db, &SQLRes, row) == UDM_OK)
  {
    double pop_rank= UDM_ATOF(row[3].val);
    UdmDSTRAppendINT4(&r, UDM_ATOI(row[0].val));
    UdmDSTRAppendINT4(&s, UDM_ATOI(row[1].val));
    UdmDSTRAppendINT4(&l, UDM_ATOI(row[2].val));
    /* FIXME: little and big endian incompatibility: */
    UdmDSTRAppend(&p, (char *)&pop_rank, sizeof(pop_rank));
  }
  UdmSQLFree(&SQLRes);

  if (use_zint4)
  {
    size_t i, nrec_ids= r.size_data / 4;
    urlid_t *rec_id= (urlid_t *)r.data;
    char *zint4_buf= UdmMalloc(nrec_ids * 5 + 5);
    UDM_ZINT4_STATE zint4_state;
    if (! zint4_buf)
    {
      rc= UDM_ERROR;
      goto ex;
    }
    udm_zint4_init(&zint4_state, zint4_buf);
    for (i= 0; i < nrec_ids; i++)
      udm_zint4(&zint4_state, rec_id[i]);
    udm_zint4_finalize(&zint4_state);
    UdmDSTRReset(&r);
    UdmDSTRAppendINT4(&r, 0xFFFFFFFF);
    UdmDSTRAppendINT4(&r, 0x00000002);
    UdmDSTRAppend(&r, (char *)zint4_state.buf,
                  zint4_state.end - zint4_state.buf);
  }

  if (odbcbind)
  {
    /* TODO: check why there is no SQLBegin for Oracle */
    if (UDM_OK != (rc= UdmSQLBegin(db)))
      goto ex;
  }

  if (pz)
    UdmDSTRRealloc(pz, p.size_data + 8 + 1);

  UdmDSTRAppendf(&buf, "DELETE FROM %s WHERE word IN ('#rec_id', '#site_id', '#last_mod_time', '#pop_rank')", table);
  UdmSQLQuery(db, NULL, buf.data);
  UdmDSTRReset(&buf);

  if (UDM_OK != (rc= UdmBlobWriteWordCmpr(db, table, "#rec_id", 0, r.data,
                                          r.size_data, &buf, pz, use_zint4)) ||
      UDM_OK != (rc= UdmBlobWriteWordCmpr(db, table, "#site_id", 0, s.data,
                                          s.size_data, &buf, pz, 0)) ||
      UDM_OK != (rc= UdmBlobWriteWordCmpr(db, table, "#last_mod_time", 0,
                                          l.data, l.size_data, &buf, pz, 0)) ||
      UDM_OK != (rc= UdmBlobWriteWordCmpr(db, table, "#pop_rank", 0, p.data,
                                          p.size_data, &buf, pz, 0)))
    goto ex;
  
  if (odbcbind || db->DBDriver == UDM_DB_ORACLE8)
  {
    if(UDM_OK != (rc= UdmSQLCommit(db)))
      goto ex;
  }

  if (UDM_OK != (rc= UdmBlobWriteLimitsInternal(A, db, table, use_deflate)) ||
      UDM_OK != (rc= UdmBlobWriteTimestamp(A, db, table)))
    goto ex;

  if (odbcbind || db->DBDriver == UDM_DB_ORACLE8)
  {
    if(UDM_OK != (rc= UdmSQLCommit(db)))
      goto ex;
  }
  
ex:
  UdmDSTRFree(&buf);
  UdmDSTRFree(&r);
  UdmDSTRFree(&s);
  UdmDSTRFree(&l);
  UdmDSTRFree(&p);
  UdmDSTRFree(&z);
  return rc;
}

#define BLOB_CACHE_SIZE 0xff

static int
UdmBlob2BlobSQL(UDM_AGENT *Indexer, UDM_DB *db, UDM_URLDATALIST *URLList)
{
  size_t t, i, use_deflate= 0, srows= 0, trows;
  udm_uint8 nbytes= 0;
  int rc;
  char buf[128];
  UDM_BLOB_CACHE cache[BLOB_CACHE_SIZE + 1];
  const char *wtable;
  int tr= db->DBType != UDM_DB_MYSQL ? 1 : 0;
  int tr_truncate= tr && (db->DBType != UDM_DB_SYBASE);
#ifdef HAVE_ZLIB
  if (UdmVarListFindBool(&db->Vars, "deflate", UDM_DEFAULT_DEFLATE))
  {
    UdmLog(Indexer, UDM_LOG_DEBUG, "Using deflate");
    use_deflate= 1;
  }
  else
    UdmLog(Indexer, UDM_LOG_DEBUG, "Not using deflate");
#endif
  /* Get table to write to */
  if (UDM_OK != (rc= UdmBlobGetWTable(db, &wtable)))
    return rc;
  /* Lock tables for MySQL */
  if (db->DBType == UDM_DB_MYSQL)
  {
    if (db->version >= 40000)
    {
      sprintf(buf, "ALTER TABLE %s DISABLE KEYS", wtable);
      if (UDM_OK != UdmSQLQuery(db, NULL, buf))
        goto ret;
    }
    udm_snprintf(buf, sizeof(buf), "LOCK TABLES bdicti READ, %s WRITE", wtable);
    if (UDM_OK != (rc= UdmSQLQuery(db, NULL, buf)))
      return rc;
  }
  /* Initialize blob cache */
  for (i= 0; i <= BLOB_CACHE_SIZE; i++)
    UdmBlobCacheInit(&cache[i]);
  /* Delete old words from bdict */
  if ((tr_truncate && UDM_OK != (rc= UdmSQLBegin(db))) ||
      UDM_OK != (rc= UdmSQLTableTruncateOrDelete(db, wtable)) ||
      (tr_truncate && UDM_OK != (rc= UdmSQLCommit(db))))
    goto ret;
  for (t= 0; t <= 0x1f; t++)
  {
    size_t rownum, nrows;
    UDM_PSTR row[2];
    UDM_SQLRES SQLRes;
    UdmLog(Indexer, UDM_LOG_DEBUG, "Loading intag%02X", t);
    udm_snprintf(buf, sizeof(buf), "SELECT url_id,intag%02X FROM bdicti WHERE state>0", t);
    if (UDM_OK != (rc= UdmSQLQuery(db, &SQLRes, buf)))
      goto ret;
    nrows= UdmSQLNumRows(&SQLRes);
    UdmLog(Indexer, UDM_LOG_ERROR, "Converting intag%02X", t);
    for (rownum= 0; rownum < nrows; rownum ++)
    {
      urlid_t url_id;
      size_t pos= 0;
      row[0].val= UdmSQLValue(&SQLRes, rownum, 0);
      row[0].len= UdmSQLLen(&SQLRes, rownum, 0);
      row[1].val= UdmSQLValue(&SQLRes, rownum, 1);
      row[1].len= UdmSQLLen(&SQLRes, rownum, 1);
      url_id= UDM_ATOI(row[0].val);
      
      if (!UdmURLSearchForConv(URLList, url_id))
        continue;
      
      while (pos < row[1].len)
      {
        char *word= &row[1].val[pos];
        udmhash32_t word_seed;
        while (pos < row[1].len && row[1].val[pos])
          pos++;
        if (++pos >= row[1].len)
          break;
        word_seed= UdmStrHash32(word) >> 8 & BLOB_CACHE_SIZE;
        while (pos < row[1].len)
        {
          unsigned char secno= (unsigned char)row[1].val[pos];
          char *intag= &row[1].val[++pos];
          size_t nintags, intaglen;
          while (pos < row[1].len && row[1].val[pos])
            pos++;
          nintags= udm_coord_len(intag);
          intaglen= row[1].val + pos - intag;
          nbytes+= intaglen;
          UdmBlobCacheAdd2(&cache[word_seed],
                           url_id, secno, word, nintags, intag, intaglen);
          if (++pos >= row[1].len || ! row[1].val[pos])
          {
            pos++;
            break;
          }
        }
      }
    }
    UdmLog(Indexer, UDM_LOG_DEBUG, "Writting intag%02X", t);
    if ((tr && UDM_OK != (rc= UdmSQLBegin(db))))
      goto ret;
    for (trows= 0, i= 0; i <= BLOB_CACHE_SIZE; i++)
    {
      if (cache[i].nwords)
      {
        srows+= cache[i].nwords;
        trows+= cache[i].nwords;
        UdmBlobCacheSort(&cache[i]);
        if (UDM_OK != (rc= UdmBlobCacheWrite(db, &cache[i], wtable, use_deflate)))
           goto ret;
      }
      UdmBlobCacheFree(&cache[i]);
      if (tr && trows > 16*1024)
      {
        if (UDM_OK != (rc= UdmSQLCommit(db)) ||
            UDM_OK != (rc= UdmSQLBegin(db)))
          goto ret;
        trows= 0;
      }
    }
    if (tr && UDM_OK != (rc= UdmSQLCommit(db)))
      goto ret;

    UdmSQLFree(&SQLRes);
  }
  UdmLog(Indexer, UDM_LOG_ERROR, "Total converted: %d records, %llu bytes", srows, nbytes);

  if (db->DBType == UDM_DB_MYSQL)
  {
    if (UDM_OK != (rc= UdmSQLQuery(db, NULL, "UNLOCK TABLES")))
      return rc;
    udm_snprintf(buf, sizeof(buf), "LOCK TABLES bdicti WRITE, %s WRITE", wtable);
    if (UDM_OK != (rc= UdmSQLQuery(db, NULL, buf)))
      return rc;
  }
  
  /* Put timestamp */
  if (UDM_OK != (rc= UdmBlobWriteTimestamp(Indexer, db, wtable)))
    goto ret;
  /* Clean bdicti */
  if ((tr && UDM_OK != (rc= UdmSQLBegin(db))) ||
      UDM_OK != (rc= UdmSQLQuery(db, NULL, "DELETE FROM bdicti WHERE state=0")) ||
      UDM_OK != (rc= UdmSQLQuery(db, NULL, "UPDATE bdicti SET state=2")) ||
      (tr && UDM_OK != (rc= UdmSQLCommit(db))))
    goto ret;
  if (db->DBType == UDM_DB_MYSQL)
  {
    UdmSQLQuery(db, NULL, "UNLOCK TABLES");
    if (db->version >= 40000)
    {
      sprintf(buf, "ALTER TABLE %s ENABLE KEYS", wtable);
      UdmSQLQuery(db, NULL, buf);
    }
  }
  for (i= 0; i <= BLOB_CACHE_SIZE; i++)
    UdmBlobCacheFree(&cache[i]);
  /* Convert URL */
  UdmLog(Indexer, UDM_LOG_ERROR, "Converting url data");
  if ((tr && UDM_OK != (rc= UdmSQLBegin(db))) ||
      UDM_OK != (rc= UdmBlobWriteURL(Indexer, db, wtable, use_deflate)) ||
      (tr && UDM_OK != (rc= UdmSQLCommit(db))))
    return rc;
  /* Switch to new blob table */
  UdmLog(Indexer, UDM_LOG_ERROR, "Switching to new blob table.");
  rc= UdmBlobSetTable(db);
  return rc;
ret:
  for (i= 0; i <= BLOB_CACHE_SIZE; i++)
    UdmBlobCacheFree(&cache[i]);
  if (db->DBType == UDM_DB_MYSQL)
    UdmSQLQuery(db, NULL, "UNLOCK TABLES");
  return rc;
}


static int
UdmMulti2BlobSQL(UDM_AGENT *Indexer, UDM_DB *db, UDM_URLDATALIST *URLList)
{
  size_t t, i, use_deflate= 0;
  int rc;
  UDM_SQLRES SQLRes;
  char buf[128];
  size_t srows = 0;
  UDM_BLOB_CACHE cache[BLOB_CACHE_SIZE + 1];
  urlid_t url_id;
  unsigned char secno;
  char *intag;
  const char *word;
  size_t nintags;
  udmhash32_t word_seed;
  UDM_PSTR row[4];
  const char *wtable;

#ifdef HAVE_ZLIB
  if (UdmVarListFindBool(&db->Vars, "deflate", UDM_DEFAULT_DEFLATE))
  {
     UdmLog(Indexer, UDM_LOG_DEBUG, "Using deflate");
     use_deflate= 1;
  }
  else
  {
     UdmLog(Indexer, UDM_LOG_DEBUG, "Not using deflate");
  }
#endif
  
  
  if (UDM_OK != (rc= UdmBlobGetWTable(db, &wtable)))
    return rc;

  /* Delete old words from bdict */
  rc = UdmSQLTableTruncateOrDelete(db, wtable);
  if (rc != UDM_OK)
  {
    return(rc);
  }

  for (i = 0; i <= BLOB_CACHE_SIZE; i++)
  {
    UdmBlobCacheInit(&cache[i]);
  }

  for (t = 0; t <= MULTI_DICTS; t++)
  {
    if (db->DBType == UDM_DB_MYSQL)
    {
      udm_snprintf(buf, sizeof(buf), "LOCK TABLES dict%02X WRITE, %s WRITE", t, wtable);
      rc = UdmSQLQuery(db, NULL, buf);
      if (rc != UDM_OK)
      {
        return(rc);
      }
    }

    UdmLog(Indexer, UDM_LOG_DEBUG, "Loading dict%02X", t);
    udm_snprintf(buf, sizeof(buf), "SELECT url_id, secno, word, intag FROM dict%02X", t);
    rc= UdmSQLExecDirect(db, &SQLRes, buf);
    if (rc != UDM_OK)
    {
      return(rc);
    }

    UdmLog(Indexer, UDM_LOG_ERROR, "Converting dict%02X", t);
    while (db->sql->SQLFetchRow(db, &SQLRes, row) == UDM_OK)
    {
      url_id = UDM_ATOI(row[0].val);
      if (!UdmURLSearchForConv(URLList, url_id))
        continue;
      secno = (unsigned char)UDM_ATOI(row[1].val);
      word = row[2].val;
      intag = row[3].val;
      nintags = udm_coord_len(intag);
      word_seed = UdmStrHash32(word ? word : "") >> 8 & MULTI_DICTS;
      UdmBlobCacheAdd(&cache[word_seed],
                      url_id, secno, word, nintags, intag, row[3].len);
    }
    UdmLog(Indexer, UDM_LOG_DEBUG, "Writting dict%02X", t);
    for (i = 0; i <= BLOB_CACHE_SIZE; i++)
    {
      srows += cache[i].nwords;
      UdmBlobCacheSort(&cache[i]);
      rc= UdmBlobCacheWrite(db, &cache[i], wtable, use_deflate);
      UdmBlobCacheFree(&cache[i]);
      if (rc != UDM_OK)
        return rc;
    }
    UdmSQLFree(&SQLRes);

    if (db->DBType == UDM_DB_MYSQL)
    {
      rc = UdmSQLQuery(db, NULL, "UNLOCK TABLES");
      if (rc != UDM_OK)
      {
        return(rc);
      }
    }
  }

  UdmLog(Indexer, UDM_LOG_ERROR, "Total records converted: %d", srows);
  if (UDM_OK != (rc= UdmBlobWriteTimestamp(Indexer, db, wtable)))
    return rc;

  UdmLog(Indexer, UDM_LOG_ERROR, "Converting url.");
  if (UDM_OK != (rc= UdmBlobWriteURL(Indexer, db, wtable, use_deflate)))
    return rc;

  UdmLog(Indexer, UDM_LOG_ERROR, "Switching to new blob table.");
  rc= UdmBlobSetTable(db);
  return rc;
}


static int
UdmSingle2BlobSQL(UDM_AGENT *Indexer, UDM_DB *db, UDM_URLDATALIST *URLList)
{
  int rc;
  char buf[128];
  UDM_PSTR row[3];
  UDM_SQLRES SQLRes;
  size_t t, u, s, w;
  UDM_BLOB_CACHE bcache;
  UDM_MULTI_CACHE mcache;
  urlid_t url_id;
  UDM_WORD words;
  const char *wtable;
  
  if (UDM_OK != (rc= UdmBlobGetWTable(db, &wtable)))
    return rc;

  /* Delete old words from bdict */
  rc= UdmSQLTableTruncateOrDelete(db, wtable);
  if (rc != UDM_OK)
  {
    return(rc);
  }

  if (db->DBType == UDM_DB_MYSQL)
  {
    udm_snprintf(buf, sizeof(buf), "LOCK TABLES dict WRITE, %s WRITE", wtable);
    rc = UdmSQLQuery(db, NULL, buf);
    if (rc != UDM_OK)
    {
      return(rc);
    }
  }

  udm_snprintf(buf, sizeof(buf), "SELECT url_id, word, intag FROM dict");
  rc= UdmSQLExecDirect(db, &SQLRes, buf);
  if (rc != UDM_OK)
  {
    return(rc);
  }

  UdmMultiCacheInit(&mcache);
  while (db->sql->SQLFetchRow(db, &SQLRes, row) == UDM_OK)
  {
    int coord;
    url_id = UDM_ATOI(row[0].val);
    words.word = row[1].val;
    coord= UDM_ATOI(row[2].val);
    words.secno= UDM_WRDSEC(coord);
    words.pos= UDM_WRDPOS(coord);
    words.hash= 0;
    UdmMultiCacheAdd(&mcache, url_id, 0, &words);
  }
  UdmSQLFree(&SQLRes);
  UdmBlobCacheInit(&bcache);
  for (t = 0; t <= MULTI_DICTS; t++)
  {
    UDM_MULTI_CACHE_TABLE *table = &mcache.tables[t];
    for (u = 0; u < table->nurls; u++)
    {
      UDM_MULTI_CACHE_URL *url = &table->urls[u];
      if (!UdmURLSearchForConv(URLList, url->url_id))
        continue;
      for (s = 0; s < url->nsections; s++)
      {
        UDM_MULTI_CACHE_SECTION *section = &url->sections[s];
        for (w = 0; w < section->nwords; w++)
	{
          UDM_MULTI_CACHE_WORD *word= &section->words[w];
          char *intag= UdmMultiCachePutIntag1(word);
          UdmBlobCacheAdd(&bcache, url->url_id, section->secno, word->word,
                          word->nintags, intag, strlen(intag));
          UDM_FREE(intag);
	}
      }
    }
  }
  UdmBlobCacheSort(&bcache);
  rc= UdmBlobCacheWrite(db, &bcache, wtable, 0);
  UdmBlobCacheFree(&bcache);
  UdmMultiCacheFree(&mcache);

  if (rc != UDM_OK)
  {
    return(rc);
  }

  if (db->DBType == UDM_DB_MYSQL)
  {
    rc = UdmSQLQuery(db, NULL, "UNLOCK TABLES");
    if (rc != UDM_OK)
    {
      return(rc);
    }
  }

  if (UDM_OK != (rc= UdmBlobWriteTimestamp(Indexer, db, wtable)))
    return rc;
    
  UdmLog(Indexer, UDM_LOG_ERROR, "Converting url.");
  if (UDM_OK != (rc= UdmBlobWriteURL(Indexer, db, wtable, 0)))
    return rc;
  
  UdmLog(Indexer, UDM_LOG_ERROR, "Switching to new blob table.");
  UdmBlobSetTable(db);
  return(UDM_OK);
}


int UdmConvert2BlobSQL(UDM_AGENT *Indexer, UDM_DB *db)
{
  int rc;
  UDM_URLDATALIST List;
  
  BuildWhere(Indexer->Conf, db);
  
  if (UDM_OK != (rc= UdmLoadURLDataFromURLForConv(Indexer, db, &List)))
    return rc;
  
  if (db->DBMode == UDM_DBMODE_MULTI)
    rc= UdmMulti2BlobSQL(Indexer, db, &List);
  else if (db->DBMode == UDM_DBMODE_SINGLE)
    rc= UdmSingle2BlobSQL(Indexer, db, &List);
  else if (db->DBMode == UDM_DBMODE_BLOB)
    rc= UdmBlob2BlobSQL(Indexer, db, &List);
  
  UdmFree(List.Item);
  return rc;
}


static int StoreWordsSingle(UDM_AGENT * Indexer,UDM_DOCUMENT * Doc,UDM_DB *db)
{
  size_t  i;
  char  qbuf[256]="";
  time_t  stmp;
  int  rc=UDM_OK;
  urlid_t  url_id = UdmVarListFindInt(&Doc->Sections, "ID", 0);
  const char      *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";

  if (UdmVarListFindInt(&Indexer->Conf->Vars, "SaveSectionSize", 1))
  {
    if (UDM_OK!= (rc= UdmWordListSaveSectionSize(Doc)))
      return rc;
  }
  
  stmp=time(NULL);
  
  /* Start transaction if supported */
  /* This is to make stuff faster   */
  
  if(db->DBType != UDM_DB_MYSQL)
    if(UDM_OK!=(rc=UdmSQLBegin(db)))
      return rc;
  
  /* Delete old words */
  sprintf(qbuf,"DELETE FROM dict WHERE url_id=%s%i%s", qu, url_id, qu);

  if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,qbuf)))
    goto unlock_StoreWordsSingle;
  
  /* Insert new words */
  if(db->DBType==UDM_DB_MYSQL)
  {
    if(Doc->Words.nwords)
    {
      size_t nstored=0;
      
      while(nstored < Doc->Words.nwords)
      {
        char * qb,*qe;
        size_t step=1024;
        size_t mlen=1024;
        size_t rstored = 0;

        qb=(char*)UdmMalloc(mlen);
        strcpy(qb,"INSERT INTO dict (word,url_id,intag) VALUES ");
        qe=qb+strlen(qb);

        for(i=nstored;i<Doc->Words.nwords;i++)
        {
          size_t len=qe-qb;
          if(!Doc->Words.Word[i].secno)
          { 
            nstored++; 
            continue;
          }
          rstored++;
        
          /* UDM_MAXWORDSIZE+100 should be enough */
          if((len + Indexer->Conf->WordParam.max_word_len + 100) >= mlen)
          {
            mlen+=step;
            qb=(char*)UdmRealloc(qb,mlen);
            qe=qb+len;
          }
          
          if(i>nstored)*qe++=',';

          if(db->DBMode==UDM_DBMODE_SINGLE)
          {
            *qe++='(';
            *qe++='\'';
            strcpy(qe,Doc->Words.Word[i].word);
            while(*qe)qe++;
            *qe++='\'';
            *qe++=',';
            qe+=sprintf(qe,"%d,%d",url_id,
                        UDM_WRDCOORD(Doc->Words.Word[i].pos,
                                     Doc->Words.Word[i].secno) /*+ 
                                     Doc->Words.Word[i].seclen_marker*/);
            *qe++=')';
            *qe='\0';
          }
          if(qe>qb+UDM_MAX_MULTI_INSERT_QSIZE)
            break;
        }
        nstored = i + 1;
        rc = (rstored > 0) ? UdmSQLQuery(db, NULL, qb) : UDM_OK;
        UDM_FREE(qb);
        if(rc!=UDM_OK) goto unlock_StoreWordsSingle;
      }
    }
  }else{
    for(i=0;i<Doc->Words.nwords;i++)
    {
      if(!Doc->Words.Word[i].secno)continue;
        
      if(db->DBMode==UDM_DBMODE_SINGLE)
      {
        sprintf(qbuf,"INSERT INTO dict (url_id,word,intag) VALUES(%s%i%s,'%s',%d)", qu, url_id, qu, 
          Doc->Words.Word[i].word,
          UDM_WRDCOORD(Doc->Words.Word[i].pos, Doc->Words.Word[i].secno) /*+
          Doc->Words.Word[i].seclen_marker*/);
      }
      if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,qbuf)))
        goto unlock_StoreWordsSingle;
    }
  }
unlock_StoreWordsSingle:
  if(db->DBType != UDM_DB_MYSQL)
    if(UDM_OK!=(rc=UdmSQLCommit(db)))
      return rc;
  return(UDM_OK);
}




static int UdmStoreWords(UDM_AGENT * Indexer,UDM_DOCUMENT *Doc,UDM_DB *db)
{
  switch(db->DBMode)
  {
    case UDM_DBMODE_BLOB:
      return(UdmStoreWordsBlob(Indexer, Doc, db));
    case UDM_DBMODE_MULTI:
      return(UdmStoreWordsMulti(Indexer, Doc, db));
    case UDM_DBMODE_SINGLE:
    default:
      return(StoreWordsSingle(Indexer, Doc, db));
  }
}


static int UdmDeleteAllFromDict(UDM_AGENT *Indexer,UDM_DB *db)
{
  size_t  i;
  int  rc=UDM_OK;
  
  switch(db->DBMode)
  {
  case UDM_DBMODE_MULTI:
    for(i = 0 ; i <= MULTI_DICTS; i++)
    {
      char tablename[32];
      sprintf(tablename, "dict%02X", i);
      if(UDM_OK != (rc= UdmSQLTableTruncateOrDelete(db, tablename)))
        return rc;
    }
    break;
  case UDM_DBMODE_BLOB:
    if(UDM_OK != (rc= UdmSQLTableTruncateOrDelete(db, "bdicti")) ||
       UDM_OK != (rc= UdmSQLTableTruncateOrDelete(db, "bdict")))
      return rc;
    
    break;
  default:
    rc= UdmSQLTableTruncateOrDelete(db, "dict");
    break;
  }
  return rc;
}


/***************** CrossWords *******************************/

static int UdmDeleteAllFromCrossDict(UDM_AGENT * Indexer,UDM_DB *db)
{
  return UdmSQLTableTruncateOrDelete(db, "crossdict");
}


static int UdmDeleteCrossWordFromURL(UDM_AGENT * Indexer,UDM_DOCUMENT *Doc,UDM_DB *db)
{
  char  qbuf[1024];
  urlid_t  url_id = UdmVarListFindInt(&Doc->Sections, "ID", 0);
  urlid_t  referrer_id  =UdmVarListFindInt(&Doc->Sections, "Referrer-ID", 0);
  int  rc=UDM_OK;
  const char      *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  
  if(url_id)
  {
    sprintf(qbuf,"DELETE FROM crossdict WHERE url_id=%s%i%s", qu, url_id, qu);
    if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,qbuf)))
      return rc;
  }
  if(referrer_id)
  {
    sprintf(qbuf,"DELETE FROM crossdict WHERE ref_id=%s%i%s", qu, referrer_id, qu);
    rc=UdmSQLQuery(db,NULL,qbuf);
  }
  return rc;
}


static int UdmStoreCrossWords(UDM_AGENT * Indexer,UDM_DOCUMENT *Doc,UDM_DB *db)
{
  UDM_DOCUMENT  U;
  size_t    i;
  char    qbuf[1024];
  const char  *lasturl="scrap";
  const char      *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  urlid_t    referrer = UdmVarListFindInt(&Doc->Sections, "ID", 0);
  urlid_t    childid = 0;
  int    rc=UDM_OK;
  UDM_HREF        Href;
  UDM_URL         docURL;
  
  UdmDocInit(&U);
  bzero((void*)&Href, sizeof(Href));
  UdmVarListReplaceInt(&Doc->Sections, "Referrer-ID", referrer);
  if(UDM_OK!=(rc=UdmDeleteCrossWordFromURL(Indexer,&U,db)))
  {
    UdmDocFree(&U);
    return rc;
  }
  
  if(Doc->CrossWords.ncrosswords==0)
  {
    UdmDocFree(&U);
    return rc;
  }
  
  UdmURLInit(&docURL);
  UdmURLParse(&docURL, UdmVarListFindStr(&Doc->Sections, "URL", ""));
  for(i=0;i<Doc->CrossWords.ncrosswords;i++)
  {
    if(!Doc->CrossWords.CrossWord[i].weight)continue;
    if(strcmp(lasturl,Doc->CrossWords.CrossWord[i].url))
    {
      Href.url = (char*)UdmStrdup(Doc->CrossWords.CrossWord[i].url);
      UdmConvertHref(Indexer, &docURL, &Doc->Spider, &Href);
      UdmVarListReplaceStr(&U.Sections, "URL", Href.url);
      UdmVarListReplaceInt(&U.Sections, "URL_ID", UdmStrHash32(Href.url));
      if(UDM_OK!=(rc=UdmFindURL(Indexer,&U,db)))
      {
        UdmDocFree(&U);
        UdmURLFree(&docURL);
        return rc;
      }
      childid = UdmVarListFindInt(&U.Sections,"ID",0);
      lasturl=Doc->CrossWords.CrossWord[i].url;
      UDM_FREE(Href.url);
    }
    Doc->CrossWords.CrossWord[i].referree_id=childid;
  }
  
  /* Begin transacttion/lock */
  if (db->DBDriver == UDM_DB_MYSQL)
  {
    sprintf(qbuf,"LOCK TABLES crossdict WRITE");
    if (UDM_OK != (rc= UdmSQLQuery(db,NULL,qbuf)))
      goto free_ex;
  }
  else
  {
    /* Don't start transaction here - 
       this function is called inside a transaction
    */
  }
  
  /* Insert new words */
  for(i=0;i<Doc->CrossWords.ncrosswords;i++)
  {
    if(Doc->CrossWords.CrossWord[i].weight && Doc->CrossWords.CrossWord[i].referree_id)
    {
      int weight=UDM_WRDCOORD(Doc->CrossWords.CrossWord[i].pos,Doc->CrossWords.CrossWord[i].weight);
      sprintf(qbuf,"INSERT INTO crossdict (ref_id,url_id,word,intag) VALUES(%s%i%s,%s%i%s,'%s',%d)",
        qu, referrer, qu, qu, Doc->CrossWords.CrossWord[i].referree_id, qu,
        Doc->CrossWords.CrossWord[i].word, weight);
      if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,qbuf)))
      {
        UdmDocFree(&U);
        goto unlock_UdmStoreCrossWords;
      }
    }
  }

unlock_UdmStoreCrossWords:

  if (db->DBDriver == UDM_DB_MYSQL)
  {
    sprintf(qbuf,"UNLOCK TABLES");
    rc=UdmSQLQuery(db,NULL,qbuf);
  }
  else
  {
    /*
      Don't do commit here - 
      this function is called inside a transaction
    */
  }

free_ex:
  UdmDocFree(&U);
  UdmURLFree(&docURL);
  return rc;
}



static void SQLResToDoc(UDM_ENV *Conf, UDM_DOCUMENT *D, UDM_SQLRES *sqlres, size_t i)
{
  time_t    last_mod_time;
  char    dbuf[128];
  const char  *format = UdmVarListFindStr(&Conf->Vars, "DateFormat", "%a, %d %b %Y, %X %Z");
  double          pr;
  
  UdmVarListReplaceStr(&D->Sections,"URL",UdmSQLValue(sqlres,i,1));
  UdmVarListReplaceInt(&D->Sections, "URL_ID", UdmStrHash32(UdmSQLValue(sqlres,i,1)));
  last_mod_time=atol(UdmSQLValue(sqlres,i,2));
  UdmVarListReplaceInt(&D->Sections, "Last-Modified-Timestamp", (int) last_mod_time);
  if (strftime(dbuf, 128, format, localtime(&last_mod_time)) == 0)
  {
    UdmTime_t2HttpStr(last_mod_time, dbuf);
  }
  UdmVarListReplaceStr(&D->Sections,"Last-Modified",dbuf);
  UdmVarListReplaceStr(&D->Sections,"Content-Length",UdmSQLValue(sqlres,i,3));
  pr= atof(UdmSQLValue(sqlres,i,3)) / 1024;
  sprintf(dbuf, "%.2f", pr);
  UdmVarListReplaceStr(&D->Sections,"Content-Length-K",dbuf);  
  last_mod_time=atol(UdmSQLValue(sqlres,i,4));
  if (strftime(dbuf, 128, format, localtime(&last_mod_time)) == 0)
  {
    UdmTime_t2HttpStr(last_mod_time, dbuf);
  }
  UdmVarListReplaceStr(&D->Sections,"Next-Index-Time",dbuf);
  UdmVarListReplaceInt(&D->Sections, "Referrer-ID", UDM_ATOI(UdmSQLValue(sqlres,i,5)));
  UdmVarListReplaceInt(&D->Sections,"crc32",atoi(UdmSQLValue(sqlres,i,6)));
  UdmVarListReplaceStr(&D->Sections, "Site_id", UdmSQLValue(sqlres, i, 7));

#if BAR_COMMA_PERIOD_ORACLE_PROBLEM
  {
	char *num= UdmSQLValue(sqlres, i, 8);
	char *comma= strchr(num, ',');
	if (comma)
	  *comma= '.';
  }
#endif

  pr = atof(UdmSQLValue(sqlres, i, 8));
  snprintf(dbuf, 128, "%.5f", pr);
  UdmVarListReplaceStr(&D->Sections, "Pop_Rank", dbuf);
}

/************************ URLs ***********************************/

static
int InsertLink(UDM_AGENT *A, UDM_DB *db, urlid_t from, urlid_t to)
{
  char qbuf[128];
  const char *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  if (from == to) /* Don't collect self links */
    return UDM_OK;
  udm_snprintf(qbuf, sizeof(qbuf),
               "INSERT INTO links (ot,k,weight) VALUES (%s%i%s,%s%d%s,0.0)",
               qu, from, qu, qu, to, qu);
  return UdmSQLQuery(db, NULL, qbuf);
}


static int UdmVarListSQLEscape(UDM_VARLIST *dst, UDM_VARLIST *src, UDM_DB *db)
{
  size_t i, nbytes= 0;
  char *tmp= NULL;
  for (i= 0; i < src->nvars; i++)
  {
    size_t len= src->Var[i].curlen;
    if (nbytes < len * 2 + 1)
    {
      nbytes= len * 2 + 1;
      tmp= (char*) UdmRealloc(tmp, nbytes);
    }
    UdmSQLEscStr(db, tmp, src->Var[i].val, len);
    UdmVarListAddStr(dst, src->Var[i].name, tmp);
  }
  UdmFree(tmp);
  return UDM_OK;
}


static int UdmAddURL(UDM_AGENT *Indexer,UDM_DOCUMENT * Doc,UDM_DB *db)
{
  char    *e_url, *qbuf;
  const char  *url;
  int    url_seed;
  int    use_crc32_url_id;
  int    usehtdburlid;
  int    rc=UDM_OK;
  size_t          len;
  const char      *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  const char *sql_export= UdmVarListFindStr(&Doc->Sections, "SQLExportHref", NULL);
  urlid_t rec_id = 0;

  if (sql_export)
  {
    char *part, *lt, *sql_export_copy= UdmStrdup(sql_export);
    UDM_DSTR d;
    UDM_VARLIST Vars;
    UdmVarListInit(&Vars);
    UdmDSTRInit(&d,256);
    
    UdmVarListSQLEscape(&Vars, &Doc->Sections, db);
    for (part= udm_strtok_r(sql_export_copy, ";", &lt) ;
         part ;
         part= udm_strtok_r(NULL, ";", &lt))
    {
      UdmDSTRParse(&d, part, &Vars);
      if(UDM_OK!= (rc= UdmSQLQuery(db, NULL, d.data)))
        return rc;
      UdmDSTRReset(&d);
    }
    UdmVarListFree(&Vars);
    UdmDSTRFree(&d);
    UdmFree(sql_export_copy);
  }
  url = UdmVarListFindStr(&Doc->Sections,"URL","");
  use_crc32_url_id = !strcasecmp(UdmVarListFindStr(&Indexer->Conf->Vars, "UseCRC32URLId", "no"), "yes");
  usehtdburlid = UdmVarListFindInt(&Indexer->Conf->Vars, "UseHTDBURLId", 0);

  len = strlen(url);
  e_url = (char*)UdmMalloc(4 * len + 1);
  if (e_url == NULL) return UDM_ERROR;
  qbuf = (char*)UdmMalloc(4 * len + 512);
  if (qbuf == NULL)
  { 
    UDM_FREE(e_url);
    return UDM_ERROR;
  }
  
  url_seed = UdmStrHash32(url) & 0xFF;
  
  /* Escape URL string */
  UdmSQLEscStr(db, e_url, url, len);
  
  if(use_crc32_url_id || usehtdburlid)
  {
    /* Auto generation of rec_id */
    /* using CRC32 algorythm     */
    if (use_crc32_url_id) rec_id = UdmStrHash32(url);
    else rec_id = UdmVarListFindInt(&Doc->Sections, "HTDB_URL_ID", 0);
    
    udm_snprintf(qbuf, 4 * len + 512, "INSERT INTO url (rec_id,url,referrer,hops,crc32,next_index_time,status,seed,bad_since_time,site_id,server_id,docsize,last_mod_time,shows,pop_rank) VALUES (%s%i%s,'%s',%s%i%s,%d,0,%d,0,%d,%d,%s%i%s,%s%i%s,%s%i%s,%li,0,0.0)",
           qu, rec_id, qu,
           e_url,
           qu, UdmVarListFindInt(&Doc->Sections,"Referrer-ID",0), qu,
           UdmVarListFindInt(&Doc->Sections,"Hops",0),
           (int)time(NULL),
           url_seed, (int)time(NULL),
           qu, UdmVarListFindInt(&Doc->Sections, "Site_id", 0), qu,
           qu, UdmVarListFindInt(&Doc->Sections, "Server_id", 0), qu,
           qu, UdmVarListFindInt(&Doc->Sections, "Content-Length", 0), qu,
           UdmHttpDate2Time_t(UdmVarListFindStr(&Doc->Sections, "Last-Modified",
                              UdmVarListFindStr(&Doc->Sections, "Date", "")))
       );
  }else{
    /* Use dabatase generated rec_id */
    /* It depends on used DBType     */
    switch(db->DBType)
    {
    case UDM_DB_SOLID:
    case UDM_DB_ORACLE8:
    case UDM_DB_SAPDB:
      /* FIXME: Dirty hack for stupid too smart databases 
       Change this for config parameter checking */
/*      if (strlen(e_url)>UDM_URLSIZE)e_url[UDM_URLSIZE]=0;*/
      /* Use sequence next_url_id.nextval */
      udm_snprintf(qbuf, 4 * len + 512, "INSERT INTO url (url,referrer,hops,rec_id,crc32,next_index_time,status,seed,bad_since_time,site_id,server_id) VALUES ('%s',%i,%d,next_url_id.nextval,0,%d,0,%d,%d,%i,%i)",
        e_url,
        UdmVarListFindInt(&Doc->Sections,"Referrer-ID",0),
        UdmVarListFindInt(&Doc->Sections,"Hops",0),
        (int)time(NULL),
         url_seed, (int)time(NULL),
         UdmVarListFindInt(&Doc->Sections, "Site_id", 0),
         UdmVarListFindInt(&Doc->Sections, "Server_id", 0)
         );
      break;
    case UDM_DB_MIMER:
      udm_snprintf(qbuf, 4 * len + 512, "INSERT INTO url (url,referrer,hops,rec_id,crc32,next_index_time,status,seed,bad_since_time,site_id,server_id) VALUES ('%s',%i,%d,NEXT_VALUE OF rec_id_GEN,0,%d,0,%d,%d,%i,%i)",
        e_url,
        UdmVarListFindInt(&Doc->Sections,"Referrer-ID",0),
        UdmVarListFindInt(&Doc->Sections,"Hops",0),
        (int)time(NULL),
         url_seed, (int)time(NULL),
         UdmVarListFindInt(&Doc->Sections, "Site_id", 0),
         UdmVarListFindInt(&Doc->Sections, "Server_id", 0)
         );
      break;    
    case UDM_DB_IBASE:
      udm_snprintf(qbuf, 4 * len + 512, "INSERT INTO url (url,referrer,hops,rec_id,crc32,next_index_time,status,seed,bad_since_time,site_id,server_id) VALUES ('%s',%i,%d,GEN_ID(rec_id_GEN,1),0,%d,0,%d,%d,%i,%i)",
        e_url,
        UdmVarListFindInt(&Doc->Sections,"Referrer-ID",0),
        UdmVarListFindInt(&Doc->Sections,"Hops",0),
        (int)time(NULL),
         url_seed, (int)time(NULL),
         UdmVarListFindInt(&Doc->Sections, "Site_id", 0),
         UdmVarListFindInt(&Doc->Sections, "Server_id", 0)
         );
      break;
    case UDM_DB_MYSQL:
      /* MySQL generates itself */
    default:  
      udm_snprintf(qbuf, 4 * len + 512, "INSERT INTO url (url,referrer,hops,crc32,next_index_time,status,seed,bad_since_time,site_id,server_id,docsize,last_mod_time,shows,pop_rank) VALUES ('%s',%s%i%s,%d,0,%d,0,%d,%d,%s%i%s,%s%i%s,%s%i%s,%li,0,0.0)",
             e_url,
             qu, UdmVarListFindInt(&Doc->Sections,"Referrer-ID",0), qu,
             UdmVarListFindInt(&Doc->Sections,"Hops",0),
             (int)time(NULL),
             url_seed, (int)time(NULL),
             qu, UdmVarListFindInt(&Doc->Sections, "Site_id", 0), qu,
             qu, UdmVarListFindInt(&Doc->Sections, "Server_id", 0), qu,
             qu, UdmVarListFindInt(&Doc->Sections, "Content-Length", 0), qu,
             UdmHttpDate2Time_t(UdmVarListFindStr(&Doc->Sections, "Last-Modified",
             UdmVarListFindStr(&Doc->Sections, "Date", "")))
         );
    }
  }

  /* Exec INSERT now */
  if(UDM_OK!=(rc=UdmSQLQuery(db, NULL, qbuf)))
    goto ex;

ex:

  UDM_FREE(qbuf);
  UDM_FREE(e_url);
  return rc;
}


static int UdmAddLink(UDM_AGENT *Indexer, UDM_DOCUMENT * Doc, UDM_DB *db)
{
  char    *e_url, *qbuf;
  UDM_SQLRES  SQLRes;
  const char  *url;
  int    use_crc32_url_id;
  int    rc=UDM_OK;
  size_t          len;
  urlid_t rec_id = 0;

  url = UdmVarListFindStr(&Doc->Sections,"URL","");
  use_crc32_url_id = !strcasecmp(UdmVarListFindStr(&Indexer->Conf->Vars, "UseCRC32URLId", "no"), "yes");

  len = strlen(url);
  e_url = (char*)UdmMalloc(4 * len + 1);
  if (e_url == NULL) return UDM_ERROR;
  qbuf = (char*)UdmMalloc(4 * len + 512);
  if (qbuf == NULL) 
  { 
    UDM_FREE(e_url); 
    return UDM_ERROR;
  }
  
  if (use_crc32_url_id)
  {
    rec_id = UdmStrHash32(url);
  }
  else
  {
    /* Escape URL string */
    UdmSQLEscStr(db, e_url, url, len);
  
    udm_snprintf(qbuf, 4 * len + 512, "SELECT rec_id FROM url WHERE url='%s'", e_url);
    if(UDM_OK != (rc = UdmSQLQuery(db, &SQLRes, qbuf))) 
      goto ex;
    if (UdmSQLNumRows(&SQLRes))
    {
      rec_id = UDM_ATOI(UdmSQLValue(&SQLRes, 0, 0));
    }
    UdmSQLFree(&SQLRes);
  }

  if (rec_id)
  {
    urlid_t from= UdmVarListFindInt(&Doc->Sections, "Referrer-ID", 0);
    UdmVarListReplaceInt(&Doc->Sections, "ID", rec_id);
    if(UDM_OK != (rc = InsertLink(Indexer, db, from, rec_id)))
      goto ex;
  }
  else
  {
    UdmLog(Indexer, UDM_LOG_ERROR, "URL not found: %s", e_url);
  }

ex:

  UDM_FREE(qbuf);
  UDM_FREE(e_url);
  return UDM_OK;
}


static int
UdmDeleteBadHrefs(UDM_AGENT *Indexer,
                  UDM_DOCUMENT *Doc,
                  UDM_DB *db,
                  urlid_t url_id)
{
  UDM_DOCUMENT  rDoc;
  UDM_SQLRES  SQLRes;
  char    q[256];
  size_t    i;
  size_t    nrows;
  int    rc=UDM_OK;
  int    hold_period= UdmVarListFindInt(&Doc->Sections,"HoldBadHrefs",0);
  const char      *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  
  if (hold_period <= 0)
    return UDM_OK;
  
  udm_snprintf(q, sizeof(q), "SELECT rec_id FROM url WHERE status > 300 AND status<>304 AND referrer=%s%i%s AND bad_since_time<%d",
    qu, url_id, qu, qu, qu, (int)time(NULL) - hold_period);
  if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLRes,q)))return rc;
  
  nrows = UdmSQLNumRows(&SQLRes);
  
  UdmDocInit(&rDoc);
  for(i = 0; i < nrows ; i++)
  {
    UdmVarListReplaceStr(&rDoc.Sections,"ID", UdmSQLValue(&SQLRes,i,0));
    if(UDM_OK!=(rc=UdmDeleteURL(Indexer, &rDoc, db)))
      break;
  }
  UdmDocFree(&rDoc);
  UdmSQLFree(&SQLRes);
  return rc;
}

static int UdmDeleteURL(UDM_AGENT *Indexer, UDM_DOCUMENT *Doc,UDM_DB *db)
{
  char  qbuf[128];
  int  rc;
  urlid_t  url_id  =UdmVarListFindInt(&Doc->Sections, "ID", 0);
  int  use_crosswords;
  const char *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  
  use_crosswords = !strcasecmp(UdmVarListFindStr(&Indexer->Conf->Vars, "CrossWords", "no"), "yes");

  if(use_crosswords)
    if(UDM_OK!=(rc=UdmDeleteCrossWordFromURL(Indexer,Doc,db)))return(rc);
  
  if(UDM_OK!=(rc=UdmDeleteWordFromURL(Indexer,Doc,db)))return(rc);
  
  sprintf(qbuf,"DELETE FROM url WHERE rec_id=%s%i%s", qu, url_id, qu);
  if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,qbuf)))return rc;
  
  sprintf(qbuf,"DELETE FROM urlinfo WHERE url_id=%s%i%s", qu, url_id, qu);
  if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,qbuf)))return rc;
  
  sprintf(qbuf,"DELETE FROM links WHERE ot=%s%i%s", qu, url_id, qu);
  if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,qbuf)))return rc;
  
  sprintf(qbuf,"DELETE FROM links WHERE k=%s%i%s", qu, url_id, qu);
  if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,qbuf)))return rc;
  
  /* remove all old broken hrefs from this document to avoid broken link collecting */
  if (UDM_OK != (rc=UdmDeleteBadHrefs(Indexer,Doc, db, url_id))) return rc;

  sprintf(qbuf,"UPDATE url SET referrer=%s0%s WHERE referrer=%s%i%s", qu, qu, qu, url_id, qu);
  return UdmSQLQuery(db,NULL,qbuf);
}

static int UdmDeleteLinks(UDM_AGENT *Indexer, UDM_DOCUMENT *Doc, UDM_DB *db)
{
  char  qbuf[128];
  urlid_t  url_id = UdmVarListFindInt(&Doc->Sections, "ID", 0);
  const char *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";

  sprintf(qbuf,"DELETE FROM links WHERE ot=%s%i%s", qu, url_id, qu);
  return UdmSQLQuery(db, NULL, qbuf);
}


static int
UdmGetCachedCopy(UDM_AGENT *Indexer, UDM_DOCUMENT *Doc, UDM_DB *db)
{
#ifdef HAVE_ZLIB
  UDM_SQLRES SQLRes;
  char buf[1024];
  const char *sname;
  const char *sval;
  int _;
  int i;

  UdmFindURL(Indexer, Doc, db);
  udm_snprintf(buf, sizeof(buf), "SELECT rec_id,url,last_mod_time,docsize,next_index_time,referrer,crc32,site_id,pop_rank FROM url WHERE rec_id=%d", UDM_ATOI(UdmVarListFindStr(&Doc->Sections, "ID", "0")));
  _ = UdmSQLQuery(db, &SQLRes, buf);
  if (_ != UDM_OK) return(_);
  if (! UdmSQLNumRows(&SQLRes)) return(UDM_ERROR);
  SQLResToDoc(Indexer->Conf, Doc, &SQLRes, 0);
  UdmSQLFree(&SQLRes);
  udm_snprintf(buf, sizeof(buf), "SELECT sname, sval FROM urlinfo WHERE url_id=%d", UDM_ATOI(UdmVarListFindStr(&Doc->Sections, "ID", "0")));
  _ = UdmSQLQuery(db, &SQLRes, buf);
  if (_ != UDM_OK) return(_);

  for (i = 0; i < UdmSQLNumRows(&SQLRes); i++)
  {
    sname = UdmSQLValue(&SQLRes, i, 0);
    sval = UdmSQLValue(&SQLRes, i, 1);
    if (! sname) continue;
    if (! sval) sval = "";

    if (! strcmp(sname, "CachedCopy")) 
    {
      size_t l;

      if (Doc->Buf.content) continue;

      UdmVarListReplaceStr(&Doc->Sections, "CachedCopyBase64", sval);
      l = strlen(sval);
      Doc->Buf.buf = UdmMalloc(UDM_MAXDOCSIZE);
      if (UDM_OK != UdmCachedCopyUnpack(Doc, sval, l))
        return UDM_ERROR;
    }
    else
    {
      UdmVarListReplaceStr(&Doc->Sections, sname, sval);
    }
  }
  
  UdmSQLFree(&SQLRes);
  return(UDM_OK);
#else
  return(UDM_ERROR);
#endif
}

static int UdmMarkForReindex(UDM_AGENT *Indexer, UDM_DOCUMENT *Doc, UDM_DB *db)
{
  char    qbuf[1024];
  const char  *where;
  UDM_SQLRES   SQLRes;
  size_t          i, j;
  int             rc;
  const char      *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  UDM_DSTR buf;

  UDM_LOCK_CHECK_OWNER(Indexer, UDM_LOCK_CONF);
  where = BuildWhere(Indexer->Conf, db);
  
  if (db->flags & UDM_SQL_HAVE_SUBSELECT &&
      db->DBType != UDM_DB_MYSQL)
  {
    udm_snprintf(qbuf,sizeof(qbuf),"UPDATE url SET next_index_time=%d WHERE rec_id IN (SELECT url.rec_id FROM url%s %s %s)",
       (int)time(NULL), db->from, (where[0]) ? "WHERE" : "", where);
    return UdmSQLQuery(db,NULL,qbuf);
  }

  udm_snprintf(qbuf, sizeof(qbuf), "SELECT url.rec_id FROM url%s %s %s", db->from, (where[0]) ? "WHERE" : "", where);
  if(UDM_OK != (rc = UdmSQLQuery(db, &SQLRes, qbuf))) return rc;

  UdmDSTRInit(&buf, 4096);
  if (db->DBSQL_IN) 
  {
    for (i = 0; i < UdmSQLNumRows(&SQLRes); i += 512) 
    {
      UdmDSTRReset(&buf);
      UdmDSTRAppendf(&buf, "UPDATE url SET next_index_time=%d WHERE rec_id IN (", (int)time(NULL));
      for (j = 0; (j < 512) && (i + j < UdmSQLNumRows(&SQLRes)); j++) 
      {
        UdmDSTRAppendf(&buf, "%s%s%s%s", (j) ? "," : "", qu, UdmSQLValue(&SQLRes, i + j, 0), qu);
      }
      UdmDSTRAppendf(&buf, ")");
      if(UDM_OK != (rc = UdmSQLQuery(db, NULL, buf.data))) 
      {
        UdmSQLFree(&SQLRes);
	UdmDSTRFree(&buf);
        return rc;
      }
    }
  } else {
    for (i = 0; i < UdmSQLNumRows(&SQLRes); i++) 
    {
      UdmDSTRReset(&buf);
      UdmDSTRAppendf(&buf, "UPDATE url SET next_index_time=%d WHERE rec_id=%s", (int)time(NULL),  UdmSQLValue(&SQLRes, i, 0));
      if(UDM_OK != (rc = UdmSQLQuery(db, NULL, buf.data))) 
      {
        UdmSQLFree(&SQLRes);
	UdmDSTRFree(&buf);
        return rc;
      }
    }
  }
  UdmDSTRFree(&buf);
  UdmSQLFree(&SQLRes);
  return UDM_OK;
}


static int UdmRegisterChild(UDM_AGENT *Indexer, UDM_DOCUMENT *Doc,UDM_DB *db)
{
  char  qbuf[1024];
  urlid_t  url_id = UdmVarListFindInt(&Doc->Sections,"ID",0);
  urlid_t  parent_id = UdmVarListFindInt(&Doc->Sections,"Parent-ID",0);
  const char      *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  
  udm_snprintf(qbuf,sizeof(qbuf),"insert into links (ot,k,weight) values(%s%i%s,%s%i%s,0.0)", qu, parent_id, qu, qu, url_id, qu);
  return UdmSQLQuery(db,NULL,qbuf);
}


static int
UdmUpdateUrl(UDM_AGENT *Indexer,UDM_DOCUMENT *Doc,UDM_DB *db)
{
  char qbuf[256];
  urlid_t  url_id = UdmVarListFindInt(&Doc->Sections, "ID", 0);
  int  status=UdmVarListFindInt(&Doc->Sections,"Status",0);
  int  prevStatus = UdmVarListFindInt(&Doc->Sections, "PrevStatus", 0);
  int  next_index_time=UdmHttpDate2Time_t(UdmVarListFindStr(&Doc->Sections,"Next-Index-Time",""));
  int  res;
  const char      *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  
  if (prevStatus != status && status > 300 && status != 304)
    sprintf(qbuf, "UPDATE url SET status=%d,next_index_time=%d,bad_since_time=%d,site_id=%s%i%s,server_id=%s%i%s WHERE rec_id=%s%i%s",
            status, next_index_time, (int)time(NULL), qu, UdmVarListFindInt(&Doc->Sections, "Site_id", 0), qu,
           qu, UdmVarListFindInt(&Doc->Sections, "Server_id",0), qu, qu, url_id, qu);
  else
    sprintf(qbuf,"UPDATE url SET status=%d,next_index_time=%d, site_id=%s%i%s,server_id=%s%i%s WHERE rec_id=%s%i%s",
            status, next_index_time, qu, UdmVarListFindInt(&Doc->Sections, "Site_id", 0), qu,
            qu, UdmVarListFindInt(&Doc->Sections, "Server_id",0), qu, qu, url_id, qu);

  if(UDM_OK!=(res=UdmSQLQuery(db,NULL,qbuf)))return res;
  
  /* remove all old broken hrefs from this document to avoid broken link collecting */
  return UdmDeleteBadHrefs(Indexer,Doc,db,url_id);
}

static int
UdmUpdateUrlWithLangAndCharset(UDM_AGENT *Indexer, UDM_DOCUMENT *Doc,UDM_DB *db)
{
  char  *qbuf;
  int  rc;
  const char  *charset;
  UDM_VAR    *var;
  int    status, prevStatus;
  urlid_t         url_id;
  size_t    i, len = 0;
  char    qsmall[64];
  const char      *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  int IndexTime= UdmVarListFindInt(&Indexer->Conf->Vars, "IndexTime", 0);
  
  status = UdmVarListFindInt(&Doc->Sections, "Status", 0);
  prevStatus = UdmVarListFindInt(&Doc->Sections, "PrevStatus", 0);
  url_id = UdmVarListFindInt(&Doc->Sections, "ID", 0);
  
  if((var=UdmVarListFind(&Doc->Sections,"Content-Language")))
  {
    if (var->val == NULL)
    {
      var->val = (char*)UdmStrdup(UdmVarListFindStr(&Doc->Sections, "DefaultLang", "en"));
    }
    len=strlen(var->val);
    for(i = 0; i < len; i++)
    {
      var->val[i] = tolower(var->val[i]);
    }
  }
  
  charset = UdmVarListFindStr(&Doc->Sections, "Charset", 
            UdmVarListFindStr(&Doc->Sections, "RemoteCharset", "iso-8859-1"));
  charset = UdmCharsetCanonicalName(charset);
  UdmVarListReplaceStr(&Doc->Sections, "Charset", charset);
  
  if (prevStatus != status && status > 300 && status != 304)
    udm_snprintf(qsmall, 64, ", bad_since_time=%d", (int)time(NULL));
  else qsmall[0] = '\0';

  if (IndexTime)
  {
    if (! prevStatus) udm_snprintf(UDM_STREND(qsmall), 64, ",last_mod_time=%li", time(NULL));
  }
  else
  {
    const char *lmsrc= UdmVarListFindStr(&Doc->Sections, "User.Date",
                       UdmVarListFindStr(&Doc->Sections, "Last-Modified",
                       UdmVarListFindStr(&Doc->Sections, "Date", "")));
    udm_snprintf(UDM_STREND(qsmall), 64, ",last_mod_time=%li", UdmHttpDate2Time_t(lmsrc));
  }
  qbuf=(char*)UdmMalloc(1024);
  
  
  udm_snprintf(qbuf, 1023, "\
UPDATE url SET \
status=%d,\
next_index_time=%li,\
docsize=%d,\
crc32=%d%s, site_id=%s%i%s, server_id=%s%i%s \
WHERE rec_id=%s%i%s",
  status,
  UdmHttpDate2Time_t(UdmVarListFindStr(&Doc->Sections,"Next-Index-Time","")),
  UdmVarListFindInt(&Doc->Sections,"Content-Length",0),
  UdmVarListFindInt(&Doc->Sections,"crc32",0),
  qsmall,
  qu, UdmVarListFindInt(&Doc->Sections,"Site_id",0), qu,
  qu, UdmVarListFindInt(&Doc->Sections, "Server_id",0), qu,
  qu, url_id, qu);
  
  rc=UdmSQLQuery(db,NULL,qbuf);
  UDM_FREE(qbuf);
  return rc;
}


static int UdmLongUpdateURL(UDM_AGENT *Indexer,UDM_DOCUMENT *Doc,UDM_DB *db)
{
  int    rc=UDM_OK;
  size_t    i;
  char    *qbuf;
  char    *arg;
  char    qsmall[128];
  size_t    len=0;
  urlid_t    url_id = UdmVarListFindInt(&Doc->Sections, "ID", 0);
  const char  *c, *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  size_t esc_multiply = (db->DBType == UDM_DB_PGSQL) ? 4 : 2;
#ifdef HAVE_ODBC
  int odbcbind= (db->DBDriver == UDM_DB_ODBC) ? 1 : 0;
#endif
  int use_crosswords;
  int use_tnx=  (db->DBMode == UDM_DBMODE_BLOB) &&
                (db->DBType == UDM_DB_PGSQL ||
                 db->DBType == UDM_DB_ORACLE8 ||
                 db->DBType == UDM_DB_MIMER ||
                 db->DBType == UDM_DB_MSSQL ||
                 db->DBType == UDM_DB_SQLITE3) ? 1 : 0;

#if defined(WIN32) && defined(HAVE_ODBC)
  if (db->DBType != UDM_DB_ORACLE8 &&
      db->DBType != UDM_DB_MIMER)
    odbcbind= 0;
#endif

  use_crosswords = !strcasecmp(UdmVarListFindStr(&Indexer->Conf->Vars, "CrossWords", "no"), "yes");

  if (use_tnx && UDM_OK!=(rc=UdmSQLBegin(db)))
    return rc;
  
  /* Now store words and crosswords */
  if(UDM_OK != (rc = UdmStoreWords(Indexer, Doc, db)))
    return rc;
  
  if(use_crosswords)
    if(UDM_OK != (rc = UdmStoreCrossWords(Indexer, Doc, db)))
      return rc;
  

  /* Copy default languages, if not given by server and not guessed */
  if(!(c=UdmVarListFindStr(&Doc->Sections,"Content-Language",NULL)))
  {
    if((c=UdmVarListFindStr(&Doc->Sections,"DefaultLang",NULL)))
      UdmVarListReplaceStr(&Doc->Sections,"Content-Language",c);
  }
  

  if(UDM_OK != (rc = UdmUpdateUrlWithLangAndCharset(Indexer, Doc, db)))
    return rc;
  
  /* remove all old broken hrefs from this document to avoid broken link collecting */
  if(UDM_OK!=(rc=UdmDeleteBadHrefs(Indexer,Doc,db,url_id)))
    return rc;
  
  sprintf(qsmall,"DELETE FROM urlinfo WHERE url_id=%s%i%s", qu, url_id, qu);
  if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,qsmall)))return rc;

/* No need delete from links here, it has been done before */
  
  if(!Doc->Sections.nvars) return UDM_OK;
  
  len=0;
  for(i=0;i<Doc->Sections.nvars;i++)
  {
    size_t l = Doc->Sections.Var[i].curlen + (Doc->Sections.Var[i].name ? strlen(Doc->Sections.Var[i].name) : 0);
    if(len < l)
      len = l;
  }
  if(!len)return UDM_OK;
  
  qbuf=(char*)UdmMalloc(esc_multiply * len + 128);
  arg=(char*)UdmMalloc(esc_multiply * len + 128);
  
  for(i=0;i<Doc->Sections.nvars;i++)
  {
    UDM_VAR *Sec=&Doc->Sections.Var[i];
    if(Sec->val && Sec->name && ((Sec->curlen && Sec->maxlen) || (!strcmp(Sec->name, "Z"))) )
    {
#ifdef HAVE_ORACLE8
      if (db->DBDriver == UDM_DB_ORACLE8 && db->sql->SQLPrepare)
      {
        sprintf(qbuf, "INSERT INTO urlinfo (url_id,sname,sval) VALUES(%i, '%s', :1)",
          url_id, Sec->name);
        db->sql->SQLPrepare(db, qbuf);
        if (db->errcode) UdmLog(Indexer, UDM_LOG_ERROR, "Cannot prepare query: errcode=%d, %s\n", db->errcode, db->errstr);
        db->sql->SQLBind(db, 1, Sec->val, (int) strlen(Sec->val), UDM_SQLTYPE_LONGVARCHAR);
        if (db->errcode) UdmLog(Indexer, UDM_LOG_ERROR, "Cannot bind value: errcode=%d, %s\n", db->errcode, db->errstr);
        db->sql->SQLExec(db);
        if (db->errcode) UdmLog(Indexer, UDM_LOG_ERROR, "Cannot execute query: errcode=%d, %s\n", db->errcode, db->errstr);
        continue;
      }
#endif
#ifdef HAVE_ODBC
      if (odbcbind && db->sql->SQLPrepare)
      {
        int bindtype= db->DBType == UDM_DB_MSSQL ||
                      db->DBType == UDM_DB_SYBASE ?
                      UDM_SQLTYPE_VARCHAR : UDM_SQLTYPE_LONGVARCHAR;
#ifdef HAVE_UNIXODBC
        if (db->DBType == UDM_DB_SYBASE)
          bindtype= UDM_SQLTYPE_LONGVARCHAR;
#endif
        sprintf(qbuf, "INSERT INTO urlinfo (url_id,sname,sval) VALUES(%i, '%s', ?)",
                url_id, Sec->name);
        if (UDM_OK != (rc= db->sql->SQLPrepare(db, qbuf)) ||
            UDM_OK != (rc= db->sql->SQLBind(db, 1, Sec->val, (int) strlen(Sec->val), bindtype)) ||
            UDM_OK != (rc= db->sql->SQLExec(db)))
          break;
        continue;
      }
#endif
      arg=UdmSQLEscStr(db,arg,Sec->val,strlen(Sec->val));
      sprintf(qbuf,"INSERT INTO urlinfo (url_id,sname,sval) VALUES (%s%i%s,'%s','%s')",
              qu, url_id, qu, Sec->name, arg);
      if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,qbuf)))break;

    }
  }
  UDM_FREE(qbuf);
  UDM_FREE(arg);

  if(use_tnx && rc==UDM_OK)
    rc= UdmSQLCommit(db);

  return rc;
}


static int
UdmUpdateClone(UDM_AGENT *Indexer,UDM_DOCUMENT *Doc,UDM_DB *db)
{
  int rc;
  int use_crosswords;
  use_crosswords = !strcasecmp(UdmVarListFindStr(&Indexer->Conf->Vars, "CrossWords", "no"), "yes");

  if (UDM_OK != (rc= UdmDeleteWordFromURL(Indexer, Doc, db)))
    return rc;
  if(use_crosswords)
  {
    if (UDM_OK != (rc= UdmDeleteCrossWordFromURL(Indexer, Doc, db)))
      return rc;
  }
  rc= UdmUpdateUrlWithLangAndCharset(Indexer, Doc, db);
  return rc;
}


static int
UdmDeleteWordsAndLinks(UDM_AGENT *Indexer, UDM_DOCUMENT *Doc, UDM_DB *db)
{
  int  rc;
  int use_crosswords;
  
  use_crosswords = !strcasecmp(UdmVarListFindStr(&Indexer->Conf->Vars, "CrossWords", "no"), "yes");
  if (use_crosswords)
    if (UDM_OK!= (rc= UdmDeleteCrossWordFromURL(Indexer,Doc,db)))
      return rc;

  if (UDM_OK != (rc= UdmDeleteWordFromURL(Indexer,Doc,db)))
    return rc;

  if (UDM_OK != (rc= UdmDeleteLinks(Indexer, Doc, db)))
    return rc;

  /* Set status, bad_since_time, etc */
  if (UDM_OK != (rc= UdmUpdateUrl(Indexer, Doc, db)))
    return rc;

  return rc;
}


static int UdmDeleteAllFromUrl(UDM_AGENT *Indexer,UDM_DB *db)
{
  int  rc;
  
  rc= UdmSQLTableTruncateOrDelete(db, "url");
  if(rc!=UDM_OK)return rc;
  
  rc= UdmSQLTableTruncateOrDelete(db, "links");
  if(rc != UDM_OK) return rc;
  
  rc= UdmSQLTableTruncateOrDelete(db, "urlinfo");
  return rc;
}



/************************ Clones stuff ***************************/
static int UdmFindOrigin(UDM_AGENT *Indexer, UDM_DOCUMENT *Doc,UDM_DB *db)
{
  size_t    i=0;
  char    qbuf[256]="";
  UDM_SQLRES  SQLRes;
  urlid_t    origin_id = 0;
  int    scrc32=UdmVarListFindInt(&Doc->Sections,"crc32",0);
  int    rc;
  
  if (scrc32==0)return UDM_OK;
  
  if (db->DBSQL_IN)
    sprintf(qbuf,"SELECT rec_id FROM url WHERE crc32=%d AND status IN (200,304,206)",scrc32);
  else
    sprintf(qbuf,"SELECT rec_id FROM url WHERE crc32=%d AND (status=200 OR status=304 OR status=206)",scrc32);
  
  if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLRes,qbuf)))
    return rc;
  
  for(i=0;i<UdmSQLNumRows(&SQLRes);i++)
  {
    const char *o;
    if((o=UdmSQLValue(&SQLRes,i,0)))
      if((!origin_id) || (origin_id > UDM_ATOI(o)))
        origin_id = UDM_ATOI(o);
  }
  UdmSQLFree(&SQLRes);
  UdmVarListReplaceInt(&Doc->Sections, "Origin-ID", origin_id);
  return(UDM_OK);
}


static void
UdmSQLTopClause(UDM_DB *db, size_t top_num,
                char *topstr, size_t topstr_size,
                char *rownumstr, size_t rownumstr_size,
                char *limitstr, size_t limitstr_size)
{
  *topstr= *rownumstr= *limitstr= '\0';
  if(db->flags & UDM_SQL_HAVE_LIMIT)
  {
    udm_snprintf(limitstr, limitstr_size, " LIMIT %d", top_num);
  }
  else if (db->flags & UDM_SQL_HAVE_TOP)
  {
    udm_snprintf(topstr, topstr_size, " TOP %d ", top_num);
  }
  else if (db->DBType == UDM_DB_ORACLE8)
  {
#if HAVE_ORACLE8
    if(db->DBDriver==UDM_DB_ORACLE8)
    {
      udm_snprintf(rownumstr, rownumstr_size, " AND ROWNUM<=%d", top_num); 
    }
#endif
    if(!rownumstr[0])
      udm_snprintf(rownumstr, rownumstr_size, " AND ROWNUM<=%d", top_num); 
  }
}

/************** Get Target to be indexed ***********************/

int UdmTargetsSQL(UDM_AGENT *Indexer, UDM_DB *db)
{
  char    sortstr[128]= "";
  char    updstr[64]="";
  char    rownumstr[64]="";
  char    limitstr[64]="";
  char    topstr[64]="";
  char    selectstr[]= "url.url,url.rec_id,docsize,status,hops,crc32,last_mod_time,seed";
  size_t    i = 0, j, start, nrows, qbuflen;
  size_t    url_num;
  UDM_SQLRES   SQLRes;
  char    smallbuf[128];
  int    rc=UDM_OK;
  const char  *where;
  char    *qbuf=NULL;
  const char      *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";

  url_num= UdmVarListFindInt(&Indexer->Conf->Vars, "URLSelectCacheSize", URL_SELECT_CACHE);
  if (Indexer->Conf->url_number < url_num)
    url_num= Indexer->Conf->url_number;
  where= BuildWhere(Indexer->Conf, db);
  qbuflen= 1024 + 4 * strlen(where);
  
  if ((qbuf = (char*)UdmMalloc(qbuflen + 2)) == NULL)
  {
      UdmLog(Indexer, UDM_LOG_ERROR, "Out of memory");
      return UDM_ERROR;
  }

  if ((Indexer->flags & (UDM_FLAG_SORT_HOPS | UDM_FLAG_SORT_EXPIRED)) ||
      !(Indexer->flags & UDM_FLAG_DONTSORT_SEED))
  {    
    sprintf(sortstr, " ORDER BY %s%s%s", 
      (Indexer->flags & UDM_FLAG_SORT_HOPS) ? "hops" : "",
      (Indexer->flags & UDM_FLAG_DONTSORT_SEED) ? "" : ((Indexer->flags & UDM_FLAG_SORT_HOPS) ? ",seed" : "seed"),
      (Indexer->flags & UDM_FLAG_SORT_EXPIRED) ? 
      ( ((Indexer->flags & UDM_FLAG_SORT_HOPS) || !(Indexer->flags & UDM_FLAG_DONTSORT_SEED)  ) ? 
        ",next_index_time" : "next_index_time") : "");
  }

  UdmSQLTopClause(db, url_num,
                  topstr, sizeof(topstr),
                  rownumstr, sizeof(rownumstr),
                  limitstr, sizeof(limitstr));

  if(1)
  {
    switch(db->DBType)
    {
      case UDM_DB_MYSQL:
        udm_snprintf(qbuf, qbuflen,
                     "INSERT INTO udm_url_tmp "
                     "SELECT url.rec_id FROM url%s "
                     "WHERE next_index_time<=%d %s%s%s%s",
                    db->from,
                    (int)time(NULL), where[0] ? "AND " : "",  where,
                    sortstr, limitstr);
        if (UDM_OK != (rc= UdmSQLQuery(db, NULL, "DROP TABLE IF EXISTS udm_url_tmp")) ||
            UDM_OK != (rc= UdmSQLQuery(db, NULL, "CREATE TEMPORARY TABLE udm_url_tmp (rec_id int not null)")) ||
            UDM_OK != (rc= UdmSQLQuery(db,NULL,"LOCK TABLES udm_url_tmp WRITE, url WRITE, urlinfo AS it WRITE, urlinfo AS il WRITE, server AS s WRITE, categories AS c WRITE")) ||
            UDM_OK != (rc= UdmSQLQuery(db, NULL, qbuf)))
          return rc;
        break;
      case UDM_DB_PGSQL:
        rc=UdmSQLQuery(db,NULL,"BEGIN WORK");
        sprintf(updstr, " FOR UPDATE ");
/*        rc=UdmSQLQuery(db,NULL,"LOCK url");*/
        break;
      case UDM_DB_ORACLE8:
        sprintf(updstr, " FOR UPDATE ");
        break;
      case UDM_DB_SAPDB:
        sprintf(updstr, " WITH LOCK ");
        break;
      default:
        break;
    }
    if(rc!=UDM_OK)return rc;
  }
  
  db->res_limit=url_num;
  if (db->DBType == UDM_DB_MYSQL)
    udm_snprintf(qbuf, qbuflen, "SELECT %s FROM url, udm_url_tmp "
                                "WHERE url.rec_id=udm_url_tmp.rec_id", selectstr);
  else
    udm_snprintf(qbuf, qbuflen, "SELECT %s%s "
                                "FROM url%s "
                                "WHERE next_index_time<=%d %s%s%s"
                                "%s%s%s",
                 topstr, selectstr, db->from,
                 (int)time(NULL), where[0] ? "AND " : "",  where, rownumstr,
                 sortstr, updstr, limitstr);
  
  if(UDM_OK != (rc= UdmSQLQuery(db,&SQLRes, qbuf)))
    goto commit;
  
  if(!(nrows = UdmSQLNumRows(&SQLRes)))
  {
    UdmSQLFree(&SQLRes);
    goto commit;
  }

  start = Indexer->Conf->Targets.num_rows;
  Indexer->Conf->Targets.num_rows += nrows;
  
  Indexer->Conf->Targets.Doc = 
    (UDM_DOCUMENT*)UdmRealloc(Indexer->Conf->Targets.Doc, sizeof(UDM_DOCUMENT)*(Indexer->Conf->Targets.num_rows + 1));
  if (Indexer->Conf->Targets.Doc == NULL)
  {
    UdmLog(Indexer, UDM_LOG_ERROR, "Out of memory at realloc %s[%d]", __FILE__, __LINE__);
    rc= UDM_ERROR;
    goto commit;
  }
  
  for(i = 0; i < nrows; i++)
  {
    char    buf[64]="";
    time_t    last_mod_time;
    UDM_DOCUMENT  *Doc = &Indexer->Conf->Targets.Doc[start + i];
    
    UdmDocInit(Doc);
    UdmVarListAddStr(&Doc->Sections,"URL",UdmSQLValue(&SQLRes,i,0));
    UdmVarListReplaceInt(&Doc->Sections, "URL_ID", UdmStrHash32(UdmSQLValue(&SQLRes,i,0)));
    UdmVarListAddInt(&Doc->Sections, "ID", UDM_ATOI(UdmSQLValue(&SQLRes,i,1)));
    UdmVarListAddInt(&Doc->Sections,"Content-Length",atoi(UdmSQLValue(&SQLRes,i,2)));
    UdmVarListAddInt(&Doc->Sections,"Status",atoi(UdmSQLValue(&SQLRes,i,3)));
    UdmVarListAddInt(&Doc->Sections,"PrevStatus",atoi(UdmSQLValue(&SQLRes,i,3)));
    UdmVarListAddInt(&Doc->Sections,"Hops",atoi(UdmSQLValue(&SQLRes,i,4)));
    UdmVarListAddInt(&Doc->Sections,"crc32",atoi(UdmSQLValue(&SQLRes,i,5)));
    last_mod_time = (time_t) atol(UdmSQLValue(&SQLRes,i,6));
    UdmTime_t2HttpStr(last_mod_time, buf);
    if (last_mod_time != 0 && strlen(buf) > 0)
    {
      UdmVarListReplaceStr(&Doc->Sections,"Last-Modified",buf);
    }
  }
  UdmSQLFree(&SQLRes);
  
  
  if (db->DBSQL_IN)
  {
    char  *urlin=NULL;
    
    if ( (qbuf = (char*)UdmRealloc(qbuf, qbuflen = qbuflen + 35 * URL_SELECT_CACHE)) == NULL)
    {
      UDM_FREE(qbuf);
      UdmLog(Indexer, UDM_LOG_ERROR, "Out of memory");
      rc= UDM_ERROR;
      goto commit;
    }
    
    if ( (urlin = (char*)UdmMalloc(35 * URL_SELECT_CACHE)) == NULL)
    {
      UDM_FREE(qbuf);
      UdmLog(Indexer, UDM_LOG_ERROR, "Out of memory");
      rc = UDM_ERROR;
      goto commit;
    }
    urlin[0]=0;
    
    for(i = 0; i < nrows; i+= URL_SELECT_CACHE)
    {

      urlin[0] = 0;

      for (j = 0; (j < URL_SELECT_CACHE) && (i + j < nrows) ; j++)
      {

      UDM_DOCUMENT  *Doc = &Indexer->Conf->Targets.Doc[start + i + j];
      urlid_t    url_id = UdmVarListFindInt(&Doc->Sections, "ID", 0);
      
      if(urlin[0])strcat(urlin,",");
      sprintf(urlin+strlen(urlin), "%s%i%s", qu, url_id, qu);
      }
      udm_snprintf(qbuf, qbuflen, "UPDATE url SET next_index_time=%d WHERE rec_id in (%s)",
             (int)(time(NULL) + URL_LOCK_TIME), urlin);
      if (UDM_OK != (rc= UdmSQLQuery(db,NULL,qbuf)))
        goto commit;
    }
    UDM_FREE(urlin);
  }
  else
  {
    for(i = 0; i < nrows; i++)
    {
      UDM_DOCUMENT  *Doc = &Indexer->Conf->Targets.Doc[start + i];
      urlid_t    url_id = UdmVarListFindInt(&Doc->Sections, "ID", 0);
      
      udm_snprintf(smallbuf, 128, "UPDATE url SET next_index_time=%d WHERE rec_id=%i",
             (int)(time(NULL) + URL_LOCK_TIME), url_id);
      if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,smallbuf)))
        goto commit;
    }
  }


commit:

  if (rc != UDM_OK)
  {
    UdmLog(Indexer, UDM_LOG_ERROR, "UdmTargetsSQL: DB error: %s", db->errstr);
  }
  if(1)
  {
    switch(db->DBType)
    {
      case UDM_DB_MYSQL:
        rc=UdmSQLQuery(db,NULL,"UNLOCK TABLES");
        break;
      case UDM_DB_PGSQL:
        rc=UdmSQLQuery(db,NULL,"END WORK");
        break;
      default:
        break;
    }
  }
  UDM_FREE(qbuf);
  return rc;
}


/************************************************************/
/* Misc functions                                           */
/************************************************************/

static int UdmGetDocCount(UDM_AGENT * Indexer, UDM_DOCUMENT *Doc, UDM_DB *db)
{
  char    qbuf[200]="";
  UDM_SQLRES  SQLres;
  int    rc;
  
  sprintf(qbuf,NDOCS_QUERY);
  if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLres,qbuf)))return rc;
  
  if(UdmSQLNumRows(&SQLres))
  {
    const char * s;
    s=UdmSQLValue(&SQLres,0,0);
    if(s)Indexer->doccount += atoi(s);
  }
  UdmSQLFree(&SQLres);
  return(UDM_OK);
}


int UdmStatActionSQL(UDM_AGENT *Indexer,UDM_STATLIST *Stats,UDM_DB *db)
{
  size_t    i,j,n;
  char    qbuf[2048];
  UDM_SQLRES  SQLres;
  int    have_group= (db->flags & UDM_SQL_HAVE_GROUPBY);
  const char  *where;
  int    rc=UDM_OK;
  const char      *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  
  if(db->DBType==UDM_DB_IBASE)
    have_group=0;

  UDM_LOCK_CHECK_OWNER(Indexer, UDM_LOCK_CONF);
  where = BuildWhere(Indexer->Conf, db);

  if(have_group)
  {
    char func[128];
    
    switch(db->DBType)
    {
      case UDM_DB_MYSQL:
        udm_snprintf(func,sizeof(func)-1, "next_index_time<=%d", Stats->time);
        break;
    
      case UDM_DB_PGSQL:
      case UDM_DB_MSSQL:
      case UDM_DB_SYBASE:
      case UDM_DB_DB2:
      case UDM_DB_SQLITE:
      case UDM_DB_SQLITE3:
      default:
        udm_snprintf(func,sizeof(func)-1, "case when next_index_time<=%d then 1 else 0 end", Stats->time);
        break;

      case UDM_DB_ACCESS:
        udm_snprintf(func,sizeof(func)-1, "IIF(next_index_time<=%d, 1, 0)", Stats->time);
        break;

      case UDM_DB_ORACLE8:
      case UDM_DB_SAPDB:
        udm_snprintf(func,sizeof(func)-1, "DECODE(SIGN(%d-next_index_time),-1,0,1,1)", Stats->time);
        break;
    }

    udm_snprintf(qbuf, sizeof(qbuf) - 1,
                 "SELECT status, SUM(%s), count(*) FROM url%s %s%s GROUP BY status ORDER BY status",
                 func, db->from, where[0] ? "WHERE " : "", where);

    if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLres,qbuf)))return rc;
    
    if((n = UdmSQLNumRows(&SQLres)))
    {
      for(i = 0; i < n; i++)
      {
        for(j=0;j<Stats->nstats;j++)
        {
          if(Stats->Stat[j].status==atoi(UdmSQLValue(&SQLres,i,0)))
          {
            Stats->Stat[j].expired += atoi(UdmSQLValue(&SQLres,i,1));
            Stats->Stat[j].total += atoi(UdmSQLValue(&SQLres,i,2));
            break;
          }
        }
        if(j==Stats->nstats)
        {
          UDM_STAT  *S;
        
          Stats->Stat=(UDM_STAT*)UdmRealloc(Stats->Stat,(Stats->nstats+1)*sizeof(Stats->Stat[0]));
          S=&Stats->Stat[Stats->nstats];
          S->status=atoi(UdmSQLValue(&SQLres,i,0));
          S->expired=atoi(UdmSQLValue(&SQLres,i,1));
          S->total=atoi(UdmSQLValue(&SQLres,i,2));
          Stats->nstats++;
        }
      }
    }
    UdmSQLFree(&SQLres);
    
  }else{
/*  
  FIXME: learn how to get it from SOLID and IBASE
  (HAVE_IBASE || HAVE_SOLID || HAVE_VIRT )
*/
    
    udm_snprintf(qbuf,sizeof(qbuf)-1,"SELECT status,next_index_time FROM url%s %s%s ORDER BY status",
      db->from, where[0] ? "WHERE " : "", where);

    if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLres,qbuf)))return rc;
    
    for(i=0;i<UdmSQLNumRows(&SQLres);i++)
    {
      for(j=0;j<Stats->nstats;j++)
      {
        if(Stats->Stat[j].status==atoi(UdmSQLValue(&SQLres,i,0)))
        {
          if((time_t)UDM_ATOU(UdmSQLValue(&SQLres,i,1)) <= Stats->time)
            Stats->Stat[j].expired++;
          Stats->Stat[j].total++;
          break;
        }
      }
      if(j==Stats->nstats)
      {
        Stats->Stat=(UDM_STAT *)UdmRealloc(Stats->Stat,sizeof(UDM_STAT)*(Stats->nstats+1));
        Stats->Stat[j].status = UDM_ATOI(UdmSQLValue(&SQLres,i,0));
        Stats->Stat[j].expired=0;
        if((time_t)UDM_ATOU(UdmSQLValue(&SQLres,i,1)) <= Stats->time)
          Stats->Stat[j].expired++;
        Stats->Stat[j].total=1;
        Stats->nstats++;
      }
    }
    UdmSQLFree(&SQLres);
  }
  return rc;
}


static int UdmGetReferers(UDM_AGENT *Indexer, UDM_DOCUMENT *Doc, UDM_DB *db)
{
  size_t    i,j;
  char    qbuf[2048];
  UDM_SQLRES  SQLres;
  const char  *where;
  int    rc;

  UDM_LOCK_CHECK_OWNER(Indexer, UDM_LOCK_CONF);
  where = BuildWhere(Indexer->Conf, db);
  
  udm_snprintf(qbuf,sizeof(qbuf),"SELECT url.status,url2.url,url.url FROM url,url url2%s WHERE url.referrer=url2.rec_id %s %s",
    db->from, where[0] ? "AND" : "", where);
  
  if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLres,qbuf)))return rc;
  
  j=UdmSQLNumRows(&SQLres);
  for(i=0;i<j;i++)
  {
    if(Indexer->Conf->RefInfo)Indexer->Conf->RefInfo(
      atoi(UdmSQLValue(&SQLres,i,0)),
      UdmSQLValue(&SQLres,i,2),
      UdmSQLValue(&SQLres,i,1)
    );
  }
  UdmSQLFree(&SQLres);
  return rc;
}


static int
UdmClearDBUsingIN(UDM_AGENT *Indexer, UDM_DB *db, UDM_URL_TMP *list)
{
  UDM_DSTR qbuf, urlin;
  int rc= UDM_OK; /* if list if empty */
  size_t part;
  size_t url_num = UdmVarListFindInt(&Indexer->Conf->Vars, "URLSelectCacheSize", URL_DELETE_CACHE);
  
  UdmDSTRInit(&qbuf, 4096);
  UdmDSTRInit(&urlin, 4096);
  
  for (part= 0; part < list->nurls; part+= url_num)
  {
    size_t offs;
    urlid_t *item= &list->urls[part];
    UdmDSTRReset(&urlin);
    for(offs= 0; (offs < url_num) && ((part + offs) < list->nurls); offs++)
    {
      if(offs) UdmDSTRAppend(&urlin,",", 1);
      UdmDSTRAppendf(&urlin, "%d", item[offs]);
    }
    
    if(UDM_OK != (rc= UdmSQLBegin(db)))
      goto ret;
    
    switch(db->DBMode)
    {
      case UDM_DBMODE_BLOB:
        UdmDSTRReset(&qbuf);
        UdmDSTRAppendf(&qbuf,
                       "DELETE FROM bdicti WHERE state=1 AND url_id IN (%s)",
                       urlin.data);
        if (UDM_OK != (rc= UdmSQLQuery(db, NULL, qbuf.data)))
          goto ret;

        UdmDSTRReset(&qbuf);
        UdmDSTRAppendf(&qbuf,
                       "UPDATE bdicti SET state=0 WHERE state=2 AND url_id IN (%s)",
                       urlin.data);
        if (UDM_OK != (rc= UdmSQLQuery(db, NULL, qbuf.data)))
          goto ret;
        break;
      
      case UDM_DBMODE_MULTI:
        {
          size_t dictnum;
          for (dictnum= 0; dictnum <= MULTI_DICTS; dictnum++)
          {
            UdmDSTRReset(&qbuf);
            UdmDSTRAppendf(&qbuf,"DELETE FROM dict%02X WHERE url_id in (%s)",
                           dictnum, urlin.data);
            if(UDM_OK != (rc= UdmSQLQuery(db, NULL, qbuf.data)))
              goto ret;
          }
        }
        break;
        
      default:
        UdmDSTRReset(&qbuf);
        UdmDSTRAppendf(&qbuf,"DELETE FROM dict WHERE url_id in (%s)",urlin.data);
        if (UDM_OK != (rc=UdmSQLQuery(db,NULL,qbuf.data)))
          goto ret;
        break;
    }
    
    UdmDSTRReset(&qbuf);
    UdmDSTRAppendf(&qbuf,"DELETE FROM url WHERE rec_id in (%s)",urlin.data);
    if (UDM_OK != (rc= UdmSQLQuery(db,NULL,qbuf.data)))
      goto ret;
    
    UdmDSTRReset(&qbuf);
    UdmDSTRAppendf(&qbuf,"DELETE FROM urlinfo WHERE url_id in (%s)",urlin.data);
    if (UDM_OK != (rc= UdmSQLQuery(db,NULL,qbuf.data)))
      goto ret;
    
    UdmDSTRReset(&qbuf);
    UdmDSTRAppendf(&qbuf,"DELETE FROM links WHERE ot in (%s)",urlin.data);
    if (UDM_OK != (rc= UdmSQLQuery(db,NULL,qbuf.data)))
      goto ret;
    
    UdmDSTRReset(&qbuf);
    UdmDSTRAppendf(&qbuf,"DELETE FROM links WHERE k in (%s)",urlin.data);
    if (UDM_OK != (rc= UdmSQLQuery(db,NULL,qbuf.data)))
      goto ret;

    if(UDM_OK != (rc= UdmSQLCommit(db)))
      goto ret;
  }
  
ret:
  UdmDSTRFree(&qbuf);
  UdmDSTRFree(&urlin);
  return rc;
}


int UdmClearDBSQL(UDM_AGENT *Indexer,UDM_DB *db)
{
  int rc, use_crosswords;
  const char *where, *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  char ClearDBHook[128];

  UDM_GETLOCK(Indexer, UDM_LOCK_CONF);
  where = BuildWhere(Indexer->Conf, db);
  use_crosswords = !strcasecmp(UdmVarListFindStr(&Indexer->Conf->Vars, "CrossWords", "no"), "yes");
  udm_snprintf(ClearDBHook, sizeof(ClearDBHook),
               UdmVarListFindStr(&Indexer->Conf->Vars, "SQLClearDBHook", ""));
  UDM_RELEASELOCK(Indexer, UDM_LOCK_CONF);
  
  if (ClearDBHook[0] && (UDM_OK != (rc= UdmSQLQuery(db, NULL, ClearDBHook))))
    return rc;
  
  if(!where[0])
  {
    if(use_crosswords)
    {
      if((UDM_OK!=(rc=UdmDeleteAllFromCrossDict(Indexer,db))))return(rc);
    }
    if((UDM_OK!=(rc=UdmDeleteAllFromDict(Indexer,db))))return(rc);
    if((UDM_OK!=(rc=UdmDeleteAllFromUrl(Indexer,db))))return(rc);
  }
  else
  {
    UDM_URL_TMP urllist;
    UDM_DSTR qbuf;
    UdmDSTRInit(&qbuf, 4096);
    
    bzero((void*) &urllist, sizeof(urllist));
    UdmDSTRAppendf(&qbuf,"SELECT url.rec_id, url.url FROM url%s WHERE url.rec_id<>%s0%s AND %s", 
                   db->from, qu, qu,  where);
    
    if (UDM_OK != (rc= UdmLoadSlowLimit(db, &urllist, qbuf.data)))
      goto fin;
    
    if (db->DBSQL_IN)
    {
      rc= UdmClearDBUsingIN(Indexer, db, &urllist);
    }
    else
    {
      size_t i;
      UDM_DOCUMENT Doc;
      bzero((void*)&Doc, sizeof(Doc));
      
      for(i=0; i < urllist.nurls; i++)
      {
        UdmVarListReplaceInt(&Doc.Sections, "ID", urllist.urls[i]);
        if(UDM_OK != (rc= UdmDeleteURL(Indexer, &Doc, db)))
         break;
      }
      UdmDocFree(&Doc);
    }
fin:
    UdmFree(urllist.urls);
    UdmDSTRFree(&qbuf);
  }
  return rc;
}


#define MAXHSIZE	1023*4	/* TUNE */

int
UdmStoreHrefsSQL(UDM_AGENT *Indexer)
{
  size_t i;
  int rc= UDM_OK;
  UDM_DOCUMENT  Doc;
  UDM_HREFLIST *Hrefs;
  UDM_DB *db= Indexer->Conf->dbl.nitems == 1 ?
              &Indexer->Conf->dbl.db[0] : (UDM_DB*) NULL;

  if (db && !(db->flags & UDM_SQL_HAVE_GOOD_COMMIT))
    db= NULL;
  
  UDM_LOCK_CHECK_OWNER(Indexer, UDM_LOCK_CONF);
 
  UdmDocInit(&Doc);

  Hrefs= &Indexer->Conf->Hrefs;

  if (db && UDM_OK != UdmSQLBegin(db))
    goto ret;
  
  for (i= 0; i < Hrefs->nhrefs; i++)
  {  
    UDM_HREF *H = &Hrefs->Href[i];
    if (!H->stored)
    {
      UdmVarListAddLst(&Doc.Sections, &H->Vars, NULL, "*");
      UdmVarListReplaceInt(&Doc.Sections, "Referrer-ID", H->referrer);
      UdmVarListReplaceUnsigned(&Doc.Sections,"Hops", H->hops);
      UdmVarListReplaceStr(&Doc.Sections,"URL",H->url?H->url:"");
      UdmVarListReplaceInt(&Doc.Sections, "URL_ID", UdmStrHash32(H->url ? H->url : ""));
      UdmVarListReplaceInt(&Doc.Sections,"Site_id", H->site_id);
      UdmVarListReplaceInt(&Doc.Sections,"Server_id", H->server_id);
      UdmVarListReplaceInt(&Doc.Sections, "HTDB_URL_ID", H->rec_id);
      if (i >= Hrefs->dhrefs)
      {
        if(UDM_OK != (rc= UdmURLActionNoLock(Indexer, &Doc, UDM_URL_ACTION_ADD)))
          goto ret;
      }
      if (H->collect_links)
      {
        if(UDM_OK != (rc= UdmURLActionNoLock(Indexer, &Doc, UDM_URL_ACTION_ADD_LINK)))
          goto ret;
      }
      
      UdmVarListFree(&Doc.Sections);
      H->stored= 1;
    }
  }

  if (db && UDM_OK != UdmSQLCommit(db))
    goto ret;

ret:

  UdmDocFree(&Doc);
  
  /*
    Remember last stored URL num
    Note that it will became 0
    after next sort in AddUrl   
  */
  Hrefs->dhrefs = Hrefs->nhrefs;
  
  /*
    We should not free URL list with onw database
    to avoid double indexing of the same document
    So, do it if compiled with SQL only          
  */
  
  /* FIXME: this is incorrect with both SQL and built-in compiled */
  if(Hrefs->nhrefs > MAXHSIZE)
    UdmHrefListFree(&Indexer->Conf->Hrefs);

  return rc;
}

/********************* Categories ************************************/
static int UdmCatList(UDM_AGENT * Indexer,UDM_CATEGORY *Cat,UDM_DB *db)
{
  size_t    i, rows;
  char    qbuf[1024];
  UDM_SQLRES  SQLres;
  int    rc;
  
  if(db->DBType==UDM_DB_SAPDB)
  {
    udm_snprintf(qbuf,sizeof(qbuf)-1,"SELECT rec_id,path,lnk,name FROM categories WHERE path LIKE '%s__'",Cat->addr);
  }else{
    udm_snprintf(qbuf,sizeof(qbuf)-1,"SELECT rec_id,path,link,name FROM categories WHERE path LIKE '%s__'",Cat->addr);
  }
  
  if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLres,qbuf)))
    return rc;
  
  if( (rows = UdmSQLNumRows(&SQLres)) )
  {
    size_t nbytes;
    
    nbytes = sizeof(UDM_CATITEM) * (rows + Cat->ncategories);
    Cat->Category=(UDM_CATITEM*)UdmRealloc(Cat->Category,nbytes);
    for(i=0;i<rows;i++)
    {
      UDM_CATITEM *r = &Cat->Category[Cat->ncategories];
      r[i].rec_id=atoi(UdmSQLValue(&SQLres,i,0));
      strcpy(r[i].path,UdmSQLValue(&SQLres,i,1));
      strcpy(r[i].link,UdmSQLValue(&SQLres,i,2));
      strcpy(r[i].name,UdmSQLValue(&SQLres,i,3));
    }
    Cat->ncategories+=rows;
  }
  UdmSQLFree(&SQLres);
  return UDM_OK;
}

static int UdmCatPath(UDM_AGENT *Indexer,UDM_CATEGORY *Cat,UDM_DB *db)
{
  size_t    i,l;
  char    qbuf[1024];
  int    rc;
  char            *head = NULL;
  
  l=(strlen(Cat->addr)/2)+1;
  Cat->Category=(UDM_CATITEM*)UdmRealloc(Cat->Category,sizeof(UDM_CATITEM)*(l+Cat->ncategories));
  head = (char *)UdmMalloc(2 * l + 1);

  if (head != NULL)
  {
    UDM_CATITEM  *r = &Cat->Category[Cat->ncategories];

    for(i = 0; i < l; i++)
    {
      UDM_SQLRES  SQLres;

      strncpy(head,Cat->addr,i*2);head[i*2]=0;

      if(db->DBType==UDM_DB_SAPDB)
      {
        udm_snprintf(qbuf,sizeof(qbuf)-1,"SELECT rec_id,path,lnk,name FROM categories WHERE path='%s'",head);
      }
      else
      {
        udm_snprintf(qbuf,sizeof(qbuf)-1,"SELECT rec_id,path,link,name FROM categories WHERE path='%s'",head);
      }
    
      if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLres,qbuf)))
        return rc;
    
      if(UdmSQLNumRows(&SQLres))
      {
        r[i].rec_id=atoi(UdmSQLValue(&SQLres,0,0));
        strcpy(r[i].path,UdmSQLValue(&SQLres,0,1));
        strcpy(r[i].link,UdmSQLValue(&SQLres,0,2));
        strcpy(r[i].name,UdmSQLValue(&SQLres,0,3));
        Cat->ncategories++;
      }
      UdmSQLFree(&SQLres);
    }
    UDM_FREE(head);
  }
  return UDM_OK;
}


/******************* Search stuff ************************************/

int UdmCloneListSQL(UDM_AGENT * Indexer, UDM_DOCUMENT *Doc, UDM_RESULT *Res, UDM_DB *db)
{
  size_t    i, nr, nadd;
  char    qbuf[256];
  UDM_SQLRES  SQLres;
  int    scrc32=UdmVarListFindInt(&Doc->Sections,"crc32",0);
  urlid_t    origin_id = UdmVarListFindInt(&Doc->Sections, "ID", 0);
  int    rc;
  const char      *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  const char  *format = UdmVarListFindStr(&Indexer->Conf->Vars, "DateFormat", "%a, %d %b %Y, %X %Z");

  if (Res->num_rows > 4) return UDM_OK;
  
  if (!scrc32)
    return UDM_OK;
  
  sprintf(qbuf,"SELECT rec_id,url,last_mod_time,docsize FROM url WHERE crc32=%d AND (status=200 OR status=304 OR status=206) AND rec_id<>%s%i%s", scrc32, qu, origin_id, qu);
  if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLres,qbuf)))
    return UDM_OK;
  
  nr = UdmSQLNumRows(&SQLres);
  if( nr == 0)
  {
    UdmSQLFree(&SQLres);
    return UDM_OK;
  }
  nadd = 5 - Res->num_rows;
  if(nr < nadd) nadd = nr;
  
  Res->Doc = (UDM_DOCUMENT*)UdmRealloc(Res->Doc, (Res->num_rows + nadd) * sizeof(UDM_DOCUMENT));
  
  for(i = 0; i < nadd; i++)
  {
    time_t    last_mod_time;
    char    buf[128];
    UDM_DOCUMENT  *D = &Res->Doc[Res->num_rows + i];
    
    UdmDocInit(D);
    UdmVarListAddInt(&D->Sections, "ID", UDM_ATOI(UdmSQLValue(&SQLres,i,0)));
    UdmVarListAddStr(&D->Sections,"URL",UdmSQLValue(&SQLres,i,1));
    UdmVarListReplaceInt(&D->Sections, "URL_ID", UdmStrHash32(UdmSQLValue(&SQLres,i,1)));
    last_mod_time=atol(UdmSQLValue(&SQLres,i,2));
    if (strftime(buf, 128, format, localtime(&last_mod_time)) == 0)
    {
      UdmTime_t2HttpStr(last_mod_time, buf);
    }
    UdmVarListAddStr(&D->Sections,"Last-Modified",buf);
    UdmVarListAddInt(&D->Sections,"Content-Length",atoi(UdmSQLValue(&SQLres,i,3)));
    UdmVarListAddInt(&D->Sections,"crc32",scrc32);
    UdmVarListAddInt(&D->Sections, "Origin-ID", origin_id);
  }
  Res->num_rows += nadd;
  UdmSQLFree(&SQLres);
  return UDM_OK;
}

/********************************************************/

static const char * UdmBlobModeInflateOrSelf(UDM_AGENT *A,
                                            UDM_DSTR *buf, const char *name,
                                            const char *src, size_t *len)
{
  int use_zint4;
  int use_deflate;
  /* fprintf(stderr, "here, len=%d src=%p\n", *len, src); */
  if (!src || *len < 8 ||
      (unsigned char)src[0] != 0xFF ||
      (unsigned char)src[1] != 0xFF ||
      (unsigned char)src[2] != 0xFF ||
      (unsigned char)src[3] != 0xFF ||
      (src[4] != 1 && src[4] != 2 && src[4] != 3) ||
      src[5] != 0x00 ||
      src[6] != 0x00 ||
      src[7] != 0x00)
    return src;
  use_zint4= src[4] == 2 || src[4] == 3;
  use_deflate= src[4] == 1 || src[4] == 3;
  src+= 8;
  *len-= 8;
  if (name)
    UdmLog(A, UDM_LOG_DEBUG, "Unpacking '%s'", name);
  if (use_deflate)
  {
    unsigned long ticks= UdmStartTimer();
    size_t i, mul[4]= {10, 100, 1000, 10000};
    size_t len0= *len;
    UdmLog(A,UDM_LOG_DEBUG, "Deflate header detected");
#ifdef HAVE_ZLIB
    for (i= 0; i < 4; i++)
    {
      size_t reslen, nbytes= len[0] * mul[i];
      /* fprintf(stderr, "allocating %d bytes\n", nbytes); */
      UdmDSTRRealloc(buf, nbytes);
      reslen= UdmInflate(buf->data, buf->size_total, src, *len);
      /* fprintf(stderr, "reslen=%d site_total=%d\n", reslen, buf->size_total);*/
      if (reslen < buf->size_total)
      {
        src= buf->data;
        len[0]= reslen;
        UdmLog(A,UDM_LOG_DEBUG, "%d to %d bytes inflated", len0, reslen);
        break;
      }
    }
    ticks= UdmStartTimer() - ticks;
    UdmLog(A, UDM_LOG_DEBUG, "Inflating done: %.2f", (float) ticks/1000);
#endif
  }
  if (*len >= 5 && use_zint4)
  {
    unsigned long ticks= UdmStartTimer();
    char *zint4_buf= UdmMalloc(*len);
    UdmLog(A, UDM_LOG_DEBUG, "zint4 header detected (zint4ed data length: %d)", *len);
    if (! zint4_buf)
    {
      UdmLog(A, UDM_LOG_ERROR, "Malloc failed. Requested %u bytes", *len);
      return(NULL);
    }
    memcpy(zint4_buf, src, *len);
    if (buf->size_total < *len * 7 && UdmDSTRRealloc(buf, *len * 7) != UDM_OK)
    {
      UdmFree(zint4_buf);
      UdmLog(A, UDM_LOG_ERROR, "UdmDSTRRealloc failed. Requested %u bytes",
             *len * 7);
      return(NULL);
    }
    *len= udm_dezint4(zint4_buf, (int4 *)buf->data, *len) * 4;
    src= buf->data;
    UdmFree(zint4_buf);
    ticks= UdmStartTimer() - ticks;
    UdmLog(A, UDM_LOG_ERROR, "dezint4ed data length: %d", *len);
    UdmLog(A, UDM_LOG_ERROR, "dezint4 done: %.2f", (float) ticks/1000);
  }
  return src;
}


static const char *UdmBlobModeInflateOrAlloc(UDM_AGENT *A,
                                             UDM_DSTR *buf, const char *name,
                                             const char *src, size_t *len)
{
  const char *res= UdmBlobModeInflateOrSelf(A, buf, name, src, len);
  if (res == src)
  {
    UdmDSTRRealloc(buf, *len);
    memcpy(buf->data, src, *len);
    buf->size_data= *len;
    res= buf->data;
  }
  return res;
}


typedef struct udm_hash_st
{
  void *base;
  size_t nmemb;
  size_t size;
  size_t (*key)(const void *);
  int (*join)(void *, void *);
} UDM_HASH;

typedef int (*hash_join)(void*, void*);

static void
UdmHashInit(UDM_HASH *hash, void* base, size_t nmemb, size_t size,
            size_t(*key)(const void*),
            int(*join)(void *, void *))
{
  hash->base= base;
  hash->nmemb= nmemb;
  hash->size= size;
  hash->key= key;
  hash->join= join;
}

static size_t
UdmHashSize(size_t n)
{
  return (n + n/10 + 10);
}



static void
UdmHashPut(UDM_HASH *hash, void *ptr)
{
  size_t key= hash->key(ptr);
  size_t pos= key % hash->nmemb;
  size_t ncoll;

  for (ncoll= 0 ; ncoll < hash->nmemb ; pos= (pos + 1) % hash->nmemb, ncoll++)
  {
    void *addr= ((char*)hash->base) + pos * hash->size;
    size_t hkey= hash->key(addr);
    if (hkey == 0)
    {
      memcpy(addr, ptr, hash->size);
      return;
    }
    if (hkey == key)
    {
      hash->join(addr, ptr);
      return;
    }
  }
}


static size_t
test_key(const void *ptr)
{
  size_t key= (size_t) ((const UDM_URLDATA*)ptr)->site_id;
  return key;
}


static int
test_join(UDM_URLDATA *dst, UDM_URLDATA *src)
{
  dst->per_site+= src->per_site;
  if (dst->coord > src->coord)
  {
    return UDM_OK;
  }
  else if (dst->coord == src->coord)
  {
    if (dst->pop_rank > src->pop_rank)
    {
      return UDM_OK;
    }
    else if (dst->pop_rank == src->pop_rank)
    {
      if (dst->url_id < src->url_id)
        return UDM_OK;
    }
  }
  dst->url_id=        src->url_id;
  dst->coord=         src->coord;
  dst->last_mod_time= src->last_mod_time;
  dst->pop_rank=      src->pop_rank;
  dst->url=           src->url;
  dst->section=       src->section;
  return UDM_OK;
}


/*
  Skip empty records and return the number of non-empty records
*/
static size_t
UdmURLDATACompact(UDM_URLDATA *dst, UDM_URLDATA *src, size_t n)
{
  UDM_URLDATA *dst0= dst, *srcend= src + n;
  for ( ; src < srcend ; src++)
  {
    if (src->site_id)
    {
      *dst++= *src;
    }
  }
  return dst - dst0;
}


#if 0
static void
UdmGroupBySiteHash(UDM_AGENT *A, UDM_URLCRDLIST *CoordList)
{
  size_t ncoords= CoordList->ncoords;
  size_t hcoords= UdmHashSize(ncoords);
  size_t hbytes= hcoords * sizeof(UDM_URLDATA);
  UDM_HASH Hash;
  UDM_URLDATA *Data= CoordList->Data, *DataEnd= Data + ncoords, *DataCur;
  UDM_URLDATA *Base= UdmMalloc(hbytes);
  unsigned long ticks;
  
  UdmHashInit(&Hash, Base, hcoords, sizeof(UDM_URLDATA),
              test_key, (hash_join) test_join);
  ticks= UdmStartTimer();
  for (DataCur= Base; DataCur < Base + hcoords; DataCur->site_id= 0, DataCur++);
  ticks= UdmStartTimer() - ticks;
  UdmLog(A, UDM_LOG_DEBUG, "Hash init done: %.2f", (float) ticks/ 1000);
  
  ticks= UdmStartTimer();
  for (DataCur= Data ; DataCur < DataEnd; DataCur++)
  {
    UdmHashPut(&Hash, DataCur);
  }
  ticks= UdmStartTimer() - ticks;
  UdmLog(A, UDM_LOG_DEBUG, "HashPut done: %.2f", (float) ticks / 1000);
  
  ticks= UdmStartTimer();
  CoordList->ncoords= UdmURLDATACompact(Data, Base, hcoords);
  ticks= UdmStartTimer() - ticks;
  UdmLog(A, UDM_LOG_DEBUG, "HashCompact done: %.2f", (float) ticks / 1000);

  UdmFree(Base);
}
#endif


static void
UdmResGroupBySite(UDM_AGENT *A, UDM_URLDATALIST *R)
{
  unsigned long ticks;
  UDM_URLDATA *Dat, *End;

  for (Dat= R->Item, End= R->Item + R->nitems;
       Dat < End; Dat++)
    Dat->per_site= 1;

#if 0
  UdmLog(A,UDM_LOG_DEBUG,"Start hash group by site_id %d words",R->CoordList.ncoords);
  ticks=UdmStartTimer();
  UdmGroupBySiteHash(A, &R->CoordList);
  ticks=UdmStartTimer()-ticks;
  UdmLog(A,UDM_LOG_DEBUG,"Stop hash group by site_id:\t%.2f",(float)ticks/1000);
#else

#if WRONG
  /* Cannot Group here, need UdmCopyCoorToData to execute first */
  UdmLog(A,UDM_LOG_DEBUG,"Start unsorted group by site_id %d words",R->CoordList.ncoords);
  ticks=UdmStartTimer();
  UdmGroupBySite(A, R);
  ticks=UdmStartTimer()-ticks;
  UdmLog(A,UDM_LOG_DEBUG,"Stop unsorted group by site_id:\t%.2f",(float)ticks/1000);
#endif

  UdmLog(A,UDM_LOG_DEBUG,"Start sort by site_id %d words", R->nitems);
  ticks=UdmStartTimer();
  UdmURLDataSortBySite(R);
  ticks=UdmStartTimer()-ticks;
  UdmLog(A,UDM_LOG_DEBUG,"Stop sort by site_id:\t%.2f",(float)ticks/1000);
  
  UdmLog(A,UDM_LOG_DEBUG,"Start group by site_id %d docs", R->nitems);
  ticks=UdmStartTimer();
  UdmURLDataGroupBySite(R);
  ticks=UdmStartTimer() - ticks;
  UdmLog(A,UDM_LOG_DEBUG,"Stop group by site_id:\t%.2f", (float)ticks/1000);
#endif
}


#define UDM_URLDATA_URL	  1
#define UDM_URLDATA_SITE  2
#define UDM_URLDATA_POP   4
#define UDM_URLDATA_LM    8 
#define UDM_URLDATA_SU   16
#define UDM_URLDATA_CACHE 32


static int
UdmLoadURLDataFromURL(UDM_AGENT *A,
                      UDM_URLDATALIST *DataList,
                      UDM_DB *db,
                      size_t num_best_rows, int flag)
{
  int rc= UDM_OK;
  char qbuf[4*1024];
  const char *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  const char *su = UdmVarListFindStr(&A->Conf->Vars, "su", NULL);
  char *esu = su ? UdmSQLEscStr(db, NULL, su, strlen(su)) : NULL;
  UDM_PSTR row[5];
  UDM_SQLRES  SQLres;
  int need_url= (flag & UDM_URLDATA_URL);
  int group_by_site= (flag & UDM_URLDATA_SITE);

  /* Check that DataList is not empty and is sorted by url_id */
  UDM_ASSERT(num_best_rows && (DataList->Item[0].url_id <= DataList->Item[num_best_rows-1].url_id));
  UdmLog(A,UDM_LOG_DEBUG,"Trying to load URL data from url");

  if (db->DBSQL_IN)
  {
    size_t j, i;
    for (j= 0; j < num_best_rows; j+= 256)
    {
      int notfirst = 0;
      sprintf(qbuf, "SELECT rec_id, site_id, pop_rank, last_mod_time%s FROM url WHERE rec_id IN (", need_url ? ",url" : "");
      for (i= 0; (i < 256) && (j + i < num_best_rows); i++)
      {
        sprintf(UDM_STREND(qbuf), "%s%s%i%s", (notfirst) ? "," : "", qu, DataList->Item[j + i].url_id, qu);
        notfirst= 1;
      }
      sprintf(UDM_STREND(qbuf), ") ORDER BY rec_id");
      if (UDM_OK != (rc= UdmSQLExecDirect(db, &SQLres, qbuf)))
        goto fin;
      for (i= 0; db->sql->SQLFetchRow(db, &SQLres, row) == UDM_OK; i++)
      {
        UDM_URLDATA *D= &DataList->Item[i + j];
        if (D->url_id != (urlid_t) UDM_ATOI(row[0].val))
        {
          UdmLog(A, UDM_LOG_ERROR, "Dat url_id (%d) != SQL url_id (%d)", 
                 D->url_id, UDM_ATOI(row[0].val));
        }
        D->site_id= UDM_ATOI(row[1].val);
        D->pop_rank= UDM_ATOF(row[2].val);
        D->last_mod_time= UDM_ATOI(row[3].val);
        D->url= need_url ? UdmStrdup(row[4].val) : NULL;
        D->section= NULL;
      }
      UdmSQLFree(&SQLres);

      if (su)
      {
        size_t nrows, s;
        notfirst = 0;
        sprintf(qbuf, "SELECT url_id, sval FROM urlinfo WHERE sname='%s' AND url_id IN (", esu);
        for (i = 0; (i < 256) && (j + i < DataList->nitems); i++)
        {
          sprintf(UDM_STREND(qbuf), "%s%s%i%s", (notfirst) ? "," : "", qu, DataList->Item[j + i].url_id, qu);
          notfirst = 1;
        }
        sprintf(UDM_STREND(qbuf), ") ORDER BY url_id");
        if (UDM_OK != (rc = UdmSQLQuery(db, &SQLres, qbuf)))
          goto fin;
        nrows = UdmSQLNumRows(&SQLres);

        i = 0;
        for(s= i + j; i < nrows; s++)
        {
          if (s == DataList->nitems) break;
          if (DataList->Item[s].url_id != (urlid_t)UDM_ATOI(UdmSQLValue(&SQLres, i, 0)))
          {
            DataList->Item[s].section= UdmStrdup("");
          }
          else
          {
            DataList->Item[s].section= UdmStrdup(UdmSQLValue(&SQLres, i, 1));
            i++;
          }
        }
        UdmSQLFree(&SQLres);
      }
    }
  }
  else
  {
    size_t i;
    for (i = 0; i < num_best_rows; i++)
    {
      UDM_URLDATA *D= &DataList->Item[i];
      udm_snprintf(qbuf, sizeof(qbuf),
                   "SELECT site_id, pop_rank, last_mod_time%s FROM url WHERE rec_id=%i",
                   need_url ? ",url" : "", DataList->Item[i].url_id);
      if (UDM_OK != (rc = UdmSQLQuery(db, &SQLres, qbuf)))
        goto fin;
      if(UdmSQLNumRows(&SQLres))
      {
        D->url_id= DataList->Item[i].url_id;
        D->site_id= UDM_ATOI(UdmSQLValue(&SQLres, 0, 0));
        D->pop_rank= UDM_ATOF(UdmSQLValue(&SQLres, 0, 1));
        D->last_mod_time= UDM_ATOI(UdmSQLValue(&SQLres, 0, 2));
        D->url= need_url ? UdmStrdup(UdmSQLValue(&SQLres, 0, 3)) : NULL;
        D->section= NULL;
      }
      UdmSQLFree(&SQLres);
    }
  }

fin:
  UDM_FREE(esu);
  if (rc == UDM_OK && group_by_site)
    UdmResGroupBySite(A, DataList);
  return rc;
}


static int
UdmLoadURLDataFromBdict(UDM_AGENT *A,
                        UDM_URLDATALIST *DataList,
                        UDM_DB *db,
                        size_t num_best_rows, int flags)
{
  int rc;
  char qbuf[4*1024];
  UDM_SQLRES SQLres;
  const char *rec_id_str= NULL;
  const char *site_id_str= NULL;
  const char *last_mod_time_str= NULL;
  const char *pop_rank_str= NULL;
  const char *table = UdmBlobGetRTable(db);
  size_t rec_id_len= 0, site_id_len= 0;
  UDM_DSTR buf, rec_id_buf, site_id_buf, pop_rank_buf, last_mod_time_buf;
  UDM_PSTR row[2];
  size_t last_mod_time_len= 0, pop_rank_len= 0;
  size_t nrecs_expected= 1, nrecs_loaded= 0;
  unsigned long ticks= UdmStartTimer();
  int need_pop_rank= (flags & UDM_URLDATA_POP);
  int need_last_mod_time= (flags & UDM_URLDATA_LM);
  int group_by_site= (flags & UDM_URLDATA_SITE);
  int need_site_id= group_by_site;
  
  UdmDSTRInit(&buf, 64);
  UdmDSTRAppendSTR(&buf, "'#rec_id'");
  if (need_pop_rank)
  {
    UdmDSTRAppendSTR(&buf, ",'#pop_rank'");
    nrecs_expected++;
  }
  if (need_last_mod_time)
  {
    UdmDSTRAppendSTR(&buf, ",'#last_mod_time'");
    nrecs_expected++;
  }
  if (need_site_id)
  {
    UdmDSTRAppendSTR(&buf, ",'#site_id'");
    nrecs_expected++;
  }

  /* Check that DataList is not empty and is sorted by url_id */
  UDM_ASSERT(num_best_rows && (DataList->Item[0].url_id <= DataList->Item[num_best_rows-1].url_id));
  UdmLog(A,UDM_LOG_DEBUG,"Trying to load URL data from bdict");
  udm_snprintf(qbuf, sizeof(qbuf), "SELECT word, intag FROM %s WHERE word IN (%s)", table, buf.data);
  UdmDSTRFree(&buf);

  
  if (UDM_OK != (rc= UdmSQLExecDirect(db, &SQLres, qbuf)))
  {
    UdmLog(A,UDM_LOG_DEBUG,"Couldn't run a query on bdict");
    return(rc);
  }

  UdmDSTRInit(&rec_id_buf, 4096);
  UdmDSTRInit(&site_id_buf, 4096);
  UdmDSTRInit(&pop_rank_buf, 4096);
  UdmDSTRInit(&last_mod_time_buf, 4096);

  while (db->sql->SQLFetchRow(db, &SQLres, row) == UDM_OK)
  {
    if (!strcmp(row[0].val, "#rec_id"))
    {
      rec_id_len= row[1].len;
      rec_id_str= UdmBlobModeInflateOrAlloc(A, &rec_id_buf, "#rec_id",
                                            row[1].val, &rec_id_len);
      nrecs_loaded++;
    }
    else if (!strcmp(row[0].val, "#site_id"))
    {
      site_id_len= row[1].len;
      site_id_str= UdmBlobModeInflateOrAlloc(A, &site_id_buf, "#site_id",
                                             row[1].val, &site_id_len);
      nrecs_loaded++;
    }
    else if (!strcmp(row[0].val, "#last_mod_time"))
    {
      last_mod_time_len= row[1].len;
      last_mod_time_str= UdmBlobModeInflateOrAlloc(A, &last_mod_time_buf,
                                                  "#last_mod_time",
                                                  row[1].val, &last_mod_time_len);
      nrecs_loaded++;
    }
    else if (!strcmp(row[0].val, "#pop_rank"))
    {
      pop_rank_len= row[1].len;
      pop_rank_str= UdmBlobModeInflateOrAlloc(A, &pop_rank_buf, "#pop_rank",
                                             row[1].val, &pop_rank_len);
      nrecs_loaded++;
    }
  }
  
  ticks= UdmStartTimer() - ticks;
  UdmLog(A, UDM_LOG_DEBUG, "Fetch from bdict done: %.2f", (float)ticks/1000);

  if (rec_id_str && rec_id_len &&
      (site_id_str || ! need_site_id) &&
      (last_mod_time_str || ! need_last_mod_time) &&
      (pop_rank_str || ! need_pop_rank))
  {
    size_t nrows, j;

    nrows= rec_id_len / 4;

    ticks= UdmStartTimer();
    UdmLog(A, UDM_LOG_DEBUG, "Unpacking URL Data %d rows\n", nrows);
    if (need_pop_rank || need_last_mod_time)
    {
      size_t i;
      for (j = 0, i = 0; i < nrows; i++)
      {
        urlid_t rec_id= udm_get_int4(rec_id_str);
        if (rec_id == DataList->Item[j].url_id)
        {
          UDM_URLDATA *D= &DataList->Item[j];
          if (need_site_id)
          D->site_id= udm_get_int4(site_id_str);
          if (need_pop_rank)
            D->pop_rank= *((const double *)pop_rank_str);
          if (need_last_mod_time)
            D->last_mod_time= udm_get_int4(last_mod_time_str);
          j++;
          if (j == DataList->nitems) break;
        }
        rec_id_str+= 4;
        site_id_str+= 4;
        pop_rank_str+= 8;
        last_mod_time_str+= 4;
      }
    }
    else if (group_by_site && need_site_id)
    {
      UDM_HASH Hash;
      UDM_URLDATA *HashData, *Item= DataList->Item, D;
      size_t hcoords= UdmHashSize(DataList->nitems);
      size_t nbytes= hcoords * sizeof(UDM_URLDATA);
      size_t i, ncoords= DataList->nitems;
      unsigned long ticks1;
      bzero(&D, sizeof(D));
      D.per_site= 1;
      HashData= (UDM_URLDATA*) UdmMalloc(nbytes);
      bzero((void*)HashData, nbytes);
      
      UdmHashInit(&Hash, HashData, hcoords, sizeof(UDM_URLDATA),
                  test_key, (hash_join) test_join);
      for (j = 0, i = 0; i < nrows; i++)
      {
        D.url_id= udm_get_int4(rec_id_str);
        if (D.url_id == Item[j].url_id)
        {
          D.site_id= udm_get_int4(site_id_str + i * 4);
          D.coord= Item[j].coord;
          UdmHashPut(&Hash, &D);
          j++;
          if (j == ncoords) break;
        }
        rec_id_str+= 4;
      }

      if (j != ncoords)
      {
        UdmFree(HashData);
        goto site_fin;
      }
      
      ticks1= UdmStartTimer();
      j= DataList->nitems= UdmURLDATACompact(DataList->Item, HashData, hcoords);
      ticks1= UdmStartTimer() - ticks1;
      UdmLog(A, UDM_LOG_DEBUG, "HashCompact %d to %d done: %.2f",
             hcoords, DataList->nitems, (float) ticks1 / 1000);
      group_by_site= 0;
      UdmFree(HashData);
    }
    else
    {
      UDM_URLDATA *Data= DataList->Item;
      size_t i, skip= 0, ncoords= DataList->nitems;

      for (j = 0, i = 0; i < nrows; i++)
      {
        urlid_t rec_id= udm_get_int4(rec_id_str);
        while (rec_id > Data[j].url_id && j < ncoords)
        {
          skip++;
          j++;
        }

        if (rec_id == Data[j].url_id)
        {
          if (need_site_id)
            Data[j].site_id= udm_get_int4(site_id_str + i*4);
          j++;
          if (j == ncoords) break;
        }
        rec_id_str+= 4;
      }
      if (j < ncoords)
      {
        skip+= (ncoords - j);
        UdmLog(A, UDM_LOG_DEBUG,
               "Warning: %d out of %d coords didn't have URL data", skip, DataList->nitems);
        UdmLog(A, UDM_LOG_DEBUG, "GroupBySite may work incorrectly");
        j= DataList->nitems;
      }
    }
site_fin:
    ticks= UdmStartTimer() - ticks;
    UdmLog(A, UDM_LOG_DEBUG, "Unpacking URL Data done: %.02f", (float)ticks/1000);
    if (j == DataList->nitems)
    {
      if (group_by_site)
        UdmResGroupBySite(A, DataList);
      rc= UDM_OK;
      goto ret;
    }
    UdmLog(A,UDM_LOG_DEBUG,"Expected to load %d URLs, loaded %d", DataList->nitems, j);
    UdmLog(A,UDM_LOG_DEBUG,"Couldn't load URL data from bdict");
  }
  else
  {
    UdmLog(A,UDM_LOG_DEBUG,"There is no URL data in bdict");
  }
  rc= UdmLoadURLDataFromURL(A, DataList, db, num_best_rows, flags);
  
ret:
  UdmSQLFree(&SQLres);
  UdmDSTRFree(&rec_id_buf);
  UdmDSTRFree(&site_id_buf);
  UdmDSTRFree(&pop_rank_buf);
  UdmDSTRFree(&last_mod_time_buf);
  return rc;
}


static int
UdmApplyUserScore(UDM_AGENT *Agent, UDM_URLSCORELIST *List, UDM_DB *db)
{
  char name[128];
  const char *us, *query;
  UDM_URL_INT4_LIST UserScoreList;
  int rc;
  size_t i;
  int4 minval, maxval;
  UDM_URL_SCORE *Coords= List->Item;
  int UserScoreFactor= UdmVarListFindInt(&Agent->Conf->Vars, "UserScoreFactor", 0);
  
  if (!UserScoreFactor ||
      !(us= UdmVarListFindStr(&Agent->Conf->Vars, "us", NULL)))
    return UDM_OK;
  udm_snprintf(name, sizeof(name), "Score.%s", us);
  if (!(query= UdmVarListFindStr(&Agent->Conf->Vars, name, NULL)))
    return UDM_OK;
  
  if ((UDM_OK != (rc= UdmLoadUserScoreList(db, &UserScoreList, query))) ||
      !UserScoreList.nitems)
    goto ret;

  minval= -1;
  maxval= 1;
  for (i= 0; i < UserScoreList.nitems; i++)
  {
    UDM_URL_INT4 *Item= &UserScoreList.Item[i];
    if (minval > Item->param)
      minval= Item->param;
    if (maxval < Item->param)
      maxval= Item->param;
  }

  for (i= 0; i < List->nitems; i++)
  {
    urlid_t url_id= Coords[i].url_id;
    uint4 coord= Coords[i].score;
    UDM_URL_INT4 *found;
    found= (UDM_URL_INT4*) UdmBSearch(&url_id,
                                      UserScoreList.Item, UserScoreList.nitems,
                                      sizeof(UDM_URL_INT4), (udm_qsort_cmp)cmpaurls);

    if (found)
    {
      if (found->param >= 0)
        coord= coord + 
              ((int4) (((float) (100000 - coord)) * found->param / maxval)) *
               UserScoreFactor / 255;
      else
        coord= coord -
               ((int4) (((float) coord) * found->param / minval)) *
               UserScoreFactor / 255;
    }

    Coords[i].score= coord;
  }

ret:
  UDM_FREE(UserScoreList.Item);
  return rc;
}


static void UdmApplyRelevancyFactors(UDM_AGENT *Agent,
                                     UDM_URLDATALIST *DataList,
                                     int RelevancyFactor,
                                     int DateFactor)
{
  int i, sum;
  time_t current_time;
  unsigned long ticks;
  UdmLog(Agent, UDM_LOG_DEBUG, "Start applying relevancy factors");
  ticks= UdmStartTimer();
  if (!(current_time= UdmVarListFindInt(&Agent->Conf->Vars, "CurrentTime", 0)))
    time(&current_time);
  sum= RelevancyFactor + DateFactor;
  sum= sum ? sum : 1;
  
  for (i= 0; i < DataList->nitems; i++)
  {
    time_t doc_time= DataList->Item[i].last_mod_time;
    uint4 *coord= &DataList->Item[i].coord;
    float rel= *coord * RelevancyFactor;
    float dat= ((doc_time < current_time) ? 
                ((float) doc_time / current_time) :
                ((float) current_time / doc_time)) * DateFactor * 100000;
    /* 100000 = 100% * 1000 = scale in db.c */
    *coord= (rel + dat) / sum;
  }
  ticks= UdmStartTimer() - ticks;
  UdmLog(Agent, UDM_LOG_DEBUG, "Stop applying relevancy factors\t\t%.2f",
      (float)ticks / 1000);
}


static inline int 
udm_int4_get(const char *s)
{
  int res;
  res=  (unsigned char) s[3]; res<<= 8;
  res+= (unsigned char) s[2]; res<<= 8;
  res+= (unsigned char) s[1]; res<<= 8;
  res+= (unsigned char) s[0];
  return res;
}


static int
UdmFetchCachedQuery(UDM_AGENT *A,
                    UDM_RESULT *Res,
                    UDM_DB *db, const char *query, int *tm)
{
  int rc;
  UDM_SQLRES SQLRes;
  UDM_URLDATALIST *DataList= &Res->URLData;
  
  if(UDM_OK != (rc= UdmSQLQuery(db,&SQLRes,query)))
      return rc;
  if (UdmSQLNumRows(&SQLRes) == 1)
  {
    size_t i, nbytes;
    const char *p;
    DataList->nitems= UdmSQLLen(&SQLRes, 0, 0) / 12;
    nbytes= DataList->nitems * sizeof(UDM_URLDATA);
    DataList->Item= (UDM_URLDATA*) UdmMalloc(nbytes);
    bzero((void*) DataList->Item, nbytes);
    for (p= UdmSQLValue(&SQLRes, 0, 0), i= 0; i < DataList->nitems; i++)
    {
      UDM_URLDATA *D= &DataList->Item[i];
      D->url_id= udm_int4_get(p); p+= 4;
      D->coord= udm_int4_get(p);  p+= 4;
      D->per_site= udm_int4_get(p); p+= 4;
    }
    if (UdmSQLNumCols(&SQLRes) >= 2)
    {
      UdmResultFromXML(A, Res, UdmSQLValue(&SQLRes, 0, 1),
                       UdmSQLLen(&SQLRes, 0,1), A->Conf->lcs);
      if (UdmSQLNumCols(&SQLRes) >= 3)
        *tm= atoi(UdmSQLValue(&SQLRes,0,2));
    }
  }
  UdmSQLFree(&SQLRes);
  return rc;
}


static int
UdmLoadCachedQueryWords(UDM_AGENT *query, UDM_RESULT *Res,
                        UDM_DB *db, const char *pqid)
{
  char qbuf[128], bqid[32];
  char *tm, *end;
  int iid, itm;
  UDM_URLDATALIST *DataList= &Res->URLData;
  
  bzero(DataList, sizeof(*DataList));
  if (!pqid)
    return UDM_OK;
  udm_snprintf(bqid, sizeof(bqid), pqid);
  if (!(tm= strchr(bqid, '-')))
    return UDM_OK;
  *tm++= '\0';
  iid= (int) strtoul(bqid, &end, 16);
  itm= (int) strtol(tm, &end, 16);
  udm_snprintf(qbuf, sizeof(qbuf),
               "SELECT doclist FROM qcache WHERE id='%d' AND tm=%d ORDER BY tm DESC LIMIT 1", iid, itm);
  return UdmFetchCachedQuery(query, Res, db, qbuf, NULL);
}


static int 
UdmApplyCachedQueryLimit(UDM_AGENT *query, UDM_URLSCORELIST *ScoreList, UDM_DB *db)
{
  UDM_RESULT CachedResult;
  UDM_URLDATALIST *CachedData= &CachedResult.URLData;
  const char *pqid= UdmVarListFindStr(&query->Conf->Vars, "pqid", NULL);
  UdmResultInit(&CachedResult);
  if (pqid && UDM_OK == UdmLoadCachedQueryWords(query, &CachedResult, db, pqid))
  {
    UdmLog(query, UDM_LOG_DEBUG,
           "Start applying pqid limit: %d docs", CachedData->nitems);
    if (CachedData->nitems)
    {
      size_t i, to;
      UdmSort(CachedData->Item, CachedData->nitems, sizeof(UDM_URLDATA), (udm_qsort_cmp) cmp_data_urls);
      for (to= 0, i= 0; i < ScoreList->nitems; i++)
      {
        if (UdmURLSearchForConv(CachedData, ScoreList->Item[i].url_id))
        {
          if (to != i)
            ScoreList->Item[to]= ScoreList->Item[i];
          to++;
        }
      }
      ScoreList->nitems= to;
    }
    else
    {
      ScoreList->nitems= 0;
    }
    UdmLog(query,UDM_LOG_DEBUG, "Stop applying pqid limit: %d docs", ScoreList->nitems);
  }
  UdmResultFree(&CachedResult);
  return UDM_OK;
}


static int
cmp_score_urlid(UDM_URL_SCORE *s1, UDM_URL_SCORE *s2)
{
  if (s1->url_id > s2->url_id) return(1);
  if (s1->url_id < s2->url_id) return(-1);
  return 0;
}
      

static void
UdmScoreListToURLData(UDM_URLDATA *D, UDM_URL_SCORE *C, size_t num)
{
  for ( ; num > 0; num--, D++, C++)
  {
    D->url_id= C->url_id;
    D->coord= C->score;
  }
}


#ifdef HAVE_DEBUG
static void
UdmURLScoreListPrint(UDM_URLSCORELIST *List)
{
  size_t i;
  for (i= 0; i < List->nitems; i++)
  {
    UDM_URL_SCORE *Item= &List->Item[i];
    fprintf(stderr, "%d:%d\n", Item->url_id, Item->score);
  }
}
#endif

static int 
UdmSortAndGroupByURL(UDM_FINDWORD_ARGS *args,
                     UDM_AGENT *A,
                     UDM_RESULT *Res,
                     UDM_SECTIONLIST *SectionList,
                     UDM_DB *db, size_t num_best_rows)
{
  UDM_URLSCORELIST ScoreList;
  UDM_URLDATALIST DataList;
  unsigned long  ticks=UdmStartTimer();
  const char *pattern= UdmVarListFindStr(&A->Conf->Vars, "s", "R");
  size_t nbytes;
  int flags= 0, rc= UDM_OK;
  const char *p;
  int RelevancyFactor= UdmVarListFindInt(&A->Conf->Vars, "RelevancyFactor", 255);
  int DateFactor= UdmVarListFindInt(&A->Conf->Vars, "DateFactor", 0);
  const char *su = UdmVarListFindStr(&A->Conf->Vars, "su", NULL);
  int group_by_site= UdmVarListFindBool(&A->Conf->Vars, "GroupBySite", 0) 
                     && UdmVarListFindInt(&A->Conf->Vars, "site", 0) == 0 ?
                     UDM_URLDATA_SITE : 0;
  size_t BdictThreshold= (size_t) UdmVarListFindInt(&A->Conf->Vars,
                                                    "URLDataThreshold", 0);
  size_t MaxResults= (size_t) UdmVarListFindInt(&db->Vars, "MaxResults", 0);
  int use_qcache= UdmVarListFindBool(&db->Vars, "qcache", 0);

  flags|= su ? UDM_URLDATA_SU : 0;
  flags|= group_by_site ? UDM_URLDATA_SITE : 0;
  flags|= DateFactor ? UDM_URLDATA_LM : 0;
  flags|= use_qcache ? UDM_URLDATA_CACHE : 0;
    
  for (p = pattern; *p; p++)
  {
    if (*p == 'U' || *p == 'u') flags|= UDM_URLDATA_URL;
    if (*p == 'P' || *p == 'p') flags|= UDM_URLDATA_POP;
    if (*p == 'D' || *p == 'd') flags|= UDM_URLDATA_LM;
  }
  
  ticks=UdmStartTimer();
  bzero((void*) &ScoreList, sizeof(ScoreList));

  UdmLog(A,UDM_LOG_DEBUG,"Start group by url_id %d sections", SectionList->nsections);
  UdmGroupByURL2(A, db, Res, SectionList, &ScoreList);

  ticks=UdmStartTimer()-ticks;
  UdmLog(A,UDM_LOG_DEBUG,"Stop group by url_id, %d docs found:\t%.2f",
         ScoreList.nitems, (float)ticks/1000);

#ifdef HAVE_DEBUG
  if (UdmVarListFindBool(&A->Conf->Vars, "DebugGroupByURL", 0))
  {
    UdmURLScoreListPrint(&ScoreList);
  }
#endif

  UdmApplyCachedQueryLimit(A, &ScoreList, db);
  if (ScoreList.nitems == 0) return UDM_OK;
  UdmApplyUserScore(A, &ScoreList, db);

  UdmLog(A,UDM_LOG_DEBUG,"Start load url data %d docs", ScoreList.nitems);
  ticks=UdmStartTimer();

  nbytes= UdmHashSize(ScoreList.nitems) * sizeof(UDM_URLDATA);
  DataList.Item = (UDM_URLDATA*)UdmMalloc(nbytes);
  bzero((void*) DataList.Item, nbytes);
  DataList.nitems= ScoreList.nitems;
  
  if (num_best_rows > ScoreList.nitems)
    num_best_rows= ScoreList.nitems;
  
  if (num_best_rows < 256 && !flags)
  {
    udm_timer_t ticks1;
    
    Res->total_found= ScoreList.nitems;
    UdmLog(A,UDM_LOG_DEBUG,"Start SORT by score %d docs", ScoreList.nitems);
    ticks1=UdmStartTimer();
    if (ScoreList.nitems > 1000)
    {
      /*
      size_t nitems= ScoreList.nitems;
      UdmURLScoreListSortByScore(&ScoreList);
      ScoreList.nitems= 1000;
      UdmURLScoreListSortByScoreThenURL(&ScoreList);
      ScoreList.nitems= nitems;
      */
      UdmURLScoreListSortByScoreThenURLTop(&ScoreList, 1000);
    }
    else
    {
      UdmURLScoreListSortByScoreThenURL(&ScoreList);
    }
    ticks1=UdmStartTimer()-ticks1;
    UdmLog(A,UDM_LOG_DEBUG,"Stop SORT by score:\t%.2f",(float)ticks1/1000);

    UdmLog(A,UDM_LOG_DEBUG,"Start load url data quick %d docs", num_best_rows);
    ticks1=UdmStartTimer();
    UdmSort((void*) ScoreList.Item, num_best_rows,
          sizeof(UDM_URL_SCORE), (udm_qsort_cmp) cmp_score_urlid);
    UdmScoreListToURLData(DataList.Item, ScoreList.Item, num_best_rows);
    rc= UdmLoadURLDataFromURL(A, &DataList, db, num_best_rows, flags);
    ticks1=UdmStartTimer()-ticks1;
    UdmLog(A,UDM_LOG_DEBUG,"Stop load url data quick:\t\t%.2f",(float)ticks1/1000);
    DataList.nitems= num_best_rows;
  }
  else if (db->DBMode == UDM_DBMODE_BLOB && !(flags&UDM_URLDATA_URL) && !su &&
           BdictThreshold < ScoreList.nitems)
  {
    UdmScoreListToURLData(DataList.Item, ScoreList.Item, DataList.nitems);
    rc= UdmLoadURLDataFromBdict(A, &DataList, db, DataList.nitems, flags);
    Res->total_found= DataList.nitems;
  }
  else
  {
    UdmScoreListToURLData(DataList.Item, ScoreList.Item, DataList.nitems);  
    rc= UdmLoadURLDataFromURL(A, &DataList, db, DataList.nitems, flags);
    Res->total_found= DataList.nitems;
  }

  /* TODO: check whether limit by site works fine */
  if (!RelevancyFactor || DateFactor)
    UdmApplyRelevancyFactors(A, &DataList, RelevancyFactor, DateFactor);

  ticks= UdmStartTimer()-ticks;
  UdmLog(A,UDM_LOG_DEBUG,"Stop load url data:\t\t%.2f",(float)ticks/1000);
  
  UdmLog(A,UDM_LOG_DEBUG,"Start SORT by PATTERN %d docs", DataList.nitems);
  ticks=UdmStartTimer();
  UdmURLDataSortByPattern(&DataList, pattern);
  ticks=UdmStartTimer()-ticks;
  UdmLog(A,UDM_LOG_DEBUG,"Stop SORT by PATTERN:\t%.2f",(float)ticks/1000);

  Res->URLData= DataList;
  UdmFree(ScoreList.Item);
  
  if (MaxResults && MaxResults < Res->total_found)
  {
    UdmLog(A, UDM_LOG_DEBUG, "Applying MaxResults=%d, total_found=%d\n",
           (int) MaxResults, (int) Res->total_found);
    Res->total_found= MaxResults;
    if (Res->URLData.nitems > MaxResults)
      Res->URLData.nitems= MaxResults;
  }
  return UDM_OK;
}


static int
LoadURL(UDM_DB *db, const char *where, UDM_URL_TMP *buf)
{
  int rc;
  UDM_SQLRES SQLRes;
  char qbuf[1024 * 4];
  size_t nrows;
  urlid_t *tmp;
  size_t i;

  if (! buf) return(UDM_ERROR);
  if (! *where) return(UDM_OK);

  udm_snprintf(qbuf, sizeof(qbuf), "SELECT url.rec_id FROM url%s WHERE %s", db->from, where);
  rc = UdmSQLQuery(db, &SQLRes, qbuf);
  if (rc != UDM_OK) return(UDM_ERROR);
  nrows = UdmSQLNumRows(&SQLRes);
  if (! nrows)
  {
    buf->empty= 1;
    UdmSQLFree(&SQLRes);
    return(UDM_OK);
  }

  tmp = UdmMalloc(sizeof(urlid_t) * nrows);
  buf->urls = UdmMalloc(sizeof(urlid_t) * nrows);
  if (tmp && buf->urls)
  {
    for (i = 0; i < nrows; i++)
    {
      tmp[i] = (urlid_t)UDM_ATOI(UdmSQLValue(&SQLRes, i, 0));
    }
    UdmSort(tmp, nrows, sizeof(urlid_t), (udm_qsort_cmp)cmpaurls);
    i = 0;
    while (i < nrows)
    {
      while (++i < nrows && tmp[i] == tmp[i - 1]);
      buf->urls[buf->nurls++] = tmp[i - 1];
    }
    UDM_FREE(tmp);
    tmp = UdmRealloc(buf->urls, sizeof(urlid_t) * buf->nurls);
    if (tmp) buf->urls = tmp;
  } else {
    UDM_FREE(buf->urls);
    UDM_FREE(tmp);
  }

  UdmSQLFree(&SQLRes);
  return(UDM_OK);
}


static int
UdmBlobLoadFastURLLimit(UDM_DB *db, const char *name, UDM_URL_TMP *buf)
{
  int rc= UDM_OK;
  UDM_SQLRES SQLRes;
  char qbuf[256], ename[130], exclude;
  size_t nrows, nurls, i, row, namelen= strlen(name);
  
  if (namelen > 64)
    return UDM_OK;
  
  UdmSQLEscStr(db, ename, name, namelen);
  exclude= buf->exclude;
  bzero((void*)buf, sizeof(*buf));
  buf->exclude= exclude;
  udm_snprintf(qbuf, sizeof(qbuf), "SELECT intag FROM bdict WHERE word LIKE '#limit#%s'", ename);
  if (UDM_OK != (rc= UdmSQLQuery(db, &SQLRes, qbuf)))
   goto ret;

  if (! (nrows= UdmSQLNumRows(&SQLRes)))
  {
    buf->empty= 1;
    goto ret;
  }
  nurls= 0;
  for (row= 0; row < nrows; row++)
    nurls+= UdmSQLLen(&SQLRes, row, 0) / 4;

  if (!(buf->urls= UdmMalloc(sizeof(urlid_t) * nurls)))
    goto ret;

  for (row= 0; row < nrows; row++)
  {
    const char *src= UdmSQLValue(&SQLRes, row, 0);
    nurls= UdmSQLLen(&SQLRes, row, 0) / 4;
    if (src && nurls)
      for (i = 0; i < nurls; i++, src+= 4)
        buf->urls[buf->nurls++]= (urlid_t) udm_get_int4(src);
  }
  if (nrows > 1)
    UdmSort(buf->urls, buf->nurls, sizeof(urlid_t), (udm_qsort_cmp)cmpaurls);
  
ret:
  UdmSQLFree(&SQLRes);
  return rc;
}


static inline int UdmBuildCmpArgSQL (UDM_DB *db, int match, const char *word,
  char *cmparg, size_t maxlen)
{
  char escwrd[1000];
  UdmSQLEscStr(db, escwrd, word, strlen(word));
  switch(match)
  {  
    case UDM_MATCH_BEGIN:
      udm_snprintf(cmparg, maxlen, " LIKE '%s%%'", escwrd);
      break;
    case UDM_MATCH_END:
      udm_snprintf(cmparg, maxlen, " LIKE '%%%s'", escwrd);
      break;
    case UDM_MATCH_SUBSTR:
      udm_snprintf(cmparg, maxlen, " LIKE '%%%s%%'", escwrd);
      break;
    case UDM_MATCH_NUMERIC_LT:
      udm_snprintf(cmparg, maxlen, " < %d", atoi(escwrd));
      break;
    case UDM_MATCH_NUMERIC_GT:
      udm_snprintf(cmparg, maxlen, " > %d", atoi(escwrd));
      break;
    case UDM_MATCH_FULL:
    default:
      udm_snprintf(cmparg, maxlen, "='%s'", escwrd);
      break;
  }
  return(UDM_OK);
}


static inline int
UdmAddOneCoord(UDM_URLCRDLIST *CoordList,
               urlid_t url_id, uint4 coord, udm_wordnum_t num)
{
  UDM_URL_CRD *C;
  if (CoordList->ncoords == CoordList->acoords)
  {
    UDM_URL_CRD *tmp;
    size_t newsize= CoordList->acoords ? CoordList->acoords * 2 : 1024;
    tmp= UdmRealloc(CoordList->Coords, newsize * sizeof(UDM_URL_CRD));
    if (! tmp)
    {
      return(UDM_ERROR);
    }
    CoordList->Coords= tmp;
    CoordList->acoords= newsize;
  }
  C= &CoordList->Coords[CoordList->ncoords];
  C->url_id= url_id;
  C->pos= UDM_WRDPOS(coord);
  C->num= num;
  C->secno= UDM_WRDSEC(coord);
  C->seclen= 0;
  CoordList->ncoords++;
  return(UDM_OK);
}



static int
UdmFindWordSingleInternal(UDM_FINDWORD_ARGS *args,
                          UDM_URLCRDLIST *CoordList,
                          const char *table, int join)
{
  char qbuf[4096];
  UDM_SQLRES SQLRes;
  size_t numrows, i, curnum;
  int rc;

  if (*args->where)
  {
    udm_snprintf(qbuf, sizeof(qbuf) - 1,"\
SELECT %s.url_id,%s.intag FROM %s, url%s \
WHERE %s.word%s AND url.rec_id=%s.url_id AND %s",
    table, table, table, args->db->from,
    table, args->cmparg, table, args->where);
  }
  else if (!join)
  {
    udm_snprintf(qbuf, sizeof(qbuf) - 1,
      "SELECT url_id,intag FROM %s WHERE word%s", table, args->cmparg);
  }
  else
  {

    udm_snprintf(qbuf,sizeof(qbuf)-1,"\
SELECT url_id,intag FROM %s,url WHERE %s.word%s AND url.rec_id=%s.url_id",
       table, table, args->cmparg, table);
  }
  
  if(UDM_OK != (rc= UdmSQLQuery(args->db, &SQLRes, qbuf)))
    return rc;
  
  if ((numrows= UdmSQLNumRows(&SQLRes)))
  {
    CoordList->Coords= (UDM_URL_CRD*) UdmMalloc(numrows*sizeof(UDM_URL_CRD));
    CoordList->acoords= numrows;
  }
  
  /* Add new found word to the list */
  for(curnum= 0, i= 0; i < numrows; i++)
  {
    uint4 coord= atoi(UdmSQLValue(&SQLRes, i, 1));
    uint4 section= UDM_WRDSEC(coord);
    uint4 weight= args->wf[section];
    
    if(weight && (!args->secno || args->secno == section))
    {
      UDM_URL_CRD *Coord= &CoordList->Coords[curnum];
      Coord->url_id= UDM_ATOI(UdmSQLValue(&SQLRes, i, 0));
      Coord->pos= UDM_WRDPOS(coord);
      Coord->secno= UDM_WRDSEC(coord);
      Coord->num= args->wordnum & 0xFF;
      Coord->seclen= 0;
      curnum++;
    }
  }
  CoordList->ncoords= curnum;
  UdmSQLFree(&SQLRes);
  UdmSortSearchWordsByURL(CoordList->Coords, CoordList->ncoords);
  return rc;
}


static int
UdmFindWordSingle (UDM_FINDWORD_ARGS *args)
{
  int rc;
  UDM_URLCRDLIST CoordList;

  bzero(&CoordList, sizeof(CoordList));

  if (UDM_OK != (rc= UdmFindWordSingleInternal(args, &CoordList, "dict", 0)))
    return rc;

  if (args->save_section_size && CoordList.ncoords)
  {
    UDM_URL_CRD *Crd= CoordList.Coords;
    UDM_URL_CRD *End= CoordList.Coords + CoordList.ncoords;
    UDM_URL_CRD *To= CoordList.Coords;
    UDM_URL_CRD *Prev= CoordList.Coords;
    urlid_t prev_url_id;
    int prev_secno;
    
    for (prev_url_id= Crd->url_id, prev_secno= Crd->secno ; Crd < End; Crd++)
    {
      UDM_URL_CRD *Next= Crd + 1;
      if (Next == End ||
          Next->url_id != prev_url_id ||
          Next->secno  != prev_secno)
      {
        for ( ; Prev < To; Prev++)
        {
          Prev->seclen= Crd->pos;
        }
        if (Next < End)
        {
          prev_url_id= Next->url_id;
          prev_secno= Next->secno;
          Prev= To;
        }
      }
      else
        *To++= *Crd;
    }
    CoordList.ncoords= To - CoordList.Coords;
  }
  if (args->urls.nurls)
    UdmApplyFastLimit(&CoordList, &args->urls);
  if (CoordList.ncoords)
  {
    args->count= CoordList.ncoords;
    UdmURLCRDListListAddWithSort2(args, &args->CoordListList, &CoordList);
  }
  return(UDM_OK);
}


static int UdmFindWordMulti (UDM_FINDWORD_ARGS *args)
{
  char qbuf[4096], secno[64]= "";
  UDM_SQLRES SQLRes;
  size_t tnum, tmin, tmax;
  unsigned long ticks;
  int rc;

  if (args->word_match != UDM_MATCH_FULL)
  {
    /* This is for substring search!  */
    /* In Multi mode: we have to scan */
    /* almost all tables except those */
    /* with to short words            */
      
    tmin= 0;
    tmax= MULTI_DICTS;
  }
  else
  {
    tmin= tmax= UdmStrHash32(args->word) & MULTI_DICTS;
  }

  if (args->secno)
    udm_snprintf(secno, sizeof(secno), " AND dict.secno=%d", args->secno);

  for(tnum= tmin; tnum <= tmax; tnum++)
  {
    if (*args->where)
    {
      udm_snprintf(qbuf, sizeof(qbuf) - 1,"\
SELECT dict.url_id,dict.secno,dict.intag \
FROM dict%02X dict, url%s \
WHERE dict.word%s \
AND url.rec_id=dict.url_id AND %s%s",
        tnum, args->db->from, args->cmparg, args->where, secno);
    }
    else
    {
      udm_snprintf(qbuf, sizeof(qbuf) - 1,
                   "SELECT url_id,secno,intag FROM dict%02X dict WHERE word%s%s",
                   tnum, args->cmparg, secno);
    }

    if (UDM_OK != (rc= UdmSQLQuery(args->db, &SQLRes, qbuf)))
      return rc;

    UdmLog(args->Agent, UDM_LOG_DEBUG, "Start UdmMultiAddCoords");
    ticks= UdmStartTimer();
    UdmMultiAddCoords(args, &SQLRes);
    ticks= UdmStartTimer() - ticks;
    UdmLog(args->Agent, UDM_LOG_DEBUG, "Stop UdmMultiAddCoords\t%.2f", (float)ticks / 1000);
    UdmSQLFree(&SQLRes);
  }
  return(UDM_OK);
}


static int UdmInflateBlobModeSQLRes(UDM_AGENT *A, UDM_SQLRES *src)
{
  UDM_DSTR ibuf;
  size_t row;
  UdmDSTRInit(&ibuf, 1024);
  for (row= 0; row < src->nRows; row++)
  {
    size_t len= UdmSQLLen(src, row, 1);
    const char *val= UdmSQLValue(src, row, 1);
    const char *iflt;
    iflt= UdmBlobModeInflateOrSelf(A, &ibuf, NULL, val, &len);
    if (iflt != val)
    {
      size_t offs= src->nCols*row + 1;
      UdmFree(src->Items[offs].val);
      src->Items[offs].val= UdmMalloc(len + 1);
      memcpy(src->Items[offs].val, iflt, len);
      src->Items[offs].len= len;
      src->Items[offs].val[len]= '\0';
    }
  }
  UdmDSTRFree(&ibuf);
  return UDM_OK;
}




static int UdmFindWordBlob (UDM_FINDWORD_ARGS *args)
{
  char qbuf[4096];
  char secno[32]= "";
  char special[32]= "";
  unsigned long ticks;
  UDM_SQLRES SQLRes;
  int rc;

  if (args->urls.empty)
  {
    UdmLog(args->Agent, UDM_LOG_DEBUG, "Not searching 'bdict': Base URL limit is empty");
    return UDM_OK;
  }
  
  ticks= UdmStartTimer();
  UdmLog(args->Agent, UDM_LOG_DEBUG, "Start fetching");
  if (args->secno)
    udm_snprintf(secno, sizeof(secno), " AND secno=%d", args->secno);
  /*
    When performing substring or number search,
    don't include special data, like '#last_mod_time' or '#rec_id'
  */
  if (args->cmparg[0] != '=')
    udm_snprintf(special, sizeof(special), " AND word NOT LIKE '#%%'");
  udm_snprintf(qbuf, sizeof(qbuf), "SELECT secno,intag,word FROM bdict WHERE word%s%s%s", args->cmparg, secno, special);
  if(UDM_OK != (rc= UdmSQLQuery(args->db, &SQLRes, qbuf)))
    return rc;
  ticks= UdmStartTimer() - ticks;
  UdmLog(args->Agent, UDM_LOG_DEBUG, "Stop fetching\t%.2f", (float) ticks / 1000);

  ticks= UdmStartTimer();
  UdmLog(args->Agent, UDM_LOG_DEBUG, "Start UdmBlobAddCoords");
  UdmInflateBlobModeSQLRes(args->Agent, &SQLRes);
  UdmBlobAddCoords2(args, &SQLRes);
  ticks= UdmStartTimer()-ticks;
  UdmLog(args->Agent, UDM_LOG_DEBUG, "Stop UdmBlobAddCoords\t%.2f", (float)ticks / 1000);
  UdmSQLFree(&SQLRes);
  return(UDM_OK);
}


#define UDM_RAWBLOB_SEARCH  0
#define UDM_RAWBLOB_DELTA   1

/*
  UDM_RAWBLOB_SEARCH mode:
  Search in raw blob mode for small databases:
  direct search from "bdicti" without having to
  run "indexer -Eblob":
  load the records with state=1 (new) or state=2 (converted).

  UDM_RAWBLOB_DELTA mode (for LiveUpdates):
  - Load the records with state=1 (new).
  - The records with state=0 (deleted) will be loaded
  as an excluding limit for "bdict".
*/
static int
UdmFindWordRawBlob(UDM_FINDWORD_ARGS *args, int flags)
{
  char qbuf[4096];
  unsigned long ticks;
  UDM_SQLRES SQLRes;
  UDM_URLCRDLIST CoordList;
  size_t intagnum= UdmStrHash32(args->word) & 0x1f;
  size_t rownum, nrows;
  int rc;
  const char *state_cmp= (flags & UDM_RAWBLOB_DELTA) ? "=" : ">=";
  UDM_URL_TMP *urls= (flags & UDM_RAWBLOB_DELTA) ?
                      &args->live_update_active_urls :
                      &args->urls;

  if (urls->empty)
  {
    UdmLog(args->Agent, UDM_LOG_DEBUG, "Not searching 'bdicti': Live URL limit is empty");
    return UDM_OK;
  }
  
  
  ticks= UdmStartTimer();
  UdmLog(args->Agent, UDM_LOG_DEBUG, "Start fetching from bdicti");
  
  if (*args->where)
  {
    udm_snprintf(qbuf, sizeof(qbuf),"\
SELECT d.url_id,d.intag%02X FROM bdicti d,url%s \
WHERE d.state%s1 AND url.rec_id=d.url_id AND %s",
                 intagnum, args->db->from, state_cmp, args->where);
  }
  else
  {
    udm_snprintf(qbuf, sizeof(qbuf), 
                 "SELECT url_id, intag%02X FROM bdicti WHERE state%s1",
                 intagnum, state_cmp);
  }

  if(UDM_OK != (rc= UdmSQLQuery(args->db, &SQLRes, qbuf)))
    return rc;
  nrows= UdmSQLNumRows(&SQLRes);
  
  bzero((void*) &CoordList, sizeof(CoordList));
  for (rownum= 0; rownum < nrows; rownum++)
  {
    CoordList.acoords+= UdmSQLLen(&SQLRes, rownum, 1);
  }
  CoordList.Coords= (UDM_URL_CRD*)UdmMalloc((CoordList.acoords) * sizeof(UDM_URL_CRD));
  
  for (rownum= 0; rownum < nrows; rownum ++)
  {
    UDM_PSTR row[2];
    urlid_t url_id;
    size_t pos= 0;
    row[0].val= UdmSQLValue(&SQLRes, rownum, 0);
    row[0].len= UdmSQLLen(&SQLRes, rownum, 0);
    row[1].val= UdmSQLValue(&SQLRes, rownum, 1);
    row[1].len= UdmSQLLen(&SQLRes, rownum, 1);
    url_id= UDM_ATOI(row[0].val);
    if (urls->nurls)
    {
      void *found= UdmBSearch(&url_id, urls->urls, urls->nurls,
                              sizeof(urlid_t), (udm_qsort_cmp)cmpaurls);
      if (found && urls->exclude)
        continue;
      if (!found && !urls->exclude)
        continue;
    }

    /*
      TODO: substring search, numeric search
    */
    
    while (pos < row[1].len)
    {
      char *word= &row[1].val[pos];
      while (pos < row[1].len && row[1].val[pos])
        pos++;
      if (++pos >= row[1].len)
        break;
      while (pos < row[1].len)
      {
        unsigned char secno= (unsigned char)row[1].val[pos];
        char *intag= &row[1].val[++pos];
        size_t lintag;
        while (pos < row[1].len && row[1].val[pos])
          pos++;
        lintag= row[1].val + pos - intag;
        if ((!args->secno || args->secno == secno) &&
            !strcmp(args->word, word) &&
            args->wf[secno])
        {
          UdmMultiAddCoordStr(args, &CoordList, url_id, secno,
                              (unsigned char*) intag, lintag);
        }
        if (++pos >= row[1].len || ! row[1].val[pos])
        {
          pos++;
          break;
        }
      }
    }
  }
  UdmSQLFree(&SQLRes);
  if (CoordList.ncoords)
  {
    args->count+= CoordList.ncoords;
    UdmSortSearchWordsByURL(CoordList.Coords, CoordList.ncoords);
    UdmURLCRDListListAddWithSort2(args, &args->CoordListList, &CoordList);
  }
  ticks= UdmStartTimer() - ticks;
  UdmLog(args->Agent, UDM_LOG_DEBUG,
         "Stop fetching from bdicti\t%.2f %d coords found",
         (float) ticks / 1000, CoordList.ncoords);
  return UDM_OK;
}


static int
UdmFindWordBlobLiveUpdates(UDM_FINDWORD_ARGS *args)
{
  int rc;
  if ((!UDM_OK == (rc= UdmFindWordBlob(args))) ||
      (!UDM_OK == (rc= UdmFindWordRawBlob(args, UDM_RAWBLOB_DELTA))))
    goto ret;

ret:
  return rc;
}


static inline int UdmFindOneWordSQL (UDM_FINDWORD_ARGS *args)
{
  char cmparg[256];
  UdmBuildCmpArgSQL(args->db, args->word_match, args->word, cmparg, sizeof(cmparg));
  args->cmparg= cmparg;
  switch (args->db->DBMode) {
    case UDM_DBMODE_SINGLE: return(UdmFindWordSingle(args));
    case UDM_DBMODE_MULTI:  return(UdmFindWordMulti(args));
    case UDM_DBMODE_BLOB:
     return args->live_updates ? UdmFindWordBlobLiveUpdates(args) :
                                 UdmFindWordBlob(args);
    case UDM_DBMODE_RAWBLOB:return(UdmFindWordRawBlob(args, UDM_RAWBLOB_SEARCH));
  }
  return(UDM_ERROR);
}



/*
  Compare two UDM_SECTIONs by url_id then secno
*/
#if 0
static int
cmp_section_by_urlid_then_secno(UDM_SECTION *s1, UDM_SECTION *s2)
{
  if (s1->url_id < s2->url_id) return -1;
  if (s1->url_id > s2->url_id) return  1;
  if (s1->secno < s2->secno) return -1;
  if (s1->secno > s2->secno) return 1;
  return 0;
}


static int
UdmSectionListListMergeUsingSort(UDM_SECTIONLISTLIST *SrcList, UDM_SECTIONLIST *Dst)
{
  size_t i, nsections;
  UDM_SECTION *p;
  
  for (i= 0, nsections=0 ; i < SrcList->nitems; i++)
    nsections+= SrcList->Item[i].nsections;
  fprintf(stderr, "skip, remain items=%d sections=%d\n",
          SrcList->nitems, nsections);
  if (!nsections)
    return UDM_OK;

  if (!(p= (UDM_SECTION*) UdmMalloc(nsections * sizeof(UDM_SECTION))))
    return UDM_ERROR;

  Dst->nsections= nsections;
  Dst->Section= p;
  
  for (i= 0; i < SrcList->nitems; i++)
  {
    UDM_SECTIONLIST *S= &SrcList->Item[i];
    size_t nbytes= S->nsections * sizeof(UDM_SECTION);
    memcpy(p, S->Section, nbytes);
    p+= S->nsections;
  }
  UdmSort((void*) Dst->Section, Dst->nsections, sizeof(UDM_SECTION),
          (udm_qsort_cmp) cmp_section_by_urlid_then_secno);
  return UDM_OK;
}
#endif

/*
  Compare two UDM_SECTIONLISTs by nsections,
  return the list with more sections first.
*/
static int
cmp_section_list_by_nsections_rev(UDM_SECTIONLIST *s1, UDM_SECTIONLIST *s2)
{
  if (s1->nsections > s2->nsections) return -1;
  if (s1->nsections < s2->nsections) return  1;
  return 0;
}

/*
  Merge sorted lists
*/
static int
UdmSectionListListMergeSorted(UDM_SECTIONLISTLIST *SrcList,
                              UDM_SECTIONLIST *Dst, int opt)
{
  UDM_SECTION **p, **e, *Section;
  size_t list, nlists, nsections;
  int rc= UDM_OK;
  urlid_t *id;
  int newver= 1;
  UDM_SECTIONLIST Short;
  
  bzero((void*) Dst, sizeof(*Dst));
  bzero((void*) &Short, sizeof(Short));
  if (!SrcList->nitems)
    return rc;
  
  if (!(p= UdmMalloc(SrcList->nitems * sizeof(UDM_SECTIONLIST*) * 2)))
    return UDM_ERROR;
  e= p + SrcList->nitems;
  if (!(id= (urlid_t*) UdmMalloc(SrcList->nitems * sizeof(urlid_t))))
    return UDM_ERROR;

  {
    const char *env= getenv ("NEWVER");
    if (env)
      newver= atoi(env);
  }
  if (newver)
  {
    UdmSort((void*) SrcList->Item, SrcList->nitems, sizeof(UDM_SECTIONLIST),
            (udm_qsort_cmp) cmp_section_list_by_nsections_rev);
  }


  /*
    Calc number of non-empty lists
    and total number of sections
  */
  nsections= nlists= 0;
  for (list= 0; list < SrcList->nitems; list++) 
  {
    size_t nsec= SrcList->Item[list].nsections;
    if (nsec)
    {
      /*fprintf(stderr, "[%d] nsec=%d\n", list, nsec); */
      
      if (opt && newver && list > 2 && nsec < SrcList->Item[0].nsections / 10)
      {
        UDM_SECTIONLISTLIST Tmp= *SrcList;
        Tmp.Item+= list;
        Tmp.nitems-= list;
        /*UdmSectionListListMergeUsingSort(&Tmp, &Short);*/
        /* fprintf(stderr, "cut\n");*/
        UdmSectionListListMergeSorted(&Tmp, &Short, 0);
        nsections+= Short.nsections;
        p[nlists]= Short.Section;
        e[nlists]= p[nlists] + Short.nsections;
        id[nlists]= p[nlists]->url_id;
        nlists++;
        /*
        fprintf(stderr, "[%d] nsec=%d ptr=%p MERGED\n",
                list, Short.nsections, Short.Section);
        */
        break;
      }
      nsections+= nsec;
      p[nlists]= SrcList->Item[list].Section;
      e[nlists]= p[nlists] + nsec;
      id[nlists]= p[nlists]->url_id;
      nlists++;
    }
  }
  if (!nlists)
    goto ret;

  if (!(Section= (UDM_SECTION*)UdmMalloc(nsections * sizeof(UDM_SECTION))))
  {
    rc= UDM_ERROR;
    goto ret;
  }
  Dst->Section= Section;
  Dst->msections= nsections;
  
  if (nlists == 1)
    goto one;

  for ( ; ; )
  {
    size_t i, min= 0;
    urlid_t p_min_url_id= id[0];

    /* Find quickly min between the first and the second lists */
    if (id[1] > p_min_url_id)
    {
      /* do nothing */
    }
    else if (id[1] < p_min_url_id)
    {
      min= 1;
      p_min_url_id= id[1];
    }
    else if (p[1]->secno < p[0]->secno)
    {
      min= 1;
    }

    /* Check the other lists */
    for (i= 2; i < nlists; i++)
    {
      if (id[i] > p_min_url_id)
        continue;
      
      if (id[i] < p_min_url_id)
      {
        min= i;
        p_min_url_id= id[i];
        continue;
      }
      
      if (p[i]->secno < p[min]->secno)
        min= i;
    }
    
    *Section++= *p[min]++;
    
    if (p[min] == e[min])
    {
      nlists--;
      p[min]= p[nlists];
      e[min]= e[nlists];
      id[min]= id[nlists];
      if (nlists == 1)
        break;
    }
    else
      id[min]= p[min]->url_id;
  }

one:
  memcpy(Section, *p, (*e - *p) * sizeof(UDM_SECTION));
  Dst->nsections= nsections;
  
ret:
  UDM_FREE(Short.Section);
  UdmFree(p);
  UdmFree(id);
  return rc;
}


static void
UdmMultiWordMergeCoords(UDM_URLCRDLIST *Phrase, size_t wordnum, size_t nparts)
{
  UDM_URL_CRD *To= Phrase->Coords;
  UDM_URL_CRD *End= Phrase->Coords + Phrase->ncoords;
  UDM_URL_CRD *From= Phrase->Coords + nparts - 1;
  UDM_URL_CRD *Prev= Phrase->Coords + nparts - 2;
  
#if 0
  fprintf(stderr, "merge: wordnum=%d nparts=%d ncoords=%d\n",
          wordnum, nparts, Phrase->ncoords);
#endif
  
  if (nparts < 2) /* If one part, keep Phrase unchanged */
    return;
  
  if (Phrase->ncoords < nparts) /* Nothing found */
  {
    Phrase->ncoords= 0;
    return;
  }
  
  for ( ; From < End ; From++, Prev++)
  {
    if (Prev->url_id == From->url_id)
    {
      size_t pos= From->pos;
      size_t sec= From->secno;
      size_t num= From->num;
      if (pos == Prev->pos + 1 &&
          sec == Prev->secno  &&
          num == Prev->num + 1)
      {
        size_t i, nmatches;
        for (nmatches= 2, i= 2; i < nparts; i++)
        {
          if (From[-i].url_id != From->url_id         ||
              From[-i].secno != sec       ||
              From[-i].pos != (pos - i) ||
              From[-i].num != (num - i))
            break;
            
          nmatches++;
        }
        if (nmatches == nparts)
        {
          To->url_id= From->url_id;
          To->pos= pos - nparts + 1;
          To->secno= sec;
          To->num= wordnum;
          To++;
        }
      }
    }
  }
  
  Phrase->ncoords= To - Phrase->Coords;
}


static int
UdmSectionListListToURLCRDList(UDM_URLCRDLIST *List,
                               UDM_SECTIONLISTLIST *SrcList)
{
  size_t list, ncoords= 0, nsections= 0;
  UDM_URL_CRD *Crd;
  for (list=0; list < SrcList->nitems; list++)
  {
    /*UdmSectionListPrint(&SrcList->Item[list]);*/
    ncoords+= SrcList->Item[list].ncoords;
    nsections+= SrcList->Item[list].nsections;
  }
  bzero((void*) List, sizeof(*List));
  if (!(List->Coords= (UDM_URL_CRD*) UdmMalloc(ncoords * sizeof(UDM_URL_CRD))))
    return UDM_ERROR;
  for (Crd= List->Coords, list=0; list < SrcList->nitems; list++)
  {
    size_t sec;
    UDM_SECTIONLIST *SectionList= &SrcList->Item[list];
    for (sec=0; sec < SectionList->nsections; sec++)
    {
      UDM_SECTION *Section= &SectionList->Section[sec];
      size_t coord;
      for (coord= 0; coord < Section->ncoords; coord++)
      {
        UDM_COORD2 *Coord2= &Section->Coord[coord];
        Crd->url_id= Section->url_id;
        Crd->seclen= Section->seclen;
        Crd->pos= Coord2->pos;
        Crd->num= Section->wordnum;
        Crd->secno= Section->secno;
        Crd++;
      }
    }
  }
  UDM_ASSERT(ncoords == Crd - List->Coords);
  List->ncoords= ncoords;
  /*UdmCoordListPrint(List);*/

  return UDM_OK;
}


static int
UdmMultiWordMergeCoords2(UDM_URLCRDLIST *Phrase, 
                         UDM_SECTIONLISTLIST *SrcList,
                         size_t orig_wordnum,
                         size_t nparts)
{
  UdmSectionListListToURLCRDList(Phrase, SrcList);
  UdmSortSearchWordsByURL(Phrase->Coords, Phrase->ncoords);
  UdmMultiWordMergeCoords(Phrase, orig_wordnum, nparts);
  /*UdmCoordListPrint(Phrase);*/
  return UDM_OK;
}


static inline int UdmFindMultiWordSQL (UDM_FINDWORD_ARGS *args)
{
  char *lt, *tmp_word, *tok;
  int rc= UDM_OK;
  UDM_SECTIONLISTLIST OriginalSectionListList;
  size_t orig_wordnum;
  size_t nparts= 0;
  const char *w, *orig_word= args->word;
  unsigned long ticks= UdmStartTimer();
  char delim[]= " \r\t_-./";

  /* Check if the word really multi-part */
  for (w= args->word; ; w++)
  {
    if (!*w)
      return UdmFindOneWordSQL(args); /* No delimiters found */
    
    if (strchr(delim, *w)) /* Delimiter found */
      break;
  }
  
  if (!(tmp_word= UdmStrdup(args->word)))
    return(UDM_ERROR);

  UdmLog(args->Agent, UDM_LOG_DEBUG, "Start searching for multiword '%s'", args->word);
  OriginalSectionListList= args->SectionListList;
  UdmSectionListListInit(&args->SectionListList);
  orig_wordnum= args->wordnum;
  
  for (tok= udm_strtok_r(tmp_word, delim, &lt) ; tok ;
       tok= udm_strtok_r(NULL, delim, &lt))
  {
    unsigned long ticks1= UdmStartTimer();
    args->word= tok;
    UdmLog(args->Agent, UDM_LOG_DEBUG, "Searching for subword '%s'", args->word);
    rc= UdmFindOneWordSQL(args);
    ticks1= UdmStartTimer() -  ticks1;
    UdmLog(args->Agent, UDM_LOG_DEBUG, "Stop searching for subword '%s' %d coords found: %.2f",
           args->word, args->count, (float) ticks1 / 1000);
    /* If the next word wasn't found - no need to search for others. */
    if (rc != UDM_OK || !args->count)
      goto ret;
    nparts++;
    args->wordnum++;
  }
  
  /* All parts returned results. Check phrase */
  {
    UDM_URLCRDLIST Phrase;
    UDM_SECTIONLIST SectionList;
    bzero((void*) &Phrase, sizeof(Phrase));
    UdmMultiWordMergeCoords2(&Phrase, &args->SectionListList, orig_wordnum, nparts);
    if (args->urls.nurls)
      UdmApplyFastLimit(&Phrase, &args->urls);
    if (Phrase.ncoords)
    {
      UdmURLCRDListToSectionList(args, &SectionList, &Phrase);
      UdmSectionListListAdd(&OriginalSectionListList, &SectionList);
    }
    UDM_FREE(Phrase.Coords);
    args->count= Phrase.ncoords;
  }
  
ret:
  UdmFree(tmp_word);
  UdmSectionListListFree(&args->SectionListList);
  args->SectionListList= OriginalSectionListList;
  args->word= orig_word;
  ticks= UdmStartTimer() - ticks;
  UdmLog(args->Agent, UDM_LOG_DEBUG, "Stop searching for multiword '%s'", args->word);
  return rc;
}


static int
UdmFindAlwaysFoundWordSQL(UDM_FINDWORD_ARGS *args)
{
  int rc= UDM_OK;
  UDM_SQLRES SQLRes;
  char qbuf[1024 * 4];
  size_t nrows;
  size_t i;
  UDM_URLCRDLIST CoordList;

  bzero((void*) &CoordList, sizeof(CoordList));

  if (*args->where)
     udm_snprintf(qbuf, sizeof(qbuf), "SELECT url.rec_id FROM url%s WHERE %s",
                  args->db->from, args->where);
  else
  {
    if (args->urls.nurls)
    {
      /*
        A fast limit is loaded.
        No needs to do "SELECT FROM url".
        Populate CoordList from the fast limit instead.
      */
      for (i= 0; i < args->urls.nurls; i++)
      {
        if (UDM_OK != (rc= UdmAddOneCoord(&CoordList, args->urls.urls[i],
                                          0x00010100, (args->wordnum & 0xFF))))
          return UDM_ERROR;
      }
      UdmURLCRDListListAddWithSort2(args, &args->CoordListList, &CoordList);
      return UDM_OK;
    }
    udm_snprintf(qbuf, sizeof(qbuf), "SELECT url.rec_id FROM url");
  }

  if ((rc= UdmSQLQuery(args->db, &SQLRes, qbuf)) != UDM_OK)
    return(rc);
  /* Note that rc is implicitly set to UDM_OK at this point. */
  if (! (nrows= UdmSQLNumRows(&SQLRes)))
    goto err;

  for (i = 0; i < nrows; i++)
    if (UDM_OK != (rc= UdmAddOneCoord(&CoordList,
                              (urlid_t)UDM_ATOI(UdmSQLValue(&SQLRes, i, 0)),
                              0x00010100, (args->wordnum & 0xFF))))
      break;

  if (args->urls.nurls)
    UdmApplyFastLimit(&CoordList, &args->urls);
  if (CoordList.ncoords)
    UdmURLCRDListListAddWithSort2(args, &args->CoordListList, &CoordList);

err:
  UdmSQLFree(&SQLRes);
  return(rc);
}


/* merging two non-empty limits together */
static int
UdmURLLimitJoin(UDM_URL_TMP *urls, UDM_URL_TMP *fl_urls)
{
  size_t src, dst;
  int bfound= fl_urls->exclude ? 0 : 1;
  int bnotfound= fl_urls->exclude ? 1 : 0;
  for (dst= 0, src= 0; src < urls->nurls; src++)
  {
    int allow= UdmBSearch(&urls->urls[src],
                          fl_urls->urls,
                          fl_urls->nurls, sizeof(urlid_t),
                          (udm_qsort_cmp)cmpaurls) ? bfound : bnotfound;
    if (allow)
    {
      urls->urls[dst]= urls->urls[src];
      dst++;
    }
  }
  urls->nurls= dst;
  if (!urls->nurls)
    urls->empty= 1;
  return UDM_OK;
}


static int
UdmURLLimitUnion(UDM_URL_TMP *a, UDM_URL_TMP *b)
{
  if (b->nurls)
  {
    size_t nbytes= sizeof(urlid_t) * (a->nurls + b->nurls);
    a->urls= (urlid_t*) UdmRealloc(a->urls, nbytes);
    memcpy(a->urls + a->nurls, b->urls, b->nurls * sizeof(urlid_t));
    a->nurls+= b->nurls;
    UdmSort(a->urls, a->nurls, sizeof(urlid_t), (udm_qsort_cmp)cmpaurls);
  }
  return UDM_OK;
}


static int
UdmURLLimitCopy(UDM_URL_TMP *a, UDM_URL_TMP *b)
{
  size_t nbytes= sizeof(urlid_t) * b->nurls;
  a->urls= (urlid_t*) UdmRealloc(a->urls, nbytes);
  a->nurls= b->nurls;
  a->exclude= b->exclude;
  memcpy(a->urls, b->urls, nbytes);
  return UDM_OK;
}


/*
  Merge two limits (some of them can be empty).
  The second limit is kept untouched.
*/
static int
UdmURLLimitMerge(UDM_URL_TMP *a, UDM_URL_TMP *b)
{
  int rc;
  if (a->exclude && b->exclude)
  {
    rc= UdmURLLimitUnion(a, b);
  }
  if (a->nurls && b->nurls)
  {
    rc= UdmURLLimitJoin(a, b);
  }
  else if (!a->nurls && b->nurls)
  {
    rc= UdmURLLimitCopy(a, b);
  }
  else if (!b->nurls)
  {
    if (!b->exclude)
      a->empty= 1;
  }
  return rc;
}


static int
UdmFindCrossWord(UDM_FINDWORD_ARGS *args)
{
  UDM_URLCRDLIST CoordList;
  char cmparg[256];
  int rc;

  bzero((void*) &CoordList, sizeof(CoordList));

  UdmBuildCmpArgSQL(args->db, args->word_match, args->word,
                    cmparg, sizeof(cmparg));
  args->cmparg= cmparg;

  if (UDM_OK != (rc= UdmFindWordSingleInternal(args, &CoordList,
                                               "crossdict", 1)))
    return rc;

  if (args->urls.nurls)
    UdmApplyFastLimit(&CoordList, &args->urls);
  if (CoordList.ncoords)
  {
    UdmURLCRDListListAddWithSort2(args, &args->CoordListList, &CoordList);
    args->count= CoordList.ncoords;
  }
  return UDM_OK;
}


static int
UdmCheckIndex(UDM_AGENT *query, UDM_DB *db)
{
  int tm, rc;
  if (UDM_OK != (rc= UdmBlobReadTimestamp(query, db, &tm, 0)))
    return rc;
  if (tm)
    return UDM_OK;
#ifdef WIN32
  sprintf(query->Conf->errstr, "Inverted word index not found. Probably you forgot to run 'Create fast index'.");
#else
  sprintf(query->Conf->errstr, "Inverted word index not found. Probably you forgot to run 'indexer -Eblob'.");
#endif
  return UDM_ERROR;
}


static int
UdmMergeWords(UDM_FINDWORD_ARGS *args, UDM_SECTIONLIST *SectionList)
{
  unsigned long ticks= UdmStartTimer();
  UDM_AGENT *A= args->Agent;
  size_t nresults;

  UdmLog(A, UDM_LOG_DEBUG, "Start merging %d lists", args->SectionListList.nitems);
  UdmSectionListListMergeSorted(&args->SectionListList, SectionList, 1);
  ticks= UdmStartTimer() - ticks;
  UdmLog(A, UDM_LOG_DEBUG, "Merged %d lists %d sections: %.2f",
         args->SectionListList.nitems, SectionList->nsections, (float) ticks / 1000);
  nresults= SectionList->nsections;

  if (!nresults && args->db->DBMode == UDM_DBMODE_BLOB)
    return UdmCheckIndex(A, args->db);
  return UDM_OK;
}


int UdmFindWordsSQL(UDM_AGENT *query, UDM_RESULT *Res, UDM_DB *db,
                    size_t num_best_rows)
{
  size_t    wordnum;
  int    word_match;
  char   wf[256];
  const char  *where, *always_found_word, *fl;
  int    use_crosswords, use_qcache;
  unsigned long   ticks=UdmStartTimer();
  int    rc= UDM_OK;
  UDM_FINDWORD_ARGS args;
  UDM_URL_TMP fl_urls;
  UDM_SECTIONLIST SectionList;

  bzero((void*) &fl_urls, sizeof(fl_urls));
  bzero((void*) &args.urls, sizeof(args.urls));
  bzero((void*) &args.live_update_deleted_urls, sizeof(args.live_update_deleted_urls));
  bzero((void*) &args.live_update_active_urls, sizeof(args.live_update_active_urls));
  bzero((void*) &args.SectionListList, sizeof(args.SectionListList));
  bzero((void*) &SectionList, sizeof(SectionList));
  UdmWideWordListInit(&args.CollationMatches);
  UdmURLCRDListListInit(&args.CoordListList);
  
  UDM_GETLOCK(query, UDM_LOCK_CONF);
  word_match = UdmMatchMode(UdmVarListFindStr(&query->Conf->Vars, "wm", "wrd"));
  where = BuildWhere(query->Conf, db);
  use_crosswords = !strcasecmp(UdmVarListFindStr(&query->Conf->Vars, "CrossWords", "no"), "yes");
  use_qcache = UdmVarListFindBool(&db->Vars, "qcache", 0);
  UdmWeightFactorsInit2(wf, &query->Conf->Vars, &db->Vars, "wf");
  always_found_word= UdmVarListFindStr(&query->Conf->Vars, "AlwaysFoundWord", NULL);
  fl= UdmVarListFindStr(&query->Conf->Vars, "fl", UdmVarListFindStr(&db->Vars, "fl", ""));
  args.save_section_size= UdmVarListFindBool(&query->Conf->Vars, "SaveSectionSize", 1);
  args.live_updates= UdmVarListFindBool(&db->Vars, "LiveUpdates", 0);

  UDM_RELEASELOCK(query, UDM_LOCK_CONF);

  if ((db->DBMode == UDM_DBMODE_BLOB && where) || fl[0])
  {
    UdmLog(query,UDM_LOG_DEBUG, "Start loading limits");
    ticks= UdmStartTimer();
    if (*where)
    {
      LoadURL(db, where, &args.urls);
      UdmLog(query, UDM_LOG_DEBUG, "WHERE limit loaded. %d URLs found", args.urls.nurls);
    }
    if (!args.urls.empty && fl[0])
    {
      char name[64];
      const char *q;
      if ((fl_urls.exclude= (fl[0] == '-')))
        fl++;
      udm_snprintf(name, sizeof(name), "Limit.%s", fl);
      if (UDM_OK != (rc= ((q= UdmVarListFindStr(&query->Conf->Vars, name, NULL)) ?
                         UdmLoadSlowLimit(db, &fl_urls, q) :
                         UdmBlobLoadFastURLLimit(db, fl, &fl_urls))))
        goto ret;
      UdmLog(query,UDM_LOG_DEBUG, "Limit '%s' loaded%s%s %d URLs",
             fl, fl_urls.exclude ? " type=excluding" : "",
             q ? " source=slow":"", fl_urls.nurls);

      UdmURLLimitMerge(&args.urls, &fl_urls);
    }
    ticks= UdmStartTimer()-ticks;
    UdmLog(query,UDM_LOG_DEBUG,"Stop loading limits, %d URLs found: %.2f",
           args.urls.nurls, (float)ticks/1000);
    if (args.urls.empty)
      goto ret;
  }

  if (db->DBMode == UDM_DBMODE_BLOB && args.live_updates)
  {
    UdmLog(query, UDM_LOG_DEBUG, "Start loading LiveUpdate url_id list");
    if (UDM_OK != (rc= UdmLoadSlowLimit(db, &args.live_update_deleted_urls,
                                        "SELECT url_id FROM bdicti WHERE state=0")))
      goto ret;
    UdmLog(query, UDM_LOG_DEBUG,
           "Stop loading LiveUpdate url_id list: %d updated docs found",
           args.live_update_deleted_urls.nurls);
    args.live_update_deleted_urls.exclude= 1;
    UdmURLLimitCopy(&args.live_update_active_urls, &args.urls);
    UdmURLLimitMerge(&args.urls, &args.live_update_deleted_urls);
  }

  
  args.Agent= query;
  args.db= db;
  args.WWList= &Res->WWList;
  args.where= where;
  args.wf= wf;
  args.word_match= word_match;


  /* Now find each word */
  for(wordnum=0; wordnum < Res->WWList.nwords; wordnum++)
  {
    UDM_WIDEWORD  *W=&Res->WWList.Word[wordnum];

    if (W->origin == UDM_WORD_ORIGIN_STOP) continue;

    args.wordnum= wordnum;
    args.count= 0;
    args.word= W->word;
    args.word_match= W->match;
    args.secno= W->secno;

    ticks=UdmStartTimer();
    UdmLog(query,UDM_LOG_DEBUG,"Start search for '%s'",Res->WWList.Word[wordnum].word);

    /*
       For now SYNONYMs only are treated as a possible multi-word
       origin. Probably it will be changed in future, so we will
       use this feature for phrase search.
     */
    if (always_found_word && !strcmp(W->word, always_found_word))
      rc= UdmFindAlwaysFoundWordSQL(&args);
    else if (W->origin == UDM_WORD_ORIGIN_SYNONYM || W->phrwidth > 0)
      rc= UdmFindMultiWordSQL(&args);
    else
      rc= UdmFindOneWordSQL(&args);

    if (rc != UDM_OK)
      goto ret;

    /*
      If CollationMatches is not empty, then we should skip
      updating word statistics here - it will be updated in
      the loop after UdmSortAndGroupByURL().
     */
    if (!args.CollationMatches.nwords)
      Res->WWList.Word[wordnum].count+= args.count;

    ticks= UdmStartTimer() - ticks;
    UdmLog(query, UDM_LOG_DEBUG, "Stop search for '%s'\t%.2f found %u coords",
           Res->WWList.Word[wordnum].word, (float)ticks / 1000, args.count);

    
    if (use_crosswords)
    {
      UdmLog(query,UDM_LOG_DEBUG,"Start search for crossword '%s'", W->word);
      args.count= 0;
      if (UDM_OK != (rc= UdmFindCrossWord(&args)))
        goto ret;
      Res->WWList.Word[wordnum].count+= args.count;
      
      ticks= UdmStartTimer()-ticks;
      UdmLog(query, UDM_LOG_DEBUG,
            "Stop search for crossword '%s'\t%.2f %d found",
             args.word, (float)ticks/1000, args.count);
    }
  }

  if (UDM_OK != (rc= UdmMergeWords(&args, &SectionList)))
    goto ret;
  
  UdmSortAndGroupByURL(&args, query, Res, &SectionList, db, num_best_rows);

  /* 
     We cannot add collation matches before
     UdmSortAndGroupByURL - to use optimized groupping
     functions when WWList->nwords==1
  */
  if (args.CollationMatches.nwords)
  {
    size_t i;
    for (i= 0; i < args.CollationMatches.nwords; i++)
    {
      UdmWideWordListAdd(&Res->WWList, &args.CollationMatches.Word[i]);
    }
  }


ret:
  UDM_FREE(fl_urls.urls);
  UDM_FREE(args.urls.urls);
  UDM_FREE(args.live_update_active_urls.urls);
  UDM_FREE(args.live_update_deleted_urls.urls);
  UdmWideWordListFree(&args.CollationMatches);
  UdmURLCRDListListFree(&args.CoordListList);
  UdmSectionListListFree(&args.SectionListList);
  UdmSectionListFree(&SectionList);
  return rc;
}


static int UdmSQLQueryOneRowInt(UDM_DB *db, int *res, const char *qbuf)
{
  UDM_SQLRES sqlRes;
  int rc;
  if (UDM_OK != (rc= UdmSQLQuery(db, &sqlRes, qbuf)))
    return rc;
  if (UdmSQLNumRows(&sqlRes) < 1)
  {
    rc= UDM_ERROR;
    *res= 0;
    sprintf(db->errstr, "Query should have returned one row");
  }
  else
    *res= UDM_ATOI(UdmSQLValue(&sqlRes, 0, 0));
  UdmSQLFree(&sqlRes);
  return rc;
}


int UdmTrackSQL(UDM_AGENT * query, UDM_RESULT *Res, UDM_DB *db)
{
  char    *qbuf;
  char    *text_escaped;
  const char  *words=UdmVarListFindStr(&query->Conf->Vars,"q",""); /* "q-lc" was here */
  const char      *IP = UdmVarListFindStr(&query->Conf->Vars, "IP", "");
  size_t          i, escaped_len, qbuf_len;
  int             res, qtime, rec_id;
  const char      *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  const char      *value= (db->DBType == UDM_DB_IBASE ||
                           db->DBType == UDM_DB_MIMER ||
                           db->DBType == UDM_DB_DB2   ||
                           db->DBType == UDM_DB_ORACLE8) ? "sval" : "value";
  
  if (*words == '\0') return UDM_OK; /* do not store empty queries */

  escaped_len = 4 * strlen(words);
  qbuf_len = escaped_len + 4096;

  if ((qbuf = (char*)UdmMalloc(qbuf_len)) == NULL) return UDM_ERROR;
  if ((text_escaped = (char*)UdmMalloc(escaped_len)) == NULL)
  { 
    UDM_FREE(qbuf);
    return UDM_ERROR;
  }
  
  /* Escape text to track it  */
  UdmSQLEscStr(db, text_escaped, words, strlen(words));
  
  if (db->DBType == UDM_DB_IBASE ||
      db->DBType == UDM_DB_MIMER ||
      db->DBType == UDM_DB_ORACLE8)
  {
    const char *next;
    switch (db->DBType)
    {
      case UDM_DB_IBASE: next= "SELECT GEN_ID(qtrack_GEN,1) FROM rdb$database"; break;
      case UDM_DB_MIMER: next= "SELECT NEXT_VALUE OF qtrack_GEN FROM system.onerow"; break;
      case UDM_DB_ORACLE8: next= "SELECT qtrack_seq.nextval FROM dual"; break;
    }
    if (UDM_OK != (res= UdmSQLQueryOneRowInt(db, &rec_id, next)))
      goto UdmTrack_exit;
    udm_snprintf(qbuf, qbuf_len - 1,
                 "INSERT INTO qtrack (rec_id,ip,qwords,qtime,wtime,found) "
                 "VALUES "
                 "(%d,'%s','%s',%d,%d,%d)",
                 rec_id, IP, text_escaped, qtime= (int)time(NULL),
                 Res->work_time, Res->total_found);
    if (UDM_OK != (res = UdmSQLQuery(db, NULL, qbuf)))
      goto UdmTrack_exit;
  }
  else
  {
    udm_snprintf(qbuf, qbuf_len - 1,
                 "INSERT INTO qtrack (ip,qwords,qtime,wtime,found) "
                 "VALUES "
                 "('%s','%s',%d,%d,%d)",
                 IP, text_escaped, qtime= (int)time(NULL),
                 Res->work_time, Res->total_found);
  
    if (UDM_OK != (res= UdmSQLQuery(db, NULL, qbuf)))
      goto UdmTrack_exit;
    
    if (db->DBType == UDM_DB_MYSQL)
      udm_snprintf(qbuf, qbuf_len - 1, "SELECT last_insert_id()");
    else
      udm_snprintf(qbuf, qbuf_len - 1, "SELECT rec_id FROM qtrack WHERE ip='%s' AND qtime=%d", IP, qtime);
    if (UDM_OK != (res= UdmSQLQueryOneRowInt(db, &rec_id, qbuf)))
      goto UdmTrack_exit;
  }
  
  for (i = 0; i < query->Conf->Vars.nvars; i++)
  {
    UDM_VAR *Var = &query->Conf->Vars.Var[i];
    if (strncasecmp(Var->name, "query.",6)==0 && strcasecmp(Var->name, "query.q") && strcasecmp(Var->name, "query.BrowserCharset")
       && strcasecmp(Var->name, "query.IP")  && Var->val != NULL && *Var->val != '\0') 
    {
      udm_snprintf(qbuf, qbuf_len,
                   "INSERT INTO qinfo (q_id,name,%s) "
                   "VALUES "
                   "(%s%i%s,'%s','%s')", 
      value, qu, rec_id, qu, &Var->name[6], Var->val);
      res= UdmSQLQuery(db, NULL, qbuf);
      if (res != UDM_OK) goto UdmTrack_exit;
    }
  }
UdmTrack_exit:
  UDM_FREE(text_escaped);
  UDM_FREE(qbuf);
  return res;
}

static int UpdateShows(UDM_DB *db, urlid_t url_id)
{
  char qbuf[64];
  udm_snprintf(qbuf, sizeof(qbuf), "UPDATE url SET shows = shows + 1 WHERE rec_id = %s%i%s",
               (db->DBType == UDM_DB_PGSQL) ? "'" : "",
               url_id,
               (db->DBType == UDM_DB_PGSQL) ? "'" : "");
  return UdmSQLQuery(db, NULL, qbuf);
}

static void SQLResToSection(UDM_SQLRES *R, UDM_VARLIST *S, size_t row)
{
  const char *sname=UdmSQLValue(R,row,1);
  const char *sval=UdmSQLValue(R,row,2);
  UdmVarListAddStr(S, sname, sval);
}


static size_t
UdmDBNum(UDM_RESULT *Res, size_t n)
{
  UDM_URLDATA *Data= &Res->URLData.Item[n + Res->first];
  return UDM_COORD2DBNUM(Data->coord);
}


static int
UdmResAddURLInfoUsingIN(UDM_RESULT *Res, UDM_DB *db, size_t dbnum,
                        const char *qbuf)
{
  int rc;
  UDM_SQLRES SQLres;
  size_t j, sqlrows;
  
  if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLres,qbuf)))
    return rc;
    
  for(sqlrows= UdmSQLNumRows(&SQLres), j=0; j<Res->num_rows; j++)
  {
    if (UdmDBNum(Res, j) == dbnum)
    {
      size_t i;
      UDM_DOCUMENT *D=&Res->Doc[j];
      urlid_t      url_id = UdmVarListFindInt(&D->Sections, "ID", 0);
      for(i = 0; i < sqlrows; i++)
      {
        if(url_id == UDM_ATOI(UdmSQLValue(&SQLres,i,0)))
          SQLResToSection(&SQLres, &D->Sections, i);
      }
    }
  }
  UdmSQLFree(&SQLres);
  return UDM_OK;
}


static int
UdmDocAddURLInfo(UDM_DOCUMENT *D, UDM_DB *db, const char *qbuf)
{
  UDM_SQLRES SQLres;
  int rc;
  size_t row;
  
  if(UDM_OK != (rc= UdmSQLQuery(db,&SQLres,qbuf)))
    return rc;

  for(row= 0; row < UdmSQLNumRows(&SQLres); row++)
    SQLResToSection(&SQLres, &D->Sections, row);
  UdmSQLFree(&SQLres);
  return rc;
}


int UdmResAddDocInfoSQL(UDM_AGENT *query, UDM_DB *db, UDM_RESULT *Res, size_t dbnum)
{
  size_t      i;
  char        instr[1024*4]="";
  char        qbuf[1024*4];
  UDM_SQLRES  SQLres;
  int         rc;
  int         use_showcnt = !strcasecmp(UdmVarListFindStr(&query->Conf->Vars, "PopRankUseShowCnt", "no"), "yes");
  int         use_category = UdmVarListFindStr(&query->Conf->Vars, "cat", NULL) ? 1 : 0;
  double      pr, ratio = 0.0;
  const       char *hi_priority= db->DBType == UDM_DB_MYSQL ? "HIGH_PRIORITY" : "";
  int         use_urlinfo= UdmVarListFindBool(&query->Conf->Vars, "LoadURLInfo", 1);
  int         use_taginfo= UdmVarListFindBool(&query->Conf->Vars, "LoadTagInfo", 0);
  if(!Res->num_rows)return UDM_OK;
  if (use_showcnt) ratio = UdmVarListFindDouble(&query->Conf->Vars, "PopRankShowCntRatio", 25.0);
  UdmLog(query, UDM_LOG_DEBUG, "use_showcnt: %d  ratio: %f", use_showcnt, ratio);
  
  for (i= 0; i < Res->num_rows; i++)
  {
    UdmVarListReplaceInt(&Res->Doc[i].Sections, "id",
                         Res->URLData.Item[i + Res->first].url_id);
  }
  
  if(db->DBSQL_IN)
  {
    size_t  j, sqlrows;
    
    /* Compose IN string and set to zero url_id field */
    for(i=0; i < Res->num_rows; i++)
    {
      if (UdmDBNum(Res, i) == dbnum)
      {
        const char *comma= instr[0] ? "," : "";
        const char *squot= db->DBType == UDM_DB_PGSQL ? "'" : "";
        sprintf(UDM_STREND(instr),"%s%s%i%s", comma, squot,
                UdmVarListFindInt(&Res->Doc[i].Sections, "ID", 0), squot);
      }
    }
    
    if (!instr[0])
      return UDM_OK;
    
    udm_snprintf(qbuf,sizeof(qbuf),"SELECT %s rec_id,url,last_mod_time,docsize,next_index_time,referrer,crc32,site_id,pop_rank FROM url WHERE rec_id IN (%s)", hi_priority,instr);
    if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLres,qbuf)))
      return rc;
    
    for(sqlrows= UdmSQLNumRows(&SQLres), j=0; j<Res->num_rows; j++)
    {
      if (UdmDBNum(Res, j) == dbnum)
      {
        UDM_DOCUMENT *D= &Res->Doc[j];
        urlid_t      url_id= UdmVarListFindInt(&D->Sections, "ID", 0);
        for(i = 0; i < sqlrows; i++)
        {
          if(url_id == UDM_ATOI(UdmSQLValue(&SQLres,i,0)))
          {
            SQLResToDoc(query->Conf, D, &SQLres, i);
            if (use_showcnt &&
                (pr= atof(UdmVarListFindStr(&D->Sections, "Score", "0.0"))) >= ratio)
                UpdateShows(db, url_id);
            break;
          }
        }
      }
    }
    UdmSQLFree(&SQLres);
    
    
    if (use_category)
    {
      udm_snprintf(qbuf, sizeof(qbuf),"SELECT u.rec_id,'Category' as sname,c.path FROM url u,server s,categories c WHERE u.rec_id IN (%s) AND u.server_id=s.rec_id AND s.category=c.rec_id", instr); 
      if (UDM_OK != (rc= UdmResAddURLInfoUsingIN(Res, db, dbnum, qbuf)))
        return rc;
    }
    
    if (use_taginfo)
    {
      udm_snprintf(qbuf, sizeof(qbuf),
                   "SELECT u.rec_id, 'tag', tag FROM url u, server s "
                   "WHERE  u.rec_id in (%s) AND u.server_id=s.rec_id",
                   instr); 
      if (UDM_OK != (rc= UdmResAddURLInfoUsingIN(Res, db, dbnum, qbuf)))
        return rc;
    }
    
    if (use_urlinfo)
    {
      udm_snprintf(qbuf,sizeof(qbuf),"SELECT url_id,sname,sval FROM urlinfo WHERE url_id IN (%s)",instr);
      if (UDM_OK != (rc= UdmResAddURLInfoUsingIN(Res, db, dbnum, qbuf)))
        return rc;
    }
  }
  else
  {
    for(i=0;i<Res->num_rows;i++)
    {
      UDM_DOCUMENT  *D=&Res->Doc[i];
      urlid_t  url_id= UdmVarListFindInt(&D->Sections, "ID", 0);
      size_t   row;
      
      if (UdmDBNum(Res, i) != dbnum)
        continue;
      
      sprintf(qbuf,"SELECT rec_id,url,last_mod_time,docsize,next_index_time,referrer,crc32,site_id,pop_rank FROM url WHERE rec_id=%i", url_id);
      if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLres,qbuf)))
        return rc;
      
      if(UdmSQLNumRows(&SQLres))
      {
        SQLResToDoc(query->Conf, D, &SQLres, 0);
          if (use_showcnt &&
              (pr= atof(UdmVarListFindStr(&D->Sections, "Score", "0.0"))) >= ratio)
              UpdateShows(db, url_id);
      }
      UdmSQLFree(&SQLres);
      
      if (use_category)
      {
        sprintf(qbuf,"SELECT u.rec_id,c.path FROM url u,server s,categories c WHERE rec_id=%i AND u.server_id=s.rec_id AND s.category=c.rec_id", url_id);
        if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLres,qbuf)))
          return rc;
        if(UdmSQLNumRows(&SQLres))
        {
          UdmVarListReplaceStr(&D->Sections, "Category", UdmSQLValue(&SQLres, i, 1));
        }
        UdmSQLFree(&SQLres);
      }
      
      if (use_taginfo)
      {
        udm_snprintf(qbuf, sizeof(qbuf),
                     "SELECT u.rec_id, 'tag', tag FROM url u, server s "
                     "WHERE  u.rec_id=%d AND u.server_id=s.rec_id", url_id); 
        if(UDM_OK != (rc= UdmDocAddURLInfo(D, db, qbuf)))
          return rc;
      }
      
      if (use_urlinfo)
      {
        sprintf(qbuf,"SELECT url_id,sname,sval FROM urlinfo WHERE url_id=%i", url_id);
        if(UDM_OK != (rc= UdmDocAddURLInfo(D, db, qbuf)))
          return rc;
      }
    }
  }
  return(UDM_OK);
}



int UdmURLDataSQL(UDM_ENV *Conf, UDM_URLDATALIST *L, UDM_DB *db)
{
  UDM_SQLRES  SQLres;
  size_t    i;
  int    rc=UDM_OK;
  
  L->nitems=0;
  L->Item=NULL;
  
  if (UDM_OK != (rc=UdmSQLQuery(db, &SQLres, "SELECT rec_id,site_id,pop_rank,last_mod_time FROM url ORDER by rec_id")))
    return rc;
  
  L->nitems = UdmSQLNumRows(&SQLres);
  L->Item = (UDM_URLDATA*)UdmMalloc(L->nitems*sizeof(UDM_URLDATA));
  if (L->Item == NULL)
  {
    L->nitems = 0;
    rc=UDM_ERROR;
    goto freeex;
  }
  for (i = 0; i < L->nitems; i++)
  {
    L->Item[i].url_id = UDM_ATOI(UdmSQLValue(&SQLres, i, 0));
    L->Item[i].site_id = UDM_ATOI(UdmSQLValue(&SQLres, i, 1));
    L->Item[i].pop_rank = UDM_ATOF(UdmSQLValue(&SQLres, i, 2));
    L->Item[i].last_mod_time = UDM_ATOU(UdmSQLValue(&SQLres, i, 3));
  }
freeex:
  UdmSQLFree(&SQLres);
  return rc;
}



/***********************************************************/
/*  HTDB stuff:  Indexing of database content              */
/***********************************************************/

#define MAXNPART 32

static void include_params(UDM_DB *db,const char *src,char *path,char *dst,
                           size_t start, int limit)
{
  size_t nparts= 0;
  char *part[MAXNPART];
  char *lt;
  
  part[0]= udm_strtok_r(path, "/", &lt);
  while (part[nparts] && nparts < MAXNPART)
  {
    part[++nparts]= udm_strtok_r(NULL, "/", &lt);
  }
  
  for (*dst='\0'; *src; )
  {
    if(*src == '\\')
    {
      *dst++= src[1];
      *dst= '\0';
      src+= 2;
      continue;
    }
    if (*src == '$')
    {
      int i= atoi(++src)-1;
      
      while((*src>='0')&&(*src<='9'))src++;
      if ((i >= 0) && (i < nparts))
      {
        UdmUnescapeCGIQuery(dst,part[i]);
        while(*dst)dst++;
      }
      continue;
    }
    *dst++=*src++;
    *dst='\0';
    continue;
  }
  
  if (limit)
  {
    switch (db->DBType)
    {
      case UDM_DB_MYSQL:
        sprintf(dst, " LIMIT %d,%d", start, limit);
        break;
      case UDM_DB_PGSQL:
      default:
        sprintf(dst, " LIMIT %d OFFSET %d", limit, start);
        break;
    }
  }
}


static int
UdmHTDBCreateHTTPResponse(UDM_AGENT *Indexer,
                          UDM_DOCUMENT *Doc,
                          UDM_SQLRES *SQLres)
{
  char *to;
  size_t i; 
  for (to= Doc->Buf.buf, i= 0; i < UdmSQLNumCols(SQLres); i++)
  {
    size_t len;
    const char *from;
    if (i > 0)
    {
      memcpy(to, "\r\n", 2);
      to+= 2;
    }
    len= UdmSQLLen(SQLres, 0, i);
    from= UdmSQLValue(SQLres, 0, i);
    if (len == 1 && *from == ' ')
    {
      /*
         Sybase+unixODBC returns space character instead
         of an empty string executing this query:
           SELECT '' FROM t1;
      */
    }
    else
    {
      memcpy(to, from, len);
      to+= len;
    }
  }
  *to= '\0';
  return UDM_OK;
}


static void
UdmRemoveWiki(char *str, char *strend)
{
  for ( ; str < strend ; str++)
  {
    if (*str == '[')
    {
      int smcount= 0;
      for (*str++= ' ' ; str < strend ; str++)
      {
        if (*str == ']')
        {
          *str++= ' ';
          break;
        }
        if (*str == '[')
          UdmRemoveWiki(str, strend);
        if (*str == ':')
        {
          *str= ' ';
          smcount++;
        }
        if (smcount < 2)
          *str= ' ';
      }
    }
  }
}


static int
UdmHTDBProcessNonHTTPResponse(UDM_AGENT *A,
                              UDM_DOCUMENT *D,
                              UDM_SQLRES *SQLRes)
{
  UDM_TEXTITEM Item;
  UDM_DSTR tbuf;
  int rc= UDM_OK, status= 200;
  size_t row, nrows, ncols= UdmSQLNumCols(SQLRes), length= 0;
  char dbuf[128]= "";
  bzero((void*)&Item, sizeof(Item));
  UdmDSTRInit(&tbuf, 1024);
  
  for (row=0, nrows= UdmSQLNumRows(SQLRes); row < nrows; row++)
  {
    size_t col;
    for (col= 0; col  < ncols; col++)
    {
      UDM_VAR *Sec;
      const char *sqlname= SQLRes->Fields[col].sqlname;
      const char *sqlval= UdmSQLValue(SQLRes, row, col);
      Item.section_name= SQLRes->Fields[col].sqlname;
      if ((Sec= UdmVarListFind(&D->Sections, Item.section_name)))
      {
        if (Sec->flags & UDM_VARFLAG_HTMLSOURCE)
        {
          UDM_HTMLTOK tag;
          const char *htok, *last;
          UdmHTMLTOKInit(&tag);
          for (htok= UdmHTMLToken(sqlval, &last, &tag);
               htok;
               htok = UdmHTMLToken(NULL, &last, &tag))
          {
            if (tag.type == UDM_HTML_TXT &&
                !tag.script && !tag.comment && !tag.style)
            {
              UdmDSTRReset(&tbuf);
              if (Sec->flags & UDM_VARFLAG_WIKI)
                UdmRemoveWiki((char*) htok, (char*) last);
              UdmDSTRAppend(&tbuf, htok, last - htok);
              Item.str= tbuf.data;
              Item.section= Sec->section;
              Item.section_name= Sec->name;
              UdmTextListAdd(&D->TextList, &Item);
            }
          }
        }
        else
        {
          Item.str= (char*) sqlval;
          Item.section= Sec->section;
          UdmTextListAdd(&D->TextList, &Item);
        }
        length+= UdmSQLLen(SQLRes, row, col);
      }
      if (!strcasecmp(sqlname, "status"))
        status= atoi(sqlval);
      else if (!strcasecmp(sqlname, "last_mod_time"))
      {
        int last_mod_time= atoi(sqlval);
        strcpy(dbuf, "Last-Modified: ");
        UdmTime_t2HttpStr(last_mod_time, dbuf + 15);
      }
    }
  }
  
  UdmDSTRFree(&tbuf);
  
  D->Buf.content_length= length;
  sprintf(D->Buf.buf, "HTTP/1.0 %d %s\r\n"
                      "Content-Type: mnogosearch/htdb\r\n%s%s\r\n",
                      status, UdmHTTPErrMsg(status),
                      dbuf[0] ? dbuf : "", dbuf[0] ? "\r\n" : "");
  return rc;
}


int UdmHTDBGet(UDM_AGENT *Indexer,UDM_DOCUMENT *Doc)
{
  char    *qbuf;
  char    *end=Doc->Buf.buf;
  UDM_SQLRES  SQLres;
  UDM_URL    realURL;
  UDM_DB      dbnew, *db= &dbnew;
  const char  *url=UdmVarListFindStr(&Doc->Sections,"URL","");
  const char  *htdblist=UdmVarListFindStr(&Doc->Sections,"HTDBList","");
  const char  *htdbdoc=UdmVarListFindStr(&Doc->Sections,"HTDBDoc","");
  const char  *htdbaddr = UdmVarListFindStr(&Doc->Sections, "HTDBAddr", NULL);
  int    usehtdburlid = UdmVarListFindInt(&Indexer->Conf->Vars, "UseHTDBURLId", 0);
  int    rc = UDM_OK;
  
  Doc->Buf.buf[0]=0;
  UdmURLInit(&realURL);
  UdmURLParse(&realURL,url);
  if ((qbuf = (char*)UdmMalloc(4 * 1024 + strlen(htdblist) + strlen(htdbdoc))) == NULL) return UDM_ERROR;
  *qbuf = '\0';
  
  
  if (htdbaddr)
  {
    UdmDBInit(&dbnew);
    if (UDM_OK != (rc= UdmDBSetAddr(db, htdbaddr, UDM_OPEN_MODE_READ)))
    {
      UdmLog(Indexer,UDM_LOG_ERROR, "Wrong HTDB address");
      return rc;
    }
  }
  else
  {
    if (Indexer->Conf->dbl.nitems != 1)
    {
      UdmLog(Indexer, UDM_LOG_ERROR, "HTDB cannot work with several DBAddr without HTDBAddr");
      return UDM_ERROR;
    }
    db= &Indexer->Conf->dbl.db[0];
  }

  if(realURL.filename != NULL)
  {
    char real_path[1024]="";
    
    udm_snprintf(real_path,sizeof(real_path)-1,"%s%s",realURL.path,realURL.filename);
    real_path[sizeof(real_path)-1]='\0';
    include_params(db,htdbdoc, real_path, qbuf, 0, 0);
    UdmLog(Indexer, UDM_LOG_DEBUG, "HTDBDoc: %s\n",qbuf);
    if(UDM_OK != (rc = UdmSQLQuery(db, &SQLres, qbuf)))
    {
      goto HTDBexit;
    }
    if(UdmSQLNumRows(&SQLres)==1)
    {
      if (!strncmp(UdmSQLValue(&SQLres,0,0),"HTTP/",5))
        UdmHTDBCreateHTTPResponse(Indexer, Doc, &SQLres);
      else
        UdmHTDBProcessNonHTTPResponse(Indexer, Doc, &SQLres);
    }
    else
    {
      sprintf(Doc->Buf.buf,"HTTP/1.0 404 Not Found\r\n\r\n");
    }
    UdmSQLFree(&SQLres);
  }
  else
  {
    size_t  i, start = 0;
    urlid_t  url_id = UdmVarListFindInt(&Doc->Sections, "ID", 0);
    const size_t  htdblimit = UdmVarListFindUnsigned(&Doc->Sections, "HTDBLimit", 0);
    int  done = 0, hops=UdmVarListFindInt(&Doc->Sections,"Hops",0);


    sprintf(Doc->Buf.buf,"HTTP/1.0 200 OK\r\nContent-type: text/html\r\n\r\n<HTML><BODY>\n");
    end=UDM_STREND(Doc->Buf.buf);
    strcpy(end,"</BODY></HTML>\n");

    while (!done)
    {
      size_t nrows;
      include_params(db,htdblist, realURL.path, qbuf, start, htdblimit);
      UdmLog(Indexer, UDM_LOG_DEBUG, "HTDBList: %s\n",qbuf);
      if(UDM_OK != (rc = UdmSQLQuery(db, &SQLres, qbuf)))
      goto HTDBexit;

      nrows= UdmSQLNumRows(&SQLres);
      done= htdblimit ? (htdblimit != nrows) : 1;
      start+= nrows;

      for(i = 0; i < nrows; i++)
      {
      UDM_HREF Href;
      UdmHrefInit(&Href);
      Href.referrer=url_id;
      Href.hops=hops+1;
      Href.url = (char*)UdmStrdup(UdmSQLValue(&SQLres,i,0));
      Href.method=UDM_METHOD_GET;
      Href.rec_id = usehtdburlid ? atoi(Href.url) : 0;
      UdmHrefListAdd(&Doc->Hrefs,&Href);
      UDM_FREE(Href.url);
      }
      UdmSQLFree(&SQLres);
      UdmDocStoreHrefs(Indexer, Doc);
      UdmHrefListFree(&Doc->Hrefs);
      UdmStoreHrefs(Indexer);
    }
  }
  end = UDM_STREND(Doc->Buf.buf);
  Doc->Buf.size=end-Doc->Buf.buf;
HTDBexit:
  if (db == &dbnew)
    UdmDBFree(db);
  UdmURLFree(&realURL);
  UDM_FREE(qbuf);
  return rc;
}


/***************************************************************************/

static int UdmPopRankCalculate(UDM_AGENT *A, UDM_DB *db)
{
  UDM_SQLRES  SQLres, Res, POPres;
  char    qbuf[1024];
  int             rc = UDM_ERROR, u;
  size_t    i, nrows;
  int    skip_same_site = !strcasecmp(UdmVarListFindStr(&A->Conf->Vars, "PopRankSkipSameSite","no"),"yes");
  int    feed_back = !strcasecmp(UdmVarListFindStr(&A->Conf->Vars, "PopRankFeedBack", "no"), "yes");
  int    use_tracking = !strcasecmp(UdmVarListFindStr(&A->Conf->Vars, "PopRankUseTracking", "no"), "yes");
  int    use_showcnt = !strcasecmp(UdmVarListFindStr(&A->Conf->Vars, "PopRankUseShowCnt", "no"), "yes");
  double          ratio = UdmVarListFindDouble(&A->Conf->Vars, "PopRankShowCntWeight", 0.01);
  const char      *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";

  if (feed_back || use_tracking)
  {
    if (use_tracking) UdmLog(A, UDM_LOG_EXTRA, "Will calculate servers weights by tracking");
    if (feed_back) UdmLog(A, UDM_LOG_EXTRA, "Will calculate feed back servers weights");

    if(UDM_OK != (rc = UdmSQLQuery(db, &Res, "SELECT rec_id FROM server WHERE command='S'")))
      goto Calc_unlock;

    nrows = UdmSQLNumRows(&Res);
    for (i = 0; i < nrows; i++)
    {

      if (use_tracking)
      {
        udm_snprintf(qbuf, sizeof(qbuf), "SELECT COUNT(*) FROM qinfo WHERE name='site' AND value='%s'", UdmSQLValue(&Res, i, 0) );
        if(UDM_OK != (rc = UdmSQLQuery(db, &SQLres, qbuf))) goto Calc_unlock;
        u = (UDM_ATOI(UdmSQLValue(&SQLres, 0, 0)) == 0);
      }
      if (feed_back && (u || !use_tracking))
      {
        udm_snprintf(qbuf, sizeof(qbuf), "SELECT SUM(pop_rank) FROM url WHERE site_id=%s%s%s", qu, UdmSQLValue(&Res, i, 0), qu);
        if(UDM_OK != (rc = UdmSQLQuery(db, &SQLres, qbuf))) goto Calc_unlock;
      }
      if (*UdmSQLValue(&SQLres, 0, 0))
      {
        udm_snprintf(qbuf, sizeof(qbuf), "UPDATE server SET weight=%s WHERE rec_id=%s%s%s", UdmSQLValue(&SQLres, 0, 0), 
         qu, UdmSQLValue(&Res, i, 0), qu);
        UdmSQLQuery(db, NULL, qbuf);
      }
      UdmSQLFree(&SQLres);
    }
    UdmSQLFree(&Res);
    UdmSQLQuery(db, NULL, "UPDATE server SET weight=1 WHERE weight=0 AND command='S'");
  }


  if(UDM_OK != (rc = UdmSQLQuery(db, &SQLres, "SELECT rec_id, url, weight FROM server WHERE command='S'")))
    goto Calc_unlock;
  
  nrows = UdmSQLNumRows(&SQLres);

  for (i = 0; i < nrows; i++)
  {
    udm_snprintf(qbuf, sizeof(qbuf), "SELECT COUNT(*) FROM url WHERE site_id=%s%s%s", qu, UdmSQLValue(&SQLres, i, 0), qu);
    if(UDM_OK != (rc = UdmSQLQuery(db, &Res, qbuf)))
    goto Calc_unlock;
    UdmLog(A, UDM_LOG_EXTRA, "Site_id: %s URL: %s Weight: %s URL count: %s",
           UdmSQLValue(&SQLres, i, 0),
           UdmSQLValue(&SQLres, i, 1),
           UdmSQLValue(&SQLres, i, 2),
           UdmSQLValue(&Res, 0, 0)); 
    if (atoi(UdmSQLValue(&Res, 0, 0)) > 0)
    {
      udm_snprintf(qbuf, sizeof(qbuf), "UPDATE server SET pop_weight=(%s/%s.0) WHERE rec_id=%s%s%s",
      UdmSQLValue(&SQLres, i, 2), UdmSQLValue(&Res, 0, 0), qu, UdmSQLValue(&SQLres, i, 0), qu);
      UdmSQLQuery(db, NULL, qbuf);
    }
    UdmSQLFree(&Res);

  }
  UdmSQLFree(&SQLres);


  UdmLog(A, UDM_LOG_EXTRA, "update links and pages weights");
  if (skip_same_site)  UdmLog(A, UDM_LOG_EXTRA, "Will skip links from same site");
  if (use_showcnt)  UdmLog(A, UDM_LOG_EXTRA, "Will add show count");

        udm_snprintf(qbuf, sizeof(qbuf), "SELECT rec_id, site_id  FROM url ORDER BY rec_id");

  if(UDM_OK != (rc = UdmSQLQuery(db, &Res, qbuf))) goto Calc_unlock;
  nrows = UdmSQLNumRows(&Res);
  for (i = 0; i < nrows; i++)
  {

    if (skip_same_site)
    {
      udm_snprintf(qbuf, sizeof(qbuf), "SELECT count(*) FROM links l, url uo, url uk WHERE uo.rec_id=l.ot AND uk.rec_id=l.k AND uo.site_id <> uk.site_id AND l.ot=%s%s%s", qu, UdmSQLValue(&Res, i, 0), qu);
    }
    else
    {
      udm_snprintf(qbuf, sizeof(qbuf), "SELECT count(*) FROM links WHERE ot=%s%s%s", qu, UdmSQLValue(&Res, i, 0), qu);
    }
    if(UDM_OK != (rc = UdmSQLQuery(db, &SQLres, qbuf))) goto Calc_unlock;
    if (*UdmSQLValue(&SQLres, 0, 0)) 
    {
      if (UDM_ATOI(UdmSQLValue(&SQLres, 0, 0)))
      {
        udm_snprintf(qbuf, sizeof(qbuf), "SELECT pop_weight FROM server WHERE rec_id=%s%s%s", qu, UdmSQLValue(&Res, i, 1), qu);
        if(UDM_OK != (rc = UdmSQLQuery(db, &POPres, qbuf))) goto Calc_unlock;
        if (UdmSQLNumRows(&POPres) != 1)
        { 
          UdmSQLFree(&POPres);
          UdmSQLFree(&SQLres);
          continue;
        }

        udm_snprintf(qbuf, sizeof(qbuf),
                     "UPDATE links SET weight = (%s/%d.0) WHERE ot=%s%s%s",
                     UdmSQLValue(&POPres, 0, 0),
                     atoi(UdmSQLValue(&SQLres, 0, 0)),
                     qu, UdmSQLValue(&Res, i, 0), qu);
        UdmSQLFree(&POPres);
        UdmSQLQuery(db, NULL, qbuf);
      }
    }
    UdmSQLFree(&SQLres);

    if (skip_same_site)
    {
      udm_snprintf(qbuf, sizeof(qbuf), "SELECT SUM(weight) FROM links l, url uo, url uk WHERE uo.rec_id=l.ot AND uk.rec_id=l.k AND uo.site_id <> uk.site_id AND l.k=%s%s%s", qu, UdmSQLValue(&Res, i, 0), qu);
    }
    else
    {
      udm_snprintf(qbuf, sizeof(qbuf), "SELECT SUM(weight) FROM links WHERE k=%s%s%s", qu, UdmSQLValue(&Res, i, 0), qu);
    }
    if(UDM_OK != (rc = UdmSQLQuery(db, &SQLres, qbuf))) goto Calc_unlock;
    if (UdmSQLValue(&SQLres,0,0) && *UdmSQLValue(&SQLres, 0, 0))
    {
      if (use_showcnt)
      {
        udm_snprintf(qbuf, sizeof(qbuf), "UPDATE url SET pop_rank=%s + (shows * %f) WHERE rec_id=%s%s%s", 
        UdmSQLValue(&SQLres, 0, 0), ratio, qu, UdmSQLValue(&Res, i, 0), qu );
      }
      else
      {
        udm_snprintf(qbuf, sizeof(qbuf), "UPDATE url SET pop_rank=%s WHERE rec_id=%s%s%s", 
        UdmSQLValue(&SQLres, 0, 0), qu, UdmSQLValue(&Res, i, 0), qu );
      }
    } else {
      udm_snprintf(qbuf, sizeof(qbuf), "UPDATE url SET pop_rank=(shows * %f) WHERE rec_id=%s%s%s", 
                   ratio, qu, UdmSQLValue(&Res, i, 0), qu );
    }
    UdmSQLQuery(db, NULL, qbuf);
    UdmSQLFree(&SQLres);
  }
  UdmSQLFree(&Res);

  rc = UDM_OK;

Calc_unlock:
  UdmLog(A, UDM_LOG_EXTRA, "Popularity rank done.");
  return rc;
}


static int
UdmWordStatQuery(UDM_AGENT *A, UDM_DB *db, const char *src)
{
  int rc;
  UDM_SQLRES SQLRes;
  size_t row, rows;
  
  if(UDM_OK!= (rc= UdmSQLQuery(db, &SQLRes, src)))
    return rc;
  
  rows=UdmSQLNumRows(&SQLRes);
  for(row=0 ; row < rows ; row++)
  {
    const char *word;
    int count;
    size_t wordlen;
    char snd[32];
    char insert[128];
    word= UdmSQLValue(&SQLRes, row, 0);
    wordlen= UdmSQLLen(&SQLRes, row, 0);
    count= UDM_ATOI(UdmSQLValue(&SQLRes, row, 1));
    UdmSoundex(A->Conf->lcs, snd, word, wordlen);
    if (snd[0])
    {
      sprintf(insert,
              "INSERT INTO wrdstat (word, snd, cnt) VALUES ('%s','%s',%d)",
              word, snd, count);
      if (UDM_OK!= (rc= UdmSQLQuery(db, NULL, insert)))
        return rc;
    }
  }
  UdmSQLFree(&SQLRes);
  return UDM_OK;
}


static int
UdmWordStatCreateMulti(UDM_AGENT *A, UDM_DB *db)
{
  int rc;
  size_t i;
  
  for (i=0; i <= 0xFF; i++)
  {
    char qbuf[128];
    UdmLog(A, UDM_LOG_EXTRA, "Processing table %02X", i);

    sprintf(qbuf, "SELECT word, count(*) FROM dict%02X GROUP BY word", i);
    if (UDM_OK != (rc= UdmWordStatQuery(A, db, qbuf)))
      return rc;
  }
  return UDM_OK;
}


static int
UdmWordStatCreateSingle(UDM_AGENT *A, UDM_DB *db)
{
  char qbuf[128];
  sprintf(qbuf, "SELECT word, count(*) FROM dict GROUP BY word");
  return UdmWordStatQuery(A, db, qbuf);
}


static int
UdmWordStatCreateBlob(UDM_AGENT *A, UDM_DB *db)
{
  char qbuf[128];
  const char *funcname;
  switch(db->DBType)
  {
    case UDM_DB_ORACLE8:
      funcname= "lengthb";
      break;
    case UDM_DB_SQLITE3:
      funcname= "length";
      break;
    default:
      funcname= "octet_length";
  }
  sprintf(qbuf, "SELECT word, sum(%s(intag)) FROM bdict GROUP BY word", funcname);
  return UdmWordStatQuery(A, db, qbuf);
}


static int
UdmWordStatCreate(UDM_AGENT *A, UDM_DOCUMENT *D, UDM_DB *db)
{
  int rc;

  UdmLog(A, UDM_LOG_ERROR, "Calculating word statistics");

  rc= UdmSQLTableTruncateOrDelete(db, "wrdstat");
  if (rc != UDM_OK)
    return rc;
  
  if (db->DBMode == UDM_DBMODE_SINGLE)
  {
    rc= UdmWordStatCreateSingle(A, db);
  }
  else if (db->DBMode == UDM_DBMODE_MULTI)
  {
    rc= UdmWordStatCreateMulti(A, db);
  }
  else if (db->DBMode == UDM_DBMODE_BLOB)
  {
    rc= UdmWordStatCreateBlob(A, db);
  }
  
  UdmLog(A, UDM_LOG_ERROR, "Word statistics done");
  return rc;
}


static int
UdmDocPerSite(UDM_AGENT *A, UDM_DOCUMENT *D, UDM_DB *db)
{
  char qbuf[1024];
  const char *s, *hostinfo= UdmVarListFindStr(&D->Sections, "Hostinfo", NULL);
  int rc, num, prevnum= UdmVarListFindInt(&D->Sections, "DocPerSite", 0);
  UDM_SQLRES SQLRes;
  
  if (!hostinfo)
    return UDM_OK;
  
  for (s= hostinfo; s[0]; s++)
  {
    /*
      Host name good characters: digits, letters, hyphen (-).
      Just check the worst characters.
    */
    if (*s == '\'' || *s == '\"')
    {
      num= 1000000;
      goto ret;
    }
  }
  udm_snprintf(qbuf, sizeof(qbuf),
               "SELECT COUNT(*) FROM url WHERE url LIKE '%s%%'", hostinfo);
  
  if(UDM_OK!= (rc= UdmSQLQuery(db, &SQLRes, qbuf)))
    return rc;
  num= prevnum + atoi(UdmSQLValue(&SQLRes, 0, 0));
  UdmSQLFree(&SQLRes);
ret:
  UdmVarListReplaceInt(&D->Sections, "DocPerSite", num);
  return UDM_OK;
}

 
static int
UdmImportSection(UDM_AGENT *A, UDM_DOCUMENT *D, UDM_DB *db)
{
  UDM_TEXTITEM Item;
  UDM_VAR *Sec;
  UDM_VARLIST Vars;
  UDM_SQLRES SQLRes;
  UDM_DSTR d;
  int rc;
  size_t row, rows, cols;
  const char *fmt= UdmVarListFindStr(&D->Sections, "SQLImportSection", NULL);
  
  if (!fmt)
    return UDM_OK;
  
  UdmDSTRInit(&d, 1024);
  UdmVarListInit(&Vars);
  UdmVarListSQLEscape(&Vars, &D->Sections, db);
  UdmDSTRParse(&d, fmt, &Vars);
  UdmVarListFree(&Vars);
  if(UDM_OK!= (rc= UdmSQLQuery(db, &SQLRes, d.data)))
    return rc;

  cols= UdmSQLNumCols(&SQLRes);
  bzero((void*)&Item, sizeof(Item));
  for (row=0, rows= UdmSQLNumRows(&SQLRes); row < rows; row++)
  {
    size_t col;
    for (col= 0; col + 1 < cols; col+= 2)
    {
      Item.section_name= (char*) UdmSQLValue(&SQLRes, row, col);
      if ((Sec= UdmVarListFind(&D->Sections, Item.section_name)))
      {
        Item.str= (char*) UdmSQLValue(&SQLRes, row, col + 1);
        Item.section= Sec->section;
        UdmTextListAdd(&D->TextList, &Item);
      }
    }
  }
  
  UdmDSTRFree(&d);
  UdmSQLFree(&SQLRes);
  return rc;
}

int (*udm_url_action_handlers[])(UDM_AGENT *A, UDM_DOCUMENT *D, UDM_DB *db)=
{
  UdmDeleteURL,           /* UDM_URL_ACTION_DELETE */
  UdmAddURL,              /* UDM_URL_ACTION_ADD */
  UdmUpdateUrl,           /* UDM_URL_ACTION_SUPDATE */
  UdmLongUpdateURL,       /* UDM_URL_ACTION_LUPDATE */
  UdmDeleteWordsAndLinks, /* UDM_URL_ACTION_DUPDATE */
  UdmUpdateClone,         /* UDM_URL_ACTION_UPDCLONE */
  UdmRegisterChild,       /* UDM_URL_ACTION_REGCHILD */
  UdmFindURL,             /* UDM_URL_ACTION_FINDBYURL */
  UdmFindMessage,         /* UDM_URL_ACTION_FINDBYMSG */
  UdmFindOrigin,          /* UDM_URL_ACTION_FINDORIG */
  UdmMarkForReindex,      /* UDM_URL_ACTION_EXPIRE */
  UdmGetReferers,         /* UDM_URL_ACTION_REFERERS */
  UdmGetDocCount,         /* UDM_URL_ACTION_DOCCOUNT */
  UdmDeleteLinks,         /* UDM_URL_ACTION_LINKS_DELETE */
  UdmAddLink,             /* UDM_URL_ACTION_ADD_LINK */
  UdmGetCachedCopy,       /* UDM_URL_ACTION_GET_CACHED_COPY */
  UdmWordStatCreate,      /* UDM_URL_ACTION_WRDSTAT */
  UdmDocPerSite,          /* UDM_URL_ACTION_DOCPERSITE */
  UdmImportSection        /* UDM_URL_ACTION_SQLIMPORTSEC */
};


static size_t
WordProximity(UDM_CHARSET *cs,
              const char *s1, size_t len1,
              const char *s2, size_t len2)
{
  size_t max= len1 > len2 ? len1 : len2;
  size_t min= len1 < len2 ? len1 : len2;
  if ((max - min)*3 > max)  /* Not more than 1/3 of length difference */
    return 0;
  return (1000 * min) / (max ? max : 1);
}


static int
UdmResSuggest(UDM_AGENT *A, UDM_DB *db, UDM_RESULT *Res, size_t dbnum)
{
  int rc= UDM_OK;
  size_t i, nwords;
  UDM_WIDEWORDLIST *WWList= &Res->WWList;
  UDM_CONV lcs_uni;
  
  UdmLog(A, UDM_LOG_DEBUG, "Generating suggestion list");
  UdmConvInit(&lcs_uni, A->Conf->lcs, &udm_charset_sys_int, UDM_RECODE_HTML);

  for (i=0, nwords= WWList->nwords; i < nwords; i++)
  {
    UDM_WIDEWORD *W= &WWList->Word[i];
    if (W->origin == UDM_WORD_ORIGIN_QUERY && !W->count)
    {
      char snd[32];
      char qbuf[128];
      char topstr[64], rowstr[64], limstr[64];
      UDM_SQLRES SQLRes;
      UDM_WIDEWORD sg;
      size_t row, rows, max_count= 0;
      size_t Wlen= W->len; /* W is not valid after UdmWideWordListAdd() */
      size_t Word= W->order;
      size_t phrpos= W->phrpos;
      const char *Wword= W->word;
      
      UdmSQLTopClause(db, 100,
                      topstr, sizeof(topstr),
                      rowstr, sizeof(rowstr),
                      limstr, sizeof(limstr));
      
      UdmSoundex(A->Conf->lcs, snd, W->word, W->len);
      UdmLog(A, UDM_LOG_DEBUG, "Suggest for '%s': '%s'", W->word, snd);
      udm_snprintf(qbuf, sizeof(qbuf),
                   "SELECT %sword, cnt FROM wrdstat WHERE snd='%s'%s"
                   " ORDER by cnt DESC%s",
                    topstr, snd, rowstr, limstr);
      
      if(UDM_OK!= (rc= UdmSQLQuery(db, &SQLRes, qbuf)))
        return rc;
      
      rows=UdmSQLNumRows(&SQLRes);
      UdmLog(A, UDM_LOG_DEBUG, "%d suggestions found", rows);
      bzero((void*)&sg, sizeof(sg));

      for(row=0 ; row < rows ; row++)
      {
        size_t nbytes, proximity_weight, count_weight, final_weight;
        int tmp[128];
        
        sg.word= (char*) UdmSQLValue(&SQLRes, row, 0);
        sg.count= UDM_ATOI(UdmSQLValue(&SQLRes, row, 1));
        sg.len= UdmSQLLen(&SQLRes, row, 0);
        if (max_count <= sg.count)
          max_count= sg.count;
        count_weight= (100 * sg.count) / (max_count ? max_count : 1);
        proximity_weight= WordProximity(A->Conf->lcs, Wword, Wlen, sg.word, sg.len); 
        final_weight= count_weight * proximity_weight;
        UdmLog(A, UDM_LOG_DEBUG, "'%s': %d/%d/%d/%d", 
               sg.word, sg.count, count_weight, proximity_weight, final_weight);
        sg.count= final_weight;

        nbytes= (sg.len + 1) * sizeof(int);
        if (nbytes < sizeof(tmp))
        {
          sg.order= Word;
          sg.phrpos= phrpos;
          sg.origin= UDM_WORD_ORIGIN_SUGGEST;
          UdmWideWordListAdd(WWList, &sg); /* W is not valid after this */
        }
      }
      UdmSQLFree(&SQLRes);
    }
  }
  return rc;
}


int UdmResActionSQL(UDM_AGENT *Agent, UDM_RESULT *Res, int cmd, UDM_DB *db, size_t dbnum)
{
  switch(cmd)
  {
    case UDM_RES_ACTION_DOCINFO:
      return UdmResAddDocInfoSQL(Agent, db, Res, dbnum);
    case UDM_RES_ACTION_SUGGEST:
      return UdmResSuggest(Agent, db, Res, dbnum);
    default:
      UdmLog(Agent, UDM_LOG_ERROR, "Unsupported Res Action SQL");
      return UDM_ERROR;
  }
}

int UdmCatActionSQL(UDM_AGENT *Agent, UDM_CATEGORY *Cat, int cmd,UDM_DB *db)
{
  switch(cmd)
  {
    case UDM_CAT_ACTION_LIST:
      return UdmCatList(Agent,Cat,db);
    case UDM_CAT_ACTION_PATH:
      return UdmCatPath(Agent,Cat,db);
    default:
              UdmLog(Agent, UDM_LOG_ERROR, "Unsupported Cat Action SQL");
      return UDM_ERROR;
  }
}

int UdmSrvActionSQL(UDM_AGENT *A, UDM_SERVERLIST *S, int cmd,UDM_DB *db){
  switch(cmd){
    case UDM_SRV_ACTION_TABLE:
      return UdmLoadServerTable(A,S,db);
    case UDM_SRV_ACTION_FLUSH:
      return UdmServerTableFlush(db);
    case UDM_SRV_ACTION_ADD:
      return UdmServerTableAdd(S, db);
    case UDM_SRV_ACTION_ID:
      return UdmServerTableGetId(A, S, db);
    case UDM_SRV_ACTION_POPRANK:
      return UdmPopRankCalculate(A, db);
    default:
      UdmLog(A, UDM_LOG_ERROR, "Unsupported Srv Action SQL");
      return UDM_ERROR;
  }
}

unsigned int   UdmGetCategoryIdSQL(UDM_ENV *Conf, char *category, UDM_DB *db)
{
  UDM_SQLRES Res;
  char qbuf[128];
  unsigned int rc = 0;

  udm_snprintf(qbuf, 128, "SELECT rec_id FROM categories WHERE path LIKE '%s'", category);
  if(UDM_OK != (rc = UdmSQLQuery(db, &Res, qbuf))) return rc;
  if ( UdmSQLNumRows(&Res) > 0)
  {
    sscanf(UdmSQLValue(&Res, 0, 0), "%u", &rc);
  }
  UdmSQLFree(&Res);
  return rc;
}


int UdmCheckUrlidSQL(UDM_AGENT *Agent, UDM_DB *db, urlid_t id)
{
  UDM_SQLRES SQLRes;
  char qbuf[128];
  unsigned int rc = 0;

  udm_snprintf(qbuf, sizeof(qbuf), "SELECT rec_id FROM url WHERE rec_id=%d", id);
  rc = UdmSQLQuery(db, &SQLRes, qbuf);
  if(UDM_OK != rc)
  {
    rc = 1;
  }
  else
  {
    if (UdmSQLNumRows(&SQLRes) != 0) rc = 1;
    else rc = 0;
  }
  UdmSQLFree(&SQLRes);
  return rc;
}

int UdmExportSQL (UDM_AGENT *Indexer, UDM_DB *db)
{
  UDM_SQLRES SQLRes;
  int rc;
  UDM_PSTR row[24];

  printf("<database>\n");
  printf("<urlList>\n");
  rc= UdmSQLExecDirect(db, &SQLRes, "SELECT * FROM url");
  if (rc != UDM_OK) return(rc);
  while (db->sql->SQLFetchRow(db, &SQLRes, row) == UDM_OK)
  {
    printf(
      "<url "
      "rec_id=\"%s\" "
      "status=\"%s\" "
      "docsize=\"%s\" "
      "next_index_time=\"%s\" "
      "last_mod_time=\"%s\" "
      "referrer=\"%s\" "
      "hops=\"%s\" "
      "crc32=\"%s\" "
      "seed=\"%s\" "
      "bad_since_time=\"%s\" "
      "site_id=\"%s\" "
      "server_id=\"%s\" "
      "shows=\"%s\" "
      "pop_rank=\"%s\" "
      "url=\"%s\" "
      "/>\n",
      row[0].val, row[1].val, row[2].val, row[3].val,
      row[4].val, row[5].val, row[6].val, row[7].val,
      row[8].val, row[9].val, row[10].val, row[11].val,
      row[12].val, row[13].val, row[14].val);
  }
  UdmSQLFree(&SQLRes);
  printf("</urlList>\n");

  printf("<linkList>\n");
  rc= UdmSQLExecDirect(db, &SQLRes, "SELECT * FROM links");
  if (rc != UDM_OK) return(rc);
  while (db->sql->SQLFetchRow(db, &SQLRes, row) == UDM_OK)
  {
    printf(
      "<link "
      "ot=\"%s\" "
      "k=\"%s\" "
      "weight=\"%s\" "
      "/>\n",
      row[0].val, row[1].val, row[2].val);
  }
  UdmSQLFree(&SQLRes);
  printf("</linkList>\n");

  printf("</database>\n");
  return(0);
}


#define QCACHEID \
"${q}.${pqid}.${SearchMode}.${orig_m}.${fl}.${wm}.${o}.${t}." \
"${cat}.${ul}.${wf}.${g}.${tmplt}.${GroupBySite}.${site}." \
"${type}.${sp}.${sy}.${dt}.${dp}.${dx}.${dm}.${dy}.${db}.${de}.${s}"

static int QueryCacheID(UDM_AGENT *A)
{
  const char *fmt= UdmVarListFindStr(&A->Conf->Vars, "QueryCacheID", QCACHEID);
  UDM_DSTR d;
  int res;
  UdmDSTRInit(&d, 256);
  UdmDSTRParse(&d, fmt, &A->Conf->Vars);
  res= (int) UdmStrHash32(d.data);
  UdmDSTRFree(&d);
  return res;
}


int UdmQueryCacheGetSQL(UDM_AGENT *A, UDM_RESULT *Res, UDM_DB *db)
{
  int use_qcache= UdmVarListFindBool(&db->Vars, "qcache", 0);
  int tm, rc, id, qtm;
  char qbuf[128];
  
  if (!use_qcache)
    return UDM_OK;
  
  if (db->DBMode != UDM_DBMODE_BLOB)
    return UDM_OK;
  if (UDM_OK != (rc= UdmBlobReadTimestamp(A, db, &tm, (int) time(0))))
    return rc;
  id= QueryCacheID(A);
  udm_snprintf(qbuf, sizeof(qbuf),
               "SELECT doclist, wordinfo, tm FROM qcache WHERE id='%d' AND tm>=%d ORDER BY tm DESC LIMIT 1", id, tm);
  rc= UdmFetchCachedQuery(A, Res, db, qbuf, &qtm);
  if (Res->URLData.nitems)
  {
    UdmLog(A, UDM_LOG_DEBUG,
           "Fetched from qcache %d documents, %d total found",
           Res->URLData.nitems, Res->total_found);
    udm_snprintf(qbuf, sizeof(qbuf), "QCache:%08X-%08X", id, qtm);
    UdmVarListReplaceStr(&A->Conf->Vars, "ResultSource", qbuf);
    udm_snprintf(qbuf, sizeof(qbuf), "%08X-%08X", id, qtm);
    UdmVarListAddStr(&A->Conf->Vars, "qid", qbuf);
  }
  return UDM_OK;
}


int UdmQueryCachePutSQL(UDM_AGENT *Indexer, UDM_RESULT *Res, UDM_DB *db)
{
  const char *usercache = UdmVarListFindStr(&db->Vars, "usercache", "");
  int prevcache= UdmVarListFindBool(&db->Vars, "qcache", 0);
  int rc= UDM_OK;
  char qbuf[64];

  if (usercache[0])
  {
    size_t i;
    for (i = 0; i < Res->URLData.nitems; i++)
    {
      sprintf(qbuf, "INSERT INTO %s VALUES(%d, %d)",
                    usercache,
                    Res->URLData.Item[i].url_id,
                    Res->URLData.Item[i].coord);
      rc= UdmSQLQuery(db, NULL, qbuf);
      if (rc != UDM_OK)
        return rc;
    }
  }
  
  if (prevcache)
  {
    UDM_DSTR buf, wwl;
    size_t i, nbytes= Res->URLData.nitems * 24, wwl_length_escaped;
    int id, tm= time(0);
    char *s;
    
    UdmLog(Indexer, UDM_LOG_DEBUG, "Putting into qcache %d documents", Res->URLData.nitems);
    id= QueryCacheID(Indexer);
    sprintf(qbuf, "%08X-%08X", id, tm);

    UdmDSTRInit(&wwl, 256);
    UdmDSTRAppendf(&wwl, "<result>"); 
    UdmDSTRAppendf(&wwl, "<totalResults>%d</totalResults>",
                   Res->total_found);
    UdmDSTRAppendf(&wwl,"<wordinfo>");
    for (i= 0; i < Res->WWList.nwords; i++)
    {
      UDM_WIDEWORD *ww= &Res->WWList.Word[i];
      UdmDSTRAppendf(&wwl, "<word id='%d' order='%d' count='%d' len='%d' "
                           "origin='%d' weight='%d' match='%d' "
                           "secno='%d' phrlen='%d' phrpos='%d'>%s</word>",
                      i, ww->order, ww->count, ww->len,
                      ww->origin, ww->weight, ww->match,
                      ww->secno, ww->phrlen, ww->phrpos, ww->word);
    }
    UdmDSTRAppendf(&wwl, "</wordinfo></result>");
    wwl_length_escaped= wwl.size_data * 5;
    UdmDSTRInit(&buf, 256);
    UdmDSTRRealloc(&buf, nbytes + 128 + wwl_length_escaped);
    UdmDSTRAppendf(&buf,
                   "INSERT INTO qcache (id, tm, doclist, wordinfo) VALUES (%d, %d, 0x",
                   id, tm);

    /* Append coord information */
    for (s= buf.data + buf.size_data, i= 0; i < Res->URLData.nitems; i++)
    {
      uint4 coord;
      id= Res->URLData.Item[i].url_id;
      sprintf(s, "%02X", id & 0xFF); s+=2; id >>= 8;
      sprintf(s, "%02X", id & 0xFF); s+=2; id >>= 8;
      sprintf(s, "%02X", id & 0xFF); s+=2; id >>= 8;
      sprintf(s, "%02X", id & 0xFF); s+=2;
      coord= Res->URLData.Item[i].coord;
      sprintf(s, "%02X", coord & 0xFF); s+=2; coord >>= 8;
      sprintf(s, "%02X", coord & 0xFF); s+=2; coord >>= 8;
      sprintf(s, "%02X", coord & 0xFF); s+=2; coord >>= 8;
      sprintf(s, "%02X", coord & 0xFF); s+=2;
      id= Res->URLData.Item[i].per_site;
      sprintf(s, "%02X", id & 0xFF); s+=2; id >>= 8;
      sprintf(s, "%02X", id & 0xFF); s+=2; id >>= 8;
      sprintf(s, "%02X", id & 0xFF); s+=2; id >>= 8;
      sprintf(s, "%02X", id & 0xFF); s+=2;      
    }
    buf.size_data+= nbytes;
    buf.data[buf.size_data]= '\0';
    
    /* Append word information */
    UdmDSTRAppend(&buf, ",'", 2);
    UdmSQLBinEscStr(db, buf.data + buf.size_data, wwl.data, wwl.size_data);
    nbytes= strlen(buf.data + buf.size_data);
    buf.size_data+= nbytes;
    UdmDSTRAppend(&buf, "'", 1);
    
    UdmDSTRAppend(&buf, ")", 1);
    rc= UdmSQLQuery(db, NULL, buf.data);
    UdmDSTRFree(&wwl);
    UdmDSTRFree(&buf);
    if (rc == UDM_OK)
      UdmVarListAddStr(&Indexer->Conf->Vars, "qid", qbuf);
  }
  return rc;
}

#endif /* HAVE_SQL */
