/*!
 * \file    OMS_ClassIdEntryDef.hpp
 * \author  IvanS, MarkusSi, PeterG
 * \brief   OMS ClassIdEntry.
 */
/*

    ========== licence begin  GPL
    Copyright (c) 2002-2005 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end



*/

#ifndef __OMS_CLASSIDENTRYDEF_HPP
#define __OMS_CLASSIDENTRYDEF_HPP

#include "Oms/OMS_CacheMissEntryDef.hpp"
#include "livecachetypes.h" // nocheck
#include "ggg250.h"
#include "ggg00.h"
#include "ggg01.h"

class OMS_Context;
class OMS_FreeListHeader;
class OMS_ContainerInfo;
class OMS_ClassInfo;
class OMS_KeyDesc;           // PTS 1122540
class OmsObjectContainer;
class OmsAbstractObject;

/// Entries in the local container dictionary.  
class OMS_ClassIdEntry {
public:
  /*! Pointer to link together entries which belong to the same hash value 
  ** (see cco13ClsIdHash::m_clsIdHead)  */
  OMS_ClassIdEntry*                                               m_clsidHashNext;
  /*! Pointer to link together entries which belong to the same hash value 
  ** (see cco13ClsIdHash::m_guidHead)  */
  OMS_ClassIdEntry*                                               m_guidHashNext;

private:
  /*! Pointer to the corresponding entry in the global container directory */
  OMS_ContainerInfo*                                             m_containerInfo;

public:
  /*! AVL-tree in which all the keyed objects are stored which are newly created in a 
  ** version. This is necessary as the kernel is not informed about the creation of such 
  ** objects and so there must be a possiblity to locate these objects without the kernel. 
  ** If the cached key feature (PTS 1117571) is switched on, then if possible objects 
  ** which are loaded into the local cache are inserted into this structure, too. But as 
  ** this is not always possible the following invariant should be valid: Every object 
  ** which is stored in this structure must be stored in the local cache but not every 
  ** object in the local cache must be stored in the search structure. The advantage of 
  ** storing the objects in this structure is, that now key accesses to these objects can 
  ** be executed without asking the kernel first and therefore this should result in a 
  ** performance improvement for accesses to objects which are already stored in the local 
  ** cache. */
  cgg250AvlTree<unsigned char*,OMS_ClassIdEntry,OMS_Context>  m_index;
  /*! Pointer to the freelist of object containers which have the appropriate size */
  OMS_FreeListHeader*                            m_freeHead;
  /*! Pointer to the freelist of object containers which have the appropriate size 
  ** (The size of containers for objects which are created in a version is different from 
  ** the normal size) */
  OMS_FreeListHeader*                            m_freeHeadInVersion;

  /// /name Members for cache miss structure      // PTS 1117571
  //@{ 
  /*! If the cached key feature is switched on, then this additional AVL-tree is used 
  ** to store the keys of those key accesses which does not find a corresponding object. 
  ** The second time a key access to such an object is started, there is no need to ask 
  ** the kernel again, as this access can be answered by this search structure.*/
  cgg250AvlTree<OMS_CacheMissEntry*,OMS_CacheMissEntry,OMS_Context> 
    m_cacheMiss;      
  /*! Is the cached key feature switched on or off? */
  bool                             m_useCachedKeys;   
  /*! If the cached key feature is switched off, then this pointer equals NULL; otherwise 
  ** it points to an item of the same type as those items which are stored in the 
  ** m_cacheMiss tree. This object is used to define a compare function for the tree 
  ** and furthermore this object is used for queries on this structure. To do this first 
  ** the search components are copied temporarily into this object and then this object is 
  ** given as parameter to the methods. (The advantage is, that the allocation of the object 
  ** occurs only once and as the copying is much cheaper this results in performance 
  ** improvement.) */
  OMS_CacheMissEntry*            m_cacheMissCmp;  
  //@}

  OMS_ClassIdEntry (OMS_ContainerInfo* classInfo,OMS_Context* c);
  inline ~OMS_ClassIdEntry();
  inline void* operator new    (size_t sz, OMS_Context* session);
#if defined(OMS_PLACEMENT_DELETE)
  inline void  operator delete (void* p, OMS_Context* session);
#endif
  inline int                     Compare(const unsigned char*, const unsigned char*);
  inline void                    DeleteSelf(OMS_Context* session);
  inline const OMS_ClassInfo*    GetClassInfoPtr() const;
  inline OMS_ContainerInfo*      GetContainerInfoPtr() const;
  inline tsp00_Uint4             GetContainerHandle() const;
  inline OmsContainerNo          GetContainerNo() const;
  inline OmsSchemaHandle         GetSchema() const;
  inline tgg01_ContainerId&      GetFileId() const;
  inline const ClassIDRef        GetGuid() const;
  inline int                     GetKeyLen() const;        
  inline bool                    IsKeyedObject() const;     // PTS 1117571
  inline const OMS_KeyDesc&      GetKeyDesc() const;        // PTS 1122540
  inline unsigned char*          GetKeyPtr(OmsObjectContainer *p);
  inline tsp00_Int4              GetObjectSize() const;
  inline tsp00_Int4              GetPersistentSize() const;
  inline void*                   GetVirtualTablePtr() const;
  inline bool                    IsDerivedClassOf(const ClassIDRef) const;
  inline bool                    IsVarObject();
  bool                           SetCachedKeys(bool, OMS_Context*);        // PTS 1117571 
  inline bool                    UseCachedKeys() const;                    // PTS 1117571

  void                           chainFree (OMS_Context&c, OmsAbstractObject* p, int caller);
  void                           chainFree (OMS_Context&c, OmsObjectContainer*& p, int caller);
  inline tgg00_BasisError        VersionAddKey(OmsObjectContainer*, OMS_Context*, bool noThrowIfExist = false);
  inline void                    VersionDelKey(OmsObjectContainer*, OMS_Context*);
  void                           VersionDelIndex(bool cachedKeysOnly, OMS_Context*);
  inline OmsObjectContainer*     VersionFindKey(const unsigned char*);
  inline void                    VersionReplaceOrAddKeyContainerPtr(OmsObjectContainer* p, OMS_Context*);
  inline OmsObjectContainer*     VersionGetInfoFromNode(const unsigned char* const*);  // PTS 1117571

  inline bool                    IsCacheMiss(const unsigned char* key);                                 // PTS 1117571
  inline void                    DeleteCacheMiss(const unsigned char* key, OMS_Context* pContext);  // PTS 1117571
  inline void                    InsertCacheMiss(const unsigned char* key, OMS_Context* pContext);  // PTS 1117571
  void                           DropCacheMisses(OMS_Context* pContext);                            // PTS 1117571
private:
  void operator delete(void*);
};

typedef OMS_ClassIdEntry* OMS_ClassIdEntryPtr;

#endif  // __OMS_CLASSIDENTRYDEF_HPP
