/*  Protocol compatible masqdialer server written in C
    Copyright (C) 1998 Charles P. Wright 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <stdio.h>
#include <sys/socket.h>
#include <sys/param.h>
#include <sys/types.h>
#include <sys/time.h>
#include <netinet/in.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <time.h>
#include <syslog.h>
#include <unistd.h>
#include <errno.h>
#include <arpa/inet.h>

#include "mserver.h"

/* prototypes for local functions */
static void create_socket(const char * ip, const char * port, int slimit);

FILE *insock;
FILE *outsock;

int sfd;
int csfd;
int nclients;
pid_t parent_pid;

int listen_sfds[255];
int num_sfds = 0;

bool demon;
bool verbose;
fd_set listen_set;

int main(int argc, char **argv)
{
	int i;

	int clen;
	int slimit;

	char temp[1024];
	char error[1024];

	char config_file[1024] = "/etc/mserver.conf";
	char exename[1024];
	
	struct sigaction cursig;

	char iplist[MAXINTERFACES][1024];
	char portlist[MAXINTERFACES][1024];
	int num_interfaces;

	char curarg;
	
	char autod[1024];

	verbose = false;
	
	while ((curarg = getopt(argc, argv, "c:vV")) != EOF)
	{
		switch (curarg)
		{
		case 'c':
			strncpy(config_file, optarg, 1024);
			break;
		case 'v':
			verbose = true;
			break;
		case 'V':
			snprintf(temp, 1024, VERSION_STR, VERSION);
			printf("%s\n", temp);
			exit(0);
			break;
		default:
			syslog(LOG_WARNING, "Unknown option: %c", curarg);
		}
	}

#ifdef DEBUG
	verbose = true;
	openlog ("mserver", LOG_NDELAY | LOG_PERROR | LOG_PID, LOG_DAEMON);
#else
	openlog ("mserver", LOG_NDELAY | LOG_PERROR | LOG_PID, LOG_DAEMON);
#endif

	memset(&cursig, 0, sizeof(cursig));
	cursig.sa_handler = child_handler;
	sigaction(SIGCHLD, &cursig, NULL);
	cursig.sa_handler = config_refresh;
	/* Even with BSD behaviour, accept will be interrupted, says Stevens */
	cursig.sa_flags = SA_RESTART;
	sigaction(SIGHUP, &cursig, NULL);
	cursig.sa_handler = mserver_halt;
	sigaction(SIGTERM, &cursig, NULL);


	if (realpath(get_exe_name(argv[0]), exename) == NULL)
	{
		syslog(LOG_ERR, "Could not resolve pathname of executable!");
		exit(1);
	}

	if (!config_init(config_file))
	{
		syslog(LOG_ERR, "Could not open configuration file: %s", config_file);
		exit(1);
	}

	slimit = config_getvalue_int ("slimit", 10);

	config_getvalue_default("listen_on", temp, "0.0.0.0", 1024);

	FD_ZERO(&listen_set);
	num_interfaces = parse_listenon(temp, iplist, portlist);

	/* Now actually create the sockets for these IP addresses and ports */
	for (i = 0; i < num_interfaces; i++)
	{
		if (util_validip(iplist[i]))
		{
			create_socket(iplist[i], portlist[i], slimit);
		}
		else
		{
			syslog(LOG_ERR, "Invalid listen_on IP address: %s", iplist[i]);
		}
	}

#ifdef DEBUG
	demon = false;
	syslog (LOG_INFO, "DEBUG selected, not forking into background!");
#else
	demon = config_getvalue_bool ("demon", false);
#endif

	if (demon)
	{
		if (chdir("/") != 0)
		{
			syslog (LOG_ERR, "Can't change directory to \"/\"!");
			mserver_cleanup();
			exit(1);
		}
			
		if (setpgrp() == -1)
		{
			syslog(LOG_ERR, "Could not create process group");
			mserver_cleanup();
			exit(1);
		}

		close(0);
		close(1);
		close(2);

		if (fork() != 0)
		{
			exit(0);
		}
	}

	shmem_init(exename);
	parent_pid = getpid();

	netload_init();

	config_getvalue_default("autodial", autod, "\0", 1024);
	if(autod[0] != '\0')
	{
		if(fork())	
		{
			auto_dial(autod);
			exit(0);
		}
	}

	while (true)
	{
#ifndef NO_FORK
		pid_t pid;
#endif
		while (1)
		{
			int i, n, found;
			struct sockaddr addr;
			int addrlen = sizeof(addr);
			fd_set lfds;

			memcpy((void *) &lfds, (void *) &listen_set, sizeof(lfds));
	
			errno = 0;
			clen = sizeof(struct sockaddr_in);
			i = util_select(256, &lfds);

			n = found = 0;

			while (found < i && n < num_sfds)
			{
				if (FD_ISSET(listen_sfds[n], &lfds))
				{
					syslog(LOG_DEBUG, "FD %d (%d) is set", listen_sfds[n], n);

					FD_CLR(listen_sfds[n], &lfds);	
					sfd = listen_sfds[n];	


					csfd = accept (sfd, &addr, &addrlen);
			
					if (csfd == -1 && errno == EINTR)
						continue;

					if (csfd == -1)
					{
						strncpy(error, strerror(errno), 1024);
						syslog(LOG_ERR, "accept error: %s", error);
						continue;
					}
			
#ifndef NO_FORK
					if ((pid = fork()) == 0)
#endif
					{
						int continue_client;
			
						cursig.sa_handler = SIG_IGN;
						sigaction(SIGCHLD, &cursig, NULL);

						insock = fdopen(csfd, "r");
						outsock = fdopen(csfd, "w");

						continue_client = true;

						while(continue_client == true)
						{	
							mserver_serve();
						}

						mserver_cleanup();
						exit(0);
					}
#ifndef NO_FORK
					else
					{	
						syslog(LOG_INFO, "Forked new client %d", pid);
						add_client(pid, csfd);
						close(csfd);
					}	
#endif
					found++;
				}
				n++;
			}
			syslog (LOG_DEBUG, "Done with while");
		}
	}
}

static void create_socket(const char * ip, const char * port, int slimit)
{
	int sockn;
	int opt_val, len, result;
	struct sockaddr_in address;
	char error[1024];
	
	sockn = socket (AF_INET, SOCK_STREAM, 0);
	
	if (sockn == -1)
	{
		syslog (LOG_ERR, "Could not create socket!");
		exit(1);
	}
	
	address.sin_family = AF_INET;
	address.sin_port = htons(atoi(port));
	address.sin_addr.s_addr = inet_addr(ip);
	len = sizeof(address);
	
	opt_val = 1;	
	setsockopt(sockn, SOL_SOCKET, SO_REUSEADDR, &opt_val, sizeof(int));
	result = bind(sockn, (struct sockaddr *) &address, len);
	
	if (result == -1)
	{
		strncpy(error, strerror(errno), 1024);
		syslog(LOG_ERR, "Can not bind socket to address (%s:%d): %s",  ip, atoi(port), error);
		exit (1);
	}
	else
	{
		
#ifdef DEBUG
		syslog(LOG_INFO, "Bound to %s:%d",  ip, atoi(port));
#endif
	}
	
	result = listen(sockn, slimit);
	
#ifndef DEBUG
	closelog();
	openlog ("mserver", LOG_NDELAY | LOG_PID, LOG_DAEMON);
#endif
	if (verbose)
	{
		syslog(LOG_INFO, "Listening on address: %s", ip);
	}
#ifndef DEBUG
	closelog();
	openlog ("mserver", LOG_NDELAY | LOG_PERROR | LOG_PID, LOG_DAEMON);
#endif
	
	if (result == -1)
	{
		syslog(LOG_ERR, "Can not listen socket to address.");
		exit (1);
	}
	
	FD_SET(sockn, &listen_set);
	listen_sfds[num_sfds++] = sockn;
}
