#   MailScanner - SMTP E-Mail Virus Scanner
#   Copyright (C) 2001  Julian Field
#
#   $Id: explode.pl,v 1.22 2002/02/01 10:18:02 jkf Exp $
#
#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this program; if not, write to the Free Software
#   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#   The author, Julian Field, can be contacted by email at
#      Jules@JulianField.net
#   or by paper mail at
#      Julian Field
#      Dept of Electronics & Computer Science
#      University of Southampton
#      Southampton
#      SO17 1BJ
#      United Kingdom
#

# Explode the MIME attachments of the messages we have been passed
# and store them all ready for virus scanning.
# Return a hash of message ids --> MIME entities.

use strict;
use MIME::Parser;
use MIME::Decoder::UU;
use POSIX qw(setsid);

my($cp) = "/bin/cp";

# Install an extra MIME decoder for badly-header uuencoded messages.
install MIME::Decoder::UU 'uuencode';

sub ExplodeMessages {
  my($QDir, $DestDir, $CantParse, $IsTNEF, $BadTNEF, @Ids) = @_;

  local(*PIPE, *OUTPUTDIR);
  my($id, $dfile, $path, $header, $parser, $filer, $entity, %entities);
  my(@tneffiles, $tneffile, $result);
  my $Df = new FileHandle;

  foreach $id (@Ids) {
    $path = "$DestDir/$id";
    $header = "$DestDir/$id.header";
    $dfile = MTA::DFileName($id);

    # Exim locks queue data file to indicate working on
    # that message... so we need to too.
    #
    # Ideally we should lock *all* the messages we're working on
    # at a higher level than this, and unlock only when we've finished
    # with them -- this way protects us from exim but not from other
    # mailscanners (another one might be re-exploding a message when
    # we come back to use the exploded bits)...
    #
    $Config::MTA eq "exim" and (Lock::openlock($Df,"$QDir/$dfile","r") or next);

    # For running on old versions of the modules
    #$parser = new MIME::Parser;
    #$parser->output_dir($path);

    # MIME::Tools Debugging output
    #MIME::Tools->debugging(1);
    #MIME::Tools->quiet(0);

    # For running on new versions of the modules
    $parser = MIME::Parser->new;
    $filer = MIME::Parser::FileInto->new($path);
    $parser->filer($filer);
    #$parser->output_dir($path);

    $parser->extract_uuencode(1); ### default is false, can read uuencode
    $parser->output_to_core('NONE');
    unless (open(PIPE, MTA::BuildMessageCmd($header,"$QDir/$dfile")." |")) {
      Log::WarnLog("Cannot build message from $header and $QDir/$dfile, %s", $!);
      next;
    }
    # For running on old versions of the modules
    #$entity = eval { $parser->parse_FH(\*PIPE) };

    # For running on new versions of the modules
    $entity = eval { $parser->parse(\*PIPE) };
    unless ($entity) {
      close PIPE;
      Log::WarnLog("Cannot parse $header and $QDir/$dfile, $@");
      push @$CantParse, $id;
      next;
    }

    close PIPE;
    $entities{$id} =  $entity;

    # JKF 24/05/01 No, don't do it this way. Need to find the winmail.dat
    # attachment in every message (if present) and set the %IsTNEF entry for
    # this message to point to it. Then we can build Entity2File and
    # File2Entity properly later. Will involve walking the tree to find
    # the winmail.dat file as we cannot guarantee where it will be.

    # Find the entity corresponding to the TNEF attachment, or undef if there
    # isn't one within this message. FindTNEFFile() is recursive.
    # JKF 7/12/2001 Only look for TNEF if we are going to expand it by hand.
    #               Sophos will just treat TNEF like any other archive.
    $IsTNEF->{$id} = FindTNEFFile($entity) if $Config::ExpandTNEF;

    # JKF 24/05/01 New code to use "tnef" to extract files from TNEF
    # attachments.
    # Look for winmail.dat files in each attachment directory $path.
    # When we find one explode it into its files and store the root MIME
    # entity into $IsTNEF{$id} so we can handle it separately later.
    # Pattern to match is actually winmail(any digits).dat(any digits) as that copes
    # with forwarded or bounced messages from mail packages that download
    # all attachments into 1 directory, adding numbers to their filenames.
    #print STDERR "TNEF = $Config::TNEF\n";
    #print STDERR "It is executable\n" if -r $Config::TNEF;
    #print STDERR "This is a TNEF message\n" if $IsTNEF->{$id};
    # Had to remove executable test from next line as people added command-line options
    # to TNEF Expander config variable
    if ($Config::TNEF && $IsTNEF->{$id}) {
      opendir(OUTPUTDIR, $path);
      @tneffiles = map { /(winmail\d*\.dat\d*)/i } readdir OUTPUTDIR;
      closedir OUTPUTDIR;
      if (@tneffiles) {
        foreach $tneffile (@tneffiles) {
          Log::InfoLog("Expanding TNEF archive at $path/$tneffile");
          $result = TryTNEFDecoder("$Config::TNEF -f $path/$tneffile -C $path --overwrite");
          unless ($result) {
            Log::WarnLog("Corrupt TNEF $tneffile that cannot be analysed in message $id");
            push @$BadTNEF, $id;
          }
        }
      }
      #$IsTNEF{$id} = $entity;
    }
    # Close and unlock the data file
    $Config::MTA eq "exim" and Lock::unlockclose($Df);
  }
  %entities;
}

# Try running the TNEF decoder with a timeout.
# Return 0 on failure, 1 on success.
sub TryTNEFDecoder
{
  my($cmd) = @_;
  local(*KID);
  my($TimedOut, $PipeReturn, $pid);

  $TimedOut = 0;

  eval {
    die "Can't fork: $!" unless defined($pid = open(KID, "-|"));
    if ($pid) {
      # In the parent
      local $SIG{ALRM} = sub { $TimedOut = 1; die "Command Timed Out" }; # 2.53
      alarm $Config::TNEFTimeout;
      close KID; # This will wait for completion
      $PipeReturn = $?;
      $pid = 0;
      alarm 0;
    } else {
      POSIX::setsid(); # 2.53
      exec $cmd or die "Can't run tnef decoder: $!";
    }
  };
  alarm 0; # 2.53

  # Note to self: I only close the KID in the parent, not in the child.

  # Catch failures other than the alarm
  Log::DieLog("TNEF decoder failed with real error: $@")
    if $@ and $@ !~ /Command Timed Out/;

  # In which case any failures must be the alarm
  if ($@ or $pid>0) {
    # Kill the running child process
    my($i);
    kill 'TERM', $pid;
    # Wait for up to 10 seconds for it to die
    for ($i=0; $i<10; $i++) {
      sleep 1;
      ($pid=0),last unless kill(0, $pid);
      #kill 'TERM', $pid;
    }
    # And if it didn't respond to 11 nice kills, we kill -9 it
    kill 'KILL', $pid if $pid;
    wait; # 2.53
  }

  # Now the child is dead, look at all the return values

  # Do we want to deliver unparsable TNEF files anyway (like we used to)
  if ($Config::DeliverBadTNEF) {
    return 0 if $TimedOut; # Ignore tnef command exit status
    return 1; # Command terminated
  } else {
    return 0 if $TimedOut || $PipeReturn; # Command failed to complete successfully
    return 1; # Command succeded and terminated
  }
}




# Find the TNEF attachment within this MIME entity tree.
# Recursive.
sub FindTNEFFile {
  my($entity) = @_;

  my(@parts, $body, $part, $path, $headfile, $tnef);

  # Find the body for this entity
  $body = $entity->bodyhandle;
  if (defined($body) && defined($body->path)) {   # data is on disk:
    $path = $body->path;
    return $entity if $path =~ /winmail\d*\.dat\d*$/i;
    #$path =~ s#^.*/([^/]*)$#$1#;
  }
  # And the head, which is where the recommended filename is stored
  # This is so we can report infections in the filenames which are
  # recommended, even if they are evil and we hence haven't used them.
  $headfile = $entity->head->recommended_filename;
  return $entity if $headfile =~ /winmail\d*\.dat\d*$/i;

  # And for all its children
  @parts = $entity->parts;
  foreach $part (@parts) {
    $tnef = FindTNEFFile($part);
    return $tnef if defined($tnef);
  }

  # Must return something.
  return undef;
}



# Print the skeleton of all the messages
sub DumpSkeletons {
  my($entities) = @_;
  my($entity, $id);

  while(($id, $entity) = each %$entities) {
    print "\nMessage $id looks like this\n";
    print "---------------------------\n";
    $entity->dump_skeleton;
  }
}

# Replace 1 infected section with clean text file attachment.
sub DisinfectEntity {
  my($infected, $parent, $filename, $report, $MsgId, $reportname) = @_;

  my($i, @parts, $infectednum, $Warning);

  # Infections applying to the entire message can't be simply disinfected.
  # Should replace entire message text here with a text/plain error.
  if (!$parent) {
    $Warning = ConstructWarning("the entire message", $report,
                                $MsgId, $reportname);
    $parts[0] = build MIME::Entity
                           Type => 'text/plain',
                           Filename => $Config::AttachmentWarningFilename,
                           Disposition => 'inline',
                           Data => $Warning,
                           Encoding => 'quoted-printable',
                           Charset => 'us-ascii',
                           Top => 0;
    $infected->parts(\@parts);
    return;
  }

  # Infection only applies to part of the message, so just replace that
  # part with an error message.
  @parts = $parent->parts;
  # Find the infected part
  $infectednum = -1;
  for ($i=0; $i<@parts; $i++) {
    ($infectednum=$i),last if $parts[$i]==$infected;
  }
  Log::WarnLog("Oh shit, missed infected entity in message :-( %m"), return
    if $infectednum<0;

  # Now to actually do something about it...
  $Warning = ConstructWarning($filename, $report,
                              $MsgId, $reportname);
  $parts[$infectednum] = build MIME::Entity
                           Type => 'text/plain',
                           Filename => $Config::AttachmentWarningFilename,
                           Disposition => 'attachment',
                           Data => $Warning,
                           Encoding => 'quoted-printable',
                           Charset => 'us-ascii',
                           Top => 0;
  $parent->parts(\@parts);
}

# Disinfect all the infected entities
sub Disinfect {
  my($Reports, $Types, $Id2Entity, $File2Entity, $Entity2Parent, $Entity2File, $IsTNEF) = @_;

  my($id,$parts,$file,$text,$entity,%MessageSigned,$type,$filename);

  # 24/05/01 If the report is about a TNEF message, then the infected
  # entity is $IsTNEF->{$id}, and %Entity2Parent and %Entity2File of this
  # entity should be perfectly valid (corresponding to winmail.dat file).
  while(($id, $parts) = each %$Reports) {
    # Disinfect the attachment that's infected
    while(($file, $text) = each %$parts) {
      # If it's a TNEF message, then use the entity of the winmail.dat
      # file, else use the entity of the infected file.
      if ($file ne "" && defined($IsTNEF->{$id})) {
        $entity = $IsTNEF->{$id};
      } else {
        $entity = $File2Entity->{$id}{$file};
      }

      # 2.60
      # Work out which message file to replace the attachment with
      $type = $Types->{"$id"}{"$file"};
      if ($type =~ /v/i) {
        # It's a virus. Should we store or delete it?
        $filename = $Config::DeletedVirusMessageText;
        $filename = $Config::StoredVirusMessageText
          if $Config::QuarantineAction =~ /stor/i;
      } elsif ($type =~ /f/i) {
        # It's a filename trap. Should we store or delete it?
        $filename = $Config::DeletedFilenameMessageText;
        $filename = $Config::StoredFilenameMessageText
          if $Config::QuarantineAction =~ /stor/i;
      } else {
        # Treat it like a virus anyway. Structure is like this as
        # $type could well have a v and an f in it.
        $filename = $Config::DeletedVirusMessageText;
        $filename = $Config::StoredVirusMessageText
          if $Config::QuarantineAction =~ /stor/i;
      }

      # 2.53 If $entity="" then message could not be parsed so no data structure
      # 2.53 exists for it. So just ignore it (this is less than ideal but way
      # 2.53 better than nothing.
      next unless $entity;
      DisinfectEntity($entity,
                      $Entity2Parent->{$entity},
                      $Entity2File->{$entity},
                      $text,
                      $id,
                      $filename);
    }

    # Mark the message as disinfected, if the user wants us to
    if ($Config::MarkInfectedMessages && !$MessageSigned{"$id"}) {
      #print STDERR "About to Sign message $id\n";
      SignWarningMessage($Id2Entity->{$id});
      # Remember, so we don't mark the message more than once
      $MessageSigned{"$id"} = 1;
    }
  }
}

# Add an inline signature to the bottom of each clean message.
# We know these are MIME messages of one sort or another.
# SignCleanMessages($CleanIds, \%MimeEntities);
sub SignCleanMessages {
  my($CleanIds, $Id2Entity) = @_;

  my($id, $entity);

  foreach $id (@$CleanIds) {
    # Sign the bottom of the MIME structure
    $entity = $Id2Entity->{"$id"};
    SignCleanMessage($entity);
    #$entity->sign('Remove'=>0, 'Signature'=>$Config::InlineSig);
  }
}

# Sign the end of a message with the given notice
sub SignCleanMessage {
  my($top) = @_;
  my($MimeType);

  return unless $top;

  # If multipart, try to sign our first part
  if ($top->is_multipart) {
    SignCleanMessage($top->parts(0));
    SignCleanMessage($top->parts(1))
      if $top->head and $top->effective_type =~ /multipart\/alternative/i;
    return;
  }

  $MimeType = $top->head->mime_type if $top->head;
  return unless $MimeType =~ m{text/}i; # Won't sign non-text message.

  # Get body data as array of newline-terminated lines
  $top->bodyhandle or return undef;
  my @body = $top->bodyhandle->as_lines;

  # Output original data back into body, followed by message
  my($line, $io);
  $io = $top->open("w");
  if ($MimeType =~ /text\/html/i) {
    foreach $line (@body) {
      $line =~ s/\<\/x?html\>/$Config::InlineHTMLSig$&/i;
      $io->print($line);
    }
    (($body[-1]||'') =~ /\n\Z/) or $io->print("\n"); # Ensure final newline
  } else {
    foreach $line (@body) { $io->print($line) }; # Original body data
    $io->print("\n$Config::InlineTextSig\n");
  }
  $io->close;
}

# Sign the beginning of a message with the given warning
sub SignWarningMessage {
  my($top) = @_;
  my($MimeType);

  return unless $top;
  #print STDERR "It's a multipart\n" if $top->is_multipart;

  # If multipart, try to sign our first part
  if ($top->is_multipart) {
    SignWarningMessage($top->parts(0));
    SignWarningMessage($top->parts(1))
      if $top->head and $top->effective_type =~ /multipart\/alternative/i;
    return;
  }

  $MimeType = $top->head->mime_type if $top->head;
  #print STDERR "It's not text\n" unless $MimeType =~ m{text/}i;
  return unless $MimeType =~ m{text/}i; # Won't sign non-text message.

  # Get body data as array of newline-terminated lines
  $top->bodyhandle or return undef;
  my @body = $top->bodyhandle->as_lines;

  #print STDERR "Signing message part\n";

  # Output message back into body, followed by original data
  my($line, $io);
  $io = $top->open("w");
  if ($MimeType =~ /text\/html/i) {
    foreach $line (@body) {
      $line =~ s/\<html\>/$&$Config::InlineHTMLWarning/i;
      $io->print($line);
    }
  } else {
    $io->print("$Config::InlineTextWarning\n");
    foreach $line (@body) { $io->print($line) }; # Original body data
  }
  (($body[-1]||'') =~ /\n\Z/) or $io->print("\n"); # Ensure final newline
  $io->close;
}


# Construct an infection warning message to the user based on the filename
sub ConstructWarning {
  my($file, $report, $id, $filename) = @_;
  my($output, $result);
  my($date) = scalar localtime;
  local(*TEXT);
  $report =~ s/^/   /gm;
  #print "Action in ConstructWarning = \"$Config::QuarantineAction\"\n";
  #$filename = $Config::DeletedMessageText;
  #$filename = $Config::StoredMessageText if $Config::QuarantineAction =~ /stor/i;

  $output = "";
  open(TEXT, $filename) or Log::WarnLog("Cannot open message file $filename");
  while (<TEXT>) {
    chomp;
    s#"#\\"#g;
    # Untainting joy...
    /(.*)/;
    $result = eval "\"$1\"";
    $output .= $result . "\n";
  }
  $output;
}

# Write out all the messages to text files
sub WriteMessages {
  my($Dir, $IdList, $entities) = @_;
  my($message);
  local(*ID);

  chdir $Dir or do {
    Log::WarnLog("Cannot chdir to $Dir to write messages, %s", $!);
    return;
  };

  foreach $message (@$IdList) {
    Log::DebugLog("Writing $Dir/$message\n");
    open(ID, ">$message") or next;
    $entities->{$message}->print_header(\*ID);
    print ID "\n";
    $entities->{$message}->print_body(\*ID);
    close ID;
  }
}

# Save the infected attachments into the quarantine area.
sub QuarantineInfections {
  my($Reports, $QuarDir, $InDir, $InQDir) = @_;
  my($day,$month,$year);
  my($TodayDir, $MsgDir);
  my($id,$dfile,$parts,$dirtyfile);

  # Construct the quarantine dir for this message
  mkdir($QuarDir, 0700) unless -d $QuarDir;
  ($day,$month,$year) = (localtime)[3,4,5];
  $month++;
  $year += 1900;
  $TodayDir = "$QuarDir/" . sprintf("%04d%02d%02d", $year, $month, $day);
  mkdir($TodayDir, 0700) unless -d $TodayDir;

  while(($id,$parts) = each %$Reports) {
    $MsgDir = "$TodayDir/$id";
    $dfile = MTA::DFileName($id);
    mkdir($MsgDir, 0700) unless -d $MsgDir;
    if (defined $parts->{""}) {
      # There is an infection report applying to the whole message,
      # so quarantine the entire thing, headers and all.
      Log::InfoLog("Saved entire message to $MsgDir");
      system(MTA::BuildMessageCmd("$InDir/$id.header","$InQDir/$dfile").
	     " > \"$MsgDir/message\"");
    } else {
      # There are infection reports applying to specific attachments.
      foreach $dirtyfile (keys %$parts) {
        # Write the infected file into this dir
        Log::InfoLog("Saved infections to $MsgDir");
        system("$cp -p \"$InDir/$id/$dirtyfile\" \"$MsgDir/$dirtyfile\"");
      }
    }
  }

  chdir $QuarDir;
}

1;
