/*
 * libsysactivity
 * http://sourceforge.net/projects/libsysactivity/
 * Copyright (c) 2009, 2010 Carlos Olmedo Escobar <carlos.olmedo.e@gmail.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <stdio.h>
#include <unistd.h>
#include <pthread.h>
#include <stdint.h>
#include <stdlib.h>
#include <string.h>
#include <inttypes.h>

#include "libsysactivity.h"

int error = 0;

void print_memory_info(struct sa_memory* memory) {
#ifdef SA_MEMORY_TOTAL
		printf("total: %"PRIu64, memory->total);
#endif
#ifdef SA_MEMORY_FREE
		printf(", free: %"PRIu64, memory->free);
#endif
#ifdef SA_MEMORY_ACTIVE
		printf(", active: %"PRIu64, memory->active);
#endif
#ifdef SA_MEMORY_INACTIVE
		printf(", inactive: %"PRIu64, memory->inactive);
#endif
#ifdef SA_MEMORY_BUFFERS
		printf(", buffers: %"PRIu64, memory->buffers);
#endif
#ifdef SA_MEMORY_SWAP_TOTAL
		printf(", swap_total: %"PRIu64, memory->swap_total);
#endif
#ifdef SA_MEMORY_SWAP_FREE
		printf(", swap_free: %"PRIu64, memory->swap_free);
#endif
#ifdef SA_MEMORY_SWAP_CACHED
		printf(", swap_cached: %"PRIu64, memory->swap_cached);
#endif
#ifdef SA_MEMORY_DIRTY
		printf(", dirty: %"PRIu64, memory->dirty);
#endif
#ifdef SA_MEMORY_CACHED
		printf(", cached: %"PRIu64, memory->cached);
#endif
#ifdef SA_MEMORY_WIRED
		printf(", wired: %"PRIu64, memory->wired);
#endif
#ifdef SA_MEMORY_EXECUTABLE
		printf(", executable: %"PRIu64, memory->executable);
#endif
#ifdef SA_MEMORY_FILES
		printf(", files: %"PRIu64, memory->files);
#endif
#ifdef SA_MEMORY_LOCKED
		printf(", locked: %"PRIu64, memory->locked);
#endif
		printf("\n");
}

void test_memory_info(struct sa_memory* memory) {
#if defined(SA_MEMORY_TOTAL)
	if (memory->total == 0) {
		printf("%s:%d ERROR: The total memory space is zero\n", __FILE__, __LINE__);
		error = 1;
	}
#endif
#if defined(SA_MEMORY_TOTAL) && defined(SA_MEMORY_FREE)
	if (memory->free >= memory->total) {
		printf("%s:%d ERROR: The free memory space is larger or equal than the total space\n", __FILE__, __LINE__);
		error = 1;
	}
#endif
#if defined(SA_MEMORY_TOTAL) && defined(SA_MEMORY_ACTIVE)
	if (memory->active > memory->total) {
		printf("%s:%d ERROR: Active memory is larger than total memory\n", __FILE__, __LINE__);
		error = 1;
	}
	if (memory->active < 1000) {
		printf("%s:%d ERROR: Active memory is lower than 1 kb\n", __FILE__, __LINE__);
		error = 1;
	}
#endif
#if defined(SA_MEMORY_TOTAL) && defined(SA_MEMORY_INACTIVE)
	if (memory->inactive > memory->total) {
		printf("%s:%d ERROR: Inactive memory is larger than total memory\n", __FILE__, __LINE__);
		error = 1;
	}
#if !defined(__NetBSD__) // In my NetBSD 5.1rc4 system vmstat -s gives 0 inactive pages so it seems to be ok
	if (memory->inactive < 1000) {
		printf("%s:%d ERROR: Inactive memory is lower than 1 kb\n", __FILE__, __LINE__);
		error = 1;
	}
#endif
#endif
#if defined(SA_MEMORY_TOTAL) && defined(SA_MEMORY_BUFFERS)
	if (memory->buffers > memory->total) {
		printf("%s:%d ERROR: Memory used in buffers is larger than total memory\n", __FILE__, __LINE__);
		error = 1;
	}
#endif
#ifdef SA_MEMORY_SWAP_TOTAL
#ifdef SA_MEMORY_SWAP_FREE
	if (memory->swap_free > memory->swap_total) {
		printf("%s:%d ERROR: Free swap space is larger than total swap space\n", __FILE__, __LINE__);
		error = 1;
	}
#endif
#ifdef SA_MEMORY_SWAP_CACHED
	if (memory->swap_cached > memory->swap_total) {
		printf("%s:%d ERROR: Cached swap space is larger than total swap space\n", __FILE__, __LINE__);
		error = 1;
	}
#endif
#endif
}

void* get_memory_info(void* arg) {
	int i;
	int ret;
	struct sa_memory memory;

#ifdef SA_OPEN_MEMORY
	ret = sa_open_memory();
	if (ret != 0) {
		printf("%s:%d ERROR: sa_open_memory(): %s\n", __FILE__, __LINE__, strerror(ret));
		exit(EXIT_FAILURE);
	}
#endif

	for (i = 0; i < 6; i++) {
		ret = sa_get_memory(&memory);
		if (ret != 0) {
			printf("%s:%d ERROR: sa_get_memory(): %s\n", __FILE__, __LINE__, strerror(ret));
			exit(EXIT_FAILURE);
		}
		print_memory_info(&memory);
		printf("\n");
		test_memory_info(&memory);
		sleep(1);
	}
#ifdef SA_CLOSE_MEMORY
	ret = sa_close_memory();
	if (ret != 0) {
		printf("%s:%d ERROR: sa_close_memory(): %s\n", __FILE__, __LINE__, strerror(ret));
		exit(EXIT_FAILURE);
	}
#endif
#ifdef SA_OPEN_MEMORY
	ret = sa_open_memory();
	if (ret != 0) {
		printf("%s:%d ERROR: sa_open_memory(): %s\n", __FILE__, __LINE__, strerror(ret));
		exit(EXIT_FAILURE);
	}
#endif
#ifdef SA_CLOSE_MEMORY
	ret = sa_close_memory();
	if (ret != 0) {
		printf("%s:%d ERROR: sa_close_memory(): %s\n", __FILE__, __LINE__, strerror(ret));
		exit(EXIT_FAILURE);
	}
#endif
	return NULL;
}

int main() {
	pthread_t thread1;
	struct timespec delay;
	delay.tv_sec = 0;
	delay.tv_nsec = 500000000;

	pthread_create(&thread1, NULL, get_memory_info, NULL);
	nanosleep(&delay, NULL);
	get_memory_info(NULL);

	if (error)
		return EXIT_FAILURE;

	return EXIT_SUCCESS;
}
