/*
 * libsysactivity
 * http://sourceforge.net/projects/libsysactivity/
 * Copyright (c) 2009-2011 Carlos Olmedo Escobar <carlos.olmedo.e@gmail.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <stdio.h>
#include <string.h>
#include <inttypes.h>
#include <stdlib.h>
#include <errno.h>
#include <ctype.h>

#include "utils.h"

static void parse_swap(struct sa_swap* __restrict__ dst, char* pos) SA_NONNULL;

extern int kernel_version;
__thread FILE* file_swaps;
__thread long swap_useful_pos;
__thread char s_line_buffer[512];

int sa_open_swap() {
	file_swaps = NULL;

	if (kernel_version < 11)
		return ENOTSUP;

	file_swaps = fopen("/proc/swaps", "r");
	if (file_swaps == NULL)
		return EIO;

	if (fgets(s_line_buffer, sizeof(s_line_buffer), file_swaps) == NULL)
		return ENOTSUP;
	swap_useful_pos = ftell(file_swaps);
	if (swap_useful_pos == -1)
		return ENOTSUP;

	return 0;
}

int sa_close_swap() {
	if (file_swaps != NULL)
		fclose(file_swaps);
	return 0;
}

int sa_count_swaps(uint16_t* number) {
	if (number == NULL)
		return EINVAL;

	if (sa_reset_swaps() != 0)
		return EIO;

	*number = 0;
	while (fgets(s_line_buffer, sizeof(s_line_buffer), file_swaps) != NULL)
		++(*number);

	return 0;
}

int sa_reset_swaps() {
	if (fflush(file_swaps) != 0)
		return EIO;
	fseek(file_swaps, swap_useful_pos, SEEK_SET);
	return 0;
}

int sa_get_swap(uint16_t index, struct sa_swap* dst) {
	if (dst == NULL)
		return EINVAL;

	int i;
	for (i = 0; fgets(s_line_buffer, sizeof(s_line_buffer), file_swaps) != NULL; ++i) {
		if (i != index)
			continue;

		errno = 0;
		parse_swap(dst, s_line_buffer);
		if (errno != 0)
			return ENOSYS;

		return 0;
	}

	return ENODEV;
}

int sa_get_swaps(struct sa_swap* dst, uint16_t dst_size, uint16_t* written) {
	if (dst == NULL || dst_size == 0 || written == NULL)
		return EINVAL;

	*written = 0;
	int i;
	for (i = 0; fgets(s_line_buffer, sizeof s_line_buffer, file_swaps) != NULL; ++i) {
		if (i >= dst_size)
			return ENOMEM;

		errno = 0;
		parse_swap(&dst[i], s_line_buffer);
		if (errno != 0)
			return ENOSYS;

		++(*written);
	}

	return 0;
}

static void parse_swap(struct sa_swap* __restrict__ dst, char* pos) {
	size_t end = strchr(pos, ' ') - pos;
	if (sizeof(dst->name) - 1 < end)
		end = sizeof(dst->name) - 1;
	strlcpy(dst->name, pos, end);
	pos = skip_value(pos);
	dst->type = strncmp(pos, "file", 4) == 0 ? 2 : 1;
	pos = skip_value(pos);
	dst->total = strtoull(pos, NULL, 10) * 1024;
	pos = skip_value(pos);
	dst->free = dst->total - strtoull(pos, NULL, 10) * 1024;
}
