/*
 * libsysactivity
 * http://sourceforge.net/projects/libsysactivity/
 * Copyright (c) 2009, 2010 Carlos Olmedo Escobar <carlos.olmedo.e@gmail.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/sysctl.h>
#include <sys/user.h>
#include <string.h>

#include "libsysactivity.h"

static int refresh_processes(size_t* len);
static void assign(struct sa_process* dst, struct kinfo_proc* kinfo_proc);

__thread struct kinfo_proc* processes = NULL;
__thread size_t processes_size = 0;

int sa_close_process() {
	if (processes != NULL) {
		free(processes);
		processes = NULL;
	}

	return 0;
}

int sa_count_processes(pid_t* number) {
	int mib[3];
	size_t len;

	mib[0] = CTL_KERN;
	mib[1] = KERN_PROC;
	mib[2] = KERN_PROC_PROC;
	if (sysctl(mib, 3, NULL, &len, NULL, 0) == -1)
		return ENOSYS;

	*number = len / sizeof(struct kinfo_proc);
	return 0;
}

int sa_get_processes_ids(pid_t* dst, pid_t dst_size, pid_t* written) {
	if (dst == NULL || dst_size <= 0 || written == NULL)
		return EINVAL;

	size_t len;
	if (refresh_processes(&len) != 0)
		return ENOSYS;

	uint16_t i;
	*written = 0;
	len /= sizeof(struct kinfo_proc);
	for (i = 0; i < len; i++) {
		if (i == dst_size)
			return ENOMEM;

		dst[i] = processes[i].ki_pid;
		(*written)++;
	}

	return 0;
}

int sa_get_process(pid_t pid, struct sa_process* dst) {
	if (pid == 0 || dst == NULL)
		return EINVAL;

	size_t len;
	if (refresh_processes(&len) != 0)
		return ENOSYS;

	uint16_t i;
	len /= sizeof(struct kinfo_proc);
	for (i = 0; i < len; i++) {
		if (processes[i].ki_pid != pid)
			continue;

		assign(dst, &processes[i]);
		return 0;
	}
	return ESRCH;
}

int sa_get_processes(struct sa_process* dst, pid_t dst_size, pid_t* written) {
	if (dst == NULL || dst_size <= 0 || written == NULL)
		return EINVAL;

	size_t len;
	if (refresh_processes(&len) != 0)
		return ENOSYS;

	uint16_t i;
	*written = 0;
	len /= sizeof(struct kinfo_proc);
	for (i = 0; i < len; i++) {
		if (i == dst_size)
			return ENOMEM;

		assign(&dst[i], &processes[i]);
		(*written)++;
	}

	return 0;
}

static int refresh_processes(size_t* len) {
	int mib[3]; // TODO make this var static?

	mib[0] = CTL_KERN;
	mib[1] = KERN_PROC;
	mib[2] = KERN_PROC_PROC;
	if (sysctl(mib, 3, NULL, len, NULL, 0) == -1)
		return ENOSYS;

	if (*len > processes_size) {
		processes = realloc(processes, *len);
		if (processes == NULL)
			return ENOMEM;
		processes_size = *len;
	}
	if (sysctl(mib, 3, processes, len, NULL, 0) == -1)
		return ENOSYS;

	return 0;
}

static void assign(struct sa_process* dst, struct kinfo_proc* kinfo_proc) {
	dst->pid = kinfo_proc->ki_pid;
	dst->uid = kinfo_proc->ki_uid;
	dst->gid = kinfo_proc->ki_rgid;
	strlcpy(dst->filename, kinfo_proc->ki_comm, SA_PROCESS_FILENAME);
	dst->parent_pid = kinfo_proc->ki_ppid;
	dst->pgrp = kinfo_proc->ki_pgid;
	dst->sid = kinfo_proc->ki_sid;
	dst->tty = kinfo_proc->ki_tpgid;
	dst->nice = kinfo_proc->ki_nice;
	dst->threads = kinfo_proc->ki_numthreads;
}
