/*
 * libsyncml - A syncml protocol implementation
 * Copyright (C) 2005  Armin Bauer <armin.bauer@opensync.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; version 
 * 2.1 of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 */
 
#include "syncml.h"
#include "syncml_internals.h"

#include "sml_support.h"

#include "config.h"

#include <glib.h>
#include <pthread.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>

GPrivate* current_tabs = NULL;

/**
 * @defgroup PublicAPI The Public API
 * 
 */
/*@{*/

/*@}*/

/**
 * @defgroup PrivateAPI The Private API
 * 
 */
/*@{*/

/*@}*/

/**
 * @defgroup SmlDebugAPI Libsyncml Debug
 * @ingroup PrivateAPI
 * @brief Debug functions used by libsyncml
 * 
 */
/*@{*/

void smlLog(const char *logname, const char *data, unsigned int size)
{
#if defined ENABLE_TRACE
	const char *trace = g_getenv("SYNCML_LOG");
	if (!trace)
		return;
	
	if (!g_file_test(trace, G_FILE_TEST_IS_DIR)) {
		printf("SYNCML_LOG argument is no directory\n");
		return;
	}
	
	int i = 0;
	char *logfile = NULL;
	for (;;i = i + 1) {
		char *logfile_tmp = g_strdup_printf("%s/%s", trace, logname);
		logfile = g_strdup_printf(logfile_tmp, i);
		g_free(logfile_tmp);
		if (!g_file_test(logfile, G_FILE_TEST_EXISTS))
			break;
		g_free(logfile);
	}
	
	GError *error = NULL;
	GIOChannel *chan = g_io_channel_new_file(logfile, "w", &error);
	if (!chan) {
		printf("unable to open %s for writing: %s\n", logfile, error->message);
		return;
	}
	
	gsize writen;
	g_io_channel_set_encoding(chan, NULL, NULL);
	if (g_io_channel_write_chars(chan, data, size, &writen, NULL) != G_IO_STATUS_NORMAL) {
		printf("unable to write trace to %s\n", logfile);
	} else
		g_io_channel_flush(chan, NULL);

	g_io_channel_shutdown(chan, TRUE, NULL);
	g_io_channel_unref(chan);
#endif
}

/*! @brief Used for tracing the application
 * 
 * use this function to trace calls. The call graph will be saved into
 * the file that is given in the SYNCML_TRACE environment variable
 * 
 * @param type The type of the trace
 * @param message The message to save
 * 
 */
void smlTrace(SmlTraceType type, const char *message, ...)
{
#if defined ENABLE_TRACE
	va_list arglist;
	char *buffer = NULL;
	
	const char *trace = g_getenv("SYNCML_TRACE");
	if (!trace)
		return;
	
	if (!g_file_test(trace, G_FILE_TEST_IS_DIR)) {
		printf("SYNCML_TRACE argument is no directory\n");
		return;
	}
	
	if (!g_thread_supported ()) g_thread_init (NULL);
	int tabs = 0;
	if (!current_tabs)
		current_tabs = g_private_new (NULL);
	else
		tabs = GPOINTER_TO_INT(g_private_get(current_tabs));
	
	unsigned long int id = (unsigned long int)pthread_self();
	char *logfile = g_strdup_printf("%s/Thread%lu.log", trace, id);
	
	va_start(arglist, message);
	buffer = g_strdup_vprintf(message, arglist);
	
	GString *tabstr = g_string_new("");
	long i = 0;
	for (i = 0; i < tabs; i++) {
		tabstr = g_string_append(tabstr, "\t");
	}

	GTimeVal curtime;
	g_get_current_time(&curtime);
	char *logmessage = NULL;
	switch (type) {
		case TRACE_ENTRY:
			logmessage = g_strdup_printf("[%li.%li]\t%s>>>>>>>  %s\n", curtime.tv_sec, curtime.tv_usec, tabstr->str, buffer);
			tabs++;
			break;
		case TRACE_INTERNAL:
			logmessage = g_strdup_printf("[%li.%li]\t%s%s\n", curtime.tv_sec, curtime.tv_usec, tabstr->str, buffer);
			break;
		case TRACE_ERROR:
			logmessage = g_strdup_printf("[%li.%li]\t%sERROR: %s\n", curtime.tv_sec, curtime.tv_usec, tabstr->str, buffer);
			break;
		case TRACE_EXIT:
			logmessage = g_strdup_printf("[%li.%li]%s<<<<<<<  %s\n", curtime.tv_sec, curtime.tv_usec, tabstr->str, buffer);
			tabs--;
			if (tabs < 0)
				tabs = 0;
			break;
		case TRACE_EXIT_ERROR:
			logmessage = g_strdup_printf("[%li.%li]%s<--- ERROR --- %s\n", curtime.tv_sec, curtime.tv_usec, tabstr->str, buffer);
			tabs--;
			if (tabs < 0)
				tabs = 0;
			break;
	}
	g_free(buffer);
	
	g_private_set(current_tabs, GINT_TO_POINTER(tabs));
	va_end(arglist);
	
	g_string_free(tabstr, TRUE);
	
	GError *error = NULL;
	GIOChannel *chan = g_io_channel_new_file(logfile, "a", &error);
	if (!chan) {
		printf("unable to open %s for writing: %s\n", logfile, error->message);
		g_free(logfile);
		g_free(logmessage);
		return;
	}
	
	gsize writen;
	g_io_channel_set_encoding(chan, NULL, NULL);
	if (g_io_channel_write_chars(chan, logmessage, strlen(logmessage), &writen, NULL) != G_IO_STATUS_NORMAL) {
		printf("unable to write trace to %s\n", logfile);
	} else
		g_io_channel_flush(chan, NULL);

	g_io_channel_shutdown(chan, TRUE, NULL);
	g_io_channel_unref(chan);
	g_free(logmessage);
	g_free(logfile);
#endif
}

/*! @brief Used for printing binary data
 * 
 * Unprintable character will be printed in hex, printable are just printed
 * 
 * @param data The data to print
 * @param len The length to print
 * 
 */
char *smlPrintBinary(const char *data, int len)
{
  int t;
  GString *str = g_string_new("");
  for (t = 0; t < len; t++) {
    if (data[t] >= ' ' && data[t] <= 'z')
      g_string_append_c(str, data[t]);
    else
      g_string_append_printf(str, " %02x ", data[t]);
  }
  return g_string_free(str, FALSE);
}

/*! @brief Used for printing binary data in just hex
 * 
 * @param data The data to print
 * @param len The length to print
 * 
 */
char *smlPrintHex(const char *data, int len)
{
  int t;
  GString *str = g_string_new("");
  for (t = 0; t < len; t++) {
    g_string_append_printf(str, " %02x", data[t]);
		if (data[t] >= ' ' && data[t] <= 'z')
    		g_string_append_printf(str, "(%c)", data[t]);
	g_string_append_c(str, ' ');
  }
  return g_string_free(str, FALSE);
}

/*! @brief Creates a random string
 * 
 * Creates a random string of given length or less
 * 
 * @param maxlength The maximum length of the string
 * @returns The random string
 * 
 */
char *smlRandStr(int maxlength, SmlBool exact)
{
	char *randchars = "abcdefghijklmnopqrstuvwxyzABCDEFGHIKLMNOPQRSTUVWXYZ1234567890";
	
	int length;
	char *retchar;
	int i = 0;

	if (exact)
		length = maxlength;
	else
		length = g_random_int_range(1, maxlength + 1);
	
	retchar = malloc(length * sizeof(char) + 1);
	retchar[0] = 0;

	for (i = 0; i < length; i++) {
		retchar[i] = randchars[g_random_int_range(0, strlen(randchars))];
		retchar[i + 1] = 0;
	}

	return retchar;
}

/*! @brief Safely mallocs
 * 
 * Mallocs or returns an error if OOM
 * 
 * @param n_bytes The size in bytes to malloc
 * @param error Pointer to a error struct
 * @returns The newly allocated memory or NULL in case of error
 * 
 */
void *smlTryMalloc0(long n_bytes, SmlError **error)
{
	void *result = g_try_malloc(n_bytes);
	if (!result) {
		smlErrorSet(error, SML_ERROR_NO_MEMORY, "No memory left");
		return NULL;
	}
	memset(result, 0, n_bytes);
	return result;
}

/*! @brief Gets the version of libsyncml as string
 * 
 * @returns The version as const string
 * 
 */
const char *smlGetVersion(void)
{
	/**
	* @version $Id$
	*/ 
	return "$Id$";
}

SmlThread *smlThreadNew(GMainContext *context, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p)", __func__, context, error);
	
	SmlThread *thread = smlTryMalloc0(sizeof(SmlThread), error);
	if (!thread)
		goto error;

	if (!g_thread_supported ()) g_thread_init (NULL);
	
	thread->started_mutex = g_mutex_new();
	thread->started = g_cond_new();
	thread->context = context;
	if (thread->context)
		g_main_context_ref(thread->context);
	thread->loop = g_main_loop_new(thread->context, FALSE);
	
	smlTrace(TRACE_EXIT, "%s: %p", __func__, thread);
	return thread;
	
error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return NULL;
}

void smlThreadFree(SmlThread *thread)
{
	smlTrace(TRACE_ENTRY, "%s(%p)", __func__, thread);
	smlAssert(thread);
	
	if (thread->started_mutex)
		g_mutex_free(thread->started_mutex);

	if (thread->started)
		g_cond_free(thread->started);
	
	if (thread->loop)
		g_main_loop_unref(thread->loop);
	
	if (thread->context)
		g_main_context_unref(thread->context);
		
	g_free(thread);
	smlTrace(TRACE_EXIT, "%s", __func__);
}

static gpointer smlThreadStartCallback(gpointer data)
{
	smlTrace(TRACE_ENTRY, "%s(%p)", __func__, data);
	SmlThread *thread = data;
	smlTrace(TRACE_INTERNAL, "+++++++++ This is the worker thread of thread %p for context %p +++++++++", thread, thread->context);
	
	g_mutex_lock(thread->started_mutex);
	g_cond_signal(thread->started);
	g_mutex_unlock(thread->started_mutex);
	
	g_main_loop_run(thread->loop);
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return FALSE;
}

static gboolean smlThreadStopCallback(gpointer data)
{
	smlTrace(TRACE_ENTRY, "%s(%p)", __func__, data);
	SmlThread *thread = data;
	smlTrace(TRACE_INTERNAL, "+++++++++ Quitting worker thread +++++++++");
	
	g_main_loop_quit(thread->loop);
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return FALSE;
}

void smlThreadStart(SmlThread *thread)
{
	smlTrace(TRACE_ENTRY, "%s(%p)", __func__, thread);
	smlAssert(thread);
	
	//Start the thread
	g_mutex_lock(thread->started_mutex);
	thread->thread = g_thread_create (smlThreadStartCallback, thread, TRUE, NULL);
	g_cond_wait(thread->started, thread->started_mutex);
	g_mutex_unlock(thread->started_mutex);
	
	smlTrace(TRACE_EXIT, "%s", __func__);
}

void smlThreadStop(SmlThread *thread)
{
	smlTrace(TRACE_ENTRY, "%s(%p)", __func__, thread);
	smlAssert(thread);
	
	GSource *source = g_idle_source_new();
	g_source_set_callback(source, smlThreadStopCallback, thread, NULL);
	g_source_attach(source, thread->context);

	g_thread_join(thread->thread);
	thread->thread = NULL;
	
	g_source_unref(source);
	
	smlTrace(TRACE_EXIT, "%s", __func__);
}

/*@}*/
