/*
 * libsyncml - A syncml protocol implementation
 * Copyright (C) 2005  Armin Bauer <armin.bauer@opensync.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; version 
 * 2.1 of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 */

#include "syncml.h"

#include "syncml_internals.h"
#include "sml_error_internals.h"

/**
 * @defgroup smlErrorPrivateAPI SyncML Error Internals
 * @ingroup PrivateAPI
 * @brief The private part of the error handling
 * 
 */
/*@{*/

/*! @brief Sets a error from a va_list
 * 
 * @param error A pointer to a error struct
 * @param type The type to set
 * @param format The message
 * @param args The arguments to the message
 * 
 */
void smlErrorSetVargs(SmlError **error, SmlErrorType type, const char *format, va_list args)
{
	return_if_fail(error);
	return_if_fail(smlErrorIsSet(error) == FALSE);
	return_if_fail(format);

	*error = smlTryMalloc0(sizeof(SmlError), NULL);
	(*error)->message = g_strdup_vprintf(format, args);
	(*error)->type = type;
	(*error)->refCount = 1;
	
	return;
}

/*@}*/

/**
 * @defgroup smlErrorAPI Libsyncml Errors
 * @ingroup PublicAPI
 * @brief Libsyncml's error reporting facilities
 * 
 */
/*@{*/

void smlErrorRef(SmlError **error)
{
	if (!smlErrorIsSet(error))
		return;
	
	g_atomic_int_inc(&(*error)->refCount);
}

void smlErrorDeref(SmlError **error)
{
	if (!smlErrorIsSet(error))
		return;
		
	if (!g_atomic_int_dec_and_test(&(*error)->refCount))
		return;
	
	if ((*error)->message)
		g_free ((*error)->message);
		
	g_free(*error);
	*error = NULL;
}

/*! @brief Checks if the error is set
 * 
 * @param error A pointer to a error struct to check
 * @returns TRUE if the error is set, FALSE otherwise
 * 
 */
SmlBool smlErrorIsSet(SmlError **error)
{
	if (!error)
		return FALSE;
		
	if (*error == NULL)
		return FALSE;
		
	return TRUE;
}

/*! @brief Returns the type of the error
 * 
 * @param error The error
 * @returns The type of the error
 * 
 */
SmlErrorType smlErrorGetType(SmlError **error)
{
	if (!smlErrorIsSet(error))
		return SML_NO_ERROR;
	
	return (*error)->type;
}

/*! @brief Returns the message of the error
 * 
 * @param error The error to print
 * @returns The message of the error or NULL if no error
 * 
 */
const char *smlErrorPrint(SmlError **error)
{
	if (!smlErrorIsSet(error))
		return NULL;
		
	return (*error)->message;
}

/*! @brief Updates the error message
 * 
 * You can use this function to update the error message on
 * a error. You can use the old error->message as a parameter
 * for this function.
 * 
 * @param error A pointer to a error struct to update
 * @param format The new message
 * 
 */
void smlErrorUpdate(SmlError **error, const char *format, ...)
{
	return_if_fail(smlErrorIsSet(error));

	va_list args;
	va_start(args, format);
	
	char buffer[1024];
	memset(buffer, 0, sizeof(buffer));
	g_vsnprintf(buffer, 1024, format, args);
	
	g_free((*error)->message);
	(*error)->message = g_strdup(buffer);
	
	va_end (args);
}

/*! @brief Duplicates the error into the target
 * 
 * 
 * @param target The target error to update
 * @param source The source error which to duplicate
 * 
 */
void smlErrorDuplicate(SmlError **target, SmlError **source)
{
	return_if_fail(target != NULL);
	return_if_fail(smlErrorIsSet(source));
	return_if_fail(!smlErrorIsSet(target));
	
	smlErrorSet(target, (*source)->type, (*source)->message);
}

/*! @brief Sets the error
 * 
 * You can use this function to set the error to the given type and message
 * 
 * @param error A pointer to a error struct to set
 * @param type The Error type to set
 * @param format The message
 * 
 */
void smlErrorSet(SmlError **error, SmlErrorType type, const char *format, ...)
{
	va_list args;
	va_start(args, format);
	smlErrorSetVargs(error, type, format, args);
	va_end (args);
}

/*! @brief Sets the type of an error
 * 
 * @param error A pointer to a error struct to set
 * @param type The Error type to set
 * 
 */
void smlErrorSetType(SmlError **error, SmlErrorType type)
{
	return_if_fail(error != NULL);
	
	(*error)->type = type;
}

/*! @brief Gets the error class
 * 
 * @param error A pointer to a error struct
 * @returns The error class
 * 
 */
SmlErrorClass smlErrorGetClass(SmlError **error)
{
	if (!smlErrorIsSet(error))
		return SML_ERRORCLASS_SUCCESS;

	if ((*error)->type == SML_NO_ERROR)
		return SML_ERRORCLASS_SUCCESS;
		
	return (int)((*error)->type / 100);
}

/*@}*/
