/* FullDescription.java
 * =========================================================================
 * This file is part of the SWIRL Library - http://swirl-lib.sourceforge.net
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 * 
 */

package be.ugent.caagt.swirl.menus;

import be.ugent.caagt.swirl.actions.*;
import javax.swing.AbstractButton;
import javax.swing.Action;
import javax.swing.Icon;
import javax.swing.JMenuItem;

/**
 * Extends a standard {@link Description} by also providing
 * the icons for the action it describes. We distinguish between two
 * different types of icon
 * <ul>
 *   <li>The <i>menu icon</i> is used when the action is installed as 
 *    a menu item.</li>
 *   <li>The <i>button icon</i> is used when the action is used on a toolbar.</li>
 * </ul>
 * In general button icons are slightly larger than the corresponding menu
 * icon.
 * For each kind of icon it is also possible to register a
 * form to be used on buttons or menu items that are disabled.
 * <p>
 * <b>Important:</b> to enable the system to distinguish between the two kinds of icon, use 
 * {@link ToolBarButton} or {@link ToolBarToggleButton} for toolbar buttons.
 * The standard Swing buttons will always use menu icons.
 */
class FullDescription extends Description {

    //
    private final Icon menuIcon;
    //
    private final Icon disabledMenuIcon;
    //
    private final Icon buttonIcon;
    //
    private final Icon disabledButtonIcon;
    //
    private final String tooltip;

    /**
     *  Create a full description by starting from the given (ordinary)
     *  description and registering the four icons.
     */
    public FullDescription(Description description, Icon menuIcon,
            Icon disabledMenuIcon, Icon buttonIcon, Icon disabledButtonIcon,
            String tooltip) {
        super(description.getCaption(),
                description.getMnemonic(),
                description.getMnemonicIndex(),
                description.getAcceleratorKey());
        this.menuIcon = menuIcon;
        this.disabledMenuIcon = disabledMenuIcon;
        this.buttonIcon = buttonIcon;
        this.disabledButtonIcon = disabledButtonIcon;
        this.tooltip = tooltip;
    }

    /**
     * Create a full description with the given description string
     * and icons.
     */
    public FullDescription(String description, Icon menuIcon,
            Icon disabledMenuIcon, Icon buttonIcon, Icon disabledButtonIcon,
            String tooltip) {
        super(description);
        this.menuIcon = menuIcon;
        this.disabledMenuIcon = disabledMenuIcon;
        this.buttonIcon = buttonIcon;
        this.disabledButtonIcon = disabledButtonIcon;
        this.tooltip = tooltip;
    }

    /**
     * Initialize caption, mnemonic, accelerator key and icons
     * for the given action, according to this description.
     */
    @Override
    public void init(Action action) {
        super.init(action);
        action.putValue(Action.SMALL_ICON, menuIcon);
        action.putValue(SimpleAction.SMALL_DISABLED_ICON, disabledMenuIcon);
        action.putValue(SimpleAction.MEDIUM_ICON, buttonIcon);
        action.putValue(SimpleAction.MEDIUM_DISABLED_ICON, disabledButtonIcon);
        action.putValue(Action.SHORT_DESCRIPTION, tooltip);
    }

    /**
     * Initialize caption, mnemonic, accelerator key and icons
     * for a menu button. For an ordinary button, only icons are
     * instantiated.
     */
    @Override
    public void init(AbstractButton button) {
        if (button instanceof JMenuItem) {
            super.init(button);
            if (menuIcon != null) {
                if (disabledMenuIcon != null) {
                    button.setDisabledIcon(disabledMenuIcon);
                }
                button.setIcon(menuIcon);
            }
        } else {
            if (buttonIcon != null) {
                if (disabledButtonIcon != null) {
                    button.setDisabledIcon(disabledButtonIcon);
                }
                button.setIcon(buttonIcon);
            }
        }
    }
}
