package org.jboss.cache.marshall;

import org.jboss.cache.CacheException;

import java.util.ArrayList;

/**
 * An efficient array-based list of referenced objects, using the reference id as a subscript for the array.
 *
 * @author <a href="mailto:manik@jboss.org">Manik Surtani (manik@jboss.org)</a>
 */
public class UnmarshalledReferences
{
   private final ArrayList<Object> referencedObjects = new ArrayList<Object>();

   /**
    * Retrieves an object referenced by an id
    *
    * @param ref reference
    * @return object
    */
   public Object getReferencedObject(int ref)
   {
      if (ref >= referencedObjects.size())
         throw new CacheException("Attempting to look up a ref that hasn't been inserted yet");
      return referencedObjects.get(ref);
   }

   /**
    * Adds a referenced object to the list of references
    *
    * @param ref reference id
    * @param o   object
    */
   public void putReferencedObject(int ref, Object o)
   {
      int sz = referencedObjects.size();
      // if we are not adding the object to the end of the list, make sure we use a specific position
      if (ref < sz)
      {
         referencedObjects.set(ref, o);
         return;
      }
      else if (ref > sz)
      {
         // if we are adding the reference to a position beyond the end of the list, make sure we expand the list first.
         // this can happen, weirdly enough, since marshallObject() can be called recursively, such as from marshallFqn().
         for (int i = sz; i < ref; i++) referencedObjects.add(null);
      }
      referencedObjects.add(o);
   }
}
