#include <errno.h>
#include <stddef.h>
#include <string.h>
#include <mb.h>

static char *me;
static char *l = NULL;
static size_t size = 0;
static size_t end = 0;
static int (*fgetc_func)(FILE *stream) = fgetc;
static int (*fputc_func)(int c, FILE *stream) = fputc;
static int (*fprintf_func)(FILE *stream, const char *format, ...) = fprintf;
static FILE *istream;
static FILE *ostream;
static long nline = 1;

static void
storec(int c)
{
  if (end >= size) {
    if (!(l = realloc(l, size + 256)))
      exit(1);

    size += 256;
  }

  l[end++] = c;
}

static char *
gets_unlimited(void)
{
  int c;

  for (end = 0 ; (c = fgetc_func(istream)) != EOF && c != '\n' ;)
    storec(c);

  storec('\0');
  return l;
}

#define CONV_MAX (256)
static mb_conv_t iconverterv[CONV_MAX] = {};
static mb_conv_t oconverterv[CONV_MAX] = {};
static mb_conv_t *converterv = iconverterv;

struct opt_st {
  mb_setup_t isetup, osetup, *setup;
  mb_cs_judge_t judge_cs;
  mb_cs_setup_t setup_cs;
  size_t n_cs_stats;
  const char *format, *mime_title, *cname;
  int lineno, iconv, oconv, which, width, cprop;
};

static void
opt_input(struct opt_st *p_opt, char *arg1st, int argrestc, char **argrestv)
{
  converterv = iconverterv;
  p_opt->setup = &p_opt->isetup;
  p_opt->iconv = 1;
}

static void
opt_output(struct opt_st *p_opt, char *arg1st, int argrestc, char **argrestv)
{
  converterv = oconverterv;
  p_opt->setup = &p_opt->osetup;
  p_opt->oconv = 1;
}

static void
opt_lineno(struct opt_st *p_opt, char *arg1st, int argrestc, char **argrestv)
{
  p_opt->lineno = 1;
}

static void
opt_tofile(struct opt_st *p_opt, char *arg1st, int argrestc, char **argrestv)
{
  if (!strcmp(arg1st, "-"))
    ostream = stdout;
  else if (!(ostream = fopen(arg1st, "w"))) {
    fprintf(stderr, "%s: fopen(\"%s\", \"w\"): %s\n", me, arg1st, strerror(errno));
    exit(1);
  }
}

static void
opt_appendtofile(struct opt_st *p_opt, char *arg1st, int argrestc, char **argrestv)
{
  if (!strcmp(arg1st, "-"))
    ostream = stdout;
  else if (!(ostream = fopen(arg1st, "a"))) {
    fprintf(stderr, "%s: fopen(\"%s\", \"a\"): %s\n", me, arg1st, strerror(errno));
    exit(1);
  }
}

static void
opt_which(struct opt_st *p_opt, char *arg1st, int argrestc, char **argrestv)
{
  p_opt->which = 1;
}

static void
opt_width(struct opt_st *p_opt, char *arg1st, int argrestc, char **argrestv)
{
  p_opt->width = 1;
}

static void
opt_cprop(struct opt_st *p_opt, char *arg1st, int argrestc, char **argrestv)
{
  p_opt->cprop = 1;
}

static void
opt_charset(struct opt_st *p_opt, char *s, int argrestc, char **argrestv)
{
  if (!mb_lang_to_detector(s, &p_opt->judge_cs, &p_opt->setup_cs, &p_opt->n_cs_stats))
    mb_setsetup(p_opt->setup, "@", s);
}

static void
opt_mimetitle(struct opt_st *p_opt, char *arg1st, int argrestc, char **argrestv)
{
  p_opt->mime_title = arg1st;
}

static void
opt_format(struct opt_st *p_opt, char *arg1st, int argrestc, char **argrestv)
{
  p_opt->format = arg1st;
}

static void
opt_cname(struct opt_st *p_opt, char *arg1st, int argrestc, char **argrestv)
{
  p_opt->cname = arg1st;
}

static void
opt_flag_unknown(const char *s, size_t n)
{
  fprintf(stderr, "%s: %.*s: unknown flag\n", me, (int)n, s);
  exit(1);
}

static void
opt_flag(struct opt_st *p_opt, char *s, int argrestc, char **argrestv)
{
  char op[2] = "|";

  switch ((unsigned char)*s) {
  case '+':
    op[0] = '|';
    ++s;
    break;
  case '-':
  case '=':
    op[0] = *s++;
  default:
    break;
  }

  mb_setsetup(p_opt->setup, op, mb_namev_to_flag(s, 0, opt_flag_unknown));
}

static void
opt_converter_unknown(const char *s, size_t n)
{
  fprintf(stderr, "%s: %.*s: unknown converter\n", me, (int)n, s);
  exit(1);
}

static void
opt_convertto(struct opt_st *p_opt, char *s, int argrestc, char **argrestv)
{
  mb_namev_to_converterv(s, converterv, CONV_MAX, opt_converter_unknown);
  mb_setsetup(p_opt->setup, "&", converterv);
}

static void opt_help(struct opt_st *p_opt, char *s, int argrestc, char **argrestv);

void
doit(struct opt_st *opt, const char *fn, int real_argc)
{
  int notstdin = (fn && strcmp(fn, "-")) ? 1 : 0;
  char *line;
  int c;

  if (opt->cname) {
    size_t size;
    mb_cs2esc_t *tab;

    size = mb_charset_cname_merge(NULL, &opt->cname, 1);
    if ((tab = malloc(size))) mb_charset_cname_merge(tab, &opt->cname, 1);
  }

  if (notstdin) {
    if (!(istream = fopen(fn, "r"))) {
      fprintf(stderr, "%s: fopen(%s, \"r\"): %s\n", me, fn, strerror(errno));
      exit(1);
    }
  }
  else
    istream = stdin;

  if (opt->iconv) {
    fgetc_func = mb_fgetc;
    mb_fbind(istream, "r!", &opt->isetup);

    if (opt->judge_cs && opt->setup_cs) {
      mb_cs_detector_t *p = mb_falloc_cs_detector(istream, BUFSIZ, 0);

      if (!p) {
	fprintf(stderr, "%s: fail to allocate detector\n", me);
	exit(1);
      }

      p->judge = opt->judge_cs;
      p->setup = opt->setup_cs;
      p->private = (void *)opt->isetup.cs;
      p->nstats = opt->n_cs_stats;
      p->flag = MB_CS_DETECT_FLAG_MKUNBOUND;
    }
  }

  if (opt->oconv) {
    fputc_func = mb_fputc;
    fprintf_func = mb_fprintf;
    mb_fbind(ostream, "a!", &opt->osetup);
  }

  if (opt->lineno && !opt->format)
    opt->format = "#%ld: %s";

  if (opt->width)
    while (*(line = gets_unlimited()))
      fprintf(ostream, "%ld\n", (long)mb_strlength(line));
  else if (opt->cprop)
    while (*(line = gets_unlimited())) {
      char *s = line;
      char propbuf[] = "S)(BN>\n";

      while (*s) {
	size_t cb = 0;
	size_t ce = MB_LEN_MAX;
	size_t pe = 0;
	int c = mb_find_char(s, &cb, &ce), prop;
	mb_char_t ch;

	mb_chartype(c, ce, &ch);
	prop = mb_charprop(&ch);

	if (prop & MB_CPROP_NEVER_EOL) propbuf[pe++] = '(';
	if (prop & MB_CPROP_IS_SPACE) propbuf[pe++] = 'S';
	if (prop & MB_CPROP_MAY_BREAK) propbuf[pe++] = 'B';
	if (prop & MB_CPROP_EOL_TO_NULL) propbuf[pe++] = 'N';
	if (prop & MB_CPROP_NEVER_BOL) propbuf[pe++] = ')';
	propbuf[pe++] = '>';
	s += ce;
	propbuf[pe++] = *s ? ' ' : '\n';
	propbuf[pe++] = '\0';
	fprintf(ostream, "<cc=0x%04X, set=%d, fc=0x%02X, prop=%s", ch.c, ch.set, ch.fc, propbuf);
      }
    }
  else if (opt->format)
    for (;; ++nline) {
      line = gets_unlimited();

      if (!*line)
	break;

      if (opt->lineno)
	fprintf_func(ostream, opt->format, nline, line);
      else
	fprintf_func(ostream, opt->format, line);

      fputc_func('\n', ostream);
    }
  else if (opt->mime_title) {
    while (*(line = gets_unlimited()))
      puts(mb_str2b64(line, opt->mime_title, &opt->osetup, ""));
  }
  else
    while ((c = fgetc_func(istream)) != EOF)
      fputc_func(c, ostream);

  if (opt->iconv && opt->which) {
    mb_info_t *info;
    const char *cs;

    mb_finfo(istream, &info, NULL);
    cs = (info && info->cs2esc && info->cs2esc->cs) ? info->cs2esc->cs : "UNKNOWN";

    if (real_argc > 1)
      fprintf(stderr, "%s: %s\n", fn ? fn : "-", cs);
    else
      fprintf(stderr, "%s\n", cs);
  }

  if (notstdin) {
    if (opt->iconv)
      mb_fclose(istream);
    else 
      fclose(istream);

    istream = stdin;
  }
}

#define INDENT "    "

static struct opt_tab_st {
  char *opt;
  size_t opt_len;
  int argc;
  void (*func)(struct opt_st *, char *, int, char **);
  char *argname, *usage;
} opt_tab[] = {
  {MB_KEY_DEF("-?"), 0, opt_help,
   NULL, "display this message and exit.\n"},
  {MB_KEY_DEF("-a"), 1, opt_appendtofile,
   "<file>", "output is appended to <file>.\n"},
  {MB_KEY_DEF("-c"), 1, opt_convertto,
   "<converters>",
   "specifies character encoding conversion.\n"
   "<converters> must be comma separated list of\n"
   INDENT "b, cn-big5\n"
   INDENT INDENT "converted to Big Five,\n"
   INDENT "c\n"
   INDENT INDENT "converted to ISO-2022-CN,\n"
   INDENT "j, a0\n"
   INDENT INDENT "converted in such a way that\n"
   INDENT INDENT "designate to G0 and invoked to GL\n"
   INDENT "k\n"
   INDENT INDENT "converted to ISO-2022-KR,\n"
   INDENT "s, sjis, shift_jis\n"
   INDENT INDENT "converted to Shift_JIS,\n"
   INDENT "b2c, big5-to-cns\n"
   INDENT INDENT "Big Five converted to CNS 11643,\n"
   INDENT "i2u, iso-to-ucs\n"
   INDENT INDENT "converted to UTF-8,\n"
   INDENT "u2b, ucs-to-big5\n"
   INDENT INDENT "UTF-8 converted to Big Five or others,\n"
   INDENT "u2c, ucs-to-cn\n"
   INDENT INDENT "UTF-8 converted to CNS 11643 or others,\n"
   INDENT "u2j, ucs-to-ja\n"
   INDENT INDENT "UTF-8 converted to JIS X 0208 or others,\n"
   INDENT "u2k, ucs-to-kr\n"
   INDENT INDENT "UTF-8 converted to KS X 1001 or others,\n"
   INDENT "u2gb, ucs-to-gb\n"
   INDENT INDENT "UTF-8 converted to GB2312 or others,\n"
   INDENT "misc\n"
   INDENT INDENT "UTF-8 converted to one of\n"
   INDENT INDENT INDENT "koi8-r,\n"
   INDENT INDENT INDENT "koi8-u,\n"
   INDENT INDENT INDENT "windows-1250, ..., or windows-1258,\n"
   INDENT "ascii\n"
   INDENT INDENT "domestic ASCII converted to US-ASCII,\n"
   INDENT "cn-gb\n"
   INDENT INDENT "converted to CN-GB,\n"
   INDENT "euc-jp\n"
   INDENT INDENT "converted to EUC-jp,\n"
   INDENT "euc-kr\n"
   INDENT INDENT "converted to EUC-kr,\n"
   INDENT "euc-tw\n"
   INDENT INDENT "converted to EUC-tw,\n"
   INDENT "charset\n"
   INDENT INDENT "converted appropriately according to\n"
   INDENT INDENT "the charset bound to the internal automaton,\n"
   INDENT "ms-latin1\n"
   INDENT INDENT "Unicode characters of code point between 0x80 and\n"
   INDENT INDENT "0x9F (both inclusive) are converted to other\n"
   INDENT INDENT "Unicode characters as if they are characters of those\n"
   INDENT INDENT "code point in Microsoft Windows Codepage 1252,\n"
   INDENT "ucs-to-johab\n"
   INDENT INDENT "converted to JOHAB.\n"
   INDENT "cn-gb-isoir165\n"
   INDENT INDENT "converted to CN-GB-ISOIR165.\n"},
  {MB_KEY_DEF("-f"), 1, opt_flag,
   "<flags>",
   "specifies flags to change behavior of conversion.\n"
   "<flags> must be comma separated list of\n"
   INDENT "28, use-0x28-for-94x94inG0\n"
   INDENT INDENT "use \"1/11 2/4 2/8 F\"\n"
   INDENT INDENT "instead of \"1/11 2/4 F\"\n"
   INDENT INDENT "to designate charsets with final octet\n"
   INDENT INDENT "4/0, 4/1, or 4/2 to G0,\n"
   INDENT "ac, ascii-at-control\n"
   INDENT INDENT "escape sequence \"1/11 2/8 4/2\" is\n"
   INDENT INDENT "output before every control character,\n"
   INDENT "uc, check-utf-8\n"
   INDENT INDENT "check overlong encoding of UTF-8,\n"
   INDENT "nossl, ignore-7bit-single-shift\n"
   INDENT INDENT "escape sequence for 7 bit single shift\n"
   INDENT INDENT "is ignored.\n"},
  {MB_KEY_DEF("-h"), 0, opt_help,
   "\n-?", NULL},
  {MB_KEY_DEF("-i"), 0, opt_input,
   NULL, "succeeding options apply to input stream.\n"},
  {MB_KEY_DEF("-m"), 1, opt_mimetitle,
   "<string>",
   "mime encoding conforming to RFC2047 is performed.\n"
   "<string> is used as charset name.\n"},
  {MB_KEY_DEF("-n"), 0, opt_lineno,
   NULL, "line number (>= 1) is inserted to beginning of each line.\n"},
  {MB_KEY_DEF("-o"), 0, opt_output,
   NULL, "succeeding options apply to output stream.\n"},
  {MB_KEY_DEF("-p"), 0, opt_cprop,
   NULL, "output properties of characters.\n"},
  {MB_KEY_DEF("-t"), 1, opt_tofile,
   "<file>", "output to file (truncated).\n"},
  {MB_KEY_DEF("-w"), 0, opt_width,
   NULL, "output width of each line.\n"},
  {MB_KEY_DEF("-cs"), 1, opt_charset,
   "<string>", "specifies charset name.\n"},
  {MB_KEY_DEF("--to"), 1, opt_tofile,
   "\n-t", NULL},
  {MB_KEY_DEF("--flag"), 1, opt_flag,
   "\n-f", NULL},
  {MB_KEY_DEF("--help"), 0, opt_help,
   "\n-?", NULL},
  {MB_KEY_DEF("--cname"), 1, opt_cname,
   "<canonical names>=<charset name>[,<charset name>,...]",
   "specifies canonical name of\n"
   "non-standard charset name.\n"},
  {MB_KEY_DEF("--input"), 0, opt_input,
   "\n-i", NULL},
  {MB_KEY_DEF("--which"), 0, opt_which,
   NULL, "output charset of input stream to stderr"},
  {MB_KEY_DEF("--width"), 0, opt_width,
   "\n-w", NULL},
  {MB_KEY_DEF("--format"), 1, opt_format,
   "<string>", "specifies output format\n"},
  {MB_KEY_DEF("--output"), 0, opt_output,
   "\n-o", NULL},
  {MB_KEY_DEF("--charset"), 1, opt_charset,
   "\n-cs", NULL},
  {MB_KEY_DEF("--append-to"), 1, opt_appendtofile,
   "\n-a", NULL},
  {MB_KEY_DEF("--convert-to"), 1, opt_convertto,
   "\n-c", NULL},
  {MB_KEY_DEF("--line-number"), 0, opt_lineno,
   "\n-n", NULL},
  {MB_KEY_DEF("--mime-charset"), 1, opt_mimetitle,
   "\n-m", NULL},
  {MB_KEY_DEF("--char-property"), 0, opt_cprop,
   "\n-p", NULL},
};

int
main(int argc, char *argv[])
{
  struct opt_st opt = {};
  struct opt_tab_st *p;
  char *eqsign;
  int i;

  opt.setup = &opt.isetup;
  opt.mime_title = opt.format = NULL;
  opt.iconv = opt.oconv = opt.lineno = 0;
  me = argv[0];
  ostream = stdout;

  for (i = 1 ; i < argc ;)
    if ((p = mb_nc_bsearch(argv[i], strlen(argv[i]), opt_tab,
			   offsetof(struct opt_tab_st, opt), offsetof(struct opt_tab_st, opt_len),
			   sizeof(opt_tab[0]), sizeof(opt_tab)))) {
      ++i;

      if (p->argc > 0) {
	if (p->argc <= argc - i)
	  p->func(&opt, argv[i], argc - i - 1, &argv[i + 1]);
	else {
	  fprintf(stderr, "%s: %s requires %d argument(s)\n", me, argv[i - 1], p->argc);
	  exit(1);
	}
      }
      else
	p->func(&opt, NULL, 0, NULL);

      i += p->argc;
    }
    else if ((eqsign = strchr(argv[i], '=')) &&
	     (p = mb_nc_bsearch(argv[i], eqsign - argv[i], opt_tab,
				offsetof(struct opt_tab_st, opt), offsetof(struct opt_tab_st, opt_len),
				sizeof(opt_tab[0]), sizeof(opt_tab)))) {
      if (p->argc > 0) {
	if (p->argc <= argc - i)
	  p->func(&opt, &eqsign[1], argc - i - 1, &argv[i + 1]);
	else {
	  fprintf(stderr, "%s: %s requires more %d argument(s)\n", me, argv[i], p->argc - 1);
	  exit(1);
	}
      }
      else {
	p->func(&opt, NULL, 0, NULL);
	++i;
      }

      i += p->argc;
    }
    else
      break;

  if (i < argc) {
    int real_argc = argc - i;

    do {
      doit(&opt, argv[i++], real_argc);
    } while (i < argc);
  }
  else
    doit(&opt, NULL, 0);

  mb_fclose(ostream);
  return 0;
}

static void
show_usage(char *usage)
{
  int j, k;

  for (j = 0 ;; j += k + 1) {
    k = strcspn(&usage[j], "\n");

    if (k || usage[j + k])
      fprintf(stderr, INDENT "%.*s\n", k, &usage[j]);

    if (!usage[j + k])
      break;
  }
}

static void
opt_help(struct opt_st *p_opt, char *s, int argrestc, char **argrestv)
{
  size_t i;

  fprintf(stderr, "%s [options] file ...\nsummary of options:\n", me);

  for (i = 0 ; i < sizeof(opt_tab) / sizeof(opt_tab[0]) ; ++i)
    if (opt_tab[i].argname) {
      if (opt_tab[i].argname[0] == '\n') {
	if (opt_tab[i].argc) {
	  if (!strncmp(opt_tab[i].opt, "--", 2))
	    fprintf(stderr, "\n%s=<arg>: same as \"%s <arg>\"\n", opt_tab[i].opt, &opt_tab[i].argname[1]);
	  else
	    fprintf(stderr, "\n%s <arg>: same as \"%s <arg>\"\n", opt_tab[i].opt, &opt_tab[i].argname[1]);
	}
	else
	  fprintf(stderr, "\n%s: same as \"%s\"\n", opt_tab[i].opt, &opt_tab[i].argname[1]);
      }
      else {
	if (!strncmp(opt_tab[i].opt, "--", 2))
	  fprintf(stderr, "\n%s=%s\n", opt_tab[i].opt, opt_tab[i].argname);
	else
	  fprintf(stderr, "\n%s %s\n", opt_tab[i].opt, opt_tab[i].argname);

	show_usage(opt_tab[i].usage);
      }
    }
    else {
      fprintf(stderr, "\n%s\n", opt_tab[i].opt);
      show_usage(opt_tab[i].usage);
    }

  exit(0);
}
