/*
 *  Copyright (C) 2016, Burt P.,
 *  All rights reserved.
 *
 *  Redistribution and use in source and binary forms, with or without modification,
 *  are permitted provided that the following conditions are met:
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *    3. The names of its contributors may not be used to endorse or promote
 *       products derived from this software without specific prior written
 *       permission.
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 *  CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 *  EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 *  PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 *  PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 *  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 *  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/*
 * This file exists to remove a dependency on the math lib in libhdcd
 * that only exists so that sin() can be used in the analyze mode tone
 * generator. This version uses simple lookup tables instead.
 *
 * HDCD is possible in CDDA (44.1kHz) and all DVD-Audio sample rates.
 * Supported rates: 44.1, 48, 88.2, 96, 176.4, and 192 kHz
 *
 * To generate tables:
 * gcc -o tone_tab -DHDCD_PRINT_TONE_TAB hdcd_analyze_tonegen.c -lm
 * ./tone_tab >tone_tab.csnip
 */

#ifdef HDCD_PRINT_TONE_TAB
#include <stdio.h>
#include <math.h>
#endif
#include <stdint.h>

static const int16_t tone_tab_192000[640] = {
    0x0000, 0x0020, 0x0040, 0x0060, 0x0080, 0x00a0, 0x00c0, 0x00e1,
    0x0101, 0x0121, 0x0141, 0x0161, 0x0181, 0x01a1, 0x01c0, 0x01e0,
    0x0200, 0x0220, 0x0240, 0x025f, 0x027f, 0x029e, 0x02be, 0x02dd,
    0x02fc, 0x031c, 0x033b, 0x035a, 0x0379, 0x0398, 0x03b7, 0x03d5,
    0x03f4, 0x0413, 0x0431, 0x044f, 0x046e, 0x048c, 0x04aa, 0x04c8,
    0x04e5, 0x0503, 0x0521, 0x053e, 0x055b, 0x0578, 0x0595, 0x05b2,
    0x05cf, 0x05ec, 0x0608, 0x0624, 0x0641, 0x065d, 0x0678, 0x0694,
    0x06b0, 0x06cb, 0x06e6, 0x0701, 0x071c, 0x0737, 0x0751, 0x076b,
    0x0785, 0x079f, 0x07b9, 0x07d3, 0x07ec, 0x0805, 0x081e, 0x0837,
    0x0850, 0x0868, 0x0880, 0x0898, 0x08b0, 0x08c7, 0x08df, 0x08f6,
    0x090c, 0x0923, 0x093a, 0x0950, 0x0966, 0x097b, 0x0991, 0x09a6,
    0x09bb, 0x09d0, 0x09e4, 0x09f9, 0x0a0d, 0x0a21, 0x0a34, 0x0a47,
    0x0a5a, 0x0a6d, 0x0a80, 0x0a92, 0x0aa4, 0x0ab6, 0x0ac7, 0x0ad8,
    0x0ae9, 0x0afa, 0x0b0a, 0x0b1b, 0x0b2a, 0x0b3a, 0x0b49, 0x0b58,
    0x0b67, 0x0b76, 0x0b84, 0x0b92, 0x0b9f, 0x0bad, 0x0bba, 0x0bc6,
    0x0bd3, 0x0bdf, 0x0beb, 0x0bf6, 0x0c02, 0x0c0d, 0x0c17, 0x0c22,
    0x0c2c, 0x0c36, 0x0c3f, 0x0c48, 0x0c51, 0x0c5a, 0x0c62, 0x0c6a,
    0x0c72, 0x0c79, 0x0c80, 0x0c87, 0x0c8d, 0x0c93, 0x0c99, 0x0c9f,
    0x0ca4, 0x0ca9, 0x0cad, 0x0cb2, 0x0cb5, 0x0cb9, 0x0cbc, 0x0cbf,
    0x0cc2, 0x0cc4, 0x0cc7, 0x0cc8, 0x0cca, 0x0ccb, 0x0ccc, 0x0ccc,
    0x0ccc, 0x0ccc, 0x0ccc, 0x0ccb, 0x0cca, 0x0cc8, 0x0cc7, 0x0cc4,
    0x0cc2, 0x0cbf, 0x0cbc, 0x0cb9, 0x0cb5, 0x0cb2, 0x0cad, 0x0ca9,
    0x0ca4, 0x0c9f, 0x0c99, 0x0c93, 0x0c8d, 0x0c87, 0x0c80, 0x0c79,
    0x0c72, 0x0c6a, 0x0c62, 0x0c5a, 0x0c51, 0x0c48, 0x0c3f, 0x0c36,
    0x0c2c, 0x0c22, 0x0c17, 0x0c0d, 0x0c02, 0x0bf6, 0x0beb, 0x0bdf,
    0x0bd3, 0x0bc6, 0x0bba, 0x0bad, 0x0b9f, 0x0b92, 0x0b84, 0x0b76,
    0x0b67, 0x0b58, 0x0b49, 0x0b3a, 0x0b2a, 0x0b1b, 0x0b0a, 0x0afa,
    0x0ae9, 0x0ad8, 0x0ac7, 0x0ab6, 0x0aa4, 0x0a92, 0x0a80, 0x0a6d,
    0x0a5a, 0x0a47, 0x0a34, 0x0a21, 0x0a0d, 0x09f9, 0x09e4, 0x09d0,
    0x09bb, 0x09a6, 0x0991, 0x097b, 0x0966, 0x0950, 0x093a, 0x0923,
    0x090c, 0x08f6, 0x08df, 0x08c7, 0x08b0, 0x0898, 0x0880, 0x0868,
    0x0850, 0x0837, 0x081e, 0x0805, 0x07ec, 0x07d3, 0x07b9, 0x079f,
    0x0785, 0x076b, 0x0751, 0x0737, 0x071c, 0x0701, 0x06e6, 0x06cb,
    0x06b0, 0x0694, 0x0678, 0x065d, 0x0641, 0x0624, 0x0608, 0x05ec,
    0x05cf, 0x05b2, 0x0595, 0x0578, 0x055b, 0x053e, 0x0521, 0x0503,
    0x04e5, 0x04c8, 0x04aa, 0x048c, 0x046e, 0x044f, 0x0431, 0x0413,
    0x03f4, 0x03d5, 0x03b7, 0x0398, 0x0379, 0x035a, 0x033b, 0x031c,
    0x02fc, 0x02dd, 0x02be, 0x029e, 0x027f, 0x025f, 0x0240, 0x0220,
    0x0200, 0x01e0, 0x01c0, 0x01a1, 0x0181, 0x0161, 0x0141, 0x0121,
    0x0101, 0x00e1, 0x00c0, 0x00a0, 0x0080, 0x0060, 0x0040, 0x0020,
    0x0000, 0xffe0, 0xffc0, 0xffa0, 0xff80, 0xff60, 0xff40, 0xff1f,
    0xfeff, 0xfedf, 0xfebf, 0xfe9f, 0xfe7f, 0xfe5f, 0xfe40, 0xfe20,
    0xfe00, 0xfde0, 0xfdc0, 0xfda1, 0xfd81, 0xfd62, 0xfd42, 0xfd23,
    0xfd04, 0xfce4, 0xfcc5, 0xfca6, 0xfc87, 0xfc68, 0xfc49, 0xfc2b,
    0xfc0c, 0xfbed, 0xfbcf, 0xfbb1, 0xfb92, 0xfb74, 0xfb56, 0xfb38,
    0xfb1b, 0xfafd, 0xfadf, 0xfac2, 0xfaa5, 0xfa88, 0xfa6b, 0xfa4e,
    0xfa31, 0xfa14, 0xf9f8, 0xf9dc, 0xf9bf, 0xf9a3, 0xf988, 0xf96c,
    0xf950, 0xf935, 0xf91a, 0xf8ff, 0xf8e4, 0xf8c9, 0xf8af, 0xf895,
    0xf87b, 0xf861, 0xf847, 0xf82d, 0xf814, 0xf7fb, 0xf7e2, 0xf7c9,
    0xf7b0, 0xf798, 0xf780, 0xf768, 0xf750, 0xf739, 0xf721, 0xf70a,
    0xf6f4, 0xf6dd, 0xf6c6, 0xf6b0, 0xf69a, 0xf685, 0xf66f, 0xf65a,
    0xf645, 0xf630, 0xf61c, 0xf607, 0xf5f3, 0xf5df, 0xf5cc, 0xf5b9,
    0xf5a6, 0xf593, 0xf580, 0xf56e, 0xf55c, 0xf54a, 0xf539, 0xf528,
    0xf517, 0xf506, 0xf4f6, 0xf4e5, 0xf4d6, 0xf4c6, 0xf4b7, 0xf4a8,
    0xf499, 0xf48a, 0xf47c, 0xf46e, 0xf461, 0xf453, 0xf446, 0xf43a,
    0xf42d, 0xf421, 0xf415, 0xf40a, 0xf3fe, 0xf3f3, 0xf3e9, 0xf3de,
    0xf3d4, 0xf3ca, 0xf3c1, 0xf3b8, 0xf3af, 0xf3a6, 0xf39e, 0xf396,
    0xf38e, 0xf387, 0xf380, 0xf379, 0xf373, 0xf36d, 0xf367, 0xf361,
    0xf35c, 0xf357, 0xf353, 0xf34e, 0xf34b, 0xf347, 0xf344, 0xf341,
    0xf33e, 0xf33c, 0xf339, 0xf338, 0xf336, 0xf335, 0xf334, 0xf334,
    0xf334, 0xf334, 0xf334, 0xf335, 0xf336, 0xf338, 0xf339, 0xf33c,
    0xf33e, 0xf341, 0xf344, 0xf347, 0xf34b, 0xf34e, 0xf353, 0xf357,
    0xf35c, 0xf361, 0xf367, 0xf36d, 0xf373, 0xf379, 0xf380, 0xf387,
    0xf38e, 0xf396, 0xf39e, 0xf3a6, 0xf3af, 0xf3b8, 0xf3c1, 0xf3ca,
    0xf3d4, 0xf3de, 0xf3e9, 0xf3f3, 0xf3fe, 0xf40a, 0xf415, 0xf421,
    0xf42d, 0xf43a, 0xf446, 0xf453, 0xf461, 0xf46e, 0xf47c, 0xf48a,
    0xf499, 0xf4a8, 0xf4b7, 0xf4c6, 0xf4d6, 0xf4e5, 0xf4f6, 0xf506,
    0xf517, 0xf528, 0xf539, 0xf54a, 0xf55c, 0xf56e, 0xf580, 0xf593,
    0xf5a6, 0xf5b9, 0xf5cc, 0xf5df, 0xf5f3, 0xf607, 0xf61c, 0xf630,
    0xf645, 0xf65a, 0xf66f, 0xf685, 0xf69a, 0xf6b0, 0xf6c6, 0xf6dd,
    0xf6f4, 0xf70a, 0xf721, 0xf739, 0xf750, 0xf768, 0xf780, 0xf798,
    0xf7b0, 0xf7c9, 0xf7e2, 0xf7fb, 0xf814, 0xf82d, 0xf847, 0xf861,
    0xf87b, 0xf895, 0xf8af, 0xf8c9, 0xf8e4, 0xf8ff, 0xf91a, 0xf935,
    0xf950, 0xf96c, 0xf988, 0xf9a3, 0xf9bf, 0xf9dc, 0xf9f8, 0xfa14,
    0xfa31, 0xfa4e, 0xfa6b, 0xfa88, 0xfaa5, 0xfac2, 0xfadf, 0xfafd,
    0xfb1b, 0xfb38, 0xfb56, 0xfb74, 0xfb92, 0xfbb1, 0xfbcf, 0xfbed,
    0xfc0c, 0xfc2b, 0xfc49, 0xfc68, 0xfc87, 0xfca6, 0xfcc5, 0xfce4,
    0xfd04, 0xfd23, 0xfd42, 0xfd62, 0xfd81, 0xfda1, 0xfdc0, 0xfde0,
    0xfe00, 0xfe20, 0xfe40, 0xfe5f, 0xfe7f, 0xfe9f, 0xfebf, 0xfedf,
    0xfeff, 0xff1f, 0xff40, 0xff60, 0xff80, 0xffa0, 0xffc0, 0xffe0,
};

static const int16_t tone_tab_176400[588] = {
    0x0000, 0x0023, 0x0046, 0x0069, 0x008c, 0x00ae, 0x00d1, 0x00f4,
    0x0117, 0x013a, 0x015d, 0x0180, 0x01a3, 0x01c5, 0x01e8, 0x020a,
    0x022d, 0x024f, 0x0272, 0x0294, 0x02b6, 0x02d9, 0x02fb, 0x031d,
    0x033f, 0x0360, 0x0382, 0x03a4, 0x03c5, 0x03e7, 0x0408, 0x0429,
    0x044a, 0x046b, 0x048c, 0x04ad, 0x04cd, 0x04ee, 0x050e, 0x052e,
    0x054e, 0x056e, 0x058d, 0x05ad, 0x05cc, 0x05eb, 0x060a, 0x0629,
    0x0647, 0x0666, 0x0684, 0x06a2, 0x06c0, 0x06de, 0x06fb, 0x0718,
    0x0735, 0x0752, 0x076f, 0x078b, 0x07a7, 0x07c3, 0x07df, 0x07fa,
    0x0816, 0x0831, 0x084c, 0x0866, 0x0880, 0x089a, 0x08b4, 0x08ce,
    0x08e7, 0x0900, 0x0919, 0x0931, 0x094a, 0x0961, 0x0979, 0x0991,
    0x09a8, 0x09bf, 0x09d5, 0x09eb, 0x0a01, 0x0a17, 0x0a2c, 0x0a41,
    0x0a56, 0x0a6b, 0x0a7f, 0x0a93, 0x0aa6, 0x0aba, 0x0acd, 0x0adf,
    0x0af2, 0x0b04, 0x0b15, 0x0b27, 0x0b38, 0x0b48, 0x0b59, 0x0b69,
    0x0b78, 0x0b88, 0x0b97, 0x0ba5, 0x0bb4, 0x0bc2, 0x0bcf, 0x0bdd,
    0x0bea, 0x0bf6, 0x0c03, 0x0c0e, 0x0c1a, 0x0c25, 0x0c30, 0x0c3b,
    0x0c45, 0x0c4f, 0x0c58, 0x0c61, 0x0c6a, 0x0c72, 0x0c7a, 0x0c82,
    0x0c89, 0x0c90, 0x0c96, 0x0c9c, 0x0ca2, 0x0ca8, 0x0cad, 0x0cb1,
    0x0cb6, 0x0cba, 0x0cbd, 0x0cc0, 0x0cc3, 0x0cc5, 0x0cc8, 0x0cc9,
    0x0ccb, 0x0ccb, 0x0ccc, 0x0ccc, 0x0ccc, 0x0ccb, 0x0ccb, 0x0cc9,
    0x0cc8, 0x0cc5, 0x0cc3, 0x0cc0, 0x0cbd, 0x0cba, 0x0cb6, 0x0cb1,
    0x0cad, 0x0ca8, 0x0ca2, 0x0c9c, 0x0c96, 0x0c90, 0x0c89, 0x0c82,
    0x0c7a, 0x0c72, 0x0c6a, 0x0c61, 0x0c58, 0x0c4f, 0x0c45, 0x0c3b,
    0x0c30, 0x0c25, 0x0c1a, 0x0c0e, 0x0c03, 0x0bf6, 0x0bea, 0x0bdd,
    0x0bcf, 0x0bc2, 0x0bb4, 0x0ba5, 0x0b97, 0x0b88, 0x0b78, 0x0b69,
    0x0b59, 0x0b48, 0x0b38, 0x0b27, 0x0b15, 0x0b04, 0x0af2, 0x0adf,
    0x0acd, 0x0aba, 0x0aa6, 0x0a93, 0x0a7f, 0x0a6b, 0x0a56, 0x0a41,
    0x0a2c, 0x0a17, 0x0a01, 0x09eb, 0x09d5, 0x09bf, 0x09a8, 0x0991,
    0x0979, 0x0961, 0x094a, 0x0931, 0x0919, 0x0900, 0x08e7, 0x08ce,
    0x08b4, 0x089a, 0x0880, 0x0866, 0x084c, 0x0831, 0x0816, 0x07fa,
    0x07df, 0x07c3, 0x07a7, 0x078b, 0x076f, 0x0752, 0x0735, 0x0718,
    0x06fb, 0x06de, 0x06c0, 0x06a2, 0x0684, 0x0666, 0x0647, 0x0629,
    0x060a, 0x05eb, 0x05cc, 0x05ad, 0x058d, 0x056e, 0x054e, 0x052e,
    0x050e, 0x04ee, 0x04cd, 0x04ad, 0x048c, 0x046b, 0x044a, 0x0429,
    0x0408, 0x03e7, 0x03c5, 0x03a4, 0x0382, 0x0360, 0x033f, 0x031d,
    0x02fb, 0x02d9, 0x02b6, 0x0294, 0x0272, 0x024f, 0x022d, 0x020a,
    0x01e8, 0x01c5, 0x01a3, 0x0180, 0x015d, 0x013a, 0x0117, 0x00f4,
    0x00d1, 0x00ae, 0x008c, 0x0069, 0x0046, 0x0023, 0x0000, 0xffdd,
    0xffba, 0xff97, 0xff74, 0xff52, 0xff2f, 0xff0c, 0xfee9, 0xfec6,
    0xfea3, 0xfe80, 0xfe5d, 0xfe3b, 0xfe18, 0xfdf6, 0xfdd3, 0xfdb1,
    0xfd8e, 0xfd6c, 0xfd4a, 0xfd27, 0xfd05, 0xfce3, 0xfcc1, 0xfca0,
    0xfc7e, 0xfc5c, 0xfc3b, 0xfc19, 0xfbf8, 0xfbd7, 0xfbb6, 0xfb95,
    0xfb74, 0xfb53, 0xfb33, 0xfb12, 0xfaf2, 0xfad2, 0xfab2, 0xfa92,
    0xfa73, 0xfa53, 0xfa34, 0xfa15, 0xf9f6, 0xf9d7, 0xf9b9, 0xf99a,
    0xf97c, 0xf95e, 0xf940, 0xf922, 0xf905, 0xf8e8, 0xf8cb, 0xf8ae,
    0xf891, 0xf875, 0xf859, 0xf83d, 0xf821, 0xf806, 0xf7ea, 0xf7cf,
    0xf7b4, 0xf79a, 0xf780, 0xf766, 0xf74c, 0xf732, 0xf719, 0xf700,
    0xf6e7, 0xf6cf, 0xf6b6, 0xf69f, 0xf687, 0xf66f, 0xf658, 0xf641,
    0xf62b, 0xf615, 0xf5ff, 0xf5e9, 0xf5d4, 0xf5bf, 0xf5aa, 0xf595,
    0xf581, 0xf56d, 0xf55a, 0xf546, 0xf533, 0xf521, 0xf50e, 0xf4fc,
    0xf4eb, 0xf4d9, 0xf4c8, 0xf4b8, 0xf4a7, 0xf497, 0xf488, 0xf478,
    0xf469, 0xf45b, 0xf44c, 0xf43e, 0xf431, 0xf423, 0xf416, 0xf40a,
    0xf3fd, 0xf3f2, 0xf3e6, 0xf3db, 0xf3d0, 0xf3c5, 0xf3bb, 0xf3b1,
    0xf3a8, 0xf39f, 0xf396, 0xf38e, 0xf386, 0xf37e, 0xf377, 0xf370,
    0xf36a, 0xf364, 0xf35e, 0xf358, 0xf353, 0xf34f, 0xf34a, 0xf346,
    0xf343, 0xf340, 0xf33d, 0xf33b, 0xf338, 0xf337, 0xf335, 0xf335,
    0xf334, 0xf334, 0xf334, 0xf335, 0xf335, 0xf337, 0xf338, 0xf33b,
    0xf33d, 0xf340, 0xf343, 0xf346, 0xf34a, 0xf34f, 0xf353, 0xf358,
    0xf35e, 0xf364, 0xf36a, 0xf370, 0xf377, 0xf37e, 0xf386, 0xf38e,
    0xf396, 0xf39f, 0xf3a8, 0xf3b1, 0xf3bb, 0xf3c5, 0xf3d0, 0xf3db,
    0xf3e6, 0xf3f2, 0xf3fd, 0xf40a, 0xf416, 0xf423, 0xf431, 0xf43e,
    0xf44c, 0xf45b, 0xf469, 0xf478, 0xf488, 0xf497, 0xf4a7, 0xf4b8,
    0xf4c8, 0xf4d9, 0xf4eb, 0xf4fc, 0xf50e, 0xf521, 0xf533, 0xf546,
    0xf55a, 0xf56d, 0xf581, 0xf595, 0xf5aa, 0xf5bf, 0xf5d4, 0xf5e9,
    0xf5ff, 0xf615, 0xf62b, 0xf641, 0xf658, 0xf66f, 0xf687, 0xf69f,
    0xf6b6, 0xf6cf, 0xf6e7, 0xf700, 0xf719, 0xf732, 0xf74c, 0xf766,
    0xf780, 0xf79a, 0xf7b4, 0xf7cf, 0xf7ea, 0xf806, 0xf821, 0xf83d,
    0xf859, 0xf875, 0xf891, 0xf8ae, 0xf8cb, 0xf8e8, 0xf905, 0xf922,
    0xf940, 0xf95e, 0xf97c, 0xf99a, 0xf9b9, 0xf9d7, 0xf9f6, 0xfa15,
    0xfa34, 0xfa53, 0xfa73, 0xfa92, 0xfab2, 0xfad2, 0xfaf2, 0xfb12,
    0xfb33, 0xfb53, 0xfb74, 0xfb95, 0xfbb6, 0xfbd7, 0xfbf8, 0xfc19,
    0xfc3b, 0xfc5c, 0xfc7e, 0xfca0, 0xfcc1, 0xfce3, 0xfd05, 0xfd27,
    0xfd4a, 0xfd6c, 0xfd8e, 0xfdb1, 0xfdd3, 0xfdf6, 0xfe18, 0xfe3b,
    0xfe5d, 0xfe80, 0xfea3, 0xfec6, 0xfee9, 0xff0c, 0xff2f, 0xff52,
    0xff74, 0xff97, 0xffba, 0xffdd,
};

/* prototype to avoid a warning. */
int _hdcd_tone16(int *sn, int rate);

int _hdcd_tone16(int *sn, int rate)
{
    (*sn)++;
    switch(rate) {
        case 192000:
            *sn %= 640;
            return tone_tab_192000[*sn];
        case 96000:
            *sn %= 320;
            return tone_tab_192000[(*sn)<<1];
        case 48000:
            *sn %= 160;
            return tone_tab_192000[(*sn)<<2];
        case 176400:
            *sn %= 588;
            return tone_tab_176400[*sn];
        case 88200:
            *sn %= 294;
            return tone_tab_176400[(*sn)<<1];
        case 44100:
        default:
            *sn %= 147;
            return tone_tab_176400[(*sn)<<2];
    }
}

#ifdef HDCD_PRINT_TONE_TAB

/** tone generator: sample_number, frequency, sample_rate, amplitude */
#define TONEGEN16(sn, f, sr, a) (int16_t)(sin((6.28318530718 * (sn) * (f)) /(sr)) * (a) * 0x7fff)

int main()
{
    int n;
    int freq = 300;
    int rate = 192000;
    int max = rate / freq;

    printf("static const int16_t tone_tab_%d[%d] = {\n    ", rate, max);
    for (n = 0; n < max; n++) {
        printf("0x%04hx, ", TONEGEN16(n, freq, rate, 0.1) );
        if (n%8 == 7) printf("\n    ");
    }
    printf("\n};\n");
    rate = 176400;
    max = rate / freq;
    printf("static const int16_t tone_tab_%d[%d] = {\n    ", rate, max);
    for (n = 0; n < max; n++) {
        printf("0x%04hx, ", TONEGEN16(n, freq, rate, 0.1) );
        if (n%8 == 7) printf("\n    ");
    }
    printf("\n};\n");
}
#endif
