/* -*- Mode: C; indent-tabs-mode: nil; c-basic-offset: 4 c-style: "K&R" -*- */

/*
   libgpiv - library for Particle Image Velocimetry

   Copyright (C) 2002, 2003, 2004 Gerber van der Graaf

   This file is part of libgpiv.

   Libgpiv is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  




---------------------------------------------------------------
FILENAME:                io.c
LIBRARY:                 libgpiv:
EXTERNAL FUNCTIONS:      gpiv_io_make_fname
                         gpiv_read_header
			 gpiv_fread_image
			 gpiv_fwrite_image
			 gpiv_read_image
                         gpiv_write_image
			 gpiv_fread_ascii_image
			 gpiv_fwrite_ascii_image
                         gpiv_read_ascii_image 
			 gpiv_write_ascii_image
			 gpiv_count_pivdata
                         gpiv_fcount_pivdata,
                         gpiv_find_pivdata_origin
			 gpiv_find_pivdata_scale
                         gpiv_read_pivdata
			 gpiv_fread_pivdata
                         gpiv_write_pivdata
			 gpiv_fwrite_pivdata
			 gpiv_fread_scdata
                         gpiv_write_scdata
			 gpiv_fwrite_scdata
			 gpiv_fwrite_griddata
			 gpiv_print_histo
			 gpiv_fprint_histo
                         gpiv_print_cumhisto_eqdatbin
                         gpiv_fprint_cumhisto_eqdatbin

                         gpiv_fread_pgm_image
                         gpiv_fwrite_pgm_image

                         gpiv_fcreate_hdf5
                         gpiv_fread_hdf5_image
                         gpiv_fwrite_hdf5_image
                         gpiv_fread_hdf5_parameters
                         gpiv_fwrite_hdf5_parameters
                         gpiv_fread_hdf5_position
                         gpiv_fwrite_hdf5_position
                         gpiv_fread_hdf5_piv_position
                         gpiv_fwrite_hdf5_piv_position
			 gpiv_fread_hdf5_pivdata
			 gpiv_fwrite_hdf5_pivdata
                         gpiv_fread_hdf5_sc_position
                         gpiv_fwrite_hdf5_sc_position
                         gpiv_fread_hdf5_scdata
			 gpiv_fwrite_hdf5_scdata
                         gpiv_fread_hdf5_histo
			 gpiv_fwrite_hdf5_histo

                         gpiv_fread_davis_image
LO
LAST MODIFICATION DATE:  $Id: io.h,v 1.14 2006/01/31 13:27:22 gerber Exp $
 --------------------------------------------------------------- */

#ifndef __LIBGPIV_IO_H__
#define __LIBGPIV_IO_H__


#define GPIV_ASCII_IMG_FMT "%d %d %f"	                        /* Ascii Image Format */
#define GPIV_PIV_FMT "%4.0f %4.0f %12.3f %12.3f %12.3f %2d\n"	/* Piv data Format */
#define GPIV_PIV_S_FMT "%-12.5f %-12.5f %-12.5f  %-12.5f  %-6.2f     %-2d\n"	/* Piv scaled data Format */
#define GPIV_SCALAR_FMT "%4.0f %4.0f %12.3f     %2d\n"	        /* Scalar data Format with PIV data positions */
#define GPIV_SCALAR_S_FMT "%-12.3f %-12.3f %-12.3f     %-2d\n"	/* Scalar data Format with PIV scaled data positions */

#define GPIV_EXT_HEADER ".h"		/* Extension of image header file name */
#define GPIV_EXT_RAW_IMAGE ".r"	        /* Extension of raw image file name */
#define GPIV_EXT_PGM_IMAGE ".pgm"	/* Extension of raw portable graymap (pgm) image file name */
#define GPIV_EXT_RAW_IMAGE_PROC ".proc.r"	/* Extension of processed raw image file name */
#define GPIV_EXT_ASCII_IMAGE ".dat"	/* Extension of ASCII image file name */
#define GPIV_EXT_PAR ".par"		/* Extension of file with used parameters */

#define GPIV_EXT_TA ".ta"               /* Extension of time averaged file name */
#define GPIV_EXT_SUBSTR ".substr"       /* Extension of file name with substracted data */

#define GPIV_EXT_GPIV ".gpi"	        /* Extension of gpiv file name (HDF 5 format) */
#define GPIV_EXT_PIV  ".piv"		/* Extension of piv file name (ASCII format) */
#define GPIV_EXT_DAVIS ".IMG"           /* DaVis image format */
#define GPIV_EXT_COV ".cov"	        /* Extension of covariance data  */
#define GPIV_EXT_INT ".int"		/* Extension of interrogation area */
#define GPIV_EXT_OLD ".old.piv"	        /* Extension of old displacements */

#define GPIV_EXT_ERR_PIV ".err.piv"	/* Extension of erroneous-corrected piv file name */
#define GPIV_EXT_ERR_STAT ".stat"	/* Extension of residu statistics */
#define GPIV_EXT_PLK  ".plk"		/* Extension of peaklocking output */
#define GPIV_EXT_UVHISTO  ".pdf"	/* Extension of displacement histogram */

#define GPIV_EXT_VOR ".vor"	        /* Extension of vorticity strain file name (ASCII format) */
#define GPIV_EXT_NSTR ".nstr"	        /* Extension of normal strain file name (ASCII format) */
#define GPIV_EXT_SSTR ".sstr"	        /* Extension of shear strain file name (ASCII format) */
#define GPIV_EXT_MANI ".ma.piv"	        /* Extension of manipulated file name (ASCII format) */
#define GPIV_EXT_SA ".sa.piv"           /* Extension of spatial averaged file name (ASCII format)*/
#define GPIV_EXT_SC ".sc.piv"           /* Extension of scaled data file name (ASCII format)*/

#define GPIV_DAC_PAR_KEY "DAC."	        /* Key of data acquisition parameters */
#define GPIV_IMAGE_PAR_KEY "IMG."	/* Key of image parameters */
#define GPIV_IMAGEPROC_PAR_KEY "IMGPROC."	/* Key of image processing parameters */
#define GPIV_EVAL_PAR_KEY "EVAL."	/* Key of evaluation parameters */
#define GPIV_VALID_PAR_KEY "VALID."	/* Key of  validation processing parameters */
#define GPIV_POST_PAR_KEY "POST."	/* Key of post processing parameters */

#define GPIV_DAVIS_IMG_DEPTH 12         /* Depth of mages generated by LaVision's Davis (general: PCO cameras) */



enum GpivDataFormat { 
  GPIV_RR = 1, 
  GPIV_DAV = 2 
} pdf;


void 
gpiv_io_make_fname(char *fname_base,
		   char *EXT,
		   char *fname_out
		   );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *  Constructs (output) filename from base name and exetsion 
 *
 * INPUTS:
 *      fname_base:     file base name
 *      EXT:            file extension name
 * OUTPUTS:
 *      fname_out:      completed filename
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/



char *
gpiv_fread_image(char *fname,
                 guint16 **img1,
                 guint16 **img2,
                 GpivImagePar image_par
		 );
/* enum FrameNr fn  */  /* concerns first or second frame (with auto-corr) */
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Read raw image data from file
 *
 * INPUTS:
 *      fname:          pointer to complete filename to be written
 *      image_par:      image parameters containing number of rows an columns
 *
 * OUTPUTS:
 *      img1:           first binary image
 *      img2:           second binary image
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/



char *
gpiv_fwrite_image(char *fname,
                  guint16 **img1,
                  guint16 **img2, 
                  GpivImagePar image_par
		  );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Write raw data image to file
 *
 * INPUTS:
 *      fname:          pointer to complete filename to be written
 *      img1:           first binary image
 *      img2:           second binary image
 *      image_par:      image parameters containing number of rows an columns
 *
 * OUTPUTS:
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/



void 
gpiv_read_image(guint16 **img1,
                guint16 **img2, 
                GpivImagePar image_par
		);
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Read raw data image from stdin
 *
 * INPUTS:
 *      image_par:      image parameters containing number of rows an columns
 *
 * OUTPUTS:
 *      img1:           first binary image
 *      img2:           second binary image
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/



void 
gpiv_write_image(guint16 **img1,
                 guint16 **img2, 
                 GpivImagePar image_par
		 );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *       Write binary image data to stdout 
 *
 * INPUTS:
 *      img1:           first binary image
 *      img2:           second binary image
 *      image_par:      image parameters containing number of rows an columns
 *
 * OUTPUTS:
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/



char *
gpiv_fread_ascii_image(char *fname,
                       guint16 **img,
                       GpivImagePar image_par
		       );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Read ASCII  data image from file
 *
 * INPUTS:
 *      fname:          pointer to complete filename to be read
 *      image_par:      image parameters containing number of rows an columns
 *
 * OUTPUTS:
 *      img:            2-dimensional matrix containing nrows x ncoloms pixels
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/



char *
gpiv_fwrite_ascii_image(char *fname,
                        guint16 **img,
                        GpivImagePar image_par
			);
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Write ASCII  data image to file
 *
 * INPUTS:
 *      fname:          pointer to complete filename to be written
 *      img:            2-dimensional matrix containing nrows x ncoloms pixels
 *      image_par:      image parameters containing number of rows an columns
 *
 * OUTPUTS:
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/



void
gpiv_read_ascii_image(guint16 **img,
                      GpivImagePar image_par
		      );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Read ASCII data image from stdin
 *
 * INPUTS:
 *      nrows:         number of rows in image
 *      ncolumns:      number of columns in image
 *
 * OUTPUTS:
 *      img:            2-dimensional matrix containing nrows x ncoloms pixels
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/



void 
gpiv_write_ascii_image(guint16 **img,
                       GpivImagePar image_par
		       );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Write ASCII data image to stdout
 *
 * INPUTS:
 *      img:            2-dimensional matrix containing nrows x ncoloms pixels
 *      nrows:          number of rows in image
 *      ncolumns:       number of columns in image
 *
 * OUTPUTS:
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/



int 
gpiv_count_pivdata(GpivPivData * piv,
                   char data_line[GPIV_MAX_LINES][GPIV_MAX_CHARS],
                   int *ndata_lines 
		   );	
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      * Reads data from STDIN to find array length of x-and y data, 
 *        provided that both x and y-data are in incremental or decremental 
 *        order
 *      * Determines if scaled data have been used in order to determine which 
 *        format/header has to be used for writing the data
 *
 * INPUTS:
 *      piv:            piv data to be read
 *
 * OUTPUTS:
 *      piv.n, piv,y:   number of x and y- data
 *      data_line:      all lines that are read from STDIN
 *      nlines:         total number of lines, including comment lines
 *
 * RETURNS:
 *      scale:          0 or 1 for (non) scaled data
 *---------------------------------------------------------------------------*/



int 
gpiv_fcount_pivdata(char *fname,
                    GpivPivData * piv 
		    );	
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      * Reads data from  ASCII file to find array length of x-and y data, 
 *        provided that both x and y-data are in incremental or decremental 
 *        order
 *      * Determines if scaled data have been used in order to determine which 
 *        format/header has to be used for writing the data
 *
 * INPUTS:
 *      fname:          pointer to complete filename containing data
 *      piv:            piv data to be read
 *
 * OUTPUTS:
 *      piv.n, piv.y:    number of x and y- data in pointer to piv structure
 *
 * RETURNS:
 *      scale:          0 or 1 for (non) scaled data
 *---------------------------------------------------------------------------*/



int 
gpiv_find_pivdata_origin(char line[GPIV_MAX_CHARS]
                         );	
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Determines the name of the program that generated the data
 *
 * INPUTS:
 *      line:           character line containing program that generated data
 *
 * OUTPUTS:
 *
 * RETURNS:
 *      pdf:            enumerator of data origin
 *---------------------------------------------------------------------------*/



int 
gpiv_find_pivdata_scale(char line[GPIV_MAX_CHARS]
			);
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Determines if the data have been scaled in order to read x-and y 
 *      positions as floating point varibles
 *
 * INPUTS:
 *      line:           character line containing program that generated data
 *
 * OUTPUTS:
 *
 * RETURNS:
 *      scale:          1 if scaled, else 0
 *---------------------------------------------------------------------------*/



void
gpiv_read_pivdata(GpivPivData * piv,
                  char data_line[GPIV_MAX_LINES][GPIV_MAX_CHARS],
                  int ndata_lines,
                  char comment_line[GPIV_MAX_LINES_C][GPIV_MAX_CHARS],
                  int *ncomment_lines
		  );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Reads PIV data from stdin
 *
 * INPUTS:
 *      fname:          complete filename
 *      piv:            struct of PIV data containing all variables in a datafile
 *                      piv.nx and piv.ny have to be known.
 *      comment_line:   lines contain comment (staring with '#' sign at first 
 *                      column)
 *      ncomment_lines: number of comment lines
 *
 * OUTPUTS:
 *      comment_line:   comment lines (lines starting with '#'), except header
 *      ncomment_lines: number of comment lines
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/



char *
gpiv_fread_pivdata(char *fname,
                   GpivPivData * piv,
                   char comment_line[GPIV_MAX_LINES_C][GPIV_MAX_CHARS],
                   int *ncomment_lines
		   );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Reads PIV data from ascii data file
 *
 * INPUTS:
 *      fname:          complete filename
 *
 * OUTPUTS:
 *      piv:            struct piv_data containing all variables in a datafile
 *                     piv.nx and piv.ny have to be known.
 *      comment_line:   lines contain comment (staring with '#' sign at first 
 *                     column)
 *      ncomment_lines: number of comment lines
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/



void
gpiv_write_pivdata(GpivPivData * piv,
                   char comment_line[GPIV_MAX_LINES_C][GPIV_MAX_CHARS],
                   int ncomment_lines,
                   int scale,
                   char *RCSID
		   );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Writes 2-d vector (piv) data to stdout
 *
 * INPUTS:
 *      piv:            struct piv_data containing all variables in a datafile
 *      comment_line:   lines contain comment (staring with '#' sign at first column)
 *      ncomment_lines: number of comment lines
 *      scale:          parameter to write piv.point_x and piv.point_y as int or as float
 *      RCSID:          identifier of program that generated datafile
 *
 * OUTPUTS:
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/



char *
gpiv_fwrite_pivdata(char *fname,
                    GpivPivData * piv,
                    char comment_line[GPIV_MAX_LINES_C][GPIV_MAX_CHARS],
                    int ncomment_lines,
                    int scale,
                    char *RCSID
		    );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Writes 2-d vector (piv) data to file
 *
 * INPUTS:
 *      fname:          complete filename
 *      piv:            struct piv_data containing all variables in a datafile
 *      comment_line:   lines contain comment (staring with '#' sign at first 
 *                      column)
 *      ncomment_lines: number of comment lines
 *      scale:          parameter to write piv.point_x and piv.point_y as int 
 *                      or as float
 *      RCSID:          identifier of program that generated datafile
 *
 * OUTPUTS:
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/



char *
gpiv_fread_scdata(char *fname,
                  GpivScalarData * scalar,
                  char comment_line[GPIV_MAX_LINES_C][GPIV_MAX_CHARS],
                  int *ncomment_lines
                  );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Reads scalar data form ascii file
 *
 * INPUTS:
 *      fname:          complete filename
 *
 * OUTPUTS:
 *      scalar_data:    struct piv_data containing all variables in a datafile
 *                      piv.nx and piv.ny have to be known.
 *      comment_line:   lines contain comment (staring with '#' sign at first column)
 *      ncomment_lines: number of comment lines
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/



void
gpiv_write_scdata(GpivScalarData * scalar,
                  char comment_line[GPIV_MAX_LINES_C][GPIV_MAX_CHARS],
                  int ncomment_lines,
                  int scale,
                  char *RCSID
		  );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Writes scalar data to stdout
 *
 * INPUTS:
 *      scalar:         struct GpivScalarData containing all variables in a datafile
 *      comment_line:   lines contain comment (staring with '#' sign at first 
 *                      column)
 *      ncomment_lines: number of comment lines
 *      scale:          parameter to write scalar.point_x and scalar.point_y as
 *                      int or as float
 *      RCSID:          identifier of program that generated datafile
 *
 * OUTPUTS:
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/



char *
gpiv_fwrite_scdata(char *fname,
                   GpivScalarData * scalar,
                   char comment_line[GPIV_MAX_LINES_C][GPIV_MAX_CHARS],
                   int ncomment_lines,
                   int scale,
                   char *RCSID
		   );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Writes scalar data to file
 *
 * INPUTS:
 *      scalar:         struct GpivScalarData containing all variables in a datafile
 *      comment_line:   lines contain comment (staring with '#' sign at first 
 *                      column)
 *      ncomment_lines: number of comment lines
 *      scale:          parameter to write scalar.point_x and scalar.point_y as
 *                      int (0) or as float (1) 
 *      RCSID:          identifier of program that generated datafile
 *
 * OUTPUTS:
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/



char *
gpiv_fwrite_sc_griddata(char *fname,
                        GpivScalarData * scalar,
                        char comment_line[GPIV_MAX_LINES_C][GPIV_MAX_CHARS],
                        int ncomment_lines,
                        int scale,
                        char *RCSID
			);
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Writes scalar data to file in grid format for gnuplot
 *
 * INPUTS:
 *      scalar:         struct GpivScalarData containing all variables in a datafile
 *      comment_line:   lines contain comment (staring with '#' sign at first 
 *                      column)
 *      ncomment_lines: number of comment lines
 *      scale:          parameter to write scalar.point_x and scalar.point_y as
 *                      int or as float
 *      RCSID:          identifier of program that generated datafile
 *
 * OUTPUTS:
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/



char *
gpiv_fwrite_sc_mtvgriddata (char *fname, 
                            GpivScalarData * scalar, 
                            char comment_line[GPIV_MAX_LINES_C][GPIV_MAX_CHARS], 
                            int ncomment_lines, 
                            int scale, 
                            char *RCSID
                            );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Writes scalar data to file in grid format for plotmtv
 *
 * INPUTS:
 *      scalar:         struct GpivScalarData containing all variables in a datafile
 *      comment_line:   lines contain comment (staring with '#' sign at first 
 *                      column)
 *      ncomment_lines: number of comment lines
 *      scale:          parameter to write scalar.point_x and scalar.point_y as
 *                      int or as float
 *
 * OUTPUTS:
 *      RCSID:          identifier of program that generated datafile
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/



void 
gpiv_print_histo(GpivBinData * klass,
                 char comment_line[GPIV_MAX_LINES_C][GPIV_MAX_CHARS],
                 int ncomment_lines,
                 int scale,
                 char *RCSID
		 );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Writes bins data to stdout in histogram format
 *
 * INPUTS:
 *      klass:          struct GpivBinData containing the values for each bin
 *      comment_line:   lines contain comment (staring with '#' sign at first
 *                      column)
 *      ncomment_lines: number of comment lines
 *      scale:          scaled (1) or unscaled (0) data
 *      RCSID:          identifier of program that generated the output datafile
 *
 * OUTPUTS:
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/



void 
gpiv_fprint_histo(char *fname,
                  GpivBinData * klass,
                  char comment_line[GPIV_MAX_LINES_C][GPIV_MAX_CHARS],
                  int ncomment_lines,
                  int scale,
                  char *RCSID
		  );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Writes bins data to file in histogram format
 *
 * INPUTS:
 *      fname:          complete filename
 *      klass:          struct bin containing the values for each bin
 *      comment_line:   lines contain comment (staring with '#' sign at first
 *                      column)
 *      ncomment_lines: number of comment lines
 *      RCSID:         identifier of program that generated the output datafile
 *
 * OUTPUTS:
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/



void
gpiv_print_cumhisto_eqdatbin(GpivBinData *klass, 
                             char comment_line[GPIV_MAX_LINES_C][GPIV_MAX_CHARS], 
                             int ncomment_lines, 
                             int scale, 
                             char *RCSID
                             );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Writing cumulative histogram data with an equal number of date per bin 
 *      or klass to stdout. Special output for validation.; work around to print
 *      float data as y-values.
 *
 * INPUTS:
 *      klass:          struct GpivBinData containing the values for each bin
 *      comment_line:   lines contain comment (staring with '#' sign at first 
 *                      column)
 *      ncomment_lines: number of comment lines
 *      RCSID:          identifier of program that generated the output datafile
 *
 * OUTPUTS:
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/



void
gpiv_fprint_cumhisto_eqdatbin(char *fname, 
                              GpivBinData *klass, 
                              char comment_line[GPIV_MAX_LINES_C][GPIV_MAX_CHARS], 
                              int ncomment_lines, 
                              int scale, 
                              char *RCSID
                              );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Writing cumulative histogram data with an equal number of date per bin 
 *      or klass to ouput file Special output for validation.; work around to
 *      print float data as y-values.
 *
 * INPUTS:
 *      fname:          complete filename
 *      klass:          struct bin containing the values for each bin
 *      comment_line:   lines contain comment (staring with '#' sign at first 
 *                      column)
 *      ncomment_lines: number of comment lines
 *      RCSID:          identifier of program that generated the output datafile
 *
 * OUTPUTS:
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/



char *
gpiv_fread_pgm_image(char *fname,
                     guint16 **img1,
                     guint16 **img2, 
                     GpivImagePar * image_par,
                     gint line_nr
                     );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Read raw portable graymap (pgm) format image data from file
 *
 * INPUTS:
 *      fname:          pointer to complete filename to be written
 *      image_par:      image parameters containing number of rows an columns
 *
 * OUTPUTS:
 *      img1:           first binary image, memory will be allocated here
 *      img2:           second binary image, memory will be allocated here
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/



char *
gpiv_fwrite_pgm_image (char *fname,
                       guint16 **img1, 
                       guint16 **img2, 
                       GpivImagePar image_par
                       );
/*  ---------------------------------------------------------------------------
 * DESCRIPTION:
 *     writes binary portable graymap (pgm) format image to file
 *
 * PROTOTYPE LOCATATION:
 *     io.h
 *
 * INPUTS:
 *      fname:          pointer to complete filename to be written
 *      img1:           first 2-dimensional matrix containing nrows x ncoloms pixels
 *      img2:           second 2-dimensional matrix containing nrows x ncoloms pixels
 *      image_par:      image parameters containing number of rows an columns
 *
 * OUTPUTS:
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/



char *
gpiv_fcreate_hdf5 (char *fname
                   );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Creates an hdf5 data file with POSTION, DATA and PARAMETERS groups
 *
 * INPUTS:
 *      fname:          complete filename
 *
 * OUTPUTS:
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/



char *
gpiv_fread_hdf5_parameters (char *fname,
                            const char *par_key,
                            void *pstruct
                            );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Reads parameters from hdf5 data file
 *
 * INPUTS:
 *      group_id:       hdf group identity
 *      par_key:        key for apropiate parameter
 *
 * OUTPUTS:
 *      pstruct:        parameter structure
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/



char *
gpiv_fread_hdf5_image(char *fname, 
#ifdef HD5_IMAGE_INT
                       int **img1, 
                       int **img2, 
#else
                       guint16 **img1, 
                       guint16 **img2, 
#endif
                      GpivImagePar * image_par
                      );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Reads image data to a hdf5 data file
 *
 * INPUTS:
 *      fname:          complete filename
 *      image_par:      image parameters
 *
 * OUTPUTS:
 *      img:            2-dimensional matrix containing nrows x ncoloms pixel 
 *                      values
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/



char *
gpiv_fwrite_hdf5_image(char *fname, 
#ifdef HD5_IMAGE_INT
                       int **img1, 
                       int **img2, 
#else
                       guint16 **img1, 
                       guint16 **img2, 
#endif
                       GpivImagePar * image_par
                       );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Writes image data to file in hdf version 5 format
 *
 * INPUTS:
 *      fname:          complete filename
 *      img:            2-dimensional matrix containing nrows x ncoloms pixel 
 *                      values
 *      image_par:      image parameters
 *
 * OUTPUTS:
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/



char *
gpiv_fwrite_hdf5_parameters (char *fname,
                             const char *par_key,
                             void *pstruct
                             );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Writes parameters to hdf5 data file
 *
 * INPUTS:
 *      group_id:       hdf group identity
 *      par_key:        key for apropiate parameter
 *      pstruct:        parameter structure
 *
 * OUTPUTS:
 *
 * RETURNS:
 *      NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/



char *
gpiv_fcount_hdf5_data(char *fname,
                      int * ny,
                      int * nx
                         );	
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Reads array lengths of 2-dimensional data
 *
 * INPUTS:
 *      fname:          complete filename
 *
 * OUTPUTS:
 *      ny:             length of first array index
 *      nx:             length of second array index
 *
 * RETURNS:
 *      NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/



char *
gpiv_fread_hdf5_piv_position(char *fname, 
                             GpivPivData *piv_data,
                             char *RCSID,
                             int *scale
                             );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Reads position data from hdf5 data file into GpivPivData struct
 *
 * INPUTS:
 *      fname:          complete filename
 *
 * OUTPUTS:
 *      piv_data:       parameter structure
 *
 * RETURNS:
 *      NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/



char *
gpiv_fwrite_hdf5_piv_position(char *fname, 
                              GpivPivData *piv_data,
                              char *RCSID,
                              int *scale 
                              );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Writess position data to hdf5 data file into GpivPivData struct
 *
 * INPUTS:
 *      fname:          complete filename
 *
 * OUTPUTS:
 *      piv_data:       parameter structure
 *
 * RETURNS:
 *      NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/



char *
gpiv_fread_hdf5_pivdata (char *fname, 
                         GpivPivData *piv_data, 
                         char *DATA_KEY,
                         char *RCSID,
                         int *scale
                         );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Reads piv data from hdf5 data file
 *
 * INPUTS:
 *      fname:          complete filename
 *
 * OUTPUTS:
 *      piv_data:       struct piv_data containing all variables in a datafile
 *                      piv.nx and piv.ny have to be known.
 *      comment_line:   lines contain comment (staring with '#' sign at first
 *                      column)
 *      ncomment_lines: number of comment lines
 *
 * RETURNS:
 *      NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/



char *
gpiv_fwrite_hdf5_pivdata(char *fname, 
                         GpivPivData *piv_data, 
                         char *DATA_KEY,
                         char *RCSID,
                         int *scale
                         );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Writes PIV data to file in hdf version 5 format
 *
 * INPUTS:
 *      fname:          complete filename
 *      piv:            struct piv_data containing all variables in a datafile
 *      DATA_KEY:       location of the data in the DATA group in the hdf file
 *      RCSID:          identifier of program that generated datafile
 *      scale:          parameter to write piv.point_x and piv.point_y as int 
 *                      or as float
 *
 * OUTPUTS:
 *
 * RETURNS:
 *      NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/



char *
gpiv_fread_hdf5_sc_position(char *fname, 
                            GpivScalarData *scdata
                            );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Reads position data from hdf5 data file into GpivScalarData struct
 *
 * INPUTS:
 *      fname:          complete filename
 *
 * OUTPUTS:
 *      scdata:         parameter structure
 *
 * RETURNS:
 *      NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/



char *
gpiv_fwrite_hdf5_sc_position(char *fname, 
                             GpivScalarData *scdata,
                              char *RCSID,
                              int *scale
                             );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Writess position data to hdf5 data file into GpivScalarData struct
 *
 * INPUTS:
 *      fname:           complete filename
 *
 * OUTPUTS:
 *       scdata:         parameter structure
 *
 * RETURNS:
 *      NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/



char *
gpiv_fread_hdf5_scdata(char *fname, 
                       GpivScalarData *scalar_data,
                       char *DATA_KEY,
                       char *RCSID
                       );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Writes SCALAR data to a hdf5 data file
 *
 * INPUTS:
 *      fname:          complete filename
 *      DATA_KEY:       key to specify data type/origin (NORMAL_STRAIN, ...)
 *      RCSID:          identifier of program that generated the output datafile
 *
 * OUTPUTS:
 *      scalar_data
 *
 * RETURNS:
 *      NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/



char *
gpiv_fwrite_hdf5_scdata(char *fname, 
                        GpivScalarData *scalar_data, 
                        char *DATA_KEY,
                        char *RCSID
                        );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Writes SCALAR data to file in hdf version 5 format
 *
 * INPUTS:
 *      scalar:         struct GpivScalarData containing all variables in a datafile
 *      DATA_KEY:       identifier of data type
 *      RCSID:          identifier of program that generated datafile
 *
 * OUTPUTS:
 *
 * RETURNS:
 *      NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/



char *
gpiv_fread_hdf5_histo(char *fname, 
                      GpivBinData *klass, 
                      char * DATA_KEY,
                      char *RCSID
                      );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Reads histogram data to ouput file in hdf version 5 format
 *
 * INPUTS:
 *      fname:          complete filename
 *      klass:          struct bin containing the values for each bin
 *      DATA_KEY:       identifier of data type (PEAK_LOCK, RESIDUS, ...)
 *      RCSID:          identifier of program that generated the output datafile
 *
 * OUTPUTS:
 *
 * RETURNS:
 *      NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/



char *
gpiv_fwrite_hdf5_histo(char *fname, 
                       GpivBinData *klass, 
                       char * DATA_KEY,
                       char *RCSID
                       );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Writes histogram data to ouput file in hdf version 5 format
 *
 * INPUTS:
 *      fname:          complete filename
 *      klass:          struct bin containing the values for each bin
 *      DATA_KEY:       identifier of data type (PEAK_LOCK, RESIDUS, ...)
 *      RCSID:          identifier of program that generated the output datafile
 *
 * OUTPUTS:
 *
 * RETURNS:
 *      NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/



char *
gpiv_fread_davis_image(char *fname,
                       guint16 **img1,
                       guint16 **img2, 
                       GpivImagePar *image_par
                       );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Reads Davis formatted image, with ext .img, from file
 *
 * INPUTS:
 *      fname:          pointer to complete filename to be read
 *      image_par:      containing image dimensions and other image specs
 *
 * OUTPUTS:
 *      img1:           image data of first image
 *      img2:           image data of second image
 *
 * RETURNS:
 *      NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/



#endif /* __LIBGPIV_IO_H__ */
