/* -*- Mode: C; indent-tabs-mode: nil; c-basic-offset: 4 c-style: "K&R" -*- */

/*-----------------------------------------------------------------------------
   libgpiv - library for Particle Image Velocimetry

   Copyright (C) 2002, 2003, 2004 Gerber van der Graaf

   This file is part of libgpiv.

   Libgpiv is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  




-------------------------------------------------------------------------------
FILENAME:                img.c
LIBRARY:                 libgpiv:
EXTERNAL FUNCTIONS:      gpiv_img_parameters_logic
                         gpiv_img_read_header
                         gpiv_img_fread_header
			 gpiv_img_check_header_read
			 gpiv_img_test_header
			 gpiv_img_print_header
			 gpiv_img_fprint_header
                         gpiv_img_cp_parameters
			 gpiv_img_fread_parameters
			 gpiv_img_fread_hdf5_parameters
			 gpiv_img_fwrite_hdf5_parameters
                         gpiv_img_fread_davis_parameters
			 
LAST MODIFICATION DATE:  $Id: img.h,v 1.13 2006/01/31 13:27:22 gerber Exp $
 --------------------------------------------------------------------------- */

#ifndef __LIBGPIV_IMG_H__
#define __LIBGPIV_IMG_H__

/*-----------------------------------------------------------------------------
 * Image header info
 */


#define GPIV_IMGPAR_DEFAULT__DEPTH 8
#define GPIV_IMGPAR_DEFAULT__NCOLUMNS 256
#define GPIV_IMGPAR_DEFAULT__NROWS 256
#define GPIV_IMGPAR_DEFAULT__X_CORR 1
#define GPIV_IMGPAR_DEFAULT__S_SCALE 1.0
#define GPIV_IMGPAR_DEFAULT__T_SCALE 1.0
#define GPIV_IMGPAR_DEFAULT__Z_OFF_X 0.0
#define GPIV_IMGPAR_DEFAULT__Z_OFF_Y 0.0

#define GPIV_IMGPAR_DEFAULT__TITLE "Developing / testing image deformation"
#define GPIV_IMGPAR_DEFAULT__CREATION_DATE "23 June 1998"
#define GPIV_IMGPAR_DEFAULT__LOCATION "Bcn"
#define GPIV_IMGPAR_DEFAULT__AUTHOR "GvdGraaf"
#define GPIV_IMGPAR_DEFAULT__SOFTWARE "gpiv / gpiv_rr"
#define GPIV_IMGPAR_DEFAULT__SOURCE "CCD camera"
#define GPIV_IMGPAR_DEFAULT__USERTEXT "You may put your comments here"
#define GPIV_IMGPAR_DEFAULT__WARNING ""
#define GPIV_IMGPAR_DEFAULT__DISCLAIMER "See The GNU General Public License (GPL)"
#define GPIV_IMGPAR_DEFAULT__COMMENT ""

typedef struct __GpivImagePar GpivImagePar;
struct __GpivImagePar {
    int depth;                  /* Image color depth in bitsize */
    int ncolumns;               /* Number of image columns */
    int nrows;                  /* Number of image rows */
    int x_corr;                 /* cross-correlation image pair */
    float s_scale;              /* spatial scale, used by gpiv_post_scale */
    float t_scale;              /* time-scale, used by gpiv_post_scale */
    float z_off_x;              /* zero offset in x (column) direction, used by gpiv_post_scale */
    float z_off_y;              /* zero offset in y (row) direction, used by gpiv_post_scale */
    char title[GPIV_MAX_CHARS];        /* Project name */
    char creation_date[GPIV_MAX_CHARS];/* Date of origin */
    char location[GPIV_MAX_CHARS];     /* Location of origin */
    char author[GPIV_MAX_CHARS];       /* Author name */
    char software[GPIV_MAX_CHARS];     /* Program that generated the image */
    char source[GPIV_MAX_CHARS];       /* Camera name and type */
    char usertext[GPIV_MAX_CHARS];     /* User comment */
    char warning[GPIV_MAX_CHARS];      /* Date of origin */
    char disclaimer[GPIV_MAX_CHARS];   /* Date of origin */
    char comment[GPIV_MAX_CHARS];      /* Comment from other image formats */

    gboolean depth_logic;
    gboolean ncolumns_logic;
    gboolean nrows_logic;
    gboolean x_corr_logic;
    gboolean s_scale_logic;
    gboolean t_scale_logic;
    gboolean z_off_x_logic;
    gboolean z_off_y_logic;

    gboolean title_logic;
    gboolean creation_date_logic;
    gboolean location_logic;
    gboolean author_logic;
    gboolean software_logic;
    gboolean source_logic;
    gboolean usertext_logic;
    gboolean warning_logic;
    gboolean disclaimer_logic;
    gboolean comment_logic;
};


void
gpiv_img_parameters_logic(GpivImagePar * image_par,
                          gboolean flag
                          );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Set flag for image_par_logic
 *
 * INPUTS:
 *     force:                 flag to enforce parameters set to defaults
 *
 * OUTPUTS:
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/



void
gpiv_img_default_parameters(GpivImagePar * image_par,
                            gboolean force
                            );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Sets default parameter values
 *
 * INPUTS:
 *     force:                 flag to enforce parameters set to defaults
 *
 * OUTPUTS:
 *     piv_img_par_default:   structure of image evaluation parameters
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/



void
gpiv_img_read_header(FILE *fp_par,
		     GpivImagePar * image_par, 
		     gboolean print_par);
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Scans the content of fp_par on image parameters without IMAGE_PAR_KEY
 *
 * INPUTS:
 *
 * OUTPUTS:
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/



void
gpiv_img_fread_header(FILE *fp_par, 
		      FILE *fp_par_out,
		      GpivImagePar * image_par, 
		      gboolean print_par);
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Reads each line of file fp_par and looks for image header parameters 
 *      without IMAGE_PAR_KEY
 *
 * INPUTS:
 *      fp_par_out:     file pointer of which parameter will have to be printed
 *      print_par:      print parameters to fp_par_out (1) or not (0)
 *
 * OUTPUTS:
 *      image_par:      structure of image parameters
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/



char *
gpiv_img_read_pgm_header(char *fname, 
                         GpivImagePar * image_par,
                         gint *line_nr
                         );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Reads each line of the raw portable graymap (pgm) format image file
 *     and retrieves image header parameters
 *     without GPIV_IMAGE_PAR_KEY
 *
 * PROTOTYPE LOCATATION:
 *     img.h
 *
 * INPUTS:
 *     fname:           file name
 *
 * OUTPUTS:
 *     image_par :     parameter structure
 *     line_nr:        line number to start image data
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/


void
gpiv_img_read_parameters(FILE *fp_h, 
			 GpivImagePar * image_par, 
			 gboolean print_par
			 );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Reads each line of file fp_h and looks for image header parameters 
 *      including IMAGE_PAR_KEY
 *
 * INPUTS:
 *      fp_h:           file pointer to image header
 *      print_par       print parameters to fp_par_out (1) or not (0)
 *
 * OUTPUTS:
 *      image_par:
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/



void
gpiv_img_fread_parameters(FILE *fp_h, 
			  FILE *fp_par_out,
			  GpivImagePar * image_par, 
			  gboolean print_par
			  );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Reads each line of the file and looks for image header parameters 
 *      including IMAGE_PAR_KEY
 *
 * INPUTS:
 *
 * OUTPUTS:
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/



char *
gpiv_img_check_header_read(GpivImagePar image_par
                           );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Check out if all image header info has been read
 *
 * INPUTS:
 *
 * OUTPUTS:
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/



char *
gpiv_img_test_header(GpivImagePar image_par);
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Check out if all header parameters have been read
 *
 * INPUTS:
 *
 * OUTPUTS:
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/



void 
gpiv_img_print_header(GpivImagePar image_par
                      );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Print header information without IMAGE_PAR_KEY
 *
 * INPUTS:
 *
 * OUTPUTS:
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/



void
gpiv_img_fprint_pgmheader(FILE * fp_par_out, 
                          GpivImagePar image_par
                          );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     prints image header parameters to PGM image format file
 *     without GPIV_IMAGE_PAR_KEY
 *
 * PROTOTYPE LOCATATION:
 *     img.h
 *
 * INPUTS:
 *     fp_par_out:     file pointer to output file
 *     image_par:      image parameter structure
 *
 * OUTPUTS:
 *---------------------------------------------------------------------------*/



void
gpiv_img_fprint_header(FILE * fp_par_out, 
		       GpivImagePar image_par);
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     prints image header parameters to file
 *     without GPIV_IMAGE_PAR_KEY
 *
 * PROTOTYPE LOCATATION:
 *     img.h
 *
 * INPUTS:
 *     fp_par_out:     file pointer to output file
 *     image_par:      image parameter structure
 *
 * OUTPUTS:
 *---------------------------------------------------------------------------*/



void
gpiv_img_cp_parameters(GpivImagePar image_par_src, 
                       GpivImagePar * image_par_dest, 
                       gboolean force,
                       gboolean print_par
                       );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Copies image parameters from src to dest
 *
 * INPUTS:
 *      piv_eval_par_src:       source piv parameters
 *      force:                  flag to force the copying, even if destination
 *                              already exists
 *      print_par:              verbose output. Not implemented yet.
 *
 * OUTPUTS:
 *      piv_eval_par_dest:     destination piv parameters
 *
 * RETURNS:
 *
 *---------------------------------------------------------------------------*/




void 
gpiv_img_print_parameters(GpivImagePar image_par
			  );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      prints image header parameters to stdout including IMAGE_PAR_KEY
 *
 * INPUTS:
 *
 * OUTPUTS:
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/



void
gpiv_img_fprint_parameters(FILE * fp_par_out, 
                           GpivImagePar image_par
                           );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      prints image header parameters to fp_par_out including IMAGE_PAR_KEY
 *
 * INPUTS:
 *
 * OUTPUTS:
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/



char *
gpiv_img_fread_hdf5_parameters(char * fname, 
			       GpivImagePar * image_par
			       );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Reads image parameters from hdf5 data file without IMAGE_PAR_KEY
 *
 * INPUTS:
 *
 * OUTPUTS:
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/



char *
gpiv_img_fwrite_hdf5_parameters(char *fname,
				GpivImagePar * image_par
				);
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Writes image parameters to an existing hdf5 data file
 *
 * INPUTS:
 *
 * OUTPUTS:
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/



char *
gpiv_img_fread_davis_parameters(char *fname,
                                GpivImagePar *image_par
                                );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Reads image specifications from Davis formatted image, with ext .IMG, 
 *      from file
 *
 * INPUTS:
 *
 * OUTPUTS:
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/



#endif /* __LIBGPIV_IMG_H__ */

