/**
 * @file libgnurdf/schema.h Schema functions
 *
 * $Id: schema.h,v 1.5 2001/09/20 21:23:37 chipx86 Exp $
 *
 * @Copyright (C) 1999-2001 The GNUpdate Project.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#ifndef _LIBGNURDF_SCHEMA_H_
#define _LIBGNURDF_SCHEMA_H_

#ifdef __cplusplus
extern "C" {
#endif

#include <libgnurdf/types.h>

/*************************************************************************/
/** @name Schema (De)allocation Functions                                */
/*************************************************************************/
/*@{*/

/**
 * Creates a new, blank RdfSchema.
 *
 * @return The new RdfSchema.
 */
RdfSchema *rdfNewSchema(void);

/**
 * Destroys an RdfSchema.
 *
 * @param schema The RdfSchema to destroy.
 */
void rdfDestroySchema(RdfSchema *schema);
/*@}*/


/*************************************************************************/
/** @name File Input/Output Functions                                    */
/*************************************************************************/
/*@{*/

/**
 * Reads an RDF file.
 * 
 * @param filename The filename to read in.
 * 
 * @return An RdfSchema containing the parsed document, or NULL on error.
 */
RdfSchema *rdfReadFile(const char *filename);

/**
 * Reads RDF from a buffer.
 *
 * @param buffer The buffer containing the RDF.
 *
 * @return An RdfSchema containing the parsed document, or NULL on error.
 */
RdfSchema *rdfReadBuffer(char *buffer);

/**
 * Writes RDF to a file.
 *
 * @param schema   The document schema to write.
 * @param filename The name of the file to write to.
 */
void rdfWriteFile(RdfSchema *schema, const char *filename);

/**
 * Writes RDF to a buffer.
 *
 * @param schema The document schema to write.
 * @param buffer The buffer to store the RDF into.
 * @param size   The new size of the buffer.
 */
void rdfWriteBuffer(RdfSchema *schema, char **buffer, int *size);

/**
 * Sets the schema's gzip compression level.
 *
 * @param schema The document schema to compress.
 * @param level  The gzip compression level.
 */
void rdfSetCompressionLevel(RdfSchema *schema, int level);

/**
 * Returns the gzip compression level of the schema.
 *
 * @param schema The document schema.
 *
 * @return The gzip compression level.
 */
int rdfGetCompressionLevel(RdfSchema *schema);

/*@}*/


/*************************************************************************/
/** @name Backwards-compatibility functions                              */
/*************************************************************************/
/*@{*/

/**
 * Reads an RDF file.
 *
 * @deprecated This function has been replaced by rdfReadFile()
 *
 * @see rdfReadFile()
 */
RdfSchema *rdfRead(const char *filename);

/**
 * Writes RDF to a file.
 *
 * @deprecated This function has been replaced by rdfWriteFile()
 *
 * @see rdfWriteFile()
 */
void rdfWrite(RdfSchema *schema, const char *filename);

/*@}*/

#ifdef __cplusplus
}
#endif

#endif /* _LIBGNURDF_SCHEMA_H_ */

