/**
 * @file property.c Property functions
 *
 * $Id: property.c,v 1.7 2001/09/20 22:47:28 chipx86 Exp $
 *
 * @Copyright (C) 1999-2001 The GNUpdate Project.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#include <gnurdf.h>
#include <libgnurdf/internal.h>

int
rdfGetProperty(RdfElement *parent, const char *property, RdfNamespace *ns,
			   char **value, RdfElement **element)
{
	RdfElement *list;
	RdfElement *best = NULL;
	int approx = 0;

	if (parent == NULL || property == NULL || ns == NULL)
		return -1;
	
	if (value != NULL)
		*value = NULL;
	
	if (element != NULL)
		*element = NULL;
	
	list = parent->xmlChildrenNode;
	
	while (list != NULL)
	{
		if ((list->ns == ns) && (!strcmp(list->name, property)))
		{
			/* Found a match. Now return it. */
			if ((list->content != NULL) && (list->xmlChildrenNode != NULL))
			{
				fprintf(stderr, "libgnurdf: rdfGetProperty: "
					"RDF Schema invalid, node %s has content and children\n",
					list->name);
			}

			if (value != NULL)
			{
				if (list->content != NULL)
					*value = xmlStrdup(list->content);
				else
					*value = xmlNodeListGetString(rdfGetElementSchema(parent),
												  list->xmlChildrenNode, 1);
			}

			if (element != NULL)
				*element = rdfFirstProperty(list);

			if (!strcmp(list->name, "Bag")) return RDF_BAG;
			if (!strcmp(list->name, "Seq")) return RDF_SEQ;
			if (!strcmp(list->name, "Alt")) return RDF_ALT;
			if (!strcmp(list->name, "Description")) return RDF_DESC;

			return 0;
		}
		else if (((list->ns == NULL) || (ns == NULL)) &&
		         (!strcmp(list->name, property)))
		{
			/* If more than one "best" match exists, ignore them */
			if (approx)
			{
				best = NULL;
			}
			else
			{
				best = list;
				approx = 1;
			}
		}

		list = list->next;
	}

	if (best != NULL)
	{
		if ((best->content != NULL) && (best->xmlChildrenNode != NULL))
		{
			fprintf(stderr, "libgnurdf: rdfGetProperty: "
				"RDF Schema invalid, node %s has content and children\n",
				best->name);
		}

		if (value != NULL)
		{
			if (best->content != NULL)
				*value = xmlStrdup(best->content);
			else
				*value = xmlNodeListGetString(rdfGetElementSchema(parent),
											  best->xmlChildrenNode, 1);
		}
		
		if (element != NULL)
			*element = best->xmlChildrenNode;

		if (!strcmp(best->name, "Bag")) return RDF_BAG;
		if (!strcmp(best->name, "Seq")) return RDF_SEQ;
		if (!strcmp(best->name, "Alt")) return RDF_ALT;
		if (!strcmp(best->name, "Description")) return RDF_DESC;

		return 0;
	}

	return -1;
}

RdfElement *
rdfSetProperty(RdfElement *parent, const char *property, RdfNamespace *ns,
			   const char *value)
{
	RdfElement *list;
	RdfSchema  *schema;
	xmlChar    *tmp;
	xmlNode    *node;

	if (parent == NULL || property == NULL || ns == NULL || value == NULL)
		return NULL;

	schema = rdfGetElementSchema(parent);
	
	list = parent->xmlChildrenNode;

	/*
	 * Search the property.
	 */
	while (list != NULL)
	{
		if ((list->ns == ns) && (!strcmp(list->name, property)))
		{
			/*
			 * Property was found, update it!
			 */
			if ((list->content != NULL) && (list->xmlChildrenNode != NULL))
			{
				fprintf(stderr, "libgnurdf: rdfSetProperty: "
					"RDF Schema invalid, node %s has content and children\n",
					list->name);
			}

			if (list->content != NULL)
				free(list->content);
			
			if (list->xmlChildrenNode != NULL)
				xmlFreeNodeList(list->xmlChildrenNode);
			
			tmp = xmlEncodeEntitiesReentrant(schema, value);

			list->xmlChildrenNode = xmlStringGetNodeList(schema, tmp);
			
			free(tmp);

			return list->xmlChildrenNode;
		}
		
		list = list->next;
	}

	 /*
	  * Create a new property.
	  */
	tmp = xmlEncodeEntitiesReentrant(schema, value);
	node = xmlNewChild(parent, ns, property, tmp);
	free(tmp);

	return node;
}

RdfElement *
rdfSetElement(RdfElement *parent, const char *property, RdfNamespace *ns,
              RdfElement *element)
{
	RdfElement *list;

	if (parent == NULL || property == NULL || ns == NULL || element == NULL)
		return NULL;
	
	list = parent->xmlChildrenNode;
	
	/*
	 * Search the property.
	 */
	while (list != NULL)
	{
		if ((list->ns == ns) && (!strcmp(list->name, property)))
		{
			/*
			 * Property was found, update it!
			 */
			if ((list->content != NULL) && (list->xmlChildrenNode != NULL))
			{
				fprintf(stderr, "libgnurdf: rdfSetElement: "
					"RDF Schema invalid, node %s has content and children\n",
					list->name);
			}
			if (list->content != NULL)
				free(list->content);
			
			if (list->xmlChildrenNode != NULL)
				xmlFreeNodeList(list->xmlChildrenNode);

			list->xmlChildrenNode = xmlCopyNode(element, 1);

			return list->xmlChildrenNode;
		}
		
		list = list->next;
	}

	 /*
	  * Create a new property.
	  */
	xmlAddChild(parent, element);

	return element;
}

void
rdfRemoveProperty(RdfElement *parent, const char *property, RdfNamespace *ns)
{
	RdfElement *list = parent->xmlChildrenNode;
	RdfElement *prev = NULL;

	if (parent == NULL || property == NULL || ns == NULL)
		return;
	
	/*
	 * Search the property
	 */
	while (list != NULL)
	{
		if ((list->ns = ns) && (!strcmp(list->name, property)))
		{
			/*
			 * Property was found, delete it!
			 */
			if (prev == NULL)
				parent->xmlChildrenNode = list->next;
			else
				prev->next = list->next;

			list->next = NULL;

			xmlFreeNode(list);
			
			return;
		}

		prev = list;
		list = list->next;
	}
}

void
rdfRemoveElement(RdfElement *element)
{
	if (element == NULL)
		return;

	if (element->prev == NULL)
		element->parent->xmlChildrenNode = element->next;
	else
		element->prev->next = element->next;

	element->next = NULL;

	xmlFreeNode(element);
}

int
rdfGetValue(RdfDescription *desc, const char *property,
			RdfNamespace *ns, char **value, RdfElement **element)
{
	return rdfGetProperty(desc, property, ns, value, element);
}

RdfElement *
rdfSetValue(RdfDescription *desc, const char *property,
			RdfNamespace *ns, const char *value)
{
	return rdfSetProperty(desc, property, ns, value);
}

