/**
 * @file libgnurdf/namespace.h Namespace functions
 *
 * $Id: namespace.h,v 1.3 2001/09/20 20:00:48 chipx86 Exp $
 *
 * @Copyright (C) 1999-2001 The GNUpdate Project.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#ifndef _LIBGNURDF_NAMESPACE_H_
#define _LIBGNURDF_NAMESPACE_H_

#include <libgnurdf/types.h>

#ifdef __cplusplus
extern "C" {
#endif

/**
 * Creates a new namespace with the specified URI and prefix.
 *
 * For example, to create a namespace such as the one used in the
 * following tag:
 *
 * @code
 * <SomeTag xmlns:myns="http://www.foo.com/myns">
 * @endcode
 *
 * You would do:
 *
 * @code
 * RdfNamespace *ns;
 *
 * ns = rdfNewNamespace(schema, "http://www.foo.com/myns", "myns");
 * @endcode
 * 
 * @param schema The document schema.
 * @param uri    The URI of the namespace.
 * @param prefix The namespace prefix.
 *
 * @return The new namespace.
 */
RdfNamespace *rdfNewNamespace(RdfSchema *schema, const char *uri,
                              const char *prefix);

/**
 * Returns the namespace with the specified URI.
 * 
 * @param schema The document schema.
 * @param uri    The URI of the namespace.
 *
 * @return The namespace, or NULL if not found.
 */
RdfNamespace *rdfGetNamespace(RdfSchema *schema, const char *uri);

/**
 * Returns the RDF namespace.
 *
 * @param schema The document schema.
 *
 * @return The namespace.
 */
RdfNamespace *rdfGetRdfNamespace(RdfSchema *schema);

/**
 * Returns the URI of the specified namespace.
 *
 * @param ns The namespace.
 *
 * @return The namespace's URI.
 */
const char *rdfGetNamespaceURI(RdfNamespace *ns);

/**
 * Returns the prefix of the specified namespace.
 *
 * @param ns The namespace.
 *
 * @return The namespace's prefix.
 */
const char *rdfGetNamespacePrefix(RdfNamespace *ns);

#ifdef __cplusplus
}
#endif

#endif /* _LIBGNURDF_NAMESPACE_H_ */

