/**
 * @file libgnurdf/element.h Element functions
 *
 * $Id: element.h,v 1.3 2001/08/23 08:51:22 chipx86 Exp $
 *
 * @Copyright (C) 1999-2001 The GNUpdate Project.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#ifndef _LIBGNURDF_ELEMENT_H_
#define _LIBGNURDF_ELEMENT_H_

#include <libgnurdf/types.h>

#ifdef __cplusplus
extern "C" {
#endif

/**
 * Returns the type of the specified element.
 *
 * @param element The element.
 * 
 * @return The element type, or 0 if it's an unknown type, or -1 if
 *         \a element is NULL.
 *
 * @see RDF_LEAF, RDF_BAG, RDF_SEQ, RDF_ALT, RDF_DESC
 */
int rdfGetElementType(RdfElement *element);

/**
 * Returns the value of the specified element.
 *
 * Note: The returned value must be freed.
 *
 * @param element The element.
 *
 * @return The element's value.
 */
char *rdfGetElementValue(RdfElement *element);

/**
 * Returns the name of the specified element.
 *
 * @param element The element.
 *
 * @return The element's name.
 */
const char *rdfGetElementPropertyName(RdfElement *element);

/**
 * Returns the namespace of the specified element.
 *
 * @param element The element.
 *
 * @return The element's namespace.
 */
RdfNamespace *rdfGetElementNamespace(RdfElement *element);

/**
 * Sets the value of the specified element.
 *
 * @param element The element.
 * @param value   The new value for the element.
 */
void rdfSetElementValue(RdfElement *element, const char *value);

/**
 * Sets an element's resource.
 *
 * @param element The element.
 * @param uri     The URI of the resource.
 */
void rdfSetElementResource(RdfElement *element, const char *uri);

/**
 * Returns an element's resource.
 *
 * Note: The returned resource must be freed!
 *
 * @param element The element.
 *
 * @return The element's resource.
 */
char *rdfGetElementResource(RdfElement *element);

/**
 * Returns an element's parent schema.
 *
 * @param element The element.
 *
 * @return The element's parent schema.
 */
RdfSchema *rdfGetElementSchema(RdfElement *element);

#ifdef __cplusplus
}
#endif

#endif /* _LIBGNURDF_ELEMENT_H_ */

