/*******************************************************************************************************************************************
 cxml.h
*******************************************************************************************************************************************/

#ifndef __CXML_H__
#define __CXML_H__

#include "cfile.h"
#include "cexception.h"

#include <libxml/tree.h>
#include <libxml/parser.h>
#include <libxml/xmlmemory.h>
#include <libxml/xmlreader.h>

// opaque class introduction, class instances xml serialization
class CSerialized;

// defined class
class CXMLDocument;
class CXMLElement;
class CXMLAttribute;

// template commodity
typedef NServices::TNode   <CXMLElement>       CXMLElementNode;
typedef NServices::TBuffer <CXMLElementNode *> CXMLElementNodes;
typedef NServices::TBuffer <CXMLAttribute>     CXMLAttributes;

//------------------------------------------------------------------------------------------------------------------------------------------
// exceptions declaration
//------------------------------------------------------------------------------------------------------------------------------------------
DECLARE_EXCEPTION (XMLPARSE, 7L);	// erreur de parsing sur document
DECLARE_EXCEPTION (XMLTYPE,  8L);	// erreur de type de document

//------------------------------------------------------------------------------------------------------------------------------------------
// xml file access
//------------------------------------------------------------------------------------------------------------------------------------------
#define XML_READ	1
#define XML_WRITE	2

//------------------------------------------------------------------------------------------------------------------------------------------
// CXMLDocument class
//------------------------------------------------------------------------------------------------------------------------------------------
class CXMLDocument : public CClass
{
	// instanciation section
	public :
		
		CXMLDocument			(const CString &inFileName, const int inMode=XML_READ,
						 const CString &inRequestedRoot=CString()) THROWABLE;
		virtual ~CXMLDocument		();

	// xml document basic functions
	public :

		CXMLElementNode	*		GetRootElement		();
		int				GetMode			() const;

	// xml serialization
	public :

		CXMLDocument &			operator <<		(CSerialized &);
		CXMLDocument &			operator >>		(CSerialized &);
		CXMLDocument &			operator <<		(CSerialized *);
		CXMLDocument &			operator >>		(CSerialized *);

		friend CXMLDocument &		operator <<		(CSerialized &, CXMLDocument &);
		friend CXMLDocument &		operator >>		(CSerialized &, CXMLDocument &);
		friend CXMLDocument &		operator <<		(CSerialized *, CXMLDocument &);
		friend CXMLDocument &		operator >>		(CSerialized *, CXMLDocument &);

	// protected attributes
	protected :

		int				m_XMLMode;
		CString				m_XMLFileName;
		xmlDocPtr			m_XMLDocument;
		xmlNodePtr			m_XMLCurrentNode;
		CXMLElementNode *		m_XMLElementNode;

		static void			BuildXMLNodes		(CXMLElementNode *, xmlNodePtr);

		SECTION_CAPSULE_METACLASS;
};

// metaclass and class tag declaration
DECLARE_CAPSULE_METACLASS ('xmld', CXMLDocument, CClass);

//------------------------------------------------------------------------------------------------------------------------------------------
// CXMLElement class
//------------------------------------------------------------------------------------------------------------------------------------------
class CXMLElement : public CClass
{
	protected :

		CXMLElement			(xmlNodePtr=NULL);

	public :

		CXMLElement			(CXMLElementNode *, const CString &, const CString & =CString());
		virtual ~CXMLElement		();

	public :

		CString				GetName			() const;
		CString				GetValue		() const;
		void				SetValue		(const CString &);

		CXMLAttributes			AddAttribute		(const CString &, const CString &);
		CXMLAttributes			GetAttributes		() const;
		CXMLAttribute			GetAttribute		(const CString &) const;

		CXMLElementNode *		GetXMLElementNode	() const;
		void				SetParentXMLElementNode	(CXMLElementNode *);

	protected :
	
		xmlNodePtr			m_XMLNodePtr;
		CXMLElementNode *		m_XMLElementNode;

		friend class			CXMLDocument;

		SECTION_CAPSULE_METACLASS;
};

// metaclass and class tag declaration
DECLARE_CAPSULE_METACLASS ('xmle', CXMLElement, CClass);

//------------------------------------------------------------------------------------------------------------------------------------------
// CXMLAttribute class
//------------------------------------------------------------------------------------------------------------------------------------------
class CXMLAttribute : public CClass
{
	public :

		CXMLAttribute			(xmlNodePtr=NULL, xmlAttrPtr=NULL);
		virtual ~CXMLAttribute		();

	public :

		CString				GetName			() const;
		CString				GetValue		() const;
		void				SetValue		(const CString &);

	protected :

		xmlNodePtr			m_XMLNodePtr;
		xmlAttrPtr			m_XMLAttrPtr;

		SECTION_CAPSULE_METACLASS;
};

// metaclass and class tag declaration
DECLARE_CAPSULE_METACLASS ('xmla', CXMLAttribute, CClass);

//------------------------------------------------------------------------------------------------------------------------------------------
// inline functions for commodity...
//------------------------------------------------------------------------------------------------------------------------------------------

inline CXMLElement * xml_node_get_element (CXMLElementNode *inNode)
{
	return inNode->GetValue();
}

inline CString xml_node_get_name (CXMLElementNode *inNode)
{ 
	return inNode->GetValue()->GetName(); 
}

inline CString xml_node_get_value (CXMLElementNode *inNode)
{ 
	return inNode->GetValue()->GetValue(); 
}

inline CXMLAttributes xml_node_get_attributes (CXMLElementNode *inNode)
{ 
	return inNode->GetValue()->GetAttributes(); 
}

inline size_t xml_node_get_attributes_number (CXMLElementNode *inNode)
{ 
	return inNode->GetValue()->GetAttributes().GetLength(); 
}

inline CXMLAttribute xml_node_get_attribute (CXMLElementNode *inNode, size_t i) 
{ 
	return *inNode->GetValue()->GetAttributes()[i]; 
}

inline CXMLAttribute xml_node_get_attribute (CXMLElementNode *inNode, CString &inName)
{
	CXMLAttributes inXMLAttributes (::xml_node_get_attributes (inNode));
	for (size_t i=0; i<inXMLAttributes.GetLength(); i++)
		if (inXMLAttributes[i]->GetName() == inName)
			return *inXMLAttributes[i];
	return CXMLAttribute();
}

inline CXMLElementNodes xml_node_get_children (CXMLElementNode *inNode) 
{ 
	return inNode->GetChildren(); 
}

inline size_t xml_node_get_children_number (CXMLElementNode *inNode) 
{ 
	return inNode->GetChildren().GetLength(); 
}

inline CXMLElementNode * xml_node_get_child (CXMLElementNode *inNode, size_t i) 
{ 
	return *inNode->GetChildren()[i]; 
}

inline CXMLElementNode * xml_node_get_child (CXMLElementNode *inNode, CString &inName)
{
	CXMLElementNodes inXMLNodes (::xml_node_get_children (inNode));
	for (size_t i=inXMLNodes.GetLength(); i>0; i--)
		if ((*inXMLNodes[i-1])->GetValue()->GetName() == inName)
			return *inXMLNodes[i-1];
	return NULL;
}

inline CXMLElementNode * xml_node_search (CXMLElementNode *inNode, CString &inName)
{
	if (inNode->GetValue()->GetName() == inName) return inNode;
	CXMLElementNodes inNodes (inNode->GetTree());
	for (size_t i=inNodes.GetLength(), j=0; i>0; i--, j++)
		if ((*inNodes[j])->GetValue()->GetName() == inName) 
			return *inNodes[j];
	return NULL;
}

inline void xml_node_set_parent (CXMLElementNode *inParent, CXMLElementNode *inChild)
{
	inChild -> GetValue() -> SetParentXMLElementNode (inParent);
}

#endif
