/*******************************************************************************************************************************************
 cchunk.h
 Chunk definition, logical data block; sequential read and write.
*******************************************************************************************************************************************/

#ifndef __CCHUNK_H__
#define __CCHUNK_H__

#include <string.h>
#include "nservices.h"

#include "cexception.h"

// chunk potential exception declaration
DECLARE_EXCEPTION (DATATYPE, 6L);

//------------------------------------------------------------------------------------------------------------------------------------------
// chunk handled data types
//------------------------------------------------------------------------------------------------------------------------------------------
typedef enum
{
	BOOL=0,		// boolean
	SINT8,		// 1 byte  signed integrer
	UINT8,		// 1 byte  unsigned integer
	SINT16,		// 2 bytes signed integer
	UINT16,		// 2 bytes unsigned integer
	SINT32,		// 4 bytes signed integer
	UINT32,		// 4 bytes unsigned integer
	SINT64,		// 8 bytes signed integer
	UINT64,		// 8 bytes unsigned integer
	FLOAT32,	// 4 bytes float point
	FLOAT64,	// 8 bytes float point
	STRING,		// null terminated string
	PVOID		// n bytes...
} TData;

// data type list definiton
typedef NServices::TBuffer <TData> TDataList; 

//------------------------------------------------------------------------------------------------------------------------------------------
// CChunk definition
//------------------------------------------------------------------------------------------------------------------------------------------
class CChunk : public CClass
{
	// instanciation section
	public :

		CChunk				();
		CChunk				(void *, const UInt32);
		CChunk				(void *, const UInt32, const TDataList &, const bool inHandleData=true);
		CChunk				(const CChunk &);
		virtual ~CChunk			();

	// general functions
	public :

		void *				GetChunk		() const;
		UInt32				GetSize			() const;
	
		TData				NextDataIs		() const;
		TDataList			GetDataList		() const;

	// sequential read and write
	public :

		// sequential write
		virtual void			WriteBool		(const Bool);
		virtual void			WriteSInt8		(const SInt8);
		virtual void			WriteUInt8		(const UInt8);
		virtual void			WriteSInt16		(const SInt16);
		virtual void			WriteUInt16		(const UInt16);
		virtual void			WriteSInt32		(const SInt32);
		virtual void			WriteUInt32		(const UInt32);
		virtual void			WriteSInt64		(const SInt64);
		virtual void			WriteUInt64		(const UInt64);
		virtual void			WriteFloat32		(const Float32);
		virtual void			WriteFloat64		(const Float64);
		virtual void			WriteString		(const SInt8 *);
		virtual void			WritePVoid		(const void *, const UInt32);

		// sequential read
		virtual bool			ReadBool		(Bool &);
		virtual bool			ReadSInt8		(SInt8 &);
		virtual bool			ReadUInt8		(UInt8 &);
		virtual bool			ReadSInt16		(SInt16 &);
		virtual bool			ReadUInt16		(UInt16 &);
		virtual bool			ReadSInt32		(SInt32 &);
		virtual bool			ReadUInt32		(UInt32 &);
		virtual bool			ReadSInt64		(SInt64 &);
		virtual	bool			ReadUInt64		(UInt64 &);
		virtual bool			ReadFloat32		(Float32 &);
		virtual bool			ReadFloat64		(Float64 &);
		virtual bool			ReadString		(SInt8 *&);
		virtual bool			ReadPVoid		(void *&, UInt32 * =NULL);

	// operators
	public :

		// forward, rewind on data
		CChunk &			operator ++		();
		CChunk &			operator ++		(int);
		CChunk &			operator -- 		();
		CChunk &			operator --		(int);

		// sequential write
		CChunk &			operator <<		(const SInt8);
		CChunk &			operator <<		(const UInt8);
		CChunk &			operator <<		(const SInt16);
		CChunk &			operator <<		(const UInt16);
		CChunk &			operator <<		(const SInt32);
		CChunk &			operator <<		(const UInt32);
		CChunk &			operator <<		(const SInt64);
		CChunk &			operator <<		(const UInt64);
		CChunk &			operator <<		(const Float32);
		CChunk &			operator <<		(const Float64);
		CChunk &			operator <<		(const SInt8 *);

		// sequential read
		CChunk &			operator >>		(SInt8 &);
		CChunk &			operator >>		(UInt8 &);
		CChunk &			operator >>		(SInt16 &);
		CChunk &			operator >>		(UInt16 &);
		CChunk &			operator >>		(SInt32 &);
		CChunk &			operator >>		(UInt32 &);
		CChunk &			operator >>		(SInt64 &);
		CChunk &			operator >>		(UInt64 &);
		CChunk &			operator >>		(Float32 &);
		CChunk &			operator >>		(Float64 &);
		CChunk &			operator >>		(SInt8 *&);
		
	// protected section
	protected :

		static bool			ReallocateChunk		(CChunk &, const SInt32 inDelta);

		void *				m_Chunk;
		UInt32				m_Size;
		UInt32				m_Offset;
		UInt32				m_iData;
		TDataList			m_DataList;
		bool				m_HandleData;

		// metaclass section
		SECTION_CAPSULE_METACLASS;
};

// metaclass and class tag declaration
DECLARE_CAPSULE_METACLASS ('cchk', CChunk, CClass);

#endif
