/* $Id: types.h 127 2006-12-21 23:16:09Z mmmaddd $ */

/*
    libg3d - 3D object loading library

    Copyright (C) 2005, 2006  Markus Dahms <mad@automagically.de>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#ifndef __G3D_TYPES_H__
#define __G3D_TYPES_H__

#include <glib.h>

#include <g3d/config.h>

#ifdef USE_LIBMAGIC
#include <magic.h>
#endif

G_BEGIN_DECLS

/*****************************************************************************
 * G3DImage
 *****************************************************************************/

#define G3D_FLAG_IMG_GREYSCALE       (1L << 1)

typedef enum {
	G3D_TEXENV_UNSPECIFIED = 0,
	G3D_TEXENV_BLEND,
	G3D_TEXENV_DECAL,
	G3D_TEXENV_MODULATE,
	G3D_TEXENV_REPLACE
} G3DTexEnv;

/**
 * G3DImage:
 * @name: name of image
 * @width: width of image in pixels
 * @height: height of image in pixels
 * @depth: depth of image in bits
 * @flags: flags
 * @pixeldata: the binary image data
 * @tex_id: the OpenGL texture id, should be unique model-wide
 * @tex_env: texture environment flags
 * @tex_scale_u: factor scaling texture width, should be 1.0 for most cases
 * @tex_scale_v: factor scaling texture height, should be 1.0 for most cases
 *
 * Object containing a two-dimensional pixel image.
 */

typedef struct {
	gchar *name;
	guint32 width;
	guint32 height;
	guint8 depth;
	guint32 flags;
	guint8 *pixeldata;

	guint32 tex_id;
	G3DTexEnv tex_env;
	gfloat tex_scale_u;
	gfloat tex_scale_v;
} G3DImage;

/*****************************************************************************
 * G3DMaterial
 *****************************************************************************/

#define G3D_FLAG_MAT_TWOSIDE    (1L << 0)

/**
 * G3DMaterial:
 * @name: name of material
 * @r: red component of color
 * @g: green component of color
 * @b: blue component of color
 * @a: alpha component of color
 * @shininess: shiny color
 * @specular: specular color
 * @flags: flags
 * @tex_image: texture image (optional, may be NULL)
 *
 * A material object.
 */

typedef struct {
	gchar *name;
	gfloat r, g, b, a;
	gfloat shininess;
	gfloat specular[4];
	guint32 flags;

	G3DImage *tex_image;
} G3DMaterial;

/*****************************************************************************
 * G3DFace
 *****************************************************************************/

#define G3D_FLAG_FAC_NORMALS    (1L << 0)
#define G3D_FLAG_FAC_TEXMAP     (1L << 1)

/**
 * G3DFace:
 * @vertex_count: number of vertices
 * @vertex_indices: indices of vertices in #G3DObject
 * @material: material to use for surface
 * @flags: flags
 * @normals: optional normal array (one vector - 3 #gfloat values - for each
 *   vertex)
 * @tex_image: optional texture image
 * @tex_vertex_count: number of texture vertices, should be 0 or match
 *   vertex_count
 * @tex_vertex_data: array of texture vertices
 *
 * An object representing a surface.
 */

typedef struct {
	guint32 vertex_count;
	guint32 *vertex_indices;

	G3DMaterial *material;

	guint32 flags;

	gfloat *normals;

	G3DImage *tex_image;
	guint32 tex_vertex_count;
	gfloat *tex_vertex_data;
} G3DFace;

/*****************************************************************************
 * G3DTransformation
 *****************************************************************************/

/**
 * G3DTransformation:
 *
 * @matrix: the transformation matrix
 * @flags: flags
 *
 * A three-dimensional matrix transformation object.
 */

typedef struct {
	gfloat matrix[16];
	guint32 flags;
} G3DTransformation;


/*****************************************************************************
 * G3DObject
 *****************************************************************************/

/**
 * G3DObject:
 * @name: name of object
 * @materials: list of materials
 * @faces: list of faces
 * @objects: list of sub-objects
 * @transformation: optional transformation
 * @hide: flag to disable object rendering
 * @vertex_count: number of vertices
 * @vertex_data: vertex vector data
 *
 * A three-dimensional object.
 */

typedef struct {
	gchar *name;

	GSList *materials;
	GSList *faces;
	GSList *objects;

	/* transformation, may be NULL */
	G3DTransformation *transformation;

	/* don't render this object */
	gboolean hide;

	/* vertices */
	guint32 vertex_count;
	gfloat *vertex_data;

	/*< private >*/
	/* FIXME: texture stuff: temporary storage, should be in G3DFace items */
	guint32 tex_vertex_count;
	gfloat *tex_vertex_data;
	G3DImage *tex_image;

	/*< private >*/
	/* some fields to speed up rendering, should not be used by plugins */
	/* FIXME: remove from API (replace with user_data pointer?) */
	gfloat *_normals;
	G3DMaterial **_materials;
	guint32  _num_faces;
	guint32 *_indices;
	guint32 *_flags;
	guint32 *_tex_images;
	gfloat *_tex_coords;
} G3DObject;

/*****************************************************************************
 * G3DContext
 *****************************************************************************/

typedef gboolean (* G3DSetBgColorFunc)(gfloat r, gfloat g, gfloat b, gfloat a,
	gpointer user_data);

typedef gboolean (* G3DUpdateInterfaceFunc)(gpointer user_data);

typedef gboolean (* G3DUpdateProgressBarFunc)(gfloat percentage,
	gboolean show, gpointer user_data);

#ifdef USE_LIBMAGIC
#define MAGIC_PTR_TYPE magic_t
#else
#define MAGIC_PTR_TYPE void *
#endif

/**
 * G3DContext:
 *
 * Internal stuff for libg3d.
 */

typedef struct {
	/*< private >*/
	GSList *plugins;
	MAGIC_PTR_TYPE magic_cookie;

	GHashTable *exts_import;
	GHashTable *exts_image;

	G3DSetBgColorFunc set_bgcolor_func;
	gpointer set_bgcolor_data;
	G3DUpdateInterfaceFunc update_interface_func;
	gpointer update_interface_data;
	G3DUpdateProgressBarFunc update_progress_bar_func;
	gpointer update_progress_bar_data;
} G3DContext;

/*****************************************************************************
 * G3DModel
 *****************************************************************************/

typedef struct {
	gchar *filename;
	G3DContext *context;

	GSList *materials;
	GSList *objects;

	GHashTable *tex_images;
} G3DModel;

G_END_DECLS

#endif /* __G3D_TYPES_H__ */
