/**
 * ===========================================
 * LibFonts : a free Java font reading library
 * ===========================================
 *
 * Project Info:  http://reporting.pentaho.org/libfonts/
 *
 * (C) Copyright 2006-2007, by Pentaho Corporation and Contributors.
 *
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation;
 * either version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * [Java is a trademark or registered trademark of Sun Microsystems, Inc.
 * in the United States and other countries.]
 *
 * ------------
 * $Id: BaseFontFontMetrics.java 3523 2007-10-16 11:03:09Z tmorgner $
 * ------------
 * (C) Copyright 2006-2007, by Pentaho Corporation.
 */

package org.jfree.fonts.itext;

import java.util.Arrays;

import com.lowagie.text.pdf.BaseFont;
import org.jfree.fonts.encoding.CodePointUtilities;
import org.jfree.fonts.registry.BaselineInfo;
import org.jfree.fonts.registry.FontMetrics;
import org.jfree.fonts.LibFontsDefaults;
import org.jfree.fonts.tools.StrictGeomUtility;

/**
 * Creation-Date: 22.07.2007, 19:04:00
 *
 * @author Thomas Morgner
 */
public class BaseFontFontMetrics implements FontMetrics
{
  private BaseFont baseFont;
  private float size;
  private long xHeight;
  private char[] cpBuffer;
  private long[] cachedWidths;
  private BaselineInfo[] cachedBaselines;
  private long ascent;
  private long descent;
  private long leading;
  private long sizeScaled;
  private long italicsAngle;
  private long maxAscent;
  private long maxDescent;
  private long maxCharAdvance;

  public BaseFontFontMetrics(final BaseFont baseFont, final float size)
  {
    if (baseFont == null)
    {
      throw new NullPointerException("BaseFont is invalid.");
    }

    this.baseFont = baseFont;
    this.size = size;
    this.cpBuffer = new char[4];
    this.cachedBaselines = new BaselineInfo[256 - 32];
    this.cachedWidths = new long[256 - 32];
    Arrays.fill(cachedWidths, -1);

    sizeScaled = StrictGeomUtility.toInternalValue(size);

    this.ascent = (long) baseFont.getFontDescriptor(BaseFont.AWT_ASCENT, sizeScaled);
    this.descent = (long) -baseFont.getFontDescriptor(BaseFont.AWT_DESCENT, sizeScaled);
    this.leading = (long) baseFont.getFontDescriptor(BaseFont.AWT_LEADING, sizeScaled);
    italicsAngle = StrictGeomUtility.toInternalValue(baseFont.getFontDescriptor(BaseFont.ITALICANGLE, size));
    maxAscent = (long) baseFont.getFontDescriptor(BaseFont.BBOXURY, sizeScaled);
    maxDescent = (long) -baseFont.getFontDescriptor(BaseFont.BBOXLLY, sizeScaled);
    maxCharAdvance = (long) baseFont.getFontDescriptor(BaseFont.AWT_MAXADVANCE, sizeScaled);

    final int[] charBBox = this.baseFont.getCharBBox('x');
    if (charBBox != null)
    {
      this.xHeight = (long) (charBBox[3] * size);
    }
    if (this.xHeight == 0)
    {
      this.xHeight = getAscent() / 2;
    }
  }

  public long getAscent()
  {
    return ascent;
  }

  public long getDescent()
  {
    return descent;
  }

  public long getLeading()
  {
    return leading;
  }

  public long getXHeight()
  {
    return xHeight;
  }

  public long getOverlinePosition()
  {
    return getLeading() - Math.max (1000, sizeScaled / 20);
  }

  public long getUnderlinePosition()
  {
    return (getLeading() + getMaxAscent()) +  Math.max (1000, sizeScaled / 20);
  }

  public long getStrikeThroughPosition()
  {
    return getMaxAscent() - (long) (LibFontsDefaults.DEFAULT_STRIKETHROUGH_POSITION * getXHeight());
  }

  public long getItalicAngle()
  {
    return italicsAngle;
  }

  public long getMaxAscent()
  {
    return maxAscent;
  }

  public long getMaxDescent()
  {
    return maxDescent;
  }

  public long getMaxHeight()
  {
    return getMaxAscent() + getMaxDescent() + getLeading();
  }

  public long getMaxCharAdvance()
  {
    return maxCharAdvance;
  }

  public long getCharWidth(final int character)
  {
    if (character >= 32 && character < 256)
    {
      // can be cached ..
      final int index = character - 32;
      final long cachedWidth = cachedWidths[index];
      if (cachedWidth >= 0)
      {
        return cachedWidth;
      }

      final int retval = CodePointUtilities.toChars(character, cpBuffer, 0);

      if (retval == 1)
      {
        final char char1 = cpBuffer[0];
        if (char1 < 128 || (char1 >= 160 && char1 <= 255))
        {
          final long width = (long) (baseFont.getWidth(char1) * size);
          cachedWidths[index] = width;
          return width;
        }
      }
      else if (retval < 1)
      {
        cachedWidths[index] = 0;
        return 0;
      }

      final long width = (long) (baseFont.getWidth(new String(cpBuffer, 0, retval)) * size);
      cachedWidths[index] = width;
      return width;
    }

    final int retval = CodePointUtilities.toChars(character, cpBuffer, 0);
    if (retval == 1)
    {
      final char char1 = cpBuffer[0];
      if (char1 < 128 || (char1 >= 160 && char1 <= 255))
      {
        return (long) (baseFont.getWidth(char1) * size);
      }
    }
    else if (retval < 1)
    {
      return 0;
    }

    return (long) (baseFont.getWidth(new String(cpBuffer, 0, retval)) * size);
  }

  public long getKerning(final int previous, final int codePoint)
  {
    return (long) (size * baseFont.getKerning((char) previous, (char) codePoint));
  }

  public BaselineInfo getBaselines(final int c, BaselineInfo info)
  {
    final boolean cacheable = (c >= 32 && c < 256);
    if (cacheable)
    {
      final BaselineInfo fromCache = cachedBaselines[c - 32];
      if (fromCache != null)
      {
        if (info == null)
        {
          info = new BaselineInfo();
        }
        info.update(fromCache);
        return info;
      }
    }

    cpBuffer[0] = (char) (c & 0xFFFF);

    if (info == null)
    {
      info = new BaselineInfo();
    }

    // If we had more data, we could surely create something better. Well, this has to be enough ..
    final long maxAscent = getMaxAscent();
    info.setBaseline(BaselineInfo.MATHEMATICAL, maxAscent - getXHeight());
    info.setBaseline(BaselineInfo.IDEOGRAPHIC, getMaxHeight());
    info.setBaseline(BaselineInfo.MIDDLE, maxAscent / 2);
    info.setBaseline(BaselineInfo.ALPHABETIC, maxAscent);
    info.setBaseline(BaselineInfo.CENTRAL, maxAscent / 2);
    info.setBaseline(BaselineInfo.HANGING, maxAscent - getXHeight());
    info.setDominantBaseline(BaselineInfo.ALPHABETIC);

    if (cacheable)
    {
      final BaselineInfo cached = new BaselineInfo();
      cached.update(info);
      cachedBaselines[c - 32] = cached;
    }

    return info;

  }
}
