/* Copyright (C) 1999 Hans Petter K. Jansson
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 * You can contact the library's author by sending e-mail to <hpj@styx.net>.
 */

#include "config.h"
#include "flux.h"

#include <stdio.h>

#include "system.h"
#include "xmlparser.h"


/* Move to .h */

enum
{
  XML_OK = 0, XML_ERR_PARSE, XML_ERR_DTD, XML_ERR_VALIDATE, XML_ERR_NOSOURCE,
  XML_ERR_MAX
};

char *xml_error_str[5] =
{
  "parsing succeeded",
  "parsing error",
  "datatype definition error",
  "validation error",
  "input source unknown"
};

int _xml_error;


#ifdef UNFINISHED

void _xml_to_ttree_warning_cb(XBit bit_warn, void *arg)
{
  int line_num, char_num;
  InputSource s;

  if (p->state == PS_validate_dtd || p->state == PS_validate_final)
  {
    for (s = p->source; s->parent; s = s->parent) ;  /* Upper source */
    
    fprintf(stderr, "End of %s: ", p->state == PS_validate_final ?
            "body" : "prolog");
  }
  else
  {
    switch(SourceLineAndChar(s, &line_num, &char_num))
    {
      case 1:
        fprintf(stderr, "%d,%d,%s: ", line_num, char_num,
                s->entity->name ? s->entity->name : "unnamed");
        break;
      case 0:
        fprintf(stderr, "%d,%d,%s (definition): ", line_num, char_num,
                s->entity->name ? s->entity->name : "unnamed");
        break;
      case -1:
        fprintf(stderr, "%s (definition): ",
                s->entity->name ? s->entity->name : "unnamed");
        break;
    }
  }
  
  
  
}

#endif


MT *_xml_to_ttree_r(Parser parser, XBit bit_in)
{
  TT *tt = 0, *tt_child;
  Attribute a;
  int i;


  /* Catch errors, datatype information and end tags */

  if (bit_in->type == XBIT_end) return(tt);
  else if (bit_in->type == XBIT_error)
  {
    _xml_error = XML_ERR_PARSE;
    return(0);
  }
  else if (bit_in->type == XBIT_dtd)
  {    
#if 0
    /* The parser was instructed to load datatype information on
     * the fly. */

    XBit bit_temp;

    bit_temp = ParseDtd(parser, parser->dtd->internal_part);
    if (bit_temp->type == XBIT_error)
    {
      _xml_error = XML_ERR_DTD;
      return(0);
    }

    bit_temp = ParseDtd(parser, parser->dtd->external_part);
    if (bit_temp->type == XBIT_error)
    {
      _xml_error = XML_ERR_DTD;
      return(0);
    }
#endif
  }
  else if (bit_in->type == XBIT_start)
  {
    /* Add node for spanning tag */
    
    tt = mt_new(MT_SPAN);
    mt_set_name_str(tt, (char *) bit_in->element_definition->name);

    /* Add attributes */

    if (bit_in->attributes)
    {
      tt = tt_new_with_parent_and_data(tt, "attr", 4);

      for (a = bit_in->attributes; a; a = a->next)
      {
        tt = tt_new_with_parent_and_data(tt, (char *) a->definition->name,
                                               strlen(a->definition->name));
        tt_new_with_parent_and_data(tt, a->value, strlen(a->value));
        tt = tt_get_parent(tt);
      }

      tt = tt_get_parent(tt);
    }

    /* Process children */

    for (i = 0; i < bit_in->nchildren; i++)
    {
      tt_child = _xml_to_ttree_r(parser, bit_in->children[i]);
      if (tt_child) mt_add_as_last_child(tt, tt_child);
      else if (_xml_error != XML_OK)
      {
        tt_del(tt);
        return(0);
      }
    }
  }
  else if (bit_in->type == XBIT_empty)
  {
    /* Add node for empty tag */

    tt = mt_new(MT_EMPTY);
    mt_set_name_str(tt, (char *) bit_in->element_definition->name);

    /* Add attributes */

    if (bit_in->attributes)
    {
      tt = tt_new_with_parent_and_data(tt, "attr", 4);

      for (a = bit_in->attributes; a; a = a->next)
      {
        tt = tt_new_with_parent_and_data(tt, (char *) a->definition->name,
                                               strlen(a->definition->name));
        tt_new_with_parent_and_data(tt, a->value, strlen(a->value));
        tt = tt_get_parent(tt);
      }
      
      tt = tt_get_parent(tt);
    }
  }
  else if (bit_in->type == XBIT_cdsect)
  {
    /* We do canonical XML; CDATA equals PCDATA. */

    tt = mt_new(MT_DATA);
    mt_data_set_str(tt, bit_in->cdsect_chars);
  }
  else if (bit_in->type == XBIT_pcdata)
  {
    /* Add node for current data */
    
    tt = mt_new(MT_DATA);
    mt_data_set_str(tt, bit_in->pcdata_chars);
  }
#if 0  /* Let's not conform. */
  else if (bit_in->type == XBIT_pi)
  {
    /* Processing instruction */

    tt = ttree_node_add(tt, "pi", 2);
    tt = ttree_node_add(tt, "name", 4);
    ttree_node_add(tt, bit_in->pi_name, strlen(bit_in->pi_name));
    tt = ttree_node_add(tt->parent, "data", 4);
    ttree_node_add(tt, bit_in->pi_chars, strlen(bit_in->pi_chars));
    tt = tt->parent->parent;
  }
#endif
#if 0  /* I've stopped caring about comments. For now. */
  else if (bit_in->type == XBIT_comment)
  {
    /* Add node for comment */
    
    tt = ttree_node_add(tt, "comment", 7);
    ttree_node_add(tt, bit_in->comment_chars, strlen(bit_in->comment_chars));
    tt = tt->parent;
  }
#endif
  
  return(tt);
}


#define XML_LOG_WARN (1 << 0)
#define XML_LOG_ERR  (1 << 1)
#define XML_STRICT   (1 << 2)

MT *mt_scan_from_xml_file(FILE *in, int validate)
{
  TT *node_root = 0;
  Parser parser;
  InputSource source;
  XBit doc_parsed;

  _xml_error = XML_OK;

  /* Setup parser/validator */

  parser = NewParser();
  ParserSetFlag(parser, ReturnDefaultedAttributes, 1);
  ParserSetFlag(parser, MergePCData, 1);
  ParserSetFlag(parser, Validate, validate);
/*
  ParserSetWarningCallback(parser, _xml_to_ttree_warning_cb);
  ParserSetFlag(parser, ErrorOnBadCharacterEntities, 1);
  ParserSetFlag(parser, ErrorOnUndefinedEntities, 1);
  ParserSetFlag(parser, ErrorOnUndefinedElements, 1);
  ParserSetFlag(parser, ErrorOnUndefinedAttributes, 1);
  ParserSetFlag(parser, ErrorOnValidityErrors, 1);
*/
  /* Load, parse and validate source */

  source = SourceFromStream("xml_input", in);
  if (source)
  {
    if (ParserPush(parser, source) != -1)
    {
      for (;;)
      {
        doc_parsed = ReadXTree(parser);

        if (doc_parsed->type == XBIT_error)
        {
          _xml_error = XML_ERR_VALIDATE;
        }
        else if (doc_parsed->type == XBIT_eof)
        {
          break;
        }
        else
        {
          /* Translate to ttree format */

          if (!(node_root =_xml_to_ttree_r(parser, doc_parsed)))
          {
            fprintf(stderr, "xml_f_to_t(): Error: _xml_to_ttree_r() failed.\n");
            tt_del(node_root);
            return(0);
          }
        
          FreeXTree(doc_parsed);
        }
      }
    }
    else _xml_error = XML_ERR_NOSOURCE;
  }
  else _xml_error = XML_ERR_NOSOURCE;

  FreeDtd(parser->dtd);
  FreeParser(parser);
  return(node_root);
}
