/* Copyright (C) 1999 Hans Petter K. Jansson
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 * You can contact the library's author by sending e-mail to <hpj@styx.net>.
 */

#include "config.h"
#include "flux.h"

#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/mman.h>


/* Communications protocol, V1
   ---------------------------

   1. Block header:

   aaaabbbbccdd...
   |   |   | |
   |   |   | |
   |   |   | |
   |   |   | `---- (word) flags.
   |   |   `------ (word) transaction ID, across several blocks.
   |   `---------- (long) serial ID, uniquifies block.
   `-------------- (long) total data length.

   All in network byte order.

   ---

   2a. Block data: organization.

   Block data is hierarchical.
  
   root
   |
   +-string
   | |
   | +-string
   | |
   | +-string
   | | |
   | | +-string
   | |
   | `-string
   |
   `-string

   The protocol does not differentiate between keyword strings and
   data strings - that is left up to the application. Example tree:

   root
   |
   `-message
     |
     +-date
     | |
     | `-19971015
     |
     `-subject
       |
       `-Test

   Another example:

   root
   |
   +-etc
   | |
   | +-type
   | | |
   | | `-dir
   | |
   | `-subs
   |   |
   |   +-sysconfig
   |   | |
   |   | `-type
   |   |   |
   |   |   `-dir
   |   |
   |   `-rc.d
   |     |
   |     `-type
   |       |
   |       `-dir
   |
   `-vmlinuz
     |
     `-type
       |
       `-file

   This abstraction allows both great freedom, flexibility, and
   robustness - also when it comes to backwards-compatible server/client
   upgrades or changes in data representaton.

   ---
  
   2b. Block data: transmission.

   The following sequence is repeated for each string (node) in the
   tree, including the root node:
  
   aaaabbbbc...
   |   |   |
   |   |   |
   |   |   `- (<aaaa> bytes) string.
   |   `--- (long) number of substrings (children).
   `------- (long) string length.

   All in network byte order. Note that there can only be one node at
   root level, and that the term "string" used here can be misleading -
   it's just a block of 8-bit bytes, and not null-terminated.
*/


TT *comm_get_next_node_in(COMM *c)
{
  TT *tt0;

  switch (c->order_in)
  {
    case COMM_ORDER_BREADTH_FIRST:
      tt0 = tt_get_next_in_breadth_with_level(c->node_in, c->block_in_level,
                                              c->block_in_level);
      if (!tt0)
      {
        c->block_in_level++;
        tt0 = tt_get_next_in_breadth_with_level(c->root_in, c->block_in_level, 0);
      }
      break;

    default:
    case COMM_ORDER_DEPTH_FIRST:
      tt0 = tt_get_next_infix(c->node_in, c->root_in);
      break;
  }

  return(tt0);
}


TT *comm_get_next_node_out(COMM *c)
{
  TT *tt0;

  switch (c->order_out)
  {
    case COMM_ORDER_BREADTH_FIRST:
      tt0 = tt_get_next_in_breadth_with_level(c->node_out, c->block_out_level,
                                              c->block_out_level);
      if (!tt0)
      {
        c->block_out_level++;
        tt0 = tt_get_next_in_breadth_with_level(c->root_out, c->block_out_level, 0);
      }
      break;

    default:
    case COMM_ORDER_DEPTH_FIRST:
      tt0 = tt_get_next_infix(c->node_out, c->root_out);
      break;
  }

  return(tt0);
}


int comm_pull(COMM *c, TT **ret_node)
{
  int ret = 0;
  int finished = 0;
  void *p;
  int size;
  int i;
  int data_size_expect;

  struct comm_node_header cnh;
  struct comm_block_header cbh;

#if 0  
  printf("Pulling.\n");
#endif
  
  _sock_feed(c->s);

  /* This is what we're expecting next */

  while (!finished)
  {
    switch (c->state_in)
    {
      case COMM_STATE_BLOCK_HEADER:
        if (sock_buf_in_size(c->s) >= sizeof(struct comm_block_header))
        {
          /* --- Get the header and set up block --- */
#if 0
          printf("Pulling block header.\n");
#endif
          c->root_in = c->node_in = tt_new();  /* Make root node */
          fifobuf_dequeue(c->s->fib_in, &cbh, sizeof(cbh));

          c->block_in_size = c->block_in_bytes_left = ntohl(cbh.bytes);
          c->trans_in = ntohs(cbh.transaction);
          cbh.flags = ntohs(cbh.flags);
          c->order_in = cbh.flags & COMM_ORDER_BREADTH_FIRST;
          c->block_in_level = 0;

          c->state_in = COMM_STATE_NODE_HEADER;
        }
        else finished = 1;
        break;

      case COMM_STATE_NODE_HEADER:
        if (sock_buf_in_size(c->s) >= sizeof(struct comm_node_header))
        {
          /* --- Get the header and set up node --- */
#if 0
          printf("Pulling node header.\n");
#endif          
          fifobuf_dequeue(c->s->fib_in, &cnh, sizeof(cnh));
          c->node_in_bytes_left = ntohl(cnh.size);
          cnh.children = ntohl(cnh.children);

          /* Make children nodes, to be filled with data later */
          for (i = 0; i < cnh.children; i++) tt_add(c->node_in, tt_new());

          c->state_in = COMM_STATE_NODE_DATA;
        }
        else finished = 1;
        break;

      case COMM_STATE_NODE_DATA:
        /* Is it enough that we care? */
        data_size_expect = c->node_in_bytes_left < 4096 ? c->node_in_bytes_left : 4096;

        if (sock_buf_in_size(c->s) >= data_size_expect)
        {
          /* --- Shuffle data chunk --- */

          printf("Pulling node data.\n");
          
          /* Grab as much as possible */
          data_size_expect = c->node_in_bytes_left < sock_buf_in_size(c->s) ?
                             c->node_in_bytes_left : sock_buf_in_size(c->s);

          FIFOBUF_FOR_DATA(c->s->fib_in, data_size_expect, p, size)
          {
            tt_data_append_bytes(c->node_in, p, size);
            c->node_in_bytes_left -= size;
            c->block_in_bytes_left -= size;
          }
          
          /* It's transferred, so drop it from socket buffers */
          fifobuf_drop(c->s->fib_in, data_size_expect);

          /* There could be more nodes waiting */
          if (c->node_in_bytes_left) finished = 1;
          else
          {
            /* Proceed immediately to next node (or block) */
            ret++;
            if (ret_node) *ret_node = c->node_in;
            c->node_in = comm_get_next_node_in(c);
            if (c->node_in) c->state_in = COMM_STATE_NODE_HEADER;
            else
            {
              /* Queue it */
              comm_enqueue_to(c, c->queue_in, c->root_in, c->trans_in, 0);
              c->root_in = 0;

              /* Block is finished; expect next */
              c->state_in = COMM_STATE_BLOCK_HEADER;
            }
          }
        }
        break;

      case COMM_STATE_NONE:
      default:
        /* This could mean the comm is uninitialized; deal with it later */
        c->state_in = COMM_STATE_BLOCK_HEADER;
        break;
    }
    
    if (ret_node && ret) break;
  }
  
  return(ret);
}


TT *comm_pull_one(COMM *c)
{
  TT *tt;
  
  tt = 0;
  comm_pull(c, &tt);
  return(tt);
}


int comm_push(COMM *c)
{
  int ret = 0;
  int finished = 0;
  int data_size_expect;
  unsigned char data[2048];

  struct comm_node_header cnh;
  struct comm_block_header cbh;

  _sock_spill(c->s);

  /* This is what we're expecting next */

  while (!finished)
  {
    switch (c->state_out)
    {
      case COMM_STATE_BLOCK_HEADER:
        if (sock_buf_out_size(c->s) < 4096 - sizeof(struct comm_block_header))
        {
          /* --- Send the block header --- */

          /* Select next tree from the out queue */
          c->root_out = c->node_out =
            comm_dequeue_from(c, c->queue_out, &(c->trans_out), &(cbh.flags));
          if (!c->root_out) finished = 1;
          else
          {
            cbh.bytes = c->block_out_size = c->block_out_bytes_left =
              tt_size_children_all(c->root_out) + tt_size(c->root_out);
            cbh.bytes = htonl(cbh.bytes);

            cbh.serial = htonl(0);

            cbh.transaction = htons(c->trans_out);

            c->order_out = cbh.flags & COMM_ORDER_BREADTH_FIRST;
            cbh.flags = htons(cbh.flags);

            fifobuf_enqueue(c->s->fib_out, &cbh, sizeof(cbh));
            c->block_out_level = 0;
            c->state_out = COMM_STATE_NODE_HEADER;
          }
        }
        else finished = 1;
        break;

      case COMM_STATE_NODE_HEADER:
        if (sock_buf_out_size(c->s) < 4096 - sizeof(struct comm_node_header))
        {
          /* --- Send the node header --- */

          c->node_out_bytes_left = tt_size(c->node_out);
          cnh.size = htonl(c->node_out_bytes_left);
          cnh.children = htonl(tt_count_children(c->node_out));
          fifobuf_enqueue(c->s->fib_out, &cnh, sizeof(cnh));

          c->state_out = COMM_STATE_NODE_DATA;
        }
        else finished = 1;
        break;

      case COMM_STATE_NODE_DATA:
        if (sock_buf_out_size(c->s) < 3072)  /* Do at least 1024 bytes */
        {
          /* --- Shuffle data chunk --- */

          /* Don't do too much at once */
          data_size_expect = c->node_out_bytes_left < 1024 ? c->node_out_bytes_left : 1024;

          tt_data_get_bytes(c->node_out, data,
                            tt_size(c->node_out) - c->node_out_bytes_left,
                            data_size_expect);
          fifobuf_enqueue(c->s->fib_out, data, data_size_expect);
          c->node_out_bytes_left -= data_size_expect;

          /* There could be more nodes waiting */
          if (c->node_out_bytes_left) finished = 1;
          else
          {
            /* Proceed immediately to next node (or block) */
            ret++;
            c->node_out = comm_get_next_node_out(c);
            if (c->node_out) c->state_out = COMM_STATE_NODE_HEADER;
            else
            {
              /* Block is finished; expect next */
              if (c->root_out)
              {
                tt_del(c->root_out);
                c->root_out = 0;
              }
              
              c->state_out = COMM_STATE_BLOCK_HEADER;
            }
          }
        }
        else finished = 1;
        break;

      case COMM_STATE_NONE:
      default:
        /* This could mean the comm is uninitialized; deal with it later */
        c->state_out = COMM_STATE_BLOCK_HEADER;
        break;
    }
  }
  
  return(ret);
}
