#!/usr/bin/perl -w

package ConfigFile;

use strict;
use Carp;
use Exporter;

our ($VERSION, @ISA, @EXPORT_OK);
@ISA = qw/Exporter/;
@EXPORT_OK = qw/read_config_file/;
$VERSION='1.2.1';


sub read_config_file($) {
    my ($line, $Conf, $file);
    $file = shift;
    open(CONF, "< " . $file) ||
        croak "Can't read configuration in $file: $!\n";
    while(<CONF>) {
        my ($conf_ele, $conf_data);
        chomp;
        next if m/^\s*#/;
        $line = $_;
        $line =~ s/(?<!\\)#.*$//;
        $line =~ s/\\#/#/g;
        next if $line =~ m/^\s*$/;
        $line =~ s{\$(\w+)}{
            exists($Conf->{$1}) ? $Conf->{$1} : "\$$1"
            }gsex;
        $line =~ m/\s*([^\s=]+)\s*=\s*(.*?)\s*$/;

        ($conf_ele, $conf_data) = ($1, $2);
        unless ($conf_ele =~ /^[\]\[A-Za-z0-9_-]+$/) {
            warn "Invalid characters in key $conf_ele - Ignoring";
            next;
        }
        $conf_ele = '$Conf->{' . join("}->{", split /[][]+/, $conf_ele) . "}";
        $conf_data =~ s!([\\\'])!\\$1!g;
        eval "$conf_ele = '$conf_data'";
    }
    close CONF;
    return $Conf;
}

1;

__END__

=pod

=head1 NAME

ConfigFile - Parse a simple configuration file

=head1 SYNOPSIS

use ConfigFile;

my $config_hash = ConfigFile::read_config_file($configuration_file);

=head1 NOTES

In versions up to 1.0, the function read_config_file was exported to the
calling program's namespace - Starting in version 1.1, nothing is exported
by default. You can either fully qualify read_config_file or explicitly
import it into your namespace:

=head2 Fully qualifying read_config_file

  use ConfigFile;

  my $config_hash = ConfigFile::read_config_file($configuration_file);

=head2 Explicitly importing read_config_file

  use ConfigFile qw(read_config_file);

  my $config_hash = read_config_file($configuration_file);

=head1 DESCRIPTION

C<read_config_file> parses a simple configuration file and stores its
values in an anonymous hash reference. The syntax of the configuration
file is quite simple:

    # This is a comment
    VALUE_ONE = foo
    VALUE_TWO = $VALUE_ONE/bar
    VALUE_THREE = The value contains a \# (hash). # This is a comment.

Options can be clustered when creating groups:

    CLUSTER_ONE[data] = data cluster one
    CLUSTER_ONE[value] = value cluster one
    CLUSTER_TWO[data] = data cluster two
    CLUSTER_TWO[value] = value cluster two

Then values can be fetched using this syntax:

    $hash_config->{CLUSTER_ONE}{data};

There can be as many sub-options in a cluster as needed.

    BIG_CLUSTER[part1][part2][part3] = data

is fetched by:
    $hash_config->{BIG_CLUSTER}{part1}{part2}{part3};

There are a couple of restrictions as for the names of the keys. First of all,
all the characters should be alphabetic, numeric, underscores or hyphens, with
square brackets allowed for the clustering. That is, the keys should conform
to /^[A-Za-z0-9_-]+$/

This means also that no space is allowed in the key part of the line.

    CLUSTER_ONE[data] = data cluster one      # Right
    CLUSTER_ONE[ data ] = data cluster one    # Wrong


=head1 Function C<read_config_file>

=head2 Syntax

    ConfigFile::read_config_file($file);

=head2 Arguments

C<$file> is the configuration file.

=head2 Return value

This function returns a hash reference. Each key of the hash is a
value defined in the configuration file.

=head2 Description

C<read_config_file> parses a configuration file a sets up some values 
in a hash reference.

=head1 AUTHOR

Development was started by Sebastien J. Gross <seb@sjgross.org>

All rights reserved.  This program is free software; you can redistribute
it and/or modify it under the terms of the GPL.

=head1 VERSION

Version 1.2.1
Copyright (c) 2002 Sebastien J. Gross. All rights reserved.
Copyright (c) 2003 Gunnar Wolf. All rights reserved.
This program is free software; you can redistribute it and/or modify
it under the terms of the GPL.

=cut
