/*
  CLAW - a C++ Library Absolutely Wonderful

  CLAW is a free library without any particular aim but being useful to 
  anyone.

  Copyright (C) 2005-2008 Julien Jorge

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: julien_jorge@yahoo.fr
*/
/**
 * \file types.hpp
 * \brief Some classes for the raw manipulation of the base types.
 * \author Julien Jorge
 */
#include <claw/meta.hpp>

#ifndef __CLAW_TYPES_HPP__
#define __CLAW_TYPES_HPP__

namespace claw
{
#ifdef CLAW_HAS_LONG_LONG

  typedef
  meta::type_list<signed long long int, meta::no_type>
  non_standard_signed_types;

  typedef
  meta::type_list<unsigned long long int, meta::no_type>
  non_standard_unsigned_types;

#else // !def CLAW_HAS_LONG_LONG

  typedef meta::no_type non_standard_signed_types;
  typedef meta::no_type non_standard_unsigned_types;

#endif // CLAW_HAS_LONG_LONG

  /** \brief This is the list of c++ signed integer types. */
  typedef meta::type_list
  < signed char,
    meta::type_list
    < signed short,
      meta::type_list<signed int, non_standard_signed_types>
  > > signed_integers;

  /** \brief This is the list of c++ unsigned integer types. */
  typedef meta::type_list
  < unsigned char,
    meta::type_list
    < unsigned short,
      meta::type_list<unsigned int, non_standard_unsigned_types>
  > > unsigned_integers;

  /**
   * \brief This meta class finds, in a list of types, the first type stored
   *        exactly with a given number of bits.
   *
   * \b Template \b parameters
   *  - \a Size The number of bits in the type to find,
   *  - \a TypeList A list of types (see meta::type_list).
   */
  template<unsigned int Size, typename TypeList>
  struct find_type_by_size
  {
  private:
    typedef typename TypeList::head_type head_type;
    typedef typename TypeList::queue_type queue_type;

  public:
    typedef
    typename meta::if_then_else
    < sizeof(head_type) * 8 == Size, head_type,
      typename find_type_by_size<Size, queue_type>::type >::result type;
  }; // find_type_by_size

  /** \brief End of the recursion of the find_type_by_size class. */
  template<unsigned int Size>
  struct find_type_by_size<Size, meta::no_type>
  {
    /** \brief This is an incomplete type that should make the compiler to
        fail. */
    struct type;
  }; // find_type_by_size

  /**
   * \brief Define the type of a signed integer stored with a given number of
   *        bits.
   * \b Template \b parameters
   *  - \a Size The number of bytes in the types.
   */
  template<unsigned int Size>
  struct integer_of_size
  {
    typedef typename find_type_by_size<Size, signed_integers>::type type;
  }; // struct integer_of_size

  /**
   * \brief Define the type of an unsigned integer stored with a given number of
   *        bits.
   * \b Template \b parameters
   *  - \a Size The number of bytes in the types.
   */
  template<unsigned int Size>
  struct unsigned_integer_of_size
  {
    typedef typename find_type_by_size<Size, unsigned_integers>::type type;
  }; // struct unsigned_integer_of_size

} // namespace claw

#endif // __CLAW_TYPES_HPP__
