/*
  CLAW - a C++ Library Absolutely Wonderful

  CLAW is a free library without any particular aim but being useful to 
  anyone.

  Copyright (C) 2005-2008 Julien Jorge

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: julien_jorge@yahoo.fr
*/
/**
 * \file multi_type_map.hpp
 * \brief This class can associate values of different types to a key.
 * \author Julien Jorge
 */
#ifndef __CLAW_MULTI_TYPE_MAP_HPP__
#define __CLAW_MULTI_TYPE_MAP_HPP__

#include <claw/meta.hpp>
#include <map>

namespace claw
{
  template<typename ReturnType, typename Map>
  class multi_type_map_wrapper;

  /**
   * \brief This class can associate values of different types to a key.
   *
   * \b Template \b parameters
   * - \a Key The type to use for the keys.
   * - \a TypeList The list of the types that the map should support.
   *
   * \b Type \b requirements
   * - \a Key must be \a LessThanComparable.
   * - The last type in \a TypeList must be claw::meta::no_type.
   * - type_list_is_a_set<TypeList>::result == true
   *
   * \b Example
   *
   * <tt>multi_type_map<int, type_list<std::string, type_list<int, no_type> > >
   * my_map;</tt>
   *
   * \author Julien Jorge
   */
  template<typename Key, typename TypeList>
  class multi_type_map
    : public multi_type_map<Key, typename TypeList::queue_type>
  {
  public:
    typedef Key key_type;
    typedef typename TypeList::head_type value_type;
    typedef multi_type_map<Key, TypeList> self_type;
    typedef multi_type_map<Key, typename TypeList::queue_type> super;
    typedef std::map<key_type, value_type> container_type;

    friend
    struct multi_type_map_wrapper< value_type,
				   multi_type_map<key_type,
						  TypeList> >::last_call;

  public:
    template<typename ValueType>
    const ValueType& get( const key_type& k ) const;

    template<typename ValueType>
    void set( const key_type& k, const ValueType& v );

    template<typename ValueType>
    bool exists( const key_type& k ) const;

  private:
    /** \brief Data stored for the first type of the list. */
    container_type m_data;

  }; // class multi_type_map

  /**
   * \brief Partial specialization, to stop the inheritance recursivity.
   */
  template<typename Key>
  class multi_type_map<Key, meta::no_type>
  {
    // empty class
  }; // class multi_type_map

} // namespace claw

#include <claw/impl/multi_type_map.tpp>

#endif // __CLAW_MULTI_TYPE_MAP_HPP__
