/*
 * Maildir folder access
 *
 * Copyright (C) 2004  Enrico Zini <enrico@debian.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#include "MaildirMailFolder.h"
#include <wibble/exception.h>
#include <buffy/System.h>

#include <sys/types.h>  // stat, opendir, readdir, utimes
#include <sys/stat.h>   // stat
#include <dirent.h>		// opendir, readdir
#include <sys/time.h>	// utimes

#include <errno.h>

using namespace std;
using namespace wibble;

namespace buffy {

static string concat(const std::string& str1, char sep, const std::string& str2)
{
	// Avoid adding a separator if str1 or str are empty, if str1 ends with the
	// separator or if str2 starts with the separator
	if (!str1.empty() && str1[str1.size() - 1] != sep && ! str2.empty() && str2[0] != sep)
		return str1 + sep + str2;
	else
		return str1 + str2;
}

MaildirMailFolder::MaildirMailFolder(const std::string& path) throw ()
	: _path(path), _stat_total(-1), _stat_unread(-1), _stat_new(-1), _stat_flagged(-1),
	  _deleted(false), _new_mtime(0), _cur_mtime(0)
{
	_name = _path;

	/// Normalize the folder name

	// Remove trailing '/'
	while (_name[_name.size() - 1] == '/')
		_name.resize(_name.size() - 1);

	// Remove leading path
	size_t lastslash = _name.find_last_of('/');
	if (lastslash != string::npos)
		_name = _name.substr(lastslash + 1);

	// Remove leading dot
	if (_name[0] == '.')
		_name = _name.substr(1);
}

MaildirMailFolder::MaildirMailFolder(const std::string& name, const std::string& path) throw ()
	: _name(name), _path(path), _stat_total(-1), _stat_unread(-1), _stat_new(-1), _stat_flagged(-1),
	  _deleted(false), _new_mtime(0), _cur_mtime(0) {}


bool MaildirMailFolder::changed()
{
	// Compute 'new' and 'cur' directory names
	string path_new = _path + "/new";
	string path_cur = _path + "/cur";

	struct stat st_new;
	if (!statIfFound(path_new, &st_new))
		if (! _deleted)
		{
			_deleted = true;
			return true;
		}

	struct stat st_cur;
	if (!statIfFound(path_cur, &st_cur))
		if (! _deleted)
		{
			_deleted = true;
			return true;
		}

	if (_deleted)
	{
		_deleted = false;
		return true;
	}

	return st_new.st_mtime > _new_mtime || st_cur.st_mtime > _cur_mtime;
}

void MaildirMailFolder::updateStatistics()
{
	int res_total = 0;
	int res_unread = 0;
	int res_new = 0;
	int res_flagged = 0;

	// Compute 'new' and 'cur' directory names
	string path_new = _path + "/new";
	string path_cur = _path + "/cur";

	// Perform consistency checks on the 'new' directory
	struct stat st_new;
	if (!statIfFound(path_new, &st_new))
	{
		_stat_total = 0;
		_stat_unread = 0;
		_stat_new = 0;
		_stat_flagged = 0;
		_deleted = true;
		return;
	}
	if (S_ISDIR(st_new.st_mode) == 0)
		throw wibble::exception::Consistency(path_new + " is not a directory");

	// Perform consistency checks on the 'cur' directory
	struct stat st_cur;
	if (!statIfFound(path_cur, &st_cur))
	{
		_stat_total = 0;
		_stat_unread = 0;
		_stat_new = 0;
		_stat_flagged = 0;
		_deleted = true;
		return;
	}
	if (S_ISDIR(st_cur.st_mode) == 0)
		throw wibble::exception::Consistency(path_cur + " is not a directory");

	if (_deleted)
		_deleted = false;

	_new_mtime = st_new.st_mtime;
	_cur_mtime = st_cur.st_mtime;

	/// Count messages in the 'new' directory

	// Count the files in the 'new' directory
	{
		Directory dir(path_new);
		while (struct dirent* d = dir.read())
		{
			if (d->d_name[0] == '.') 
				continue;
			res_total++;
			res_new++;
		}
	}

	// Restore the access time of the mailbox for other checking programs
	struct timeval t[2];
	t[0].tv_sec = st_new.st_atime;
	t[0].tv_usec = 0;
	t[1].tv_sec = st_new.st_mtime;
	t[1].tv_usec = 0;
	utimes(path_new.c_str(), t);


	/// Count messages in the 'cur' directory

	// Count the files in the 'cur' directory
	{
		Directory dir(path_cur);
		while (struct dirent *d = dir.read())
		{
			if (d->d_name[0] == '.') 
				continue;

			res_total++;

			// Look for an `info' block in the name
			char* info = strrchr(d->d_name, ':');
			if (info == 0)
				continue;

			// Ensure that the info block is in the right format
			if (strncmp(info, ":2,", 3) != 0)
				continue;

			// Look for the 'S' flag (it should not be there)
			info += 3;
			if (strchr(info, 'S') == 0)
				res_unread++;
			if (strchr(info, 'F') != 0)
				res_flagged++;
		}
	}

	// Restore the access time of the mailbox for other checking programs
	t[0].tv_sec = st_cur.st_atime;
	t[0].tv_usec = 0;
	t[1].tv_sec = st_cur.st_mtime;
	t[1].tv_usec = 0;
	utimes(path_cur.c_str(), t);


	// Return the values
	_stat_total = res_total;
	_stat_unread = res_unread + res_new;
	_stat_new = res_new;
	_stat_flagged = res_flagged;
}


static bool isMaildir(const std::string& pathname)
{
	try {
		// It must be a directory
		struct stat st;
		stat(pathname, &st);
		if (S_ISDIR(st.st_mode) == 0)
			return false;

		// It must contain cur, new and tmp subdirectories
		char* subdirs[3] = { "cur", "new", "tmp" };
		for (int i = 0; i < 3; i++)
		{
			string subdir = concat(pathname, '/', subdirs[i]);

			struct stat st;
			if (stat(subdir.c_str(), &st) != 0)
				return false;
			if (S_ISDIR(st.st_mode) == 0)
				return false;
		}

		// It appears to be a maildir directory
		return true;
	} catch (wibble::exception::File& e) {
		// If we can't even stat() it, then it's not a maildir
		return false;
	}
}

MailFolder MaildirMailFolder::accessFolder(const std::string& path)
{
	try {
		if (isMaildir(path))
			return MailFolder(new MaildirMailFolder(path));
	} catch (wibble::exception::System& e) {
		// FIXME: cerr << e.type() << ": " << e.fullInfo() << endl;
	}
	return MailFolder();
}

static void enumerateSubfolders(
		const std::string& parent,
		const std::string& name,
		MailFolderConsumer& cons,
		InodeSet seen = InodeSet())
{
	try {
		struct stat st;
		try {
			stat(parent, &st);
		} catch (wibble::exception::File& e) {
			// If we can't even stat() it, then we don't try to enumerate its subfolders
			// FIXME: cerr << e.type() << ": " << e.fullInfo() << endl;
			return;
		}
		// It must be a directory
		if (S_ISDIR(st.st_mode) == 0)
			return;

		// Check that we aren't looping
		if (seen.has(st.st_ino))
			return;

		if (isMaildir(parent))
		{
			MailFolder f(new MaildirMailFolder(name, parent));
			cons.consume(f);
		}

		// Recursively enumerate the Maildirs in the directory
		Directory dir(parent);
		while (struct dirent *d = dir.read())
		{
			if (strcmp(d->d_name, ".") == 0)
				continue;
			if (strcmp(d->d_name, "..") == 0)
				continue;
			if (strcmp(d->d_name, "tmp") == 0)
				continue;
			if (strcmp(d->d_name, "cur") == 0)
				continue;
			if (strcmp(d->d_name, "new") == 0)
				continue;

			enumerateSubfolders(
					concat(parent, '/', d->d_name),
					concat(name, '.', d->d_name),
					cons, seen + st.st_ino);
		}
	} catch (wibble::exception::Generic& e) {
		// FIXME: cerr << e.type() << ": " << e.fullInfo() << endl;
	}
}

void MaildirMailFolder::enumerateFolders(const std::string& parent, MailFolderConsumer& cons)
{
	// Remove trailing slash from the parent directory
	// The root name is empty if parent is not a maildir
	//   else, it is the last component of parent's path
	string root;
	string rootName;

	size_t pos = parent.rfind('/');
	if (pos == string::npos)
		root = rootName = parent;
	else if (pos == parent.size() - 1)
	{
		pos = parent.rfind('/', pos - 1);
		root = parent.substr(0, parent.size() - 1);
		rootName = parent.substr(pos+1, parent.size() - pos - 2);
	}
	else
	{
		root = parent;
		rootName = parent.substr(pos + 1);
	}

	if (!isMaildir(parent))
		rootName = string();

	enumerateSubfolders(root, rootName, cons);
}

}

//#ifdef COMPILE_TESTSUITE

#include <tests/test-utils.h>
#include <buffy/MailboxMailFolder.h>

namespace tut {
using namespace buffy;
using namespace wibble::tests;

struct buffy_maildirmailfolder_shar {
	buffy_maildirmailfolder_shar() {
	}
	~buffy_maildirmailfolder_shar() {
	}
};
TESTGRP(buffy_maildirmailfolder);

class MailFolderCounter : public MailFolderConsumer
{
	size_t m_count;
public:
	MailFolderCounter() : m_count(0) {}
	void consume(MailFolder& f) { ++m_count; }
	size_t count() const { return m_count; }
};

// Check an empty maildir
template<> template<>
void to::test<1>()
{
	MailFolder test(MaildirMailFolder::accessFolder("maildir/empty"));
	ensure_equals(test.name(), "empty");
	ensure_equals(test.path(), "maildir/empty");
	ensure_equals(test.type(), "Maildir");

	ensure_equals(test.getMsgTotal(), -1);
	ensure_equals(test.getMsgUnread(), -1);
	ensure_equals(test.getMsgNew(), -1);
	ensure_equals(test.getMsgFlagged(), -1);
	ensure_equals(test.changed(), true);

	test.updateStatistics();
	ensure_equals(test.getMsgTotal(), 0);
	ensure_equals(test.getMsgUnread(), 0);
	ensure_equals(test.getMsgNew(), 0);
	ensure_equals(test.getMsgFlagged(), 0);
	ensure_equals(test.changed(), false);

	MailFolderCounter counter;
	MaildirMailFolder::enumerateFolders("maildir/empty", counter);
	ensure_equals(counter.count(), 1u);
}

// Check a non empty maildir
template<> template<>
void to::test<2>()
{
	MailFolder test(MaildirMailFolder::accessFolder("maildir/test"));
	ensure_equals(test.name(), "test");
	ensure_equals(test.path(), "maildir/test");
	ensure_equals(test.type(), "Maildir");

	ensure_equals(test.getMsgTotal(), -1);
	ensure_equals(test.getMsgUnread(), -1);
	ensure_equals(test.getMsgNew(), -1);
	ensure_equals(test.getMsgFlagged(), -1);
	ensure_equals(test.changed(), true);

	test.updateStatistics();
	ensure_equals(test.getMsgTotal(), 3);
	ensure_equals(test.getMsgUnread(), 0);
	ensure_equals(test.getMsgNew(), 0);
	ensure_equals(test.getMsgFlagged(), 1);
	ensure_equals(test.changed(), false);

	MailFolderCounter counter;
	MaildirMailFolder::enumerateFolders("maildir/test", counter);
	ensure_equals(counter.count(), 1u);
}

// Check a maildir whose root is a broken symlink
template<> template<>
void to::test<3>()
{
	MailFolder test(MaildirMailFolder::accessFolder("maildir/broken"));
	ensure_equals((bool)test, false);

	MailFolderCounter counter;
	MaildirMailFolder::enumerateFolders("maildir/broken", counter);
	ensure_equals(counter.count(), 0u);
}

// Check a maildir whose cur, new and tmp directories are broken symlinks
template<> template<>
void to::test<4>()
{
	MailFolder test(MaildirMailFolder::accessFolder("maildir/broken1"));
	ensure_equals((bool)test, false);

	MailFolderCounter counter;
	MaildirMailFolder::enumerateFolders("maildir/broken1", counter);
	ensure_equals(counter.count(), 0u);
}

// Check a maildir whose root is a symlink pointing to a nonexisting directory
template<> template<>
void to::test<5>()
{
	MailFolder test(MaildirMailFolder::accessFolder("maildir/loop"));
	ensure_equals((bool)test, false);

	MailFolderCounter counter;
	MaildirMailFolder::enumerateFolders("maildir/loop", counter);
	ensure_equals(counter.count(), 0u);
}

// Check a maildir which has a submaildir that is a symlink pointing up
template<> template<>
void to::test<6>()
{
	MailFolder test(MaildirMailFolder::accessFolder("maildir/loop1"));
	ensure_equals((bool)test, true);

	MailFolderCounter counter;
	MaildirMailFolder::enumerateFolders("maildir/loop1", counter);
	ensure_equals(counter.count(), 1u);
}

// Check a maildir which has a submaildir that is a symlink pointing up
template<> template<>
void to::test<7>()
{
	MailFolderCounter counter;
	MaildirMailFolder::enumerateFolders("maildir", counter);
	ensure_equals(counter.count(), 4u);
}

}

//#endif

// vim:set ts=4 sw=4:
