#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <confdb.h>
#include <fviews.h>
#include <subsys.h>
#include <configf.h>
#include "../../paths.h"
#include "conectiva.h"

static HELP_FILE help_clock ("conectiva","clock");
static HELP_FILE help_keyboard ("conectiva","keyboard");
static HELP_FILE help_network ("conectiva","network");
static HELP_FILE help_sendmail ("conectiva","sendmail");
static HELP_FILE help_sysctl ("conectiva","sysctl");

static CONFIG_FILE f_clock("/etc/sysconfig/clock",help_clock
	,CONFIGF_MANAGED|CONFIGF_OPTIONNAL
	,"root","root",0644);

static CONFIG_FILE f_keyboard("/etc/sysconfig/keyboard",help_keyboard
	,CONFIGF_MANAGED|CONFIGF_OPTIONNAL
	,"root","root",0644);

static CONFIG_FILE f_network("/etc/sysconfig/network",help_network
	,CONFIGF_MANAGED|CONFIGF_OPTIONNAL
	,"root","root",0644,subsys_netclient);

static CONFIG_FILE f_routed("/etc/sysconfig/routed",help_network
	,CONFIGF_MANAGED|CONFIGF_OPTIONNAL
	,"root","root",0644,subsys_netclient);

static CONFIG_FILE f_sendmail("/etc/sysconfig/sendmail",help_sendmail
	,CONFIGF_MANAGED|CONFIGF_OPTIONNAL
	,"root","root",0644,subsys_netclient);

static CONFIG_FILE f_sysctl (ETC_SYSCTL_CONF,help_sysctl
	,CONFIGF_MANAGED|CONFIGF_OPTIONNAL
	,"root","root",0644,subsys_netclient);

enum CONECTIVA_FILE { CL_clock,CL_keyboard,CL_network,CL_routed,
	CL_sendmail, CL_sysctl, CL_dummy };

enum CONECTIVA_TYPE {CONECTIVA_STR,CONECTIVA_YESNO,CONECTIVA_ONOFF,CONECTIVA_VAL0, CONECTIVA_BOOL01};

struct CLLK{
	const char *key1;
	const char *key2;
	const char *var;
	CONECTIVA_FILE file;
	CONECTIVA_TYPE type;	
};

static CLLK tblk[]={
	{"linuxconf",	"keymap",		"KEYTABLE",		CL_keyboard,	CONECTIVA_STR},
	{"datetime",	"universal",	"UTC",			CL_clock,		CONECTIVA_YESNO},
	{"datetime",	"alphaarc",		"ARC",			CL_clock,		CONECTIVA_YESNO},
	{"nis",			"domain",		"NISDOMAIN",	CL_network,		CONECTIVA_STR},
	{"routing",		"gateway",		"GATEWAY",		CL_network,		CONECTIVA_STR},
	{"routing",		"gatewaydev",	"GATEWAYDEV",	CL_network,		CONECTIVA_STR},
	// Linuxconf maintains both the old and new IP forwarding
	// The first one is used to read, both are used to update
	// The CONFDB_CONECTIVA constructor disables the first one if
	// /etc/sysctl.conf does not exist (i.e. on releases
	// prior to X.X (not yet!))
	{"routing",		"ipv4",			"net.ipv4.ip_forward",	CL_sysctl,		CONECTIVA_BOOL01},
	//{"routing",		"ipv4",			"FORWARD_IPV4",	CL_network,		CONECTIVA_YESNO},
	{"routed",		"gateway",		"EXPORT_GATEWAY",CL_routed,		CONECTIVA_YESNO},
	{"routed",		"silent",		"SILENT",		CL_routed,		CONECTIVA_YESNO},

	{"ipx",			"active",		"IPX",			CL_network,		CONECTIVA_YESNO},
	{"ipx",			"internal_netnum",
									"IPXINTERNALNETNUM",		CL_network,		CONECTIVA_STR},
	{"ipx",			"internal_nodenum",
									"IPXINTERNALNODENUM",		CL_network,		CONECTIVA_STR},
	{"ipx",			"primary_auto",	"IPXAUTOPRIMARY",	CL_network,		CONECTIVA_ONOFF},
	{"ipx",			"frame_auto",	"IPXAUTOFRAME",		CL_network,		CONECTIVA_ONOFF},

	{"mailconf",	"queuedelay",	"QUEUE",			CL_sendmail,	CONECTIVA_VAL0},
	/* Special dummy entry. We are using pseudo entry in conf.linuxconf */
	/* to handle distribution "preferences" */
	{"--distrib--",	"sysvrestart",	NULL,			CL_dummy,		CONECTIVA_STR},
	{"--distrib--",	"privgroup",	NULL,			CL_dummy,		CONECTIVA_STR},
	{"--distrib--",	"NETWORKING",	"NETWORKING",	CL_network,		CONECTIVA_YESNO},
	{"--distrib--",	"HOSTNAME",		"HOSTNAME",		CL_network,		CONECTIVA_STR},
	{"--distrib--",	"sysvenh",		NULL,			CL_dummy,		CONECTIVA_STR},
	{"--distrib--",	"rpmvendor",	NULL,			CL_dummy,		CONECTIVA_STR},
	{"--distrib--",	"rpmdistribution",	NULL,		CL_dummy,		CONECTIVA_STR},
	{"--distrib--",	"release",		NULL,		CL_dummy,		CONECTIVA_STR},
	{"--distrib--",	"runlevel_text",	NULL,		CL_dummy,		CONECTIVA_STR},
	{"--distrib--",	"runlevel_graph",	NULL,		CL_dummy,		CONECTIVA_STR},
	{"--distrib--",	"runlevel_single",	NULL,		CL_dummy,		CONECTIVA_STR},
};
#define NBLK (sizeof(tblk)/sizeof(tblk[0]))


class CONFDB_CONECTIVA: public CONFDB{
	CONFDB *orig;	// Original CONFDB object managing conf.linuxconf
	struct {
		VIEWITEMS *clock;
		VIEWITEMS *keyboard;
		VIEWITEMS *network;
		VIEWITEMS *routed;
		VIEWITEMS *sendmail;
		VIEWITEMS *sysctl;
		bool clock_modified,keyboard_modified;
		bool network_modified,routed_modified;
		bool sendmail_modified;
		bool sysctl_modified;
	} sys;
	SSTRINGS tbstr;		// Old temporary strings returned
						// to caller of getval
	/*~PROTOBEG~ CONFDB_CONECTIVA */
public:
	CONFDB_CONECTIVA (CONFDB *_orig);
	void add (const char *prefix,
		 const char *key,
		 const char *val);
	int archive (SSTREAM&ss, const char *_sys);
	int extract (SSTREAM&ss, const char *_sys);
	int getall (const char *prefix,
		 const char *key,
		 SSTRINGS&lst,
		 bool copy);
	const char *getval (const char *prefix,
		 const char *key,
		 const char *defval);
private:
	const char *getvalfromrh (CLLK *rh);
public:
	void removeall (const char *prefix,
		 const char *key);
private:
	void reset_modified (void);
public:
	int save (PRIVILEGE *priv);
	void setcursys (const char *_subsys);
private:
	void updaterh (CLLK *rh, const char *val);
public:
	~CONFDB_CONECTIVA (void);
	/*~PROTOEND~ CONFDB_CONECTIVA */
};

PRIVATE void CONFDB_CONECTIVA::reset_modified()
{
	sys.keyboard_modified = false;
	sys.clock_modified = false;
	sys.network_modified = false;
	sys.routed_modified = false;
	sys.sendmail_modified = false;
	sys.sysctl_modified = false;
}

static CLLK *filter_lookup (
	const char *key1,
	const char *key2,
	unsigned &start)		// Start here in the search
{
	CLLK *ret = NULL;
	CLLK *pt = tblk+start;
	for (unsigned i=start; i<NBLK; i++, pt++){
		if (strcmp(pt->key1,key1)==0
			&& strcmp(pt->key2,key2)==0){
			ret = pt;
			start = i+1;
			break;
		}
	}
	return ret;
}

PUBLIC CONFDB_CONECTIVA::CONFDB_CONECTIVA(CONFDB *_orig)
{
	orig = _orig;
	sys.clock = NULL;
	sys.keyboard = NULL;
	sys.network = NULL;
	sys.routed = NULL;
	sys.sendmail = NULL;
	sys.sysctl = NULL;
	reset_modified();
	if (!f_sysctl.exist()){
		unsigned start = 0;
		CLLK *rh = filter_lookup ("routing","ipv4",start);
		if (rh != NULL){
			rh->key1 = "?????";	// Disable the entry
		}
	}
}

PUBLIC CONFDB_CONECTIVA::~CONFDB_CONECTIVA()
{
	delete orig;
	delete sys.clock;
	delete sys.keyboard;
	delete sys.network;
	delete sys.routed;
	delete sys.sendmail;
	delete sys.sysctl;
}

PUBLIC void CONFDB_CONECTIVA::setcursys (const char *_subsys)
{
	orig->setcursys (_subsys);
}

static void saveconf (
	VIEWITEMS *&items,
	CONFIG_FILE &cf,
	PRIVILEGE *priv,
	bool modified)
{
	if (items != NULL){
		if (modified) items->write (cf,priv);
		delete items;
		items = NULL;
	}
}


PUBLIC int CONFDB_CONECTIVA::save(PRIVILEGE *priv)
{
	int ret = orig->save(priv);
	if (ret != -1){
		saveconf (sys.clock,f_clock,priv,sys.clock_modified);
		saveconf (sys.keyboard,f_keyboard,priv,sys.keyboard_modified);
		saveconf (sys.network,f_network,priv,sys.network_modified);
		saveconf (sys.routed,f_routed,priv,sys.routed_modified);
		saveconf (sys.sendmail,f_sendmail,priv,sys.sendmail_modified);
		saveconf (sys.sysctl,f_sysctl,priv,sys.sysctl_modified);
		reset_modified();
	}
	tbstr.remove_all();
	return ret;
}

/*
	Return the release of the conectiva distribution
	release 5.1 becomes 501
*/

static CONFIG_FILE f_release("/etc/verso-conectiva",help_nil
	,CONFIGF_PROBED|CONFIGF_NOARCH|CONFIGF_OPTIONAL);

/*
	Return the conectiva release in format XYY (5.1 -> 501)
*/
int conectiva_release()
{
	static int ret = -1;	// Read the file only once
	if (ret == -1){
		ret = 501;
		FILE_CFG *fin = f_release.fopen ("r");
		if (fin != NULL){
			char line[200];
			while (fgets (line,sizeof(line)-1,fin) != NULL){
				char *pt = strstr (line,"Linux");
				if (pt != NULL){
					const char *ver = str_skip (pt+5);
					int num1 = atoi(ver);
					int num2 = 0;
					const char *pt = strchr(ver,'.');
					if (pt != NULL) num2 = atoi(pt+1);
					ret = num1*100 + num2;
					break;
				}
			}
			fclose (fin);
		}
	}
	return ret;
}



PRIVATE const char *CONFDB_CONECTIVA::getvalfromrh(CLLK *rh)
{
	const char *ret = NULL;
	VIEWITEMS **rhf = NULL;
	CONFIG_FILE *cf = NULL;
	if (rh->file == CL_dummy){
		int release = conectiva_release();
		if (strcmp(rh->key2,"sysvrestart")==0){
			ret = release >= 401 ? "1" : "0";
		}else if (strcmp(rh->key2,"runlevel_text")==0){
			ret = "3";
		}else if (strcmp(rh->key2,"runlevel_graph")==0){
			ret = "5";
		}else if (strcmp(rh->key2,"runlevel_single")==0){
			ret = "1";
		}else if (strcmp(rh->key2,"privgroup")==0){
			ret = "1";
		}else if (strcmp(rh->key2,"sysvenh")==0){
			ret = release > 401 ? "1" : "0";
		}else if (strcmp(rh->key2,"rpmvendor")==0){
			ret = vendor_getid();
		}else if (strcmp(rh->key2,"rpmdistribution")==0){
			ret = vendor_getdistid();
		}else if (strcmp(rh->key2,"release")==0){
			static char release_str[10];
			sprintf (release_str,"%d.%d",release/100,release%100);
			ret = release_str;
		}
	}else if (rh->file == CL_clock){
		rhf = &sys.clock;
		cf = &f_clock;
	}else if (rh->file == CL_keyboard){
		rhf = &sys.keyboard;
		cf = &f_keyboard;
	}else if (rh->file == CL_network){
		rhf = &sys.network;
		cf = &f_network;
	}else if (rh->file == CL_routed){
		rhf = &sys.routed;
		cf = &f_routed;
	}else if (rh->file == CL_sendmail){
		rhf = &sys.sendmail;
		cf = &f_sendmail;
	}else if (rh->file == CL_sysctl){
		rhf = &sys.sysctl;
		cf = &f_sysctl;
	}
	if (rhf != NULL){
		if (*rhf == NULL){
			*rhf = new VIEWITEMS;
			(*rhf)->read (*cf);
		}
		char tmp[1000];
		const char *val = (**rhf).locateval (rh->var,tmp);
		if (val != NULL){
			if (rh->type == CONECTIVA_YESNO
				|| rh->type == CONECTIVA_ONOFF
				|| rh->type == CONECTIVA_BOOL01){
				ret = stricmp(val,"yes")==0
					|| stricmp(val,"true")==0
					|| stricmp(val,"on")==0
					|| stricmp(val,"1")==0
					? "1" : "0";
			}else if (rh->type == CONECTIVA_VAL0 && val[0] == '\0'){
				ret = "0";
			}else{
				SSTRING *n = new SSTRING (val);
				tbstr.add (n);
				ret = n->get();
			}
		}
	}
	return ret;
}		

PRIVATE void CONFDB_CONECTIVA::updaterh(CLLK *rh, const char *val)
{
	VIEWITEMS **rhf = NULL;
	CONFIG_FILE *cf = NULL;
	const char *quote = "\"";
	if (rh->type == CONECTIVA_YESNO){
		val = strcmp(val,"1")==0 ? "yes" : "no";
	}else if (rh->type == CONECTIVA_ONOFF){
		val = strcmp(val,"1")==0 ? "on" : "off";
	}else if (rh->type == CONECTIVA_VAL0){
		if (strcmp(val,"0")==0) val = "";
	}else if (rh->type == CONECTIVA_BOOL01){
		quote = "";
	}
	if (rh->file == CL_clock){
		rhf = &sys.clock;
		cf = &f_clock;
		sys.clock_modified = true;
	}else if (rh->file == CL_keyboard){
		rhf = &sys.keyboard;
		cf = &f_keyboard;
		sys.keyboard_modified = true;
	}else if (rh->file == CL_network){
		rhf = &sys.network;
		cf = &f_network;
		sys.network_modified = true;
	}else if (rh->file == CL_routed){
		rhf = &sys.routed;
		cf = &f_routed;
		sys.routed_modified = true;
	}else if (rh->file == CL_sendmail){
		rhf = &sys.sendmail;
		cf = &f_sendmail;
		sys.sendmail_modified = true;
	}else if (rh->file == CL_sysctl){
		rhf = &sys.sysctl;
		cf = &f_sysctl;
		sys.sysctl_modified = true;
	}
	if (*rhf == NULL){
		*rhf = new VIEWITEMS;
		(*rhf)->read (*cf);
	}
	char newline[strlen(rh->var)+1+1+strlen(val)+1+1];
	sprintf (newline,"%s=%s%s%s",rh->var,quote,val,quote);
	VIEWITEM *it = (*rhf)->locateassign (rh->var);
	if (it == NULL){
		it = new VIEWITEM (newline);
		(*rhf)->add (it);
	}else{
		it->line.setfrom (newline);
	}
}		
	

PUBLIC const char *CONFDB_CONECTIVA::getval(
	const char *prefix,
	const char *key,
	const char *defval)
{
	unsigned start = 0;
	CLLK *rh = filter_lookup (prefix,key,start);
	const char *ret = defval;
	if (rh != NULL){
		/* #Specification: conectiva files and conf.linuxconf / strategy
			A migration of some stuff stored in /etc/conf.linuxconf
			is done. The goal is to use from now on the conectiva specific
			config file (in /etc/sysconfig) directly. A second goal
			is to let linuxconf users migrate to this strategy painlessly.

			To achieve that, we will use a strategy giving priority
			to the content of /etc/conf.linuxconf. Normally
			for all new linuxconf installation, /etc/conf.linuxconf
			won't have anything conflicting with the sysconfig file, so
			giving priority to it means nothing. For old linuxconf
			installation, conf.linuxconf does contain the more accurate
			(more recent) data.

			So the strategy is for reading

			#
			-read in conf.linuxconf first.
			-if there is something there, use it
			-if there is nothing, read the appropriate sysconfig file
			#

			And for writing

			#
			-erase all relevant entries in conf.linuxconf
			-update the appropriate sysconfig file
			#

			This strategy should provide a completly transparent
			migration.
		*/
		ret = orig->getval (prefix,key,NULL);
		if (ret == NULL){
			ret = getvalfromrh (rh);
			if (ret == NULL) ret = defval;
		}
	}else{
		ret = orig->getval (prefix,key,defval);
	}
	return ret;
}

PUBLIC int CONFDB_CONECTIVA::getall (
	const char *prefix,
	const char *key,
	SSTRINGS &lst,
	bool copy)	// Take a copy of the values
{
	return orig->getall (prefix,key,lst,copy);
}

PUBLIC void CONFDB_CONECTIVA::removeall (const char *prefix, const char *key)
{
	orig->removeall (prefix,key);
}
PUBLIC void CONFDB_CONECTIVA::add (
	const char *prefix,
	const char *key,
	const char *val)
{
	/* #specification: Cleaning conf.linuxconf / strategy
		All updates in conf.linuxconf are done using the same
		sequence

		#
		removeall
		add
		#

		The "removeall"s in the conectiva conf.linuxconf filter are going
		through normally. The "add"s are filtered. Some special entries
		are rerouted to the conectiva sysconfig files.

		This strategy, combined with the one for the getval() function
		provide the following functionnality:

		#
		-conf.linuxconf has priority to provide information
		-conf.linuxconf lose information at the first upgrade of this
		 information as the sysconfig file are updated
		#
	*/
	bool found = false;
	unsigned pos = 0;
	while (1){
		CLLK *rh = filter_lookup (prefix,key,pos);
		if (rh != NULL){
			updaterh (rh,val);
			found = true;
		}else{
			break;
		}
	}
	if (!found)	orig->add (prefix,key,val);
}

PUBLIC int CONFDB_CONECTIVA::archive (SSTREAM &ss, const char *_sys)
{
	return orig->archive (ss,_sys);
}


PUBLIC int CONFDB_CONECTIVA::extract (SSTREAM &ss, const char *_sys)
{
	return orig->extract (ss,_sys);
}

CONFDB *filter_fctnew (CONFDB *orig)
{
	return new CONFDB_CONECTIVA(orig);
}

/*
	Obtain a value from /etc/sysconfig/network
*/
static void filter_getval(const char *var, char *val)
{
	val[0] = '\0';
	VIEWITEMS items;
	if (items.read (f_network)!=-1){
		char tmp[1000];
		const char *pt = items.locateval (var,tmp);
		if (pt != NULL) strcpy (val,pt);
	}
}

/*
	Set the Host name in /etc/sysconfig/network
*/
int filter_setval(const char *var, const char *val)
{
	int ret = -1;
	VIEWITEMS items;
	if (items.read (f_network)!=-1){
		items.update (var,val);
		ret = items.write (f_network,NULL);
	}
	return ret;
}


static const char K_HOSTNAME[]="HOSTNAME";
/*
	Obtain the Host name from /etc/sysconfig/network
*/
void filter_gethostname(char *name)
{
	filter_getval (K_HOSTNAME,name);
}

/*
	Set the Host name in /etc/sysconfig/network
*/
int filter_sethostname(const char *name)
{
	return filter_setval (K_HOSTNAME,name);
}

#if 0
static const char K_GATEWAY[]="GATEWAY";
static const char K_GATEWAYDEV[]="GATEWAYDEV";
/*
	Obtain the Default gateway from /etc/sysconfig/network
*/
void filter_getgateway (char *gtw)
{
	filter_getval (K_GATEWAY,gtw);
}

/*
	Set the default gateway in /etc/sysconfig/network
*/
int filter_setgateway(const char *gtw, const char *device)
{
	return filter_setval (K_GATEWAY,gtw)
		| filter_setval (K_GATEWAYDEV,device);
}
#endif

