#include <stdlib.h>
#include <string.h>
#include <limits.h>
#include <sys/stat.h>
#include "misc.h"
#include "../paths.h"

static char *lang = strdup("eng");
static HELP_FILE *first = NULL;

/*
	Record the language to use for help files
*/
void help_setlang (const char *_lang, const char *envvar)
{
	free (lang);
	if (envvar != NULL){
		const char *pt = getenv(envvar);
		if (pt != NULL) _lang = pt;
	}
	lang = strdup(_lang);
}

/*
	Record the spec of a config file that is maintain (or used) by linuxconf
*/
PUBLIC HELP_FILE::HELP_FILE(
	const char *_subdir,
	const char *_fname)
{
	/* #Specification: help file / path
		All help file of linuxconf are under the same directory
		USR_LIB_CONF. Each subject (netconf, userconf, etc...) has
		its own subdirectory under this directory. In these
		subdirectories, we have file ending with the extension help.
	*/
	subdir = _subdir;
	fname = _fname;
	path = NULL;
	next = first;
	first = this;
}

/*
	Resolve the path of a help file by picking the first available
	langage: The one selected by the user or english if the first
	is available.

	We know which one by looking for the .help file. But the
	caller is only interested in the base path
*/
PRIVATE void HELP_FILE::getpaths (
	char abspath[PATH_MAX],
	char relpath[PATH_MAX])
{
	const char *trylang = lang;
	for (int i=0; i<2; i++, trylang = "eng"){
		/* #Specification: help file / path / exception
			HELP_FILE object for modules may be defined with a subdir
			component being an absolute path. The standard
			linuxconf help base path is not used then. The subdir (which
			is indeed a full absolute path) is concatenated with
			this way to create the basepath of the help file.

			#
			subdir/help.lang/filename
			#
		*/
		if (subdir[0] == '/'){
			snprintf (abspath,PATH_MAX-1,"%s/help.%s/%s",subdir,trylang,fname);
			strcpy (relpath,abspath);
		}else{
			snprintf (relpath,PATH_MAX-1,"help.%s/%s/%s",trylang,subdir,fname);
			snprintf (abspath,PATH_MAX-1,"%s/%s",HELP_BASEPATH,relpath);
		}
		char tmp[PATH_MAX];
		// Check for the html version which is used, the .help (text version)
		// is certainly there also
		snprintf (tmp,PATH_MAX-1,"%s.html",abspath);
		if (file_exist(tmp)) break;
	}
}

/*
	Return the path of the configuration file
*/
PUBLIC const char *HELP_FILE::getpath()
{
	if (path == NULL && fname != NULL){
		char abspath[PATH_MAX],relpath[PATH_MAX];
		getpaths(abspath,relpath);
		path = strdup (abspath);
	}	
	return path;
}
/*
	Provide the relative path of a help file (relative to /usr/lib/linuxconf)
*/
PUBLIC void HELP_FILE::getrpath(char *rpath)
{
	rpath[0] = '\0';
	if (fname != NULL){
		char abspath[PATH_MAX];
		getpaths(abspath,rpath);
	}
}

/*
	Is this is the help_nil object
*/
PUBLIC bool HELP_FILE::is_nil() const
{
	return fname == NULL;
}

class HELP_FILE_STATUS: public ARRAY_OBJ{
public:
	HELP_FILE *help;
	HELP_FILE_STATUS(HELP_FILE *f){
		help = f;
	}
};
static int cmp_help_file (const ARRAY_OBJ *o1, const ARRAY_OBJ *o2)
{
	HELP_FILE_STATUS *f1 = (HELP_FILE_STATUS*)o1;
	HELP_FILE_STATUS *f2 = (HELP_FILE_STATUS*)o2;
	return strcmp(f1->help->getpath(),f2->help->getpath());
}

class HELP_FILE_STATUSS: public ARRAY{
public:
	HELP_FILE_STATUS* getitem(int no) const
	{
		return (HELP_FILE_STATUS*)ARRAY::getitem(no);
	}
};
/*
	Check if all the help file of the application are there.
	This is a runtime check even if it is only requiered for testing
	the software.
*/
void helpf_checkall()
{
	HELP_FILE *f = first;
	HELP_FILE_STATUSS tb;
	while (f != NULL){
		if (f->getpath() != NULL){
			tb.add (new HELP_FILE_STATUS(f));
		}
		f = f->next;
	}
	tb.sort (cmp_help_file);
	for (int i=0; i<tb.getnb(); i++){
		HELP_FILE *f = tb.getitem(i)->help;
		const char *path = f->getpath();
		char abspath[PATH_MAX];
		sprintf (abspath,"%s.help",path);
		struct stat st;
		printf ("%s%s\n",stat(abspath,&st)!=-1 ? "\t" : "***\t"
			,abspath);
	}
}

HELP_FILE help_nil (NULL,NULL);
